/*
 * getsize.c --- get the size of a partition.
 * 
 * Copyright (C) 1995 Theodore Ts'o.  This file may be
 * redistributed under the terms of the GNU Public License.
 */

#ifdef OS2
#define INCL_DOS
#define INCL_DOSDEVIOCTL
#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>

#include <string.h>

#endif

#include <stdio.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#include <fcntl.h>
#ifdef HAVE_LINUX_FS_H
#include <linux/fs.h>
#endif
#ifdef HAVE_LINUX_FD_H
#include <sys/ioctl.h>
#include <linux/fd.h>
#endif
#ifdef HAVE_SYS_DISKLABEL_H
#include <sys/ioctl.h>
#include <sys/disklabel.h>
#endif /* HAVE_SYS_DISKLABEL_H */

#include <linux/ext2_fs.h>
#include "ext2fs.h"

#ifndef OS2
static int valid_offset (int fd, ext2_loff_t offset)
{
	char ch;

	if (ext2_llseek (fd, offset, 0) < 0)
		return 0;
	if (read (fd, &ch, 1) < 1)
		return 0;
	return 1;
}

/*
 * Returns the number of blocks in a partition
 */
errcode_t ext2fs_get_device_size(const char *file, int blocksize,
				 blk_t *retblocks)
{
	int	fd;
	int	size;
	ext2_loff_t high, low;
#ifdef FDGETPRM
	struct floppy_struct this_floppy;
#endif
#ifdef HAVE_SYS_DISKLABEL_H
	struct disklabel lab;
	struct partition *pp;
	char ch;
#endif /* HAVE_SYS_DISKLABEL_H */

	fd = open(file, O_RDONLY);
	if (fd < 0)
		return errno;

#ifdef BLKGETSIZE
	if (ioctl(fd, BLKGETSIZE, &size) >= 0) {
		close(fd);
		*retblocks = size / (blocksize / 512);
		return 0;
	}
#endif
#ifdef FDGETPRM
	if (ioctl(fd, FDGETPRM, &this_floppy) >= 0) {
		close(fd);
		*retblocks = this_floppy.size / (blocksize / 512);
		return 0;
	}
#endif
#ifdef HAVE_SYS_DISKLABEL_H
	size = strlen(file) - 1;
	if (size >= 0) {
		ch = file[size];
		if (isdigit(ch))
			size = 0;
		else if (ch >= 'a' && ch <= 'h')
			size = ch - 'a';
		else
			size = -1;
	}
	if (size >= 0 && (ioctl(fd, DIOCGDINFO, (char *)&lab) >= 0)) {
		pp = &lab.d_partitions[size];
		if (pp->p_size) {
			close(fd);
			*retblocks = pp->p_size / (blocksize / 512);
			return 0;
		}
	}
#endif /* HAVE_SYS_DISKLABEL_H */

	/*
	 * OK, we couldn't figure it out by using a specialized ioctl,
	 * which is generally the besy way.  So do binary search to
	 * find the size of the partition.
	 */
	low = 0;
	for (high = 1024; valid_offset (fd, high); high *= 2)
		low = high;
	while (low < high - 1)
	{
		const ext2_loff_t mid = (low + high) / 2;

		if (valid_offset (fd, mid))
			low = mid;
		else
			high = mid;
	}
	valid_offset (fd, 0);
	close(fd);
	*retblocks = (low + 1) / blocksize;
	return 0;
}
#else
#pragma pack(1)
struct parm {
    unsigned char command;
    unsigned char drive;
};
#define FLAGS_DASD   1		/* 1 : OPEN_FLAGS_DASD open         0 : normal open */
#define FLAGS_LOCKED 2		/* 1 : IOCtl DSK_LOCKDRIVE called   0 : not called  */
#define FLAGS_FORMAT 4          /* 1 : IOCtl DSK_BEGINFORMAT called 0 : not called  */

struct data {
    BIOSPARAMETERBLOCK bpb;
    unsigned short     nr_cyl;
    unsigned char      devtype;
    unsigned short     devattr;
};
#pragma pack(4)
errcode_t ext2fs_get_device_size(const char *file, int blocksize,
				 blk_t *retblocks)
{
    APIRET rc;
    FILESTATUS3 falloc;
    int    namelen;
    int    disknum;


    namelen = strlen(file);

    if ((namelen == 2) && (file[1] == ':') && (((file[0] >= 'A') && (file[0] <= 'Z')) || ((file[0] >= 'a') && (file[0] <= 'z')))) {
        struct parm p;
        struct data d;
        unsigned long plen = sizeof(p);
        unsigned long dlen = sizeof(d);

        disknum = toupper((int)(file[0])) - 'A';

        p.command = 0;
        p.drive   = (unsigned char)disknum;
        if ((rc = DosDevIOCtl(
                              (HFILE)-1,
                              IOCTL_DISK,
                              DSK_GETDEVICEPARAMS,
                              &p, plen, &plen,
                              &d, dlen, &dlen
                             )) != NO_ERROR) {
            fprintf(stderr, "DosDevIOCtl failed with rc = %d in file %s at line %d\n", rc, __FILE__, __LINE__);
            os2_err(rc);
            return -1;
        }
        fprintf(stderr, "\tOS2 : cSectors          = %u\n", d.bpb.cSectors);
        fprintf(stderr, "\t      cLargeSectors     = %u\n", d.bpb.cLargeSectors);
        fprintf(stderr, "\t      usBytesPerSector  = %u\n", d.bpb.usBytesPerSector);
        fprintf(stderr, "\t      usSectorsPerTrack = %u\n", d.bpb.usSectorsPerTrack);
        fprintf(stderr, "\t      cHeads            = %u\n", d.bpb.cHeads);
        fprintf(stderr, "\t      cHiddenSectors    = %u\n", d.bpb.cHiddenSectors);
        fprintf(stderr, "\t      cCylinders        = %u\n", d.bpb.cCylinders);

        if ((d.bpb.cSectors == 0) && (d.bpb.cLargeSectors)){
             long long tmp;

             tmp = (long long) d.bpb.cLargeSectors * (long long)d.bpb.usBytesPerSector;
             if (tmp > 2147483647) {
                 fprintf(stderr, "Cannot handle devices larger than 2 Gb yet\n");
                 *retblocks = 0;
                 errno = EFBIG;
                 return errno;
             }

            *retblocks = (blk_t)(tmp / blocksize);
        } else if ((d.bpb.cSectors) && (d.bpb.cLargeSectors == 0)) {
             long long tmp;

             tmp = (long long) d.bpb.cSectors * (long long)d.bpb.usBytesPerSector;
             if (tmp > 2147483647) {
                 fprintf(stderr, "Cannot handle devices larger than 2 Gb yet\n");
                 *retblocks = 0;
                 errno = EFBIG;
                 return errno;
             }

            *retblocks = (blk_t)(tmp / blocksize);
        } else {
	    fprintf(stderr, "OS2 FATAL ERROR while retrieving device size\n");
            errno = -EACCES;
            return errno;
        }

        return 0;
    } else {
                         


        /*
         * This is NOT a drive letter
         */

        rc = DosQueryPathInfo(file, FIL_STANDARD, &falloc, sizeof(FILESTATUS3));
        if (rc != NO_ERROR) {
            fprintf(stderr, "DosQueryPathInfo failed with rc = %d in file %s at line %d\n", rc, __FILE__, __LINE__);
            os2_err(rc);
            return -1;
        }
        *retblocks = falloc.cbFile / blocksize;
        printf("\tOS2 : size is %d - ret = %d\n", falloc.cbFile, *retblocks);
        return 0;
    }

}
#endif
