//
// $Header: D:/ext2-os2/vfs/RCS/ll_rwblk.c,v 6.0 1996/01/24 00:00:21 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


//
//
// This file contains an "emulation" of Linux ll_rw_block I/O routine, 100%
// specific to OS/2. Its purpose is to emulate Linux block device I/O by
// rerouting the requests to the relevant OS/2 device driver, using either
// FSH_DOVOLIO (strategy 1) or directly through the strategy 2 DD entry point
// (if supported).
//

// #pragma code_seg("EXT2_FIXED_CODE", "FIXED_CODE")
#pragma data_seg("EXT2_FIXED_DATA", "FIXED_DATA")

#define os2_panic(msg) ext2_os2_panic(0, msg)

#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>                // From the "Developer Connection Device Driver Kit" version 2.0
#include <strat2.h>             // From the "Developer Connection Device Driver Kit" version 2.0

#include <fsh.h>


#include <os2/types.h>
#include <os2/os2proto.h>
#include <os2/errors.h>
#include <os2/volume.h>
#include <os2/log.h>

#include <linux/fs.h>
#include <linux/fs_proto.h>
#include <linux/sched.h>
#include <linux/locks.h>

#include <os2/magic.h>     // Magic numbers


//
// WARNING !!! Unlike Linux ll_rw_block, here ll_rw_block is SYNCHRONOUS because FSH_DOVOLIO is.
// If we wanted an asynchronous version we could for instance use the extended strategy interface
// if the block device supports it.
//
void ll_rw_block_standard(int rw, int nr, struct buffer_head **bh, struct super_block *sb) {
    int rc;
    int i;
    int nb_sec;
    int rwmode;

    switch (rw) {
        case READ  :
        case READA :
            rwmode = DVIO_OPREAD;
            break;
        case WRITE  :
        case WRITEA :
            rwmode = DVIO_OPWRITE;
            break;
        default :
            os2_panic("ll_rw_block() - Invalid flag");
    }
    for (i = 0 ; i < nr ; i++) {
        lock_buffer(bh[i]);
        bh[i]->b_uptodate = 0;
        nb_sec = (int)sb->sectors_per_block;
        if ((rc = FSH_DOVOLIO(
                               rwmode,
                               DVIO_ALLFAIL | DVIO_ALLABORT | DVIO_ALLRETRY,
                               bh[i]->b_dev,
                               bh[i]->b_data,
                               &nb_sec,
                               bh[i]->b_blocknr * sb->sectors_per_block
                              )) != NO_ERROR) {
            kernel_printf("FSH_DOVOLIO() - rc = 0x%04X - blk_no = %lu", rc, bh[i]->b_blocknr);
        } else {
            bh[i]->b_uptodate = 1;
            if (rwmode == DVIO_OPWRITE) {
                mark_buffer_clean(bh[i]);
            }
        }
        unlock_buffer(bh[i]);

    }
}

/****************************************************************************************/
/***                        Strategy 2 I/O                                            ***/
/****************************************************************************************/

#define PAGE_SIZE 61440L  // 65536 - 4096, to avoid segment upper boundary

struct RW_private {
    magic_t                magic;    // Magic signature always equal to RW_PRIVATE_MAGIC
    struct reqlist        *reqlist;  // reqlist ptr - to be able to free it when notified by OS2DASD of completion
    int                    wait_for_all;
    unsigned long          wait_list;
    int                    nb_bh;    // Number of buffer_head structure(s)
    struct buffer_head    *bh[1];    // Corresponding buffer_head structure(s)
};


//
// reqlist structure - for allocating/deallocating request lists
//
struct reqlist {
    magic_t                  magic;          //  Magic signature always equal to REQLIST_MAGIC
    struct reqlist          *pnext;
    struct reqlist          *pprevious;
    struct _Req_List_Header  rwlist;
};

//
// Variables needed for allocating/deallocating request lists
//
// **** WARNING *** These data will be accessed in interrupt context
//
int nreqlist     = 0;                     // Total number of request lists
int nfreereqlist = 0;                     // Number of free request lists
int nusedreqlist = 0;                     // Number of used request lists


static struct reqlist *freereqlist = 0;          // Free request list chain head
static struct reqlist *usedreqlist = 0;          // Used request list chain head
static unsigned long reqlist_wait  = 0;          // Block ID to wait on when nfreereqlist = 0

static void insert_req_free(struct reqlist *req) {
    _disable();
    req->pnext     = freereqlist;
    req->pprevious = 0;
    if (req->pnext)
        req->pnext->pprevious = req;
    freereqlist = req;
    _enable();

}

int init_reqlist(void) {
    int i, k;
    int rc;
    long rwlist_size;
    struct reqlist *req;
    int nr;

    if (nreqlist)
        return NO_ERROR;

    /*
     * Maximum size of a request list : 32 requests with one SG entry per request (worst case)
     */
    rwlist_size = sizeof(struct reqlist) + 32 * (sizeof(struct _PB_Read_Write) +
                                                 sizeof(struct _SG_Descriptor) +
                                                 sizeof(struct RW_private));

#if 1
    nr = (int) (PAGE_SIZE / rwlist_size);
#else
    nreqlist     = (int) (PAGE_SIZE / rwlist_size);
    nfreereqlist = nreqlist;
    nusedreqlist = 0;
#endif

    for (k = 0; k < 2; k++) {
        /*
         * NON swappable memory is needed here : request lists are accessed in interrupt context
         */
        if ((req = (struct reqlist *)G_malloc_fixed((long)nr * rwlist_size)) == 0) {
            os2_panic("init_reqlist - couldn't allocate request list array");
        } /* end if */

        for (i = 0 ; i < nr ; i++, req = (struct reqlist *)((char *)req + rwlist_size)) {
            memset(req, 0, (int)rwlist_size);
            req->magic       = REQLIST_MAGIC;
            insert_req_free(req);
        }
        nreqlist     += nr;
        nfreereqlist += nr;
    }

//    if ((rc = FSH_FORCENOSWAP(SELECTOROF(init_reqlist))) != NO_ERROR) {
//        kernel_printf("init_reqlist - FSH_FORCENOSWAP returned %d for code segment", rc);
//        return rc;
//    }

    /*
     * Locks the 'EXT2_FIXED_DATA' data segment in physical memory
     */
    if ((rc = FSH_FORCENOSWAP((__segment)(&nreqlist))) != NO_ERROR) {
        kernel_printf("init_reqlist - FSH_FORCENOSWAP returned %d for data segment", rc);
        return rc;
    }

    kernel_printf("init_reqlist : there are %d strategy 2 request lists of size %lu", nreqlist, rwlist_size);

    return NO_ERROR;
}

struct reqlist *alloc_reqlist(void) {
    long            retrying;
    struct reqlist *curfree,
                   *nextfree,
                   *previousfree,
                   *curused,
                   *nextused,
                   *previousused;
/*
    fs_log("alloc_reqlist()");
*/
    retrying = 0;
retry:
    if (nfreereqlist == 0) {
        kernel_printf("alloc_reqlist() : no more request lists ... retrying (%lu)", ++retrying);
        sleep_on(&reqlist_wait);
        goto retry;
    }


    _disable();
    curfree      = freereqlist;
    curused      = usedreqlist;
    nextfree     = (curfree == 0 ? 0 : freereqlist->pnext);
    nextused     = (curused == 0 ? 0 : usedreqlist->pnext);
    previousfree = (curfree == 0 ? 0 : freereqlist->pprevious);
    previousused = (curused == 0 ? 0 : usedreqlist->pprevious);

    usedreqlist            = curfree;
    usedreqlist->pnext     = curused;
    usedreqlist->pprevious = 0;
    freereqlist            = nextfree;
    if (nextfree != 0) {
        freereqlist->pprevious = 0;
    } /* endif */
    if (curused != 0) {
        curused->pprevious = curfree;           /* (5) */
    } /* endif */

    nusedreqlist++;
    nfreereqlist--;
//    usedreqlist->busy = 1;
    _enable();

    /*
     * Post conditions check.
     */
    if (!usedreqlist) {
        os2_panic("alloc_reqlist() is returning a NULL request list");
    }
    if (usedreqlist->magic != REQLIST_MAGIC) {
        os2_panic("alloc_reqlist() is returning a reqlist structure with an invalid magic number");
    }

    return usedreqlist;
}


int free_reqlist(struct reqlist *reqlist) {

    struct reqlist *curfree,
                   *nextfree,
                   *previousfree,
                   *curused,
                   *nextused,
                   *previousused;


    if (!reqlist) {
        os2_panic("free_reqlist() called with a NULL request list");
    }
    if (reqlist->magic != REQLIST_MAGIC) {
        os2_panic("free_reqlist() reqlist structure with invalid magic number");
    }

    _disable();
    curfree      = freereqlist;
    curused      = reqlist;
    nextfree     = (curfree == 0 ? 0 : freereqlist->pnext);
    nextused     = (curused == 0 ? 0 : reqlist->pnext);
    previousfree = (curfree == 0 ? 0 : freereqlist->pprevious);
    previousused = (curused == 0 ? 0 : reqlist->pprevious);

    freereqlist        = curused;
    freereqlist->pnext = curfree;
    freereqlist->pprevious = 0;
    if (previousused != 0) {
        previousused->pnext = nextused;
    } else {
        usedreqlist = nextused;                 /* (7) */
    }
    if (nextused != 0) {
        nextused->pprevious = previousused;
    }
    if (curfree != 0) {
        curfree->pprevious = curused;            /* (6) */
    }

    nusedreqlist--;
    nfreereqlist++;
    _enable();

    wake_up(&reqlist_wait);

    return NO_ERROR;
}




//
// Request list completed callback
//
// *** WARNING *** This routine is called in interrupt context
//
int request_list_completed(struct  _Req_List_Header *request_list) {
    __segment ReqSeg;
    struct _Req_List_Header __based(ReqSeg) *pRLH;        // Request list header
    struct _PB_Read_Write   __based(ReqSeg) *pRLE;        // Request list entry
    struct _SG_Descriptor   __based(ReqSeg) *pSG;        // Scatter-gather descriptor
    struct RW_private       __based(ReqSeg) *pBH;        // ext2fs specific part of RLE

    ReqSeg         = (__segment)request_list;

    (USHORT)pRLH   = OFFSETOF(request_list);
    (USHORT)pRLE   = (USHORT)pRLH + (USHORT)sizeof(struct _Req_List_Header);
    (USHORT)pSG    = (USHORT)pRLE + (USHORT)sizeof(struct _PB_Read_Write);
    (USHORT)pBH    = (USHORT)pSG  + (USHORT)sizeof(struct _SG_Descriptor) * (USHORT)pRLE->SG_Desc_Count ;

    if (pBH->magic != RW_PRIVATE_MAGIC) {
       os2_panic("request_list_completed - request list with an invalid magic number");
    }
    if (pRLH->Lst_Status & RLH_All_Req_Done) {
        if (!pBH->wait_for_all) {
#if 0
            if (pBH->reqlist->busy) {
                pBH->reqlist->busy = 0;
            } else {
                os2_panic("EXT2-os2 fatal error in request_list_completed() -  pBH->reqlist->busy = 0");
            }
            return NO_ERROR;
#endif
            return free_reqlist(pBH->reqlist);
        } else {
            wake_up(&(pBH->wait_list));
            return NO_ERROR;
        }
    } else {
        return NO_ERROR;
    }
}

//
// Request completed callback
//
// *** WARNING *** This routine is called in interrupt context
//
int request_completed(struct Strat2_Read_Write *request) {
    __segment ReqSeg;
    struct _PB_Read_Write   __based(ReqSeg) *pRLE;     // Request list entry
    struct _SG_Descriptor   __based(ReqSeg) *pSG;      // Scatter-gather descriptor
    struct RW_private       __based(ReqSeg) *pBH;      // ext2fs specific part of RLE
    int status;
    int err_status;
    int i;

    ReqSeg         = (__segment)request;
    (USHORT)pRLE   = OFFSETOF(request);
    (USHORT)pSG    = (USHORT)pRLE + (USHORT)sizeof(struct _PB_Read_Write);
    (USHORT)pBH    = (USHORT)pSG  + (USHORT)sizeof(struct _SG_Descriptor) * (USHORT)pRLE->SG_Desc_Count ;


    if (pBH->magic != RW_PRIVATE_MAGIC) {
       os2_panic("request_completed - request list with an invalid magic number");
    }

    status         = pRLE->RqHdr.Status & 0x0F;
    err_status     = pRLE->RqHdr.Status & 0xF0;

    //
    // First case : the request is finished
    //
    if (status == RH_DONE) {
        switch (err_status) {
            case RH_NO_ERROR :
                break;
            case RH_RECOV_ERROR :
//                os2_panic("request_completed - Recoverable error");
//                kernel_printf("request_completed() - Recoverable error 0x%0X occured for block %lu - status = RH_DONE", pRLE->RqHdr.Error_Code, pBH->bh->b_blocknr);
                break;
            default :
//                kernel_printf("request_completed() - RH_DONE & 0x%0X", err_status);
//                os2_panic("request_completed - Unrecoverable error");
                for (i = 0 ; i < pBH->nb_bh ; i++) {
                    unlock_buffer(pBH->bh[i]);
                }
                return NO_ERROR;
        }
        switch (pRLE->RqHdr.Command_Code) {
            case PB_WRITE_X :
//                nb_bh = pBH->nb_bh;
//                __asm {int 3};

                for (i = 0 ; i < pBH->nb_bh ; i++) {
                     if (pBH->bh[i]->b_dirt) {
                         mark_buffer_clean(pBH->bh[i]);
                    }
                }
                break;

                case PB_READ_X :
                    break;
                default :
                      break;
        }
        for (i = 0 ; i < pBH->nb_bh ; i++) {
            pBH->bh[i]->b_uptodate = 1;
            unlock_buffer(pBH->bh[i]);
        }
        return NO_ERROR;
    }

    //
    // Second case : the request is NOT finished
    //
    switch (err_status) {
        //
        // In this case we should be notified later of the real status of the request.
        //
        case RH_NO_ERROR    :
        case RH_RECOV_ERROR :
            return NO_ERROR;
        //
        // In this case there is no more chance ...
        //
        default :
            for (i = 0 ; i < pBH->nb_bh ; i++) {
                unlock_buffer(pBH->bh[i]);
            }
            return NO_ERROR;
    }
}


extern void stub_request_list_completed(void);
extern void stub_request_completed(void);

struct _Req_List_Header *ll_rw_block_strat2(int rw, int nr, struct buffer_head **bh, int wait_for_all, struct super_block *sb) {
    int k, l, n;
    int nb_extents;
    int i;
//    struct _Req_List_Header *rw_req;
    unsigned char command;
    void (*strat2)();
    struct reqlist *reqlist;
    unsigned long sectors_per_block;
    struct buffer_head *bh2[32];

    __segment ReqSeg;
    struct _Req_List_Header __based(ReqSeg) *pRLH;        // Request list header
    struct _PB_Read_Write   __based(ReqSeg) *pRLE;        // Request list entry
    struct _SG_Descriptor   __based(ReqSeg) *pSG;        // Scatter-gather descriptor
    struct RW_private       __based(ReqSeg) *pBH;        // ext2fs specific part of RLE


    //
    // No buffer_heads, then nothing to do !!
    //
    if (nr == 0) {
       return NULL;
    }

    //
    // Save the strat2 function pointer
    //
    if (!(strat2 = sb->s_strat2)) {
        return NULL;
    } /* endif */


    switch (rw) {
        case READ  :
        case READA :
            command = PB_READ_X;
            break;
        case WRITE  :
        case WRITEA :
            command = PB_WRITE_X;
            break;
        default :
            os2_panic("ll_rw_block_strat2() - Invalid flag");
    }

    //
    // - We suppose that all buffer_head structures passed to ll_rw_block are for the same drive ...
    // - We also suppose that the bh array will result in a request list < 64 Kb
    // - We also suppose that there are less than 32 items in the bh array

    //
    // Let's sort the bh array by ascending disk block order (dumb bubble sort ... I will change
    // this later !)
    //
    memcpy(bh2, bh, sizeof(struct buffer_head *) * nr);
    {
         int found = 1;
            while (found) {
                found = 0;
                for (i = 0 ; i < nr - 1 ; i++) {
                    if (bh2[i]->b_blocknr > bh2[i + 1]->b_blocknr) {
                        struct buffer_head *tmp;
                        tmp = bh2[i];
                        bh2[i] = bh2[i + 1];
                        bh2[i + 1] = tmp;
                        found = 1;
                    }
                }
            }
    }

    //
    // Counts the number of extents (an extent is a set of consecutive disk blocks)
    //
    k = 0;
    n = 0;
    while (k < nr) {
        l = k + 1;
        while ((l < nr) && (bh2[l]->b_blocknr == bh2[k]->b_blocknr + l - k)) l++;
        if (k < nr) {
            n++;
        }
        k = l;
    }
    nb_extents = n;
//    kernel_printf("ll_rw_block : nr = %d, extents = %d", nr, n);

    /*
     * Allocates a request list.
     */
    reqlist = alloc_reqlist();
//    rw_req  = &(reqlist->rwlist);
    ReqSeg  = (__segment)reqlist;
    (USHORT)pRLH   = OFFSETOF(reqlist) + (USHORT)sizeof(struct reqlist) - (USHORT)sizeof(struct _Req_List_Header);
    (USHORT)pRLE   = (USHORT)pRLH + (USHORT)sizeof(struct _Req_List_Header);


    //
    // We initialize the request list header
    //
    pRLH->Count           = nb_extents;                     // number of requests in Req List
    pRLH->Notify_Address  = stub_request_list_completed;    // 16:16 address of notification routine
    if (nb_extents == 1) {
        pRLH->Request_Control = RLH_Notify_Done |
                                RLH_Notify_Err  |
                                RLH_Single_Req;             // Notify on completion or on error
    } else {
        pRLH->Request_Control = RLH_Notify_Done |
                                RLH_Notify_Err;             // Notify on completion or on error
    }
    pRLH->Block_Dev_Unit  = sb->s_unit;                     // logical unit number of volume (pvpfsi->vpi_unit)


    //
    // We now create a request for each extent
    //
    k = 0;    // First blk of extent
    n = 0;    // Number of extents
//    l = 0;  // Last blk of extent
    while (k < nr) {
        l = k + 1;
        while ((l < nr) && (bh2[l]->b_blocknr == bh2[k]->b_blocknr + l - k)) {
            l++;
        }
        //
        // Now the current extent is [k, l[
        // n is the current extent number (starting from 0)
        //
        if (k < nr) {
            (USHORT)pSG = (USHORT)pRLE + (USHORT)sizeof(struct _PB_Read_Write);
            (USHORT)pBH = (USHORT)pSG  + (USHORT)sizeof(struct _SG_Descriptor) * (USHORT)(l - k);

            //
            // We lock the buffers of the current extent
            //
            for (i = k ; i < l ; i++) {
                lock_buffer(bh2[i]);
                bh2[i]->b_uptodate = 0;
            }

            //
            // We initialize the request header
            //
            pRLE->RqHdr.Length         = (n == nb_extents - 1 ? RH_LAST_REQ : (USHORT)pBH  + (USHORT)sizeof(struct RW_private) + (USHORT)(l - k - 1) * (USHORT)sizeof(struct buffer_head *) - (USHORT)pRLE);        // offset of the next request
            pRLE->RqHdr.Old_Command    = PB_REQ_LIST;                          // always 1Ch
            pRLE->RqHdr.Command_Code   = command;                              // command code
            pRLE->RqHdr.Head_Offset    = (ULONG)pRLE - (ULONG)pRLH;            // offset from begin of Req List Header
            pRLE->RqHdr.Req_Control    = RH_NOTIFY_DONE | RH_NOTIFY_ERROR;     // Notify on completion or error
            pRLE->RqHdr.Priority       = (BYTE)(pLocInfoSeg->LIS_CurThrdPri >> 8);   // Priority of request
            pRLE->RqHdr.Status         = RH_NOT_QUEUED | RH_NO_ERROR;          // status bitfield
            pRLE->RqHdr.Error_Code     = 0;                                    // I24 error code
            pRLE->RqHdr.Notify_Address = stub_request_completed;               // 16:16 address of notification routine
            pRLE->RqHdr.Hint_Pointer   = 0xFFFFFFFF;                           // 16:16 pointer to req packet in list

            //
            // We initialize the command specific fields
            //
            sectors_per_block    = bh2[k]->b_size / 512;
            pRLE->Start_Block    = bh2[k]->b_blocknr * sectors_per_block;     // disk block number
            pRLE->Block_Count    = sectors_per_block * (l - k);               // number of blocks to transfer
            pRLE->Blocks_Xferred = 0;                                         // number of blocks transferred
            pRLE->RW_Flags       = 0;                                         // command specific control flags
            pRLE->SG_Desc_Count  = l - k;                                     // number of SG descriptors

            //
            // We initialize the scatter-gather descriptors
            //
            for (i = k ; i < l ; i++) {
                pSG[i - k].BufferSize = bh2[i]->b_size;                       // size of the buffer in bytes
                pSG[i - k].BufferPtr  = bh2[i]->b_physaddr;                   // physical address of buffer
            }

            //
            // We initialize the bh field and req_list field
            //
            pBH->nb_bh        = l - k;
            pBH->reqlist      = reqlist;
            pBH->wait_for_all = wait_for_all;
            pBH->magic        = RW_PRIVATE_MAGIC;
            for (i = k ; i < l ; i++) {
                pBH->bh[i - k] = bh2[i];
            }


            //
            // Next request list entry
            //
            (USHORT)pRLE      += (USHORT)pRLE->RqHdr.Length;

            //
            // Next extent
            //
            n++;
        }
        k = l;
    }

    //
    // We now call the device driver's extended strategy routine
    //
    __asm {
        push ds
        push es
        push si
        push di
        mov ax, ReqSeg
        mov es, ax
        mov bx, pRLH
        call [strat2]
        pop di
        pop si
        pop es
        pop ds
    };
    return MAKEP(ReqSeg, pRLH);
}



void ll_rw_block(int rw, int nr, struct buffer_head **bh) {
    struct super_block *sb;
    int                 i;

    if (nr == 0) {
        kernel_printf("ll_rw_block - Nothing to do");
        return;
    }

    switch (rw) {
        case READ  :
        case READA :
            break;

        case WRITE  :
        case WRITEA :
            if (!Read_Write) {
                for (i = 0; i < nr; i++) {
                    mark_buffer_clean(bh[i]);
                }
                return;
            }
            break;

        default :
            os2_panic("ll_rw_block() - Invalid flag");
    }


//    for (i = 0; i < nr; i++) {
//        if (bh[i]->b_magic != BUFFER_HEAD_MAGIC) {
//            os2_panic("ll_rw_block - buffer_head with an invalid magic number");
//        }
//    }

#if 0
    //
    // Checks for access to sector 0 ... debug only
    //
    for (i = 0; i < nr; i++) {
        if (bh[i]->b_blocknr == 0) {
            if ((rw == WRITE) || (rw == WRITEA)) {
                kernel_printf("ll_rw_block() - WARNING - Attempt to WRITE to sector 0 - aborting ...");
                return;
            }
        }
    }
#endif

    for (i = 0 ; i < nr ; i++) {
        if (bh[i]->b_dev == 0) {
            os2_panic("ll_rw_block - NULL device");
        }
        if (bh[i]->b_dev == 0xffff) {
            os2_panic("ll_rw_block - device is 0xFFFF");
        }
        if (bh[0]->b_dev != bh[i]->b_dev) {
            os2_panic("ll_rw_block - Multi-device request NOT allowed");
        }
    }
    if ((sb = getvolume(bh[0]->b_dev)) == 0) {
        os2_panic("ll_rw_block : can't retrieve superblock");
    }

    if (sb->s_strat2) {
        ll_rw_block_strat2(rw, nr, bh, 0, sb);
//        kernel_printf("u [%d] f [%d] m [%lu] r [%lu] a [%lu]", nusedreqlist, nfreereqlist, mallocated, reallocated, asis);
    } else {
        ll_rw_block_standard(rw, nr, bh, sb);
    }
}








#if 0
void ll_rw_block_and_wait_for_all(int rw, int nr, struct buffer_head **bh) {
    int i;
    struct _Req_List_Header  *rw_req;

    if (nr == 0) {
        kernel_printf("ll_rw_block - Nothing to do");
        return;
    }

#if 0
    //
    // Checks for access to sector 0 ... debug only
    //
    for (i = 0; i < nr; i++) {
        if (bh[i]->b_blocknr == 0) {
            if ((rw == WRITE) || (rw == WRITEA)) {
                kernel_printf("ll_rw_block() - WARNING - Attempt to WRITE to sector 0 - aborting ...");
                return;
            }
        }
    }
#endif

    if (bh[0]->dev->s_strat2) {
        if ((rw_req = ll_rw_block_strat2(rw, nr, bh, 1)) != NULL) {
            sleep_on(&(rw_req->rw[0].wait_list));
            free_reqlist(rw_req->rw[0].reqlist);
        }
    } else {
        ll_rw_block_standard(rw, nr, bh);
        for (i = 0 ; i < nr ; i++) {
            wait_on_buffer(bh[i]);
        }

    }
}
#endif
