//
// $Header: D:/ext2-os2/RCS/fs_dir.c,v 6.0 1996/01/23 23:54:51 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to     
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996 Matthieu WILLM 
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


/*****************************************************************************/
/***                                                                       ***/
/***  This file contains the following functions :                         ***/
/***      FS_CHDIR                                                         ***/
/***      FS_MKDIR                                                         ***/
/***      FS_RMDIR                                                         ***/
/***                                                                       ***/
/*****************************************************************************/

#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>                // From the "Developer Connection Device Driver Kit" version 2.0

#include <fsd.h>
#include <fsh.h>

#include <linux/stat.h>

#include <os2/ifsdbg.h>
#include <os2/filefind.h>
#include <os2/cdfsd.h>
#include <os2/errors.h>

#include <os2/log.h>         /* Prototypes des fonctions de log.c                      */
#include <os2/volume.h>      /* Prototypes des fonctions de volume.c                   */
#include <os2/files.h>       /* Prototypes des fonctions de files.c                    */

#include <os2/os2misc.h>
#include <os2/os2proto.h>
#include <linux/fs.h>
#include <linux/fs_proto.h>
#include <linux/e2_fs.h>
#include <linux/e2_proto.h>
#include <linux/sched.h>

#define THISFILE FILE_FS_DIR_C

extern char trace_FS_CHDIR;
extern char trace_FS_RMDIR;
extern char trace_FS_MKDIR;

/*************************************************************************************************/
/*** FS_CHDIR - IFS change directory entry point                                               ***/
/*************************************************************************************************/
int _FS_ENTRY FS_CHDIR(
                       unsigned short         flag,
                       struct cdfsi   _FS_PTR pcdfsi,
                       struct cdfsd   _FS_PTR pcdfsd,
                       pchar                  pDir,
                       unsigned short         iCurDirEnd
                      )
{
    int     rc;
    pfile   p_file;
    struct super_block * p_volume;
    UINT32       DOSmode;
    int valid;

    DOSmode = (is_case_retensive() ? OPENMODE_DOSBOX : 0);

    switch(flag) {
        case CD_EXPLICIT :
#ifdef FS_TRACE
            kernel_printf("FS_CHDIR( CD_EXPLICIT, %s )", pDir);
#endif
            if (iCurDirEnd != ~0) {
                kernel_printf("\tOld CDS is %s", pcdfsi->cdi_curdir);
            }

            //
            // Gets the superblock from pcdfsi
            //
            p_volume = getvolume(pcdfsi->cdi_hVPB);

            if ((p_file = _open_by_name(p_volume, pDir, OPENMODE_READONLY | DOSmode)) == 0) {
                fs_err(FUNC_FS_CHDIR, FUNC_OPEN_BY_NAME, ERROR_PATH_NOT_FOUND, THISFILE, __LINE__);
                return ERROR_PATH_NOT_FOUND;
            } else {
                if (!S_ISDIR(p_file->f_inode->i_mode)) {
                    kernel_printf("FS_CHDIR( %s ) Not a directory", pDir);
                    vfs_close(p_file);
                    return ERROR_ACCESS_DENIED;
                }

                ((_cdfsd _FS_PTR)pcdfsd)->is_valid = 1;
                ((_cdfsd _FS_PTR)pcdfsd)->p_file   = p_file;

                return NO_ERROR;
            } /* end if */

        case CD_VERIFY   :
#ifdef FS_TRACE
            fs_log("FS_CHDIR : flag = CD_VERIFY");
#endif
            //
            // Gets the superblock from pcdfsi
            //
            p_volume = getvolume(pcdfsi->cdi_hVPB);
            valid    = 1;
            if ((p_file = _open_by_name(p_volume, pcdfsi->cdi_curdir, OPENMODE_READONLY | DOSmode)) == 0) {
                fs_err(FUNC_FS_CHDIR, FUNC_OPEN_BY_NAME, ERROR_PATH_NOT_FOUND, THISFILE, __LINE__);
                valid = 0;
            } else {
            if (!S_ISDIR(p_file->f_inode->i_mode)) {
                kernel_printf("FS_CHDIR( %s ) Not a directory", pDir);
                vfs_close(p_file);
                valid = 0;
            }

            if (pcdfsi->cdi_flags & CDI_ISVALID) {
                vfs_close(((_cdfsd _FS_PTR)pcdfsd)->p_file);
            }
            if (valid) {
                ((_cdfsd _FS_PTR)pcdfsd)->is_valid = 1;
                ((_cdfsd _FS_PTR)pcdfsd)->p_file   = p_file;
                return NO_ERROR;
            } else {
                vfs_close(p_file);
                return ERROR_PATH_NOT_FOUND;
            }
            }

        case CD_FREE :
#ifdef FS_TRACE
            kernel_printf("FS_CHDIR( CD_FREE )");
#endif
            /*******************************************************************/
            /*** Gets the superblock from pcdfsd                             ***/
            /*******************************************************************/
//            p_volume = getvolume(((_cdfsd _FS_PTR)pcdfsd)->p_file->hVPB);
           /*******************************************************************/

            if (((_cdfsd *)pcdfsd)->is_valid == 1) {
                ((_cdfsd *)pcdfsd)->is_valid = 0;

                p_volume = getvolume(((_cdfsd *)pcdfsd)->p_file->f_inode->i_sb->s_dev);
                if ((rc = vfs_close(((_cdfsd *)pcdfsd)->p_file)) != NO_ERROR) {
                    fs_err(FUNC_FS_CHDIR, FUNC_CLOSE, rc, THISFILE, __LINE__);
                }
                return rc;
            } else {
                return NO_ERROR;
            } /* end if */

        default :
            fs_log("FS_CHDIR : invalid flag");
            return ERROR_INVALID_PARAMETER;

    } /* end switch */
    /**************************************************************/

}
/*************************************************************************************************/


/*************************************************************************************************/
/*** FS_MKDIR - IFS make directory entry point                                                 ***/
/*************************************************************************************************/
int _FS_ENTRY FS_MKDIR(
                           struct cdfsi   _FS_PTR pcdfsi,
                           struct cdfsd   _FS_PTR pcdfsd,
                           char           _FS_PTR pName,
                           unsigned short         iCurDirEnd,
                           char           _FS_PTR pEABuf,
                           unsigned short         flags
                          )
{
    int                 rc;
    struct super_block *p_volume;
    char                parent[CCHMAXPATH];
    char                name[CCHMAXPATH];
    struct file        *p_file;
    ino_t               ino_no;
    struct inode       *dir;
    UINT32              mode;


    if (trace_FS_MKDIR) {
        kernel_printf("FS_MKDIR pre-invocation : %s", pName);
    }


    if (Read_Write) {
        /*
         * Checks if we must do case sensitive compares
         */
        mode = (is_case_retensive() ? OPENMODE_DOSBOX : 0);

        /*
         * Gets the superblock from pcdfsi
         */
        if ((p_volume = getvolume(pcdfsi->cdi_hVPB)) != 0) {

            ExtractPath(pName, parent);
            ExtractName(pName, name);
            /*
             * Opens the parent directory
             */
            if ((p_file = _open_by_name(p_volume, parent, OPENMODE_READONLY | mode)) != NULL) {
                ino_no = p_file->f_inode->i_ino;
                if ((rc = vfs_close(p_file)) == NO_ERROR) {
                    dir = iget(p_volume, ino_no);
                    dir->i_count++;
                    down(&dir->i_sem);
                    rc = dir->i_op->mkdir(dir, name, strlen(name), 0777);
                    up(&dir->i_sem);
                    iput(dir);
                    rc = map_err(rc);        // rc was a Linux error code (from linux/errno.h)
                } else {
                    fs_err(FUNC_FS_MKDIR, FUNC_CLOSE, rc, THISFILE, __LINE__);
                    rc = ERROR_CANNOT_MAKE;
                }
            } else {
                fs_err(FUNC_FS_MKDIR, FUNC_OPEN_BY_NAME, -1, THISFILE, __LINE__);
                rc = ERROR_CANNOT_MAKE;
            } /* end if */
        } else {
            kernel_printf("FS_MKDIR - couldn't retrieve superblock");
            rc = ERROR_INVALID_PARAMETER;
        }
    } else {                 // !Read_Write
        rc = ERROR_WRITE_PROTECT;
    }

    /*
     * Some apps (software installer...) do rely on this, which seems to be the default
     * behavior on HPFS.
     */
    if (rc == ERROR_FILE_EXISTS) {
        rc = ERROR_ACCESS_DENIED;
    }

    if (trace_FS_MKDIR) {
        kernel_printf("FS_MKDIR post-invocation : rc = %d", rc);
    }
    return rc;
}
/*************************************************************************************************/

/*************************************************************************************************/
/*** FS_RMDIR - IFS remove directory entry point                                               ***/
/*************************************************************************************************/
int _FS_ENTRY FS_RMDIR(
                           struct cdfsi   _FS_PTR pcdfsi,
                          struct cdfsd   _FS_PTR pcdfsd,
                           char           _FS_PTR pName,
                           unsigned short         iCurDirEnd
                          )
{
    int                 rc;       /* return code       */
    struct super_block *sb;       /* volume descriptor */
    struct inode       *dir;
    struct file        *filp;
    ino_t               ino_no;
    char              parent[CCHMAXPATH];
    char              name[CCHMAXPATH];
    umode_t                i_mode;
    UINT32              mode;

    if (trace_FS_RMDIR) {
        kernel_printf("FS_RMDIR pre-invocation : %s", pName);
    }

    if (Read_Write) {
        mode = (is_case_retensive() ? OPENMODE_DOSBOX : 0);
        //
        // Gets the superblock from pcdfsi
        //
        if ((sb = getvolume(pcdfsi->cdi_hVPB)) != 0) {

            ExtractPath(pName, parent);
            ExtractName(pName, name);
            if ((filp = _open_by_name(sb, parent, OPENMODE_READONLY | mode)) != NULL) {
                ino_no = filp->f_inode->i_ino;
                i_mode = filp->f_inode->i_mode;
                if ((rc = vfs_close(filp)) == NO_ERROR) {

                    if (S_ISDIR(i_mode)) {
                        if (((i_mode & S_IWUSR)) ||
                            ((i_mode & S_IWGRP)) ||
                            ((i_mode & S_IWOTH))) {

                            dir = iget(sb, ino_no);
                            rc = dir->i_op->rmdir(dir, name, strlen(name));
                            rc = map_err(rc);        // rc was a Linux error code (from linux/errno.h)
                        } else {
                            kernel_printf("FS_RMDIR - %s is read only", pName);
                            rc = ERROR_ACCESS_DENIED;
                        }
                    } else {
                        kernel_printf("FS_RMDIR - %s is not a directory", pName);
                        rc = ERROR_ACCESS_DENIED;
                    }
                } else {
                    fs_err(FUNC_FS_RMDIR, FUNC_CLOSE, rc, THISFILE, __LINE__);
                }
            } else {
                fs_err(FUNC_FS_RMDIR, FUNC_OPEN_BY_NAME, -1, THISFILE, __LINE__);
                rc = ERROR_OPEN_FAILED;
            } /* end if */
        } else {
            kernel_printf("FS_RMDIR - Couldn't get the superblock");
            rc = ERROR_INVALID_PARAMETER;
        }
    } else {
        rc = ERROR_WRITE_PROTECT;
    }

    if (trace_FS_RMDIR) {
        kernel_printf("FS_RMDIR post-invocation : rc = %d", rc);
    }
    return rc;
}
/*************************************************************************************************/
