//
// $Header: D:/ext2-os2/vfs/RCS/inode.c,v 1.7 1995/08/08 21:16:10 Willm Exp Willm $
//

// Linux ext2 file system driver for OS/2 2.x and WARP - Allows OS/2 to     
// access your Linux ext2fs partitions as normal drive letters.
// OS/2 implementation : Copyright (C) 1995  Matthieu WILLM
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

/*
 *  linux/fs/inode.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#ifndef OS2
#include <linux/stat.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>

#include <asm/system.h>
#else
#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>		// From the "Developer Connection Device Driver Kit" version 2.0

#include <os2/types.h>
#include <os2/os2proto.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/fs_proto.h>
#include <linux/stat.h>
#include <os2/errors.h>
extern unsigned long event;
#define const
#endif



#ifdef OS2
#define Panic(x) FSH_INTERR((x), sizeof(x))
#define inline _inline
#endif

static struct inode_hash_entry {
	struct inode * inode;
	int updating;
} hash_table[NR_IHASH];

#ifndef OS2
static struct inode * first_inode;
static struct wait_queue * inode_wait = NULL;
static int nr_inodes = 0, nr_free_inodes = 0;
#else
static struct inode * first_inode;
static unsigned long inode_wait = NULL;
static long nr_inodes = 0, nr_free_inodes = 0;
#endif

#ifndef OS2
static inline int const hashfn(dev_t dev, unsigned int i)
{
	return (dev ^ i) % NR_IHASH;
}
#else
static inline long const hashfn(dev_t dev, unsigned long i)
{
	return (long)(((unsigned long)dev ^ i) % NR_IHASH);
}
#endif

#ifndef OS2
static inline struct inode_hash_entry * const hash(dev_t dev, int i)
{
	return hash_table + hashfn(dev, i);
}
#else
static inline struct inode_hash_entry * const hash(dev_t dev, long i)
{
	return hash_table + hashfn(dev, (unsigned long)i);
}
#endif
static void insert_inode_free(struct inode *inode)
{
        
        if (!(inode)) {
            Panic("insert_inode_free - 1");
        } /* endif */
	inode->i_next = first_inode;
        if (!(inode)) {
            Panic("insert_inode_free - 2");
        } /* endif */
	inode->i_prev = first_inode->i_prev;
        if (!(inode->i_next)) {
            Panic("insert_inode_free - 3");
        } /* endif */
	inode->i_next->i_prev = inode;
        if (!(inode->i_prev)) {
            Panic("insert_inode_free - 4");
        } /* endif */
	inode->i_prev->i_next = inode;
        if (!(first_inode)) {
            Panic("insert_inode_free - 5");
        } /* endif */
	first_inode = inode;

}

static void remove_inode_free(struct inode *inode)
{
	if (first_inode == inode)
		first_inode = first_inode->i_next;
	if (inode->i_next)
		inode->i_next->i_prev = inode->i_prev;
	if (inode->i_prev)
		inode->i_prev->i_next = inode->i_next;
	inode->i_next = inode->i_prev = NULL;
}

void insert_inode_hash(struct inode *inode)
{
	struct inode_hash_entry *h;
	h = hash(inode->i_dev, inode->i_ino);

	inode->i_hash_next = h->inode;
	inode->i_hash_prev = NULL;
	if (inode->i_hash_next)
		inode->i_hash_next->i_hash_prev = inode;
	h->inode = inode;
}

static void remove_inode_hash(struct inode *inode)
{
	struct inode_hash_entry *h;
	h = hash(inode->i_dev, inode->i_ino);

	if (h->inode == inode)
		h->inode = inode->i_hash_next;
	if (inode->i_hash_next)
		inode->i_hash_next->i_hash_prev = inode->i_hash_prev;
	if (inode->i_hash_prev)
		inode->i_hash_prev->i_hash_next = inode->i_hash_next;
	inode->i_hash_prev = inode->i_hash_next = NULL;
}

#ifndef OS2
static void put_last_free(struct inode *inode)
{
	remove_inode_free(inode);
	inode->i_prev = first_inode->i_prev;
	inode->i_prev->i_next = inode;
	inode->i_next = first_inode;
	inode->i_next->i_prev = inode;
}
#else
static void put_last_free(struct inode *inode)
{
        if (!inode) {
            Panic("put_last_free - 0");
        } /* endif */

	remove_inode_free(inode);
        if (!inode) {
            Panic("put_last_free - 1");
        } /* endif */
	inode->i_prev = first_inode->i_prev;
        if (!(inode->i_prev)) {
            Panic("put_last_free - 2");
        } /* endif */
	inode->i_prev->i_next = inode;
        if (!inode) {
            Panic("put_last_free - 3");
        } /* endif */

	inode->i_next = first_inode;
        if (!(inode->i_next)) {
            Panic("put_last_free - 4");
        } /* endif */

	inode->i_next->i_prev = inode;
}
#endif
#ifndef OS2
void grow_inodes(void)
{
	struct inode * inode;
	int i;

	if (!(inode = (struct inode*) get_free_page(GFP_KERNEL)))
		return;

	i=PAGE_SIZE / sizeof(struct inode);
	nr_inodes += i;
	nr_free_inodes += i;

	if (!first_inode)
		inode->i_next = inode->i_prev = first_inode = inode++, i--;

	for ( ; i ; i-- )
		insert_inode_free(inode++);

}
#else
void grow_inodes(void)
{
	struct inode * inode;
	long i;

	if (!(inode = (struct inode*) G_malloc(65000L))) {
        	kernel_printf("grow_inodes() - erreur G_malloc");
                FS_DevHelp_Beep(1000, 200);
		return;
        }
	i=65000L / sizeof(struct inode);
	kernel_printf("********* grow_inode() from %lu to %lu", nr_inodes, i);
	nr_inodes += i;
	nr_free_inodes += i;
#if 0
	if (!first_inode)
		inode->i_next = inode->i_prev = first_inode = inode++, i--;
#else
//
// If I don't do it that way with MS Visual C++, first_inode->i_prev = NULL ... TRAP D !!!!
//
	if (!first_inode) {
                first_inode   = inode; 
		inode->i_next = inode;
                inode->i_prev = inode;
                inode++;
                i--;
        }
        if ((!first_inode)         || 
            (!first_inode->i_prev) || 
            (!first_inode->i_next)) 
                Panic("grow_inodes - 1");
#endif

	for ( ; i ; i-- ) {
		insert_inode_free(inode++);
        	if ((!first_inode) || (!first_inode->i_prev) || (!first_inode->i_next)) Panic("grow_inodes - 2");
        }
}
#endif
unsigned long inode_init(unsigned long start, unsigned long end)
{
	memset(hash_table, 0, sizeof(hash_table));
	first_inode = NULL;
	return start;
}

void __wait_on_inode(struct inode *);

inline void wait_on_inode(struct inode * inode)
{
	if (inode->i_lock)
		__wait_on_inode(inode);
}

static inline void lock_inode(struct inode * inode)
{
	wait_on_inode(inode);
	inode->i_lock = 1;
}

static inline void unlock_inode(struct inode * inode)
{
	inode->i_lock = 0;

	wake_up(&inode->i_wait);
}

/*
 * Note that we don't want to disturb any wait-queues when we discard
 * an inode.
 *
 * Argghh. Got bitten by a gcc problem with inlining: no way to tell
 * the compiler that the inline asm function 'memset' changes 'inode'.
 * I've been searching for the bug for days, and was getting desperate.
 * Finally looked at the assembler output... Grrr.
 *
 * The solution is the weird use of 'volatile'. Ho humm. Have to report
 * it to the gcc lists, and hope we can do this more cleanly some day..
 */
#ifndef OS2
void clear_inode(struct inode * inode)
{
	struct wait_queue * wait;

	wait_on_inode(inode);
	remove_inode_hash(inode);
	remove_inode_free(inode);
	wait = ((volatile struct inode *) inode)->i_wait;
	if (inode->i_count)
		nr_free_inodes++;
	memset(inode,0,sizeof(*inode));
	((volatile struct inode *) inode)->i_wait = wait;
	insert_inode_free(inode);
}
#else
void clear_inode(struct inode * inode)
{
	unsigned long wait;

	wait_on_inode(inode);
	remove_inode_hash(inode);
	remove_inode_free(inode);
	wait = ((volatile struct inode *) inode)->i_wait;
	if (inode->i_count)
		nr_free_inodes++;
	memset(inode,0,sizeof(*inode));
	((volatile struct inode *) inode)->i_wait = wait;
        if (!inode) Panic("clear_inode() - insert_inode_free(NULL)");
	insert_inode_free(inode);
}
#endif

#ifndef OS2
int fs_may_mount(dev_t dev)
{
	struct inode * inode, * next;
	int i;

	next = first_inode;
	for (i = nr_inodes ; i > 0 ; i--) {
		inode = next;
		next = inode->i_next;	/* clear_inode() changes the queues.. */
		if (inode->i_dev != dev)
			continue;
		if (inode->i_count || inode->i_dirt || inode->i_lock)
			return 0;
		clear_inode(inode);
	}
	return 1;
}

int fs_may_umount(dev_t dev, struct inode * mount_root)
{
	struct inode * inode;
	int i;

	inode = first_inode;
	for (i=0 ; i < nr_inodes ; i++, inode = inode->i_next) {
		if (inode->i_dev != dev || !inode->i_count)
			continue;
		if (inode == mount_root && inode->i_count == 1)
			continue;
		return 0;
	}
	return 1;
}

int fs_may_remount_ro(dev_t dev)
{
	struct file * file;
	int i;

	/* Check that no files are currently opened for writing. */
	for (file = first_file, i=0; i<nr_files; i++, file=file->f_next) {
		if (!file->f_count || !file->f_inode ||
		    file->f_inode->i_dev != dev)
			continue;
		if (S_ISREG(file->f_inode->i_mode) && (file->f_mode & 2))
			return 0;
	}
	return 1;
}
#endif
static void write_inode(struct inode * inode)
{
#ifdef OS2
	if (inode->i_ino == INODE_DASD) 
		return;
#endif
	if (!inode->i_dirt)
		return;
	wait_on_inode(inode);
	if (!inode->i_dirt)
		return;
	if (!inode->i_sb || !inode->i_sb->s_op || !inode->i_sb->s_op->write_inode) {
		inode->i_dirt = 0;
		return;
	}
	inode->i_lock = 1;	
	inode->i_sb->s_op->write_inode(inode);
	unlock_inode(inode);
}

static void read_inode(struct inode * inode)
{
#ifdef OS2
	if (inode->i_ino == INODE_DASD) {
		inode->i_mode    = S_IFREG | S_IRUSR | S_IWUSR;
		inode->i_size    = inode->i_sb->sector_size * inode->i_sb->nb_sectors;
		inode->i_blksize = inode->i_sb->s_blocksize;
		inode->i_blocks  = inode->i_size / inode->i_sb->s_blocksize;
		inode->i_op      = &ext2_file_inode_operations;
		return;
        }
#endif
	lock_inode(inode);
	if (inode->i_sb && inode->i_sb->s_op && inode->i_sb->s_op->read_inode)
		inode->i_sb->s_op->read_inode(inode);
	unlock_inode(inode);
}

#ifndef OS2
/* POSIX UID/GID verification for setting inode attributes */
int inode_change_ok(struct inode *inode, struct iattr *attr)
{
	/* Make sure a caller can chown */
	if ((attr->ia_valid & ATTR_UID) &&
	    (current->fsuid != inode->i_uid ||
	     attr->ia_uid != inode->i_uid) && !fsuser())
		return -EPERM;

	/* Make sure caller can chgrp */
	if ((attr->ia_valid & ATTR_GID) &&
	    (!in_group_p(attr->ia_gid) && attr->ia_gid != inode->i_gid) &&
	    !fsuser())
		return -EPERM;

	/* Make sure a caller can chmod */
	if (attr->ia_valid & ATTR_MODE) {
		if ((current->fsuid != inode->i_uid) && !fsuser())
			return -EPERM;
		/* Also check the setgid bit! */
		if (!fsuser() && !in_group_p((attr->ia_valid & ATTR_GID) ? attr->ia_gid :
					     inode->i_gid))
			attr->ia_mode &= ~S_ISGID;
	}

	/* Check for setting the inode time */
	if ((attr->ia_valid & ATTR_ATIME_SET) &&
	    ((current->fsuid != inode->i_uid) && !fsuser()))
		return -EPERM;
	if ((attr->ia_valid & ATTR_MTIME_SET) &&
	    ((current->fsuid != inode->i_uid) && !fsuser()))
		return -EPERM;


	return 0;
}

/*
 * Set the appropriate attributes from an attribute structure into
 * the inode structure.
 */
void inode_setattr(struct inode *inode, struct iattr *attr)
{
	if (attr->ia_valid & ATTR_UID)
		inode->i_uid = attr->ia_uid;
	if (attr->ia_valid & ATTR_GID)
		inode->i_gid = attr->ia_gid;
	if (attr->ia_valid & ATTR_SIZE)
		inode->i_size = attr->ia_size;
	if (attr->ia_valid & ATTR_ATIME)
		inode->i_atime = attr->ia_atime;
	if (attr->ia_valid & ATTR_MTIME)
		inode->i_mtime = attr->ia_mtime;
	if (attr->ia_valid & ATTR_CTIME)
		inode->i_ctime = attr->ia_ctime;
	if (attr->ia_valid & ATTR_MODE) {
		inode->i_mode = attr->ia_mode;
		if (!fsuser() && !in_group_p(inode->i_gid))
			inode->i_mode &= ~S_ISGID;
	}
	inode->i_dirt = 1;
}

/*
 * notify_change is called for inode-changing operations such as
 * chown, chmod, utime, and truncate.  It is guaranteed (unlike
 * write_inode) to be called from the context of the user requesting
 * the change.  It is not called for ordinary access-time updates.
 * NFS uses this to get the authentication correct.  -- jrs
 */

int notify_change(struct inode * inode, struct iattr *attr)
{
	int retval;

	if (inode->i_sb && inode->i_sb->s_op  &&
	    inode->i_sb->s_op->notify_change) 
		return inode->i_sb->s_op->notify_change(inode, attr);

	if ((retval = inode_change_ok(inode, attr)) != 0)
		return retval;

	inode_setattr(inode, attr);
	return 0;
}

/*
 * bmap is needed for demand-loading and paging: if this function
 * doesn't exist for a filesystem, then those things are impossible:
 * executables cannot be run from the filesystem etc...
 *
 * This isn't as bad as it sounds: the read-routines might still work,
 * so the filesystem would be otherwise ok (for example, you might have
 * a DOS filesystem, which doesn't lend itself to bmap very well, but
 * you could still transfer files to/from the filesystem)
 */
int bmap(struct inode * inode, int block)
{
	if (inode->i_op && inode->i_op->bmap)
		return inode->i_op->bmap(inode,block);
	return 0;
}
#endif

void invalidate_inodes(dev_t dev)
{
	struct inode * inode, * next;
#ifdef OS2
	long i;
#else
	int i;
#endif
	next = first_inode;
	for(i = nr_inodes ; i > 0 ; i--) {
		inode = next;
		next = inode->i_next;		/* clear_inode() changes the queues.. */
		if (inode->i_dev != dev)
			continue;
		if (inode->i_count || inode->i_dirt || inode->i_lock) {
#ifndef OS2
			printk("VFS: inode busy on removed device %d/%d\n", MAJOR(dev), MINOR(dev));
#else
			printk("VFS: inode busy on removed device 0x%0X", dev);
#endif
			continue;
		}
		clear_inode(inode);
	}
}

void sync_inodes(dev_t dev)
{
#ifdef OS2
	long i;
#else
	int i;
#endif
	struct inode * inode;

	inode = first_inode;
	for(i = 0; i < nr_inodes*2; i++, inode = inode->i_next) {
		if (dev && inode->i_dev != dev)
			continue;
		wait_on_inode(inode);
		if (inode->i_dirt)
			write_inode(inode);
	}
}

void iput(struct inode * inode)
{
	if (!inode)
		return;
#ifdef FS_TRACE
	kernel_printf("iput(%lu) - prev. i_count = %d", inode->i_ino, inode->i_count);
#endif
	wait_on_inode(inode);
	if (!inode->i_count) {
   #ifndef OS2
		printk("VFS: iput: trying to free free inode\n");
		printk("VFS: device %d/%d, inode %lu, mode=0%07o\n",
			MAJOR(inode->i_rdev), MINOR(inode->i_rdev),
					inode->i_ino, inode->i_mode);
#else
		printk("VFS: iput: trying to free free inode %lu", inode->i_ino);
#endif
		return;
	}
#ifndef OS2
	if (inode->i_pipe)
		wake_up_interruptible(&PIPE_WAIT(*inode));
#endif
repeat:
	if (inode->i_count>1) {
		inode->i_count--;
		return;
	}
	wake_up(&inode_wait);

#ifndef OS2
	if (inode->i_pipe) {
		unsigned long page = (unsigned long) PIPE_BASE(*inode);
		PIPE_BASE(*inode) = NULL;
		free_page(page);
	}
#endif
#ifdef OS2
        if (inode->i_ino != INODE_DASD) {
#endif
	if (inode->i_sb && inode->i_sb->s_op && inode->i_sb->s_op->put_inode) {
		inode->i_sb->s_op->put_inode(inode);
		if (!inode->i_nlink)
			return;
	}
	if (inode->i_dirt) {
		write_inode(inode);	/* we can sleep - so do again */
		wait_on_inode(inode);
		goto repeat;
	}
#ifdef OS2
        }
#endif
	inode->i_count--;
#ifndef OS2
	if (inode->i_mmap) {
		printk("iput: inode %lu on device %d/%d still has mappings.\n",
			inode->i_ino, MAJOR(inode->i_dev), MINOR(inode->i_dev));
		inode->i_mmap = NULL;
	}
#endif
	nr_free_inodes++;
	return;
}

struct inode * get_empty_inode(void)
{
	struct inode * inode, * best;
	int i;

	if (nr_inodes < NR_INODE && nr_free_inodes < (nr_inodes >> 2))
		grow_inodes();
repeat:
	inode = first_inode;
	best = NULL;
	for (i = 0; i<nr_inodes; inode = inode->i_next, i++) {
		if (!inode->i_count) {
			if (!best)
				best = inode;
			if (!inode->i_dirt && !inode->i_lock) {
				best = inode;
				break;
			}
		}
	}
	if (!best || best->i_dirt || best->i_lock)
		if (nr_inodes < NR_INODE) {
			grow_inodes();
			goto repeat;
		}
	inode = best;

	if (!inode) {
		printk("VFS: No free inodes - contact Linus\n");
		sleep_on(&inode_wait);
		goto repeat;
	}

	if (inode->i_lock) {
		wait_on_inode(inode);
		goto repeat;
	}
	if (inode->i_dirt) {
		write_inode(inode);
		goto repeat;
	}
	if (inode->i_count)
		goto repeat;
	clear_inode(inode);
	inode->i_count = 1;
	inode->i_nlink = 1;
	inode->i_version = ++event;
	inode->i_sem.count = 1;
	nr_free_inodes--;
	if (nr_free_inodes < 0) {
		printk ("VFS: get_empty_inode: bad free inode count.\n");
		nr_free_inodes = 0;
	}
	return inode;
}

#ifndef OS2
struct inode * get_pipe_inode(void)
{
	struct inode * inode;
	extern struct inode_operations pipe_inode_operations;

	if (!(inode = get_empty_inode()))
		return NULL;
	if (!(PIPE_BASE(*inode) = (char*) __get_free_page(GFP_USER))) {
		iput(inode);
		return NULL;
	}
	inode->i_op = &pipe_inode_operations;
	inode->i_count = 2;	/* sum of readers/writers */
	PIPE_WAIT(*inode) = NULL;
	PIPE_START(*inode) = PIPE_LEN(*inode) = 0;
	PIPE_RD_OPENERS(*inode) = PIPE_WR_OPENERS(*inode) = 0;
	PIPE_READERS(*inode) = PIPE_WRITERS(*inode) = 1;
	PIPE_LOCK(*inode) = 0;
	inode->i_pipe = 1;
	inode->i_mode |= S_IFIFO | S_IRUSR | S_IWUSR;
	inode->i_uid = current->fsuid;
	inode->i_gid = current->fsgid;
	inode->i_atime = inode->i_mtime = inode->i_ctime = CURRENT_TIME;
	inode->i_blksize = PAGE_SIZE;
	return inode;
}
#endif

#ifndef OS2
struct inode * __iget(struct super_block * sb, int nr, int crossmntp)
{
	static struct wait_queue * update_wait = NULL;
#else
struct inode * __iget(struct super_block * sb, ino_t nr, int crossmntp)
{
	static unsigned long update_wait = 0;
#endif
	struct inode_hash_entry * h;
	struct inode * inode;
	struct inode * empty = NULL;

	if (!sb)
#ifndef OS2
		panic("VFS: iget with sb==NULL");
#else
		Panic("VFS: iget with sb==NULL");
#endif
	h = hash(sb->s_dev, nr);
repeat:
	for (inode = h->inode; inode ; inode = inode->i_hash_next)
		if (inode->i_dev == sb->s_dev && inode->i_ino == nr)
			goto found_it;
	if (!empty) {
		h->updating++;
		empty = get_empty_inode();
		if (!--h->updating)
			wake_up(&update_wait);
		if (empty)
			goto repeat;
		return (NULL);
	}
	inode = empty;
	inode->i_sb = sb;
	inode->i_dev = sb->s_dev;
	inode->i_ino = nr;
	inode->i_flags = sb->s_flags;
	put_last_free(inode);
	insert_inode_hash(inode);
	read_inode(inode);
	goto return_it;

found_it:
	if (!inode->i_count)
		nr_free_inodes--;
	inode->i_count++;
	wait_on_inode(inode);
#ifdef OS2
        if (!inode || (SELECTOROF(inode) == -1)) Panic("__iget - 1");
#endif
	if (inode->i_dev != sb->s_dev || inode->i_ino != nr) {
		printk("Whee.. inode changed from under us. Tell Linus\n");
		iput(inode);
		goto repeat;
	}
#ifndef OS2
	if (crossmntp && inode->i_mount) {
		struct inode * tmp = inode->i_mount;
		tmp->i_count++;
		iput(inode);
		inode = tmp;
		wait_on_inode(inode);
	}
#endif
	if (empty)
		iput(empty);

return_it:
	while (h->updating)
		sleep_on(&update_wait);
	return inode;
}

/*
 * The "new" scheduling primitives (new as of 0.97 or so) allow this to
 * be done without disabling interrupts (other than in the actual queue
 * updating things: only a couple of 386 instructions). This should be
 * much better for interrupt latency.
 */
#ifndef OS2
static void __wait_on_inode(struct inode * inode)
{
	struct wait_queue wait = { current, NULL };

	add_wait_queue(&inode->i_wait, &wait);
repeat:
	current->state = TASK_UNINTERRUPTIBLE;
	if (inode->i_lock) {
		schedule();
		goto repeat;
	}
	remove_wait_queue(&inode->i_wait, &wait);
	current->state = TASK_RUNNING;
}
#else

void __wait_on_inode(struct inode * inode)
{

#ifdef FS_TRACE_LOCKS
    kernel_printf("wait_on_inode(%lu)", &(inode->i_wait));
#endif
    cli();
    while (inode->i_lock) {
        FS_DevHelp_ProcBlock((unsigned long)(&(inode->i_wait)));
        cli();
    }
    sti();
}
#endif


