
(*
 * TEIniFile - Enhanced TIniFile component.
 *
 * Derived from Borland Delphi's TIniFile component by
 * Mark R. Holbrook - System Dynamics - 10/16/1995
 *
 * No warranties expressed or implied are made about the usefulness or
 * destructive nature of this component.  The author assumes no liability for
 * any damage to your computer hardware or software that this component may
 * cause.  This component has been tested under Windows95 and
 * and appears to work as expected.
 *
 * This component is released into the public domain.  I only ask that you
 * distribute it with these comments intact.  If you feel included to donate
 * for it's usefulness, please contact me at the email address below.
 *
 * I would appreciate comments, questions, enhancements, and observations
 * about this code.  Please feel free to write:  markh@sysdyn.com
 *
 * This version has all the functionality (since it is derived from) of
 * TIniFile.  It has been modified to have the following additional
 * features:
 *
 *   Search the entire .INI file for an item.
 *   Replace an item in a .INI file.
 *   Delete an item from a .INI file.
 *   Add an item in a .INI file.
 *
 * These four features are notable in that they will work with SYSTEM.INI.
 * The [386Enh] section of SYSTEM.INI contains multiple lines of the form:
 *
 *   device=...
 *
 * These lines mess up the normally used Windows functions
 * GetPrivateProfileString and WritePrivateProfileString since they are
 * looking for unique identifiers. Since the ReadString and WriteString
 * methods of TIniFile are based on those functions they mess up too.
 *
 * The methods to add, delete, and replace .INI file items accomplish there
 * actions by making copies of the file.  This means that your directory will
 * have back copies of the .INI file you modify.  It is up to you to decide if
 * you want to clean these up or leave them around.
 *
 * I wrote this to go along with TInstall from Bill White.  Our application
 * must have a .386 VXD installed to work correctly.  Every .INI component
 * I looked at did not handle the wierdness of SYSTEM.INI.  So here it is:
 *
 *)

unit EIniFile;

interface

uses WinTypes, Classes, IniFiles;

type

	(*
   * Derived version of TIniFile
   *)
  TEIniFile = class(TIniFile)
  private

  	(*
     * Obtain full pathname to the current .INI file
     *)
  	function			GetIniPathName : string;

    (*
     * Obtain a backup .INI pathname
     *)
		function			GetBackupIniPathName( iniName : string ) : string;

  public

  	(*
     * Searches entire .INI file for a specified item.
     *
     * The item to search for can be a substring.
     *
     * If found the function returns true and the Section
     * and Line parameters are filled to hold what was found.
     *)
    function			ItemExists(		const		Item		: string;
    														var			Section : string;
                              	var			Line		: string ) : boolean;


		(*
     * Add a new item to a section of a .INI
     *
     * This function is useful for that wierd section of SYSTEM.INI
     * where there are multiple lines like:  device=...
     *
     * This function will add a new item.  It first renames the
     * .INI to .0 or something that works.  Then it copies the entire
     * file line by line looking for the specified section.  Once found
     * it adds the new Line as the first item of the specified section.
     *
     * Pass in the section that you expect to find Item in WITHOUT the
     * braces.  IE for [386Enh] pass in 386Enh.
     *
     * Finally pass in the ENTIRE line to be added.  This means
     * you must pass in the WHOLE line as in:  device=c:\windows\test.386
     *)
		function			AddItem( 			const		Section : string;
    														const		Line		: string ) : boolean;

		(*
     * Replace a specific item in a .INI file.
     *
     * This function is useful for that wierd section of SYSTEM.INI
     * where there are multiple lines like:  device=...
     *
     * This function will replace an item.  It first renames the
     * .INI to .0 or something that works.  Then it copies the entire
     * file line by line looking for the specified line.  Once found
     * it substitutes Line for the found one.
     *
     * To make this function work correctly pass in the item you want
     * to replace in 'Item'.  This can be a substring but be careful to
     * make it unique.
     *
     * Pass in the section that you expect to find Item in WITHOUT the
     * braces.  IE for [386Enh] pass in 386Enh.
     *
     * Finally pass in the ENTIRE line to replace it with.  This means
     * you must pass in the WHOLE line as in:  device=c:\windows\test.386
     *)
		function			ReplaceItem(  const		Item		: string;
    														const		Section	: string;
                                const		Line		: string ) : boolean;

		(*
     * Delete a specific item from a .INI file.
     *
     * This function is useful for that wierd section of SYSTEM.INI
     * where there are multiple lines like:  device=...
     *
     * This function will delete an item.  It first renames the
     * .INI to .0 or something that works.  Then it copies the entire
     * file line by line looking for the specified line.  Once found
     * it simply does not copy it to the new .INI.
     *
     * To make this function work correctly pass in the item you want
     * to delete in 'Item'.  This can be a substring but be careful to
     * make it unique.
     *
     * Pass in the section that you expect to find Item in WITHOUT the
     * braces.  IE for [386Enh] pass in 386Enh.
     *)
		function			DeleteItem(		const		Item		: string;
    														const		Section	: string ) : boolean;

  end;



implementation

uses WinProcs, SysUtils;


(*
 * This function determines if a path needs
 * to be appended to the .INI filename.
 *
 * By definition, if there is no path info in the .INI file
 * name, the GetProfile functions assume it is in the Windows
 * directory.
 *)
function	TEIniFile.GetIniPathName : string;
var
	winDir	: array[0..127] of char;
begin
	{ Use TIniFile property for filename }
  Result	:= FileName;

  { Check to see if it has a full path. If not then assume windows directory }
	if Pos('\',Result) = 0 then
   	begin
	    GetWindowsDirectory( winDir, 127 );
   		Result	:= StrPas( winDir );
	    Result	:= Result + '\' + FileName;
		end;
end;


(*
 * This function takes the passed in .INI filename
 * and determines a backup filename by adding a simple
 * numerical suffix.
 *)
function	TEIniFile.GetBackupIniPathName( iniName : string ) : string;
var
	c			: integer;
  i			: integer;
begin
	{ We need to rename it to a backup name }
  c	:= -1;
  repeat
  	{ Get our name }
	  Result	:= iniName;

    { Increment the extension counter }
    inc(c);

  	{ Locate the .INI part and delete it }
		i	:= Pos('.INI',Result);
    Delete(Result,i,4);

    { Add a new extention }
    Result	:= Result + '.' + IntToStr(c);
  until not FileExists(Result);
end;


(*
 * Searches entire .INI file for a specified item.
 *
 * The item to search for can be a substring.
 *
 * If found the function returns true and the Section
 * and Line parameters are filled to hold what was found.
 *)
function	TEIniFile.ItemExists( const		Item			: string;
																var			Section		: string;
                                var			Line			: string ) : boolean;
var
	iniFile	: text;
  iniName	: string;
  curItem	: string;
  theItem	: string;
  i				: integer;
begin
	try
  	{ Initialize return }
    Result	:= false;
    Section	:= '';
    Line		:= '';

    { Convert searched for item to upper case }
    theItem	:= UpperCase( Item );

    { Get ini file path and name }
    iniName	:= GetIniPathName;

		{ Point to the file and open it }
  	AssignFile( iniFile, iniName );
	  Reset( iniFile );

    repeat
		  { Read a line }
			readln( iniFile, curItem );

      { See if this is a section - Something like: [386Enh] }
      if (Pos('[',curItem)>0) and (Pos(']',curItem)>0) then
      	begin
        	{ Save it for our caller }
	      	Section	:= curItem;

          { Remove left brace }
          i	:= Pos('[',Section);
          Delete( Section, i, 1 );

          { Remove right brace }
          i	:= Pos(']',Section);
          Delete( Section, i, 1 );
				end
			else
      	{ Search for our item in this line }
	      Result	:= (Pos(theItem,UpperCase(curItem)) > 0);
		until Result or eof(iniFile);

    { If we found it then return the line }
    if Result then
    	Line		:= curItem
		else
    	Section	:= '';

	finally
  	CloseFile( iniFile );
	end;
end;


(*
 * Add a new item to a section of a .INI
 *
 * This function is useful for that wierd section of SYSTEM.INI
 * where there are multiple lines like:  device=...
 *
 * This function will add a new item.  It first renames the
 * .INI to .0 or something that works.  Then it copies the entire
 * file line by line looking for the specified section.  Once found
 * it adds the new Line as the first item of the specified section.
 *
 * Pass in the section that you expect to find Item in WITHOUT the
 * braces.  IE for [386Enh] pass in 386Enh.
 *
 * Finally pass in the ENTIRE line to be added.  This means
 * you must pass in the WHOLE line as in:  device=c:\windows\test.386
 *)
function	TEIniFile.AddItem(			const		Section		: string;
	                                const		Line			: string ) : boolean;
var
	iniFile	: text;
  outFile	: text;
  iniName	: string;
  outName	: string;
  savName	: string;
  iniItem	: string;
  curItem	: string;
  curSect	: string;
  theSect	: string;
  i				: integer;
  AddCnt	: integer;
begin
	try
  	{ Assume no additions }
    Result	:= false;
    AddCnt	:= 0;

    { Convert searched for items to upper case }
    theSect	:= UpperCase( Section );

    { Get ini file path and name }
    iniName	:= GetIniPathName;
    outName	:= iniName;
    savName	:= GetBackupIniPathName( iniName );

		{ Rename the input file }
    RenameFile( iniName, savName );

    { Open input file }
  	AssignFile( iniFile, savName );
	  Reset( iniFile );

    { Open the output file }
    AssignFile( outFile, outName );
    Rewrite( outFile );

    repeat
		  { Read a line }
			readln( iniFile, iniItem );

      { Upper case it }
      curItem	:= UpperCase(iniItem);

      { See if this is a section }
      if (Pos('[',iniItem)>0) and (Pos(']',iniItem)>0) then
      	begin
        	{ Save it - Uppercase for later compare }
	      	curSect	:= curItem;

          { Remove left brace }
          i	:= Pos('[',curSect);
          Delete( curSect, i, 1 );

          { Remove right brace }
          i	:= Pos(']',curSect);
          Delete( curSect, i, 1 );

          { Write it to the output file }
          writeln( outFile, iniItem );
				end
			else
      	begin
					{ See if this is the section they wanted }
          if (curSect=theSect) and (AddCnt=0) then
           	begin
             	{ Ok output our newline }
               writeln( outFile, Line );

              { Count it }
              inc(AddCnt);
            end;

					{ Output the line we read }
         	writeln( outFile, iniItem );
				end;
		until eof(iniFile);

	finally
  	CloseFile( outFile );
  	CloseFile( iniFile );
	end;

  { Return status }
  Result	:= (AddCnt > 0);
end;



(*
 * Replace a specific item in a .INI file.
 *
 * This function is useful for that wierd section of SYSTEM.INI
 * where there are multiple lines like:  device=...
 *
 * The standard WriteString member of this class will ADD a new
 * line but it will not allow a replace.
 *
 * This function will allow a replace.  It first renames the
 * .INI to .0 or something that works.  Then it copies the entire
 * file line by line looking for the specified line.  Once found
 * it substitutes Line for the found one.
 *
 * To make this function work correctly pass in the item you want
 * to replace in 'Item'.  This can be a substring but be careful to
 * make it unique.
 *
 * Pass in the section that you expect to find Item in WITHOUT the
 * braces.  IE for [386Enh] pass in 386Enh.
 *
 * Finally pass in the ENTIRE line to replace it with.  This means
 * you must pass in the WHOLE line as in:  device=c:\windows\test.386
 *)
function	TEIniFile.ReplaceItem(	const		Item			: string;
																	const		Section		: string;
	                                const		Line			: string ) : boolean;
var
	iniFile	: text;
  outFile	: text;

  iniName	: string;
  outName	: string;
  savName	: string;

  iniItem	: string;
  curItem	: string;
  theItem	: string;
  curSect	: string;
  theSect	: string;

  i				: integer;
  RepCnt	: integer;

begin
	try
  	{ Assume no replacements }
    Result	:= false;
    RepCnt	:= 0;

    { Convert searched for items to upper case }
    theItem	:= UpperCase( Item );
    theSect	:= UpperCase( Section );

    { Get ini file path and name }
    iniName	:= GetIniPathName;
    outName	:= iniName;
    savName	:= GetBackupIniPathName( iniName );

		{ Rename the input file }
    RenameFile( iniName, savName );

    { Open input file }
  	AssignFile( iniFile, savName );
	  Reset( iniFile );

    { Open the output file }
    AssignFile( outFile, outName );
    Rewrite( outFile );

    repeat
		  { Read a line }
			readln( iniFile, iniItem );

      { Upper case it }
      curItem	:= UpperCase(iniItem);

      { See if this is a section }
      if (Pos('[',iniItem)>0) and (Pos(']',iniItem)>0) then
      	begin
        	{ Save it - Uppercase for later compare }
	      	curSect	:= curItem;

          { Remove left brace }
          i	:= Pos('[',curSect);
          Delete( curSect, i, 1 );

          { Remove right brace }
          i	:= Pos(']',curSect);
          Delete( curSect, i, 1 );

          { Write it to the output file }
          writeln( outFile, iniItem );
				end
			else
      	begin
	      	{ Search for our item in this line }
		      if Pos(theItem,curItem) > 0 then
          	begin
							{ See if this is the section they wanted }
              if (curSect=theSect) and (RepCnt=0) then
              	begin
                	{ Ok output our newline if anything there }
                  if length(Line) > 0 then
	                  writeln( outFile, Line );

                  { Count it }
                  inc(RepCnt);
                end
							else
              	writeln( outFile, iniItem );
            end
					else
           	writeln( outFile, iniItem );
				end;

		until eof(iniFile);

	{ Set return }
  Result	:= (RepCnt > 0);

	finally
  	CloseFile( outFile );
  	CloseFile( iniFile );
	end;
end;


(*
 * Delete a specific item from a .INI file.
 *
 * This function is useful for that wierd section of SYSTEM.INI
 * where there are multiple lines like:  device=...
 *
 * This function will delete an item.  It first renames the
 * .INI to .0 or something that works.  Then it copies the entire
 * file line by line looking for the specified line.  Once found
 * it simply does not copy it to the new .INI.
 *
 * To make this function work correctly pass in the item you want
 * to delete in 'Item'.  This can be a substring but be careful to
 * make it unique.
 *
 * Pass in the section that you expect to find Item in WITHOUT the
 * braces.  IE for [386Enh] pass in 386Enh.
 *)
function	TEIniFile.DeleteItem(	const		Item		: string;
    														const		Section	: string ) : boolean;
begin
	{ Call ReplaceItem with a zero length string to do the work }
	Result	:= ReplaceItem(	Item, Section, '' );
end;



end.


