;;; cp850-iso.el --- Enable editing of cp850 texts on an ISO 8859-1 system

;; Copyright (C) 1996 Eberhard Mattes

;; Author: Eberhard Mattes <mattes@azu.informatik.uni-stuttgart.de>
;; Keywords: i18n

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Written by Eberhard Mattes.

;;; Code:

(require 'disp-table)

(defvar cp850-iso-enabled nil)

(defconst cp850-iso-table
  ;; CP850   ISO 8859-1                 ; POSTSCRIPT
  '((?\200 . ?\307)                     ; Ccedilla
    (?\201 . ?\374)                     ; udieresis
    (?\202 . ?\351)                     ; eacute
    (?\203 . ?\342)                     ; acircumflex
    (?\204 . ?\344)                     ; adieresis
    (?\205 . ?\340)                     ; agrave
    (?\206 . ?\345)                     ; aring
    (?\207 . ?\347)                     ; ccedilla
    (?\210 . ?\352)                     ; ecircumflex
    (?\211 . ?\353)                     ; edieresis
    (?\212 . ?\350)                     ; egrave
    (?\213 . ?\357)                     ; idieresis
    (?\214 . ?\356)                     ; icircumflex
    (?\215 . ?\354)                     ; igrave
    (?\216 . ?\304)                     ; Adieresis
    (?\217 . ?\305)                     ; Aring
    (?\220 . ?\311)                     ; Eacute
    (?\221 . ?\346)                     ; ae
    (?\222 . ?\306)                     ; AE
    (?\223 . ?\364)                     ; ocircumflex
    (?\224 . ?\366)                     ; odieresis
    (?\225 . ?\362)                     ; ograve
    (?\226 . ?\373)                     ; ucircumflex
    (?\227 . ?\371)                     ; ugrave
    (?\230 . ?\377)                     ; ydieresis
    (?\231 . ?\326)                     ; Odieresis
    (?\232 . ?\334)                     ; Udieresis
    (?\233 . ?\370)                     ; oslash
    (?\234 . ?\243)                     ; sterling
    (?\235 . ?\330)                     ; Oslash
    (?\236 . ?\327)                     ; multiply
    (?\237 . nil)                       ; florin
    (?\240 . ?\341)                     ; aacute
    (?\241 . ?\355)                     ; iacute
    (?\242 . ?\363)                     ; oacute
    (?\243 . ?\372)                     ; uacute
    (?\244 . ?\361)                     ; ntilde
    (?\245 . ?\321)                     ; Ntilde
    (?\246 . ?\252)                     ; ordfeminine
    (?\247 . ?\272)                     ; ordmasculine
    (?\250 . ?\277)                     ; questiondown
    (?\251 . ?\256)                     ; registered
    (?\252 . ?\254)                     ; logicalnot
    (?\253 . ?\275)                     ; onehalf
    (?\254 . ?\274)                     ; onequarter
    (?\255 . ?\241)                     ; exclamdown
    (?\256 . ?\253)                     ; guillemotleft
    (?\257 . ?\273)                     ; guillemotright
    (?\260 . nil)                       ; (gray)
    (?\261 . nil)                       ; (gray)
    (?\262 . nil)                       ; (gray)
    (?\263 . nil)                       ; (line)
    (?\264 . nil)                       ; (line)
    (?\265 . ?\301)                     ; Aacute
    (?\266 . ?\302)                     ; Acircumflex
    (?\267 . ?\300)                     ; Agrave
    (?\270 . ?\251)                     ; copyright
    (?\271 . nil)                       ; (line)
    (?\272 . nil)                       ; (line)
    (?\273 . nil)                       ; (line)
    (?\274 . nil)                       ; (line)
    (?\275 . ?\242)                     ; cent
    (?\276 . ?\245)                     ; yen
    (?\277 . nil)                       ; (line)
    (?\300 . nil)                       ; (line)
    (?\301 . nil)                       ; (line)
    (?\302 . nil)                       ; (line)
    (?\303 . nil)                       ; (line)
    (?\304 . nil)                       ; (line)
    (?\305 . nil)                       ; (line)
    (?\306 . ?\343)                     ; atilde
    (?\307 . ?\303)                     ; Atilde
    (?\310 . nil)                       ; (line)
    (?\311 . nil)                       ; (line)
    (?\312 . nil)                       ; (line)
    (?\313 . nil)                       ; (line)
    (?\314 . nil)                       ; (line)
    (?\315 . nil)                       ; (line)
    (?\316 . nil)                       ; (line)
    (?\317 . ?\244)                     ; currency
    (?\320 . ?\360)                     ; eth
    (?\321 . ?\320)                     ; Eth
    (?\322 . ?\312)                     ; Ecircumflex
    (?\323 . ?\313)                     ; Edieresis
    (?\324 . ?\310)                     ; Egrave
    (?\325 . ?\220)                     ; dotlessi
    (?\326 . ?\315)                     ; Iacute
    (?\327 . ?\316)                     ; Icircumflex
    (?\330 . ?\317)                     ; Idieresis
    (?\331 . nil)                       ; (line)
    (?\332 . nil)                       ; (line)
    (?\333 . nil)                       ; (block)
    (?\334 . nil)                       ; (block)
    (?\335 . ?\246)                     ; brokenbar
    (?\336 . ?\314)                     ; Igrave
    (?\337 . nil)                       ; (block)
    (?\340 . ?\323)                     ; Oacute
    (?\341 . ?\337)                     ; germandbls
    (?\342 . ?\324)                     ; Ocircumflex
    (?\343 . ?\322)                     ; Ograve
    (?\344 . ?\365)                     ; otilde
    (?\345 . ?\325)                     ; Otilde
    (?\346 . ?\265)                     ; mu
    (?\347 . ?\336)                     ; Thorn
    (?\350 . ?\376)                     ; thorn
    (?\351 . ?\332)                     ; Uacute
    (?\352 . ?\333)                     ; Ucircumflex
    (?\353 . ?\331)                     ; Ugrave
    (?\354 . ?\375)                     ; yacute
    (?\355 . ?\335)                     ; Yacute
    (?\356 . ?\257)                     ; macron
    (?\357 . ?\264)                     ; acute
    (?\360 . ?\255)                     ; hyphen
    (?\361 . ?\261)                     ; plusminus
    (?\362 . nil)                       ; (double underscore)
    (?\363 . ?\276)                     ; threequarters
    (?\364 . ?\266)                     ; paragraph
    (?\365 . ?\247)                     ; section
    (?\366 . ?\367)                     ; divide
    (?\367 . ?\270)                     ; cedilla (ogonek?)
    (?\370 . ?\260)                     ; degree (ring?)
    (?\371 . ?\250)                     ; dieresis
    (?\372 . ?\267)                     ; periodcentered
    (?\373 . ?\271)                     ; onesuperior
    (?\374 . ?\263)                     ; threesuperior
    (?\375 . ?\262)                     ; twosuperior
    (?\376 . nil)                       ; (centered square)
    (?\377 . ?\240)                     ; (NBS)
    ))

(defun cp850-iso-display-enable ()
  "Enable translation of code page 850 to ISO 8859-1 for the display."
  (mapcar (function (lambda (pair)
                      (standard-display-ascii (car pair)
                                              (if (cdr pair)
                                                  (char-to-string (cdr pair))
                                                "{?}"))
                      nil))
          cp850-iso-table))

(defun cp850-iso-display-disable ()
  (standard-display-default 128 159)
  (standard-display-8bit 160 255))

(defun cp850-iso-keyboard-enable ()
  "Enable translation of ISO 8859-1 to code page 850 for the keyboard."
  (setq keyboard-translate-table (make-vector 256 nil))
  (mapcar (function (lambda (pair)
                      (if (cdr pair)
                          (aset keyboard-translate-table
                                (cdr pair) (car pair)))
                      nil))
          cp850-iso-table))

(defun cp850-iso-keyboard-disable ()
  "Disable translation of ISO 8859-1 to code page 850 for the keyboard."
  (setq keyboard-translate-table nil))

(defun cp850-iso (arg)
  "Toggle translation for editing code page 850 texts on an ISO 8859-1 system.
With prefix argument, enable translation iff arg is positive."
  (interactive "P")
  (if (or (<= (prefix-numeric-value arg) 0)
	  (and (null arg)
	       keyboard-translate-table))
      (progn
        (cp850-iso-keyboard-disable)
        (cp850-iso-display-disable))
    (cp850-iso-keyboard-enable)
    (cp850-iso-display-enable)))

(provide 'cp850-iso)

;;; cp850-iso.el ends here
