/*
 * FILE parse.c 
 * 
 * Copyright 1996, Hanns Kucer
 * 
 */

/*********************** INCLUDES */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "lang.h"
#include "common.h"
#include "gtc.h"

#define LINES_PER_ROUND 1000

/*********************** DATA */

token token_head=NULL;    /* empty head of token list */
token token_tail=NULL;    /* last token of list       */

token refer_head=NULL;    /* referenced labels        */
token defin_head=NULL;    /* defined labels           */

key_t *keytable=NULL;
key_t main_key;
key_t keyword_keytable[NUM_KEYWORDS]; /* fast lookup table */
int key_table_len=0;


/*********************** PROTOTYPES */

int readtokens(FILE *, int);
void writeoutput(FILE*, int);
void writeheader(FILE*);

int getnum(char*,int,int*);
char *getstring(FILE *);

void puttasmstring(FILE *, char *);
token puttasmCPU0(FILE*, token);
token puttasmCPU1(FILE*, token);
token puttasmCPU2(FILE*, token);
token puttasmCPU3(FILE*, token);
token puttasmCPUX(FILE*, token);
char *gettasmsize(token);

int isseparator(char);
int myisspace(char);

void token_init(void);
token token_new(void);
void token_free(token);
void token_killtokenlist();
void token_insertlast(char *, int);
void token_insert(token, char *, int);
void token_out(token);
void token_delnext(token);
void token_delprev(token);
token token_resolve(token);
void token_entypenum(token);

token token_DIR(token);
token token_CPU(token);
token token_SPEC(token);
token token_LABL(token);

void key_init(void);
key_t key_newkey(char*);
char *key_getsym(key_t);
key_t key_getkey(int);
key_t key_insertsym(char *);
int key_stringiskeyw(char *);
int key_group(token);
void key_printkeys(void);

extern unsigned int hashstring(char *);

void label_insert(token, token);
void label_tryinsert(token, token);
int label_inlist(token, token);

extern char *infname;
extern char *outfname;
extern int mem_max;
extern int mem_diff;


/*********************** PUBLIC CODE */
void p_parse(FILE *inf, FILE *out) {  
   int lines=0, l=LINES_PER_ROUND;
   token_init();      /* initializes token, label, scope */

   main_key=key_insertsym("_main"); /* so we can check ifdef _main later */
   
   while(l==LINES_PER_ROUND){
      l=readtokens(inf,lines+1);
      token_resolve(token_head->next);
      writeoutput(out, lines);
      token_killtokenlist();
      lines+=l;
   }
   writeheader(out);
}

/***************** PRIVATE CODE **********************************/

int readtokens(FILE *inf, int lines){
   static char cn=EOF;
   char tok[MAX_TOKENLEN+1], *tmp;
   int i=0, lno=lines, type, size; 
   
   if(cn==EOF) cn=fgetc(inf); /* inherit the old cn if any */
   
   while(cn!=EOF && lno!=(lines+LINES_PER_ROUND)) {
      if(i>=MAX_TOKENLEN)
      	pc_error(FAT_ERR, "token too long\n", NULL); 
      if(i!=0 && cn=='.') cn='_';  /* dots not allowed within labels */
      if(myisspace(cn) || isseparator(cn)) {
	 tok[i]=0;         /* terminate the token */ 
	 
	 if(i>2 && (size=(!key_stringiskeyw(tok))*tok[i-1])) {
	    switch(size) {
	     case 'l': type=T_DWORD; tok[--i]=0; break;
	     case 'w': type=T_WORD; tok[--i]=0; break;
	     case 'b': type=T_BYTE; tok[--i]=0; break;
	     default: type=T_NONE;
	    }
	    if(type!=T_NONE && !key_stringiskeyw(tok)) {
	       tok[i++]=size;
	       type=T_NONE;
	    }
	    token_insertlast(tok, lno);
	    token_tail->type=type;        /* default is notype */
	    token_entypenum(token_tail);  /* could be numeric  */
	 }
	 else if(i!=0) {
	    token_insertlast(tok,lno);
	    token_entypenum(token_tail);
	 }
	 else if(cn=='"') {
	    token_insertlast(tmp=getstring(inf), lno);
	    token_tail->type=T_STR;
	    mfree(tmp, strlen(tmp)+1);
	    cn=fgetc(inf);
	 }
	 
	 if(isseparator(cn)) {
	    tok[0]=cn;
	    tok[1]=0;
	    token_insertlast(tok, lno);   /* no type */
	    cn=fgetc(inf); /* get past the separator */
	 }
	 while(cn!=EOF && myisspace(cn)) {
	    if(cn=='\n') lno++;
	    cn=fgetc(inf);
	 }
	 i=0;
      }
      else {
	 tok[i++]=cn;
	 cn=fgetc(inf);
      }
   } /* while */
   
   lno-=lines;
   token_insertlast("TERM", -1);  /* terminator token to prevent various */ 
   return lno;                    /* routines from looking past end of list */
}

void writeoutput(FILE *out, int lno) {
   token tk=token_head->next;
   
   while(tk!=token_tail) {  /* token_tail is the terminator */
      if(tk->lno!=lno) fprintf(out, "\n");
      lno=tk->lno;
      switch(key_group(tk)) {
       case T_DIR:
	 if(GETKEY(tk->key)==T_GLOBAL) {
	    fprintf(out, "\t%s\t%s:NEAR", tasmwords[GETKEY(tk->key)], key_getsym(tk->next->key));
	    tk=tk->next;
	    break;
	 }
	 fprintf(out,"\t%s\t", tasmwords[GETKEY(tk->key)]);
	 if(GETKEY(tk->key)==T_SPACE) 
	  token_insert(tk->next,  " DUP(0)", tk->next->lno);
	 break;
       case T_CPU:
	 if(ISCPU0(GETKEY(tk->key))) tk=puttasmCPU0(out, tk);
	 else if(ISCPU1(GETKEY(tk->key))) tk=puttasmCPU1(out, tk);
	 else if(ISCPU2(GETKEY(tk->key))) tk=puttasmCPU2(out, tk);
	 else if(ISCPUX(GETKEY(tk->key))) tk=puttasmCPUX(out, tk);
	 else pc_warning("Wrong operand type for", tk);
	 break;
       case T_REG:
       case T_SPEC:
	 if(GETKEY(tk->key)==T_COMM || GETKEY(tk->key)==T_LCOMM) {
	    fprintf(out, "\tGLOBAL\t%s:NEAR\n\tALIGN\t4\n%s%s\n\tDB\t%s DUP(0)",
		    key_getsym(tk->next->key), 
		    key_getsym(tk->next->key),
		    key_getsym(tk->next->next->key),
		    key_getsym(tk->next->next->next->key));
		    tk=tk->next->next->next;
	 }
	 else if(GETKEY(tk->key)==T_FILE) 
	   fprintf(out, "%s \"%s\"\n",tasmwords[GETKEY(tk->key)], infname);
	 else
	   fprintf(out, "%s", tasmwords[GETKEY(tk->key)]);
	 break;
       default:
	 if(tk->type==T_STR)
	   puttasmstring(out, key_getsym(tk->key));
	 else
	   fprintf(out, "%s", key_getsym(tk->key));
      }
      tk=tk->next;
   }   
   fflush(out);
}

void writeheader(FILE *out) {
   token tk;
   char *sym;
   void *data;
   unsigned long len;
   int ok;

   /* say we're done ! */
   fprintf(out, "\nEND\n");
   fflush(out);
   len=ftell(out);
   rewind(out);
   
   ok=(len==fread(data=mmalloc(len), 1, len, out));
   rewind(out);
   
   /* put the header stuff */
   if(ok) {
      fprintf(out,";*********************************************************\n");
      fprintf(out,"; %s, Generated by GTC, (C) 1996 Hanns Kucer\n", outfname);
      fprintf(out,";*********************************************************\n\n");
      
         /****** put model directives    *********/
      fprintf(out,"\t.386P\n\t.MODEL FLAT\n\t.CODE\n\n");
      
      tk=refer_head->next;
      while(tk!=NULL) {
	 if(tk->key->def!=T_DEFINED) {
	    sym=key_getsym(tk->key);
	    fprintf(out, "\tEXTRN \t%s:NEAR\n", sym);
	    if(sym[0]!='_') pc_warning("Missing label ??", tk);
	 }
	 tk=tk->next;
      }
      fflush(out);
      
      if(fwrite(data, 1, len, out)!=len)
      	pc_error(WR_ERR, outfname, NULL); 
   }
   else 
     pc_error(RD_ERR, outfname, NULL);
   mfree(data, len);
}

/*******************************************************************
                     TOKEN STUFF
********************************************************************/
void token_init(void){
   key_init();

   token_head=token_new();
   token_head->key=key_insertsym("TOKEN_HEAD");  /* must have a key */
   token_tail=token_head;
   
   refer_head=token_new();
   refer_head->key=key_insertsym("REFER_HEAD");
   defin_head=token_new();
   defin_head->key=key_insertsym("DEFIN_HEAD");
}

token token_new(void){
   token tmp_tk;

   tmp_tk=(token)mmalloc(sizeof(struct token_struct));
   return tmp_tk;
}

void  token_free(token tk){
   mfree(tk, sizeof(struct token_struct));
}

/* delete all head & all it's sucessors */ 
void token_killtokenlist(void) {
   /* now, reset the list */
   token_head->next=NULL;
   token_head->prev=NULL;
   token_tail=token_head;
}

void token_insert(token tk, char *s, int lno){
/* create new token with s, inserted after tk */
   token tmp_tk;

   if(tk!=NULL && s!=NULL && s[0]!=0) {
      tmp_tk=tk->next;
      tk->next=token_new();
      tk->next->next=tmp_tk;
      if(tmp_tk!=NULL) tmp_tk->prev=tk->next;
      tk->next->prev=tk;

      tk->next->key=key_insertsym(s);
      tk->next->lno=lno;
   }
   else pc_error(FAT_ERR, "internal error 1", NULL);
}

void token_insertlast(char *s, int lno){
/* create new token with s, inserted last in list */
   token_insert(token_tail, s, lno);
   token_tail=token_tail->next;
}

void token_out(token tk){
/* unlink the token tk */
     if(tk->prev!=NULL)
        tk->prev->next=tk->next;
     if(tk->next!=NULL)
        tk->next->prev=tk->prev;
}

void token_delnext(token tk){
/* delete the token following tk */
     token tmp_tk;

     if(tk->next!=NULL){
        tmp_tk=tk->next;
        tk->next=tk->next->next;
        token_free(tmp_tk);
        if(tk->next!=NULL) tk->next->prev=tk;
     }
}

void token_delprev(token tk){
/* delete the token following tk */
     token tmp_tk;

     if(tk->prev!=NULL){
        tmp_tk=tk->prev;
        tk->prev=tk->prev->prev;
        token_free(tmp_tk);
        if(tk->prev!=NULL) tk->prev->next=tk;
     }
}

/* try to extract the numerical type of token */
void token_entypenum(token tk) {
   char *sym;
   int foo;
   
   if(tk->type==T_NONE && key_group(tk)==T_LABL) {
      if(getnum(sym=key_getsym(tk->key),16, &foo)) {
	 tk->type=T_NUM;
	 sprintf(sym,"0%xH",foo); /* there is room for this */
      }
      else if(getnum(sym,10, &foo))
      	tk->type=T_NUM;
   }
}

token token_resolve(token tk){
   if(tk!=NULL && tk!=token_tail) {
      switch(key_group(tk)) {
       case T_UDEF:
	 pc_error(FAT_ERR, "internal error 2", NULL);
	 return NULL;
       case T_DIR:
	 return token_resolve(token_DIR(tk));
       case T_SPEC:
	 return token_resolve(token_SPEC(tk));
       case T_CPU:
	 return token_resolve(tk->next);
       case T_REG:
	 return token_resolve(tk->next);
       case T_LABL:
	 return token_resolve(token_LABL(tk));
      }
   }
   else if(tk==NULL) pc_error(FAT_ERR, "internal error 3", NULL);
   return NULL;
}

token token_CPU(token tk) {
   return tk->next;
}

token token_DIR(token tk) {   
   switch(GETKEY(tk->key)) {
    case T_ALIGN:
      if(tk->next->next!=NULL && GETKEY(tk->next->next->key)==T_COMMA) {
	 token_delnext(tk->next);   /* skip Y in ".align X,Y" */ 
	 token_delnext(tk->next);
      }
    default:
      return tk->next;
   }
}

token token_SPEC(token tk) {
   token tmp=tk;
   switch(GETKEY(tk->key)) {
    case T_DATA:
    case T_CODE:
      tmp=tk->next;
      token_delnext(tk->prev);
      return tmp;
    case T_FILE:
      token_delnext(tk);
      return tk->next;
    case T_COMM:
    case T_LCOMM:
      tk->next->next->key=key_getkey(T_COLON);  /* make it look like a label */
      return tk->next;
    case T_DOLLAR:
      if((tk=tk->next)->type==T_NUM || GETKEY(tk->key)==T_MINUS)
      	token_delprev(tk);
      return tk;
    case T_LBRACE:
      if(GETKEY(tk->next->key)==T_COMMA) token_delnext(tk);
      while(GETKEY((tk=tk->next)->key)!=T_RBRACE) {
	 if(GETKEY(tk->key)==T_COMMA) {
	    if(tk->next->type==T_NUM) tk->key=key_getkey(T_SCALE);
	    else tk->key=key_getkey(T_PLUS);
	 }
      }
    default:
      return tk->next;
   }
}

token token_LABL(token tk) {
   token tkn=tk->next, tkp=tk->prev;
   
   /* used to check if tkp->key==T_GLOBAL aswell */
   if((tkn==NULL || GETKEY(tkn->key)!=T_COLON) && (tk->type==T_NONE) && (tk->key->def==0)){
      tk->key->def=T_REFERED;
      token_insert(refer_head, key_getsym(tk->key), tk->lno);
   }
   else if(tkn!=NULL && GETKEY(tkn->key)==T_COLON && tk->type==T_NONE)
     tk->key->def=T_DEFINED; 
   return tkn;
}

/*******************************************************************
                     KEY STUFF
********************************************************************/
void key_init(void) {
   int i;
   keytable=mmalloc(HASH_SIZE*sizeof(struct key_struct));
   for(i=0; i<NUM_KEYWORDS; i++)
     keyword_keytable[i]=key_insertsym(keywords[i]);
}

key_t key_newkey(char *sym) {
   key_t tmp;
   
   tmp=mmalloc(sizeof(struct key_struct));
   tmp->sym=strcpy(mmalloc(strlen(sym)+1),sym);
   tmp->key=key_table_len;
}

char *key_getsym(key_t key) {
   return key->sym;
}

key_t key_getkey(int key) {
   if(ISKEYW(key)) return keyword_keytable[key];
   else pc_error(FAT_ERR, "internal error 4", NULL);
}

key_t key_insertsym(char *sym) {   
   unsigned int k;
   key_t key, tmp;
   
   k=hashstring(sym)%(HASH_SIZE);   
   tmp=key=keytable[k];
   
   while(key!=NULL && strcmp(key->sym, sym)!=0){
      tmp=key;
      key=key->next;
   }
   /* key already in table ? */
   if(key!=NULL) return key;       
   
   /* else we got a new key...  */           
   if(keytable[k]==NULL) tmp=keytable[k]=key_newkey(sym);
   else tmp=(tmp->next=key_newkey(sym));
   key_table_len++;
   return tmp;
}

int key_stringiskeyw(char *str) {
   key_t key;
   
   key=keytable[hashstring(str)%(HASH_SIZE)];
   while(key!=NULL) {
      if(ISKEYW(key->key) && strcmp(key->sym, str)==0)
      	return 1;
      key=key->next;
   }
   return 0;
}

void key_printkeys(void) {
   key_t key;
   int i;
   
   for(i=0; i<HASH_SIZE; i++) {
      key=keytable[i];
      printf("\nIN %d:", i);
      while(key!=NULL) {
	 printf("%s,", key->sym);
	 key=key->next;
      }
   }
}

int key_group(token tk){   
   if(tk==NULL) 
     return T_UDEF;
   else if(ISUDEF(GETKEY(tk->key)))
     return T_UDEF;
   else if(ISDIR(GETKEY(tk->key)))
     return T_DIR;
   else if(ISSPEC(GETKEY(tk->key)))
     return T_SPEC;
   else if(ISCPU(GETKEY(tk->key)))
     return T_CPU;
   else if(ISREG(GETKEY(tk->key)))
     return T_REG;
   else
     return T_LABL;
}


/*******************************************************************
                     STRING STUFF
********************************************************************/

/* getnumber of base 1..16, (must have prefix "0x" when base 16) */ 
int getnum(char *str, int base, int *num){
   int n=0, i=0, sign=1;
   
   if(str==NULL || strlen(str)==0)
    return 0;
   else if(*str=='-') { sign=-1; str++; }
   else if(*str=='+') str++;
   
   if(strlen(str)<1) 
    return 0;
   else if(strlen(str)>=3 && base==16 && 
	   str[0]=='0' && (str[1]=='x' || str[1]=='X'))
    str+=2;
   else if(base==16) 
    return 0;
   
   while((base==16 && isxdigit(*str)) || isdigit(*str)) {
      if(isdigit(*str)) n=n*base+((*str)-'0');
      else if(base==16 && islower(*str)) n=n*base+(10+(*str)-'a');
      else if(base==16) n=n*base+(10+(*str)-'A');
      else return 0; 
      str++;
      i++;
   }
   *num=(n*sign);
   return i!=0;
}

char *getstring(FILE *inf){
   int i=0;
   char *tmp, cn=0,co=0;

   while(!((cn=fgetc(inf))=='"' && co!='\\')) {
      co=cn;
      i++;
   }
   fseek(inf, -(++i), SEEK_CUR);
   tmp=(char*)mmalloc(i+2);
   fread(tmp+1,1,i,inf);

   tmp[0]='"';
   tmp[i+1]='\0';
   return tmp;
}

void puttasmstring(FILE *out, char *str) {
   int num,alnum=0;
   
   str++;
   while(*str!='"') {
      if(*str=='\\') {
	 if(getnum(++str, 8, &num)) {
	    if(alnum==1) fprintf(out, "\",%d", num);
	    else if(alnum==2) fprintf(out, ",%d", num);
	    else fprintf(out,"%d",num);
	    while(isdigit(*str)) str++;
	 }
	 else {
	    if(alnum==1) fprintf(out, "\",'%c'", *str);
	    else if(alnum==2) fprintf(out,",'%c'", *str);
	    else fprintf(out, "'%c'", *str);
	    str++;
	 }
	 alnum=2;
      }
      else {
	 if(alnum==1) fputc(*str, out);
	 else if(alnum==2) fprintf(out, ",\"%c", *str);
	 else fprintf(out, "\"%c", *str);
	 alnum=1;
	 str++;
      }
   }
   if(alnum==1) fputc('"', out);
}

char *gettasmsize(token tk) {
   if(tk->type==T_DWORD) return "dword ptr ";
   else if(tk->type==T_WORD) return "word ptr ";
   else if(tk->type==T_BYTE) return "byte ptr ";
   else if(GETKEY(tk->next->key)==T_STAR) return "dword ptr ";
   else if(!ISBRANCH(GETKEY(tk->key))) pc_warning("wrong operand type",tk);
   return "";
}
   
/* output an sole instruction the tasm way */
token puttasmCPU0(FILE *out, token tk) {
   switch(GETKEY(tk->key)) {
    case T_REP:
    case T_REPE:
    case T_REPZ:
    case T_REPNE:
    case T_REPNZ:
      tk->next->lno=tk->lno;    /* do cmpsb etc. on the same line */     
      fprintf(out, "\t%s", tasmwords[GETKEY(tk->key)]);
      return tk;
   }
   fprintf(out, "\t%s\t", tasmwords[GETKEY(tk->key)]);
   return tk;
}

/* output a unary instruction the tasm way */
token puttasmCPU1(FILE *out, token tk) {
   token tmp=tk->next;
   int key=GETKEY(tk->key);
   int type=tk->type;
   
   fprintf(out, "\t%s\t", tasmwords[GETKEY(tk->key)]);
   while(tmp->lno==tk->lno && GETKEY(tmp->key)!=T_LBRACE) tmp=tmp->next;
   if(tmp->lno==tk->lno || (tmp->prev==tk->next && ISLABL(tk->next))) 
     fprintf(out, gettasmsize(tk));
   if((key==T_PUSH || key==T_PUSH) && (type!=T_NONE && type!=T_DWORD))
     pc_warning("Push/Pop other than 32bit", tk);
   return tk;
}
   
/* output a binary instruction the tasm way */
token puttasmCPU2(FILE *out, token tk) {
   token tmp=tk->next, s=tk->next;
   int ovr1=0, ovr2=0;
   key_t key, dest;
   int type=0, al=0, dw, save;
   char ins[60];
   
   /* swap operands */
   while(GETKEY(tmp->key)!=T_COMMA) {
      if(GETKEY(tmp->key)==T_LBRACE) ovr2=1;
      tmp=tmp->next; 
   }
   tk->next=tmp->next;   
   tmp->next=s;
   s=tk->next;
   while(s->lno==tk->lno) {
      if(GETKEY(s->key)==T_LBRACE) ovr1=1;
      s=s->next;
   }
   s->prev->next=tmp;
   tmp->prev->next=s;
   
   /* now tk->next == op1 and tmp->next == op2 */
   
   key=tk->key;
   if(GETKEY(key)==T_MOVSB || GETKEY(key)==T_MOVZB) {
      dest=tk->next->key;
      if(!ISREG(GETKEY(dest))) pc_warning("Destination is not a register", tk);
      al=GETKEY(tmp->next->key)==T_AL;
      save=1*(GETKEY(dest)!=T_AX && GETKEY(dest)!=T_EAX);
      dw=2*((type=tk->type)==T_DWORD);
      if(GETKEY(key)==T_MOVSB) {
	 if(save) fprintf(out, "\tPUSH\teax\n");
	 switch(save+dw) {
	  case 0: sprintf(ins, "\n\tCBW"); break;
	  case 1: sprintf(ins, "\n\tCBW\n\tMOV\t%s,ax\n\tPOP\teax", tasmwords[GETKEY(dest)]); break;
	  case 2: sprintf(ins, "\n\tCBW\n\tMOVSX\teax,ax"); break;
	  case 3: sprintf(ins, "\n\tCBW\n\tMOVSX\t%s,ax\n\tPOP\teax", tasmwords[GETKEY(dest)]); break;
	 }
      }
      else {
	 if(save) fprintf(out, "\tPUSH\teax\n");
	 switch(save+dw) {
	  case 0: sprintf(ins, "\n\tXOR\tah,ah"); break;
	  case 1: sprintf(ins, "\n\tXOR\tah,ah\n\tMOV\t%s,ax\n\tPOP\teax", tasmwords[GETKEY(dest)]); break;
	  case 2: sprintf(ins, "\n\tXOR\tah,ah\n\tMOVZX\teax,ax"); break;
	  case 3: sprintf(ins, "\n\tXOR\tah,ah\n\tMOVZX\t%s,ax\n\tPOP\teax", tasmwords[GETKEY(dest)]); break;
	 }
      }
      if(al) ins[0]=' '; /* skip initial '\n' when no initial mov */
      token_insert(tmp->prev, ins, tk->lno);
      tk->next->key=key_getkey(T_AL);
      tk->key=key_getkey(T_MOV);
      tk->type=T_BYTE;
   }
   else if(GETKEY(key)==T_MOVZW || GETKEY(key)==T_MOVSW) {
      type=tk->type;
      tk->type=T_WORD;
   }
   else if(GETKEY(key)==T_MOV &&
	   (ISREG(GETKEY(tk->next->key)) && 
	    tk->next->key==tmp->next->key))
     return tmp->prev;        /* don't do "mov reg,reg" */  
   
   if(al) return tmp->prev;   /* skip initial move if AL is source opr */
   if(type==T_BYTE) {         /* type is zero if not movsb, movzw, etc */
      tk->key=key;            /* put back the original instruction     */
      pc_warning("Can't handle byte dest for", tk);
   }
   
   fprintf(out, "\t%s\t", tasmwords[GETKEY(tk->key)]);
   
   /* tk->next == op1 and tmp->next == op2 */
   if(!(ISREG(GETKEY(tk->next->key)) || ISREG(GETKEY(tmp->next->key)))) {
      if(ovr1 || ISLABL(tmp->prev) || ISLABL(tk->next)) 
      	fprintf(out, "%s", gettasmsize(tk));
      if(ovr2) 
      	token_insert(tmp, gettasmsize(tk), tk->lno);
   }
   else {
      if(ovr1 || ISLABL(tk->next))
      	fprintf(out, "%s", gettasmsize(tk));
      if(ovr2 || ISLABL(tmp->next))
      	token_insert(tmp, gettasmsize(tk), tk->lno);
   }
   return tk;
}

/* output a trinary instruction the tasm way
 * (can only handle imul-3 for the moment */
token puttasmCPU3(FILE *out, token tk) {
   token im, cm, tmp;
   
   /* error if not imul-3 */
   if(GETKEY(tk->key)!=T_IMUL) 
    pc_error(FAT_ERR, "Can only do 3 operands for imul", tk);
   
   tmp=tk;
   token_out((im=tk->next));   /* always an immediate for imul-3 */
   token_out((cm=tk->next));   /* the first comma */
   puttasmCPU2(out, tk);       /* do an imul-2 */
   while(tmp->next->lno==tk->lno) tmp=tmp->next;    /* get next instruction */
   token_insert(tmp, key_getsym(im->key), im->lno); /* insert the immedate */
   token_insert(tmp, key_getsym(cm->key), cm->lno); /* insert the comma */ 
   return tk;
}

/* for those who's number of opr varies */
token puttasmCPUX(FILE *out, token tk) {
   int i=1;
   token tmp=tk->next;
   
   while(tmp->lno==tk->lno) {
      if(GETKEY(tmp->key)==T_COMMA) i++;
      tmp=tmp->next;
   }
   if(i==1) return puttasmCPU1(out, tk);
   else if(i==2) return puttasmCPU2(out, tk);
   else if(i==3) return puttasmCPU3(out, tk);
   pc_error(FAT_ERR, "Can't handle the number of operands", tk);
   return tk;
}

int isseparator(char c){
   int i=0;
   while(i<NUM_SEPARATORS)
     if(c==separators[i++]) return 1;
   return 0;
}

/* an 'isspace' that handles '\r' correctly 
*  which don't seem to happen under DOS !
*/
myisspace(char c) {
   return ((isspace(c)!=0) || c=='\r');
}
