/*
 * FILE gtc.c 
 * 
 * Copyright 1996, Hanns Kucer
 * 
 */

/************************ INCLUDES */
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "parse.h"
#include "gtc.h"


/************************ PROTOTYPES */
void pc_init(int, char*[]);
void pc_exit(void);

/********************* TYPES */


/************************ DATA */
int pc_errno=0;
int pc_warno=0;
int quiet_mode=0;

char *Usage_string= \
"gtc, GNU- to TASM-assembly Converter 0.0, (C) Hanns Kucer 1996\n\r" \
"     Usage: gtc [-h]|[<infile> [-q] [-o <outfile>]]\n\r" \
"            -q     suppress non-errors upon success\n\r" \
"            -o     output file; Default: infile.asm\n\r" \
"            -h     give this help and then exit(0)";

char *errtypes[] = {
   "bad option '%s' (use -h for help)\n",
   "can't open '%s'\n",
   "%s\n",                         /* fatal error   */
   "can't write to '%s'\n",
   "can't read from '%s'\n",
   "%s\n",                         /* general error */
};

FILE *in, *out;
char *infname=NULL;
char *outfname=NULL;

int mem_diff=0;
int mem_max=0;

/************************ CODE */
main(int argc, char *argv[]) {
   pc_init(argc, argv);
   p_parse(in, out);
   pc_exit();
}

void pc_init(int argc, char *argv[]){
   int i, o_flag=0;
   char *arg, *tmp_out=NULL, *tmp_in=NULL;
   
   if(argc>1){
      for(i=1; i<=argc-1; i++) {
	 arg=argv[i];
	 if(*arg=='-') {
	    if(strcmp(arg+1,"h")==0) {
	       printf("%s\n", Usage_string);
	       exit(0);
	    }
	    else if(!quiet_mode && strcmp(arg+1,"q")==0)
	      quiet_mode=1;
	    else if(!o_flag && strcmp(arg+1,"o")==0)
	      o_flag=1;
	    else
	      pc_error(ARG_ERR, arg, NULL);
	 }
	 else if(tmp_out==NULL && o_flag)
	   tmp_out=arg;
	 else if(tmp_in==NULL && !(o_flag^(tmp_out!=NULL)))
	   tmp_in=arg;
	 else
	   pc_error(FAT_ERR, "too many arguments", NULL);
      }
   }
   else pc_error(FAT_ERR, Usage_string, NULL);
   
   if(tmp_in==NULL || (o_flag && tmp_out==NULL)) 
     pc_error(FAT_ERR, "missing argument", NULL);
   
   if(o_flag) sprintf(outfname=mmalloc(strlen(tmp_out)+5), "%s.asm", tmp_out);
   else sprintf(outfname=mmalloc(strlen(tmp_in)+5), "%s.asm", tmp_in);
   sprintf(infname=mmalloc(strlen(tmp_in)+3),"%s.s", tmp_in);
   if((out=fopen(outfname, "w+"))==NULL)
     pc_error(FOP_ERR, outfname, NULL);
   if((in=fopen(infname, "r"))==NULL) {
      tmp_in=infname;
      infname=NULL;
      pc_error(FOP_ERR, tmp_in, NULL);
   }
}

void pc_exit(){
   fclose(in);
   fclose(out);
   
   if(!quiet_mode || pc_errno!=0) {
      printf("%d error(s), %d warning(s)", pc_errno, pc_warno);
      if(pc_errno!=0) {
	 printf(", deleting %s\n", outfname);
	 remove(outfname);
      }
      else
      	printf(", output written to %s\n", outfname);
   }
   mfree(infname, MAX_LINE);
   mfree(outfname, MAX_LINE);
   exit(pc_errno);
}

void pc_warning(char *estr, token tk) {
   if(!quiet_mode) {
      fprintf(stderr, "%s:%d: warning: %s for '%s'\n",
	      infname,
	      tk->lno, 
	      estr,
	      key_getsym(tk->key));
      fflush(stderr);
   }
   pc_warno++;
}
   
void pc_error(int etype, char *estr, token tk){
   if(infname!=NULL) fprintf(stderr, "%s: ", infname);
   if(tk!=NULL) fprintf(stderr, "%d: ", tk->lno);
   if(ERR_ISFATAL(etype)){
      fprintf(stderr, errtypes[etype], estr);
      exit(pc_errno+1);
   }
   fprintf(stderr, errtypes[etype], estr);
   fflush(stderr);
   if((pc_errno++)>MAX_ERRS) pc_error(FAT_ERR, "too many errors", NULL);
}

void *mmalloc(int nb){
   void *m;
   
   if((m=calloc(nb, 1))==NULL) 
     pc_error(FAT_ERR, "out of memory", NULL);
   else {
      mem_diff=mem_diff+nb;
      if(mem_diff>=mem_max) mem_max=mem_diff;
   }
   return m;
}

void mfree(void *m, int nb){
   /* mem_diff=mem_diff-nb; */
   /* free(m);              */
}
