///////////////////////////////////////////////////////////////////////
//
// default.ch - Default arguments for Clipper, ala C++, and DEFAULT TO.
//
// Copyright (C) 1994 All rights reserved. Paul Long
//
// CompuServe: 72607,1506
// Internet: Paul_Long@ortel.org
//           pci!plong@ormail.intel.com
//
// Intro
// -----
// If you #include this header file at the top of a Clipper 5 .PRG file,
// you can specify default values alongside their respective parameters
// in the function statement like this:
//    FUNCTION Skip(nRecs := 1, bFilter := {|| .T.}, xStart, xEnd := xStart)
// This saves keystrokes and reduces the likelihood of introducing an
// error due to repeating parameter names in the function statement and
// then in subsequent initialization code.
//
// Syntax
// ------
// Using the convention used in the Clipper manuals, here is the syntax
// of the function statement that incorporates the new default-argument
// syntax.
//    [STATIC] FUNCTION <idFunction>[([ <parameter> [:= <default>]
//       [, <parameter2> [:= <default2>] ...] ])]
// Basically, code your function statements the way you always have,
// except to specify a default value, follow the parameter with the
// inline-assignment operator (:=) and an expression.
//
// Semantics
// ---------
// All parameters contain their as-called values before the default-to
// code is evaluated, which is performed left to right.  For parameters
// that have defaults specified, the default value is assigned to the
// parameter if the parameter is NIL; otherwise, the parameter is not
// modified.  This should only be of concern if a default expression
// contains a reference to a parameter, as is correctly done in the
// above skip function.  Below is an example of how this might trip you
// up:
//    DoIt()   // Prints NIL 1
//    FUNCTION DoIt(nThis := nThat, nThat := 1)
//    ? nThis, nThat
//    RETURN NIL
// Here, the first parameter is assigned its default before the second
// parameter has been assigned its default.  The result is that nThis
// has the value of NIL--the value of nThat before it was assigned its
// default of 1.  This is probably not what was intended.
//
// Caveats
// -------
// 1. This syntax and concept is foreign to Clipper, although there is
// precedence for this in C++.
// 2. For functions with a parameter list--even if no default values are
// specified--a dummy, otherwise useless local variable with the name
// xDFDmy is created.  It is never referred to by name, so if, in the
// remote likelihood that there is a naming conflict with one of your
// variables, change xDFDmy to something else.
// 3. The long and short forms of the FUNCTION and PROCEDURE keywords
// and the long form of the STATIC keyword are recognized, but not
// anything else. Specifically, the following are recognized:
//    STATIC FUNC FUNCTION PROC PROCEDURE
// But, for example, the following are not:
//    STAT FUNCT PROCEDU
//
///////////////////////////////////////////////////////////////////////

// Write a dummy local variable that "hosts" the default-to code
// in its initializer expression.  If the argument list is empty, i.e.,
// DFxASGN(), the local-variable definition is still written, but
// without any default-to code.  Regardless, NIL is always assigned to
// the local.
#xtranslate DFxASGN( <id1> [ , <idN> ] ) => ;
      LOCAL xDFDmy := ( DFxDEFTOC( <id1> ) [ DFxDEFTOC( <idN> ) ] NIL)


// Function parameters in the output function definition are stripped of
// any default-argument syntax, i.e., := <def>.
#xtranslate DFxARG( <id>          ) => <id>
#xtranslate DFxARG( <id> := <def> ) => <id>

// If there is no default-argument specifier for a parameter, no code is
// written.  Otherwise, default-to code is written, followed by a comma.
// (The harmless unary plus operator is necessary to work around a bug
// in the Clipper preprocessor.  The preprocessor sometimes reports a
// circular UDC error when a UDC translates to nothing.)
#xtranslate DFxDEFTOC( <id>          ) => +
#xtranslate DFxDEFTOC( <id> := <def> ) => DFxDEFTO(<id>, <def>),

// The core default-to translation directive.
#xtranslate DFxDEFTO( <var>, <def> ) => IF(<var> == NIL, <var> := <def>, )


// Prototype for all function definitions.
#xtranslate DFxFUNC <func> ( <id1> [ , <idN> ] ) => ;
      FUNCTI <func>(DFxARG( <id1> ) [, DFxARG( <idN> )]);;
      DFxASGN( <id1> [ , <idN> ] )

// Function UDCs for various combinations of long and short forms of the
// STATIC and FUNCTION keywords, with and without the STATIC designation.
#xcommand FUNC <func> ( <id1> [, <idN> ] ) => ;
      DFxFUNC <func> ( <id1> [ , <idN> ] )
#xcommand FUNCTION <func> ( <id1> [, <idN> ] ) => ;
      DFxFUNC <func> ( <id1> [ , <idN> ] )
#xcommand STATIC <mod: FUNC, FUNCTION> <func> ( <id1> [, <idN> ] ) => ;
      STATIC DFxFUNC <func> ( <id1> [ , <idN> ] )


// Prototype for all procedure definitions.
#xtranslate DFxPROC <proc> ( <id1> [ , <idN> ] ) => ;
      PROCEDU <proc>(DFxARG( <id1> ) [, DFxARG( <idN> )]);;
      DFxASGN( <id1> [ , <idN> ] )

// Procedure UDCs for various combinations of long and short forms of the
// STATIC and PROCEDURE keywords, with and without the STATIC designation.
#xcommand PROC <proc> ( <id1> [, <idN> ] ) => ;
      DFxPROC <proc> ( <id1> [ , <idN> ] )
#xcommand PROCEDURE <proc> ( <id1> [, <idN> ] ) => ;
      DFxPROC <proc> ( <id1> [ , <idN> ] )
#xcommand STATIC <mod: PROC, PROCEDURE> <proc> ( <id1> [, <idN> ] ) => ;
      STATIC DFxPROC <proc> ( <id1> [ , <idN> ] )


// The ever-popular DEFAULT TO.  Takes one or more default clauses.
#xcommand DEFAULT <var1> TO <def1> [, <varN> TO <defN> ] => ;
      DFxDEFTO( <var1>, <def1> ) [; DFxDEFTO( <varN>, <defN> ) ]
