unit Manager;
interface
uses Employee, EmployeeCaller;
type
	(* TManagerSuperClasses is a set containing an enumeration of
	the superclasses. this works like a multi-dimensional boolean
	(the flag for the dimension is set if its corresponding value
	in the enumeration is in the set) *)
	TManagerSuperClasses	=	set of ( EmployeeSuperClass );

	(* the TManager class a class that semanticly inherits from
	the TEmployee class, but by using aggregation *)
	TManager	=	class
		public
			(* the Create constructor creates all the superclasses
			and then continues initialization of the object *)
			constructor Create;
			destructor Destroy; override;
			(* all of the virtual methods from the superclass must be
			redefined here in case any derived classes want to
			override them later. they must of course be declared
			as virtual here the same way they were in the base class.
			notice that they are declared virtual and not override
			since they're syntacticly not declared in this family
			before but rather forwarded by a member class *)
			procedure Work( Hours : integer ); virtual;
			function Wage : real; virtual;
			(* since we're not formally inheriting, this object
			cannot be casted into one of the superclasses. therefore,
			we provide functions that return the superclasses of the
			object. since all the virtual methods will be forwarded,
			we are in effect returning a pointer to the superclass
			part of ourself - a kind of casting *)
			function AsEmployee : TEmployee; virtual;
			(* the new functions *)
			function PlansMade : integer; virtual;
		protected
			(* the CreateWithSuperClass constructor creates the
			object using an already constructed superclass *)
			constructor CreateWithSuperClass(
				SuperClass	:	TEmployeeCaller );
		private
			(* instead of deriving from the TEmployee class the
			usual way, we make the superclass a member inside the
			subclass. (has-a instead of is-a relationship) *)
			m_SuperClass			:	TEmployeeCaller;
			(* this member is a multi-dimensional flag that tells us
			which of the superclasses that have already been
			destructed *)
			m_Destructed		:	TManagerSuperClasses;
			(* this member keeps record of which superclasses we have
			created ourself, so we know which superclasses we'll have
			to destruct and which will be destructed elsewhere *)
			m_NotVirtual		:	TManagerSuperClasses;
			(* regular members of the class *)
			m_PlansMade		:	integer;
	end;

implementation
const
	g_ManagerSalary	=	20;
	g_PlansPerHour		=	2;

type
	(* the purpose of the TManagerEmployee class is to have
	a proxy that can forward virtual methods from the TEmployee
	superclass to the TManager subclass *)
	TManagerEmployee	=	class( TEmployeeCaller )
			(* since the memory of the superclass and the subclass
			is located apart, we need to maintain a reference to
			the subclass so we can forward virtual methods *)
			m_SubClass	:	TManager;
			(* we need a constructor for this class since we have
			to set up the link to the subclass. it takes a
			reference to the subclass followed by all the
			parameters that the superclass needs to be created *)
			constructor Create(
				BackPtr	:	TManager;
				Salary	:	real );
			(* we must override the destructor so that we can forward
			the call to the destructor in the subclass *)
			destructor Destroy; override;
			(* all the virtual methods must be overriden here so
			that they can be forwarded to the subclass *)
			procedure Work( Hours : integer ); override;
			function Wage : real; override;
	end;

(* this constructor initialize all the members specific for this
class (i.e. the subclass reference) and construct the superclass
with the parameters given *)
constructor TManagerEmployee.Create;
begin
	m_SubClass := BackPtr;
	(* construct the superclass with the rest of the parameters
	given. we don't modify the parameters sent to the superclass
	at all - if any other behavior is wanted, this should be
	taken care of in the subclass, not here. the only purpose
	of this class is to provide a bridge between the superclass
	and the subclass *)
	inherited Create( Salary );
end;

(* this destructor is invoked when someone destructs the
TManager object through its TEmployee superclass (which is
a TManagerEmployee)
it shall NOT take down the superclass, since the superclasses
must be destructed in opposite order of creation, something only
the subclass can do since it knows about the other superclasses
*)
destructor TManagerEmployee.Destroy;
begin
	(* we have to tell the subclass that the destructor of this
	superclass already is called by putting the enumeration
	representing this class into the set of destructed superclasses
	*)
	m_SubClass.m_Destructed	:=
		m_SubClass.m_Destructed + [ EmployeeSuperClass ];
	(* the subclass shall coordinate the destruction of the
	entire object. hence the call to a destructor must be forwarded
	*)
	m_SubClass.Destroy;
end;

(* the virtual methods only forward the call to the corresponding
method in the subclass. this is also a good place to do name
resolution (in case methods in different superclasses have the
same name) *)
procedure TManagerEmployee.Work;
begin
	m_SubClass.Work( Hours );
end;

function TManagerEmployee.Wage;
begin
	Result	:=	m_SubClass.Wage;
end;

constructor TManager.Create;
begin
	(* we create the superclass ourself, and pass it to the
	constructor that makes the object reference the superclass *)
	CreateWithSuperClass(
		TManagerEmployee.Create( Self, g_ManagerSalary ) );
	(* keep note of that we've made the superclass ourselves and
	therefore, we'll have to destruct it ourselves too *)
	m_NotVirtual	:=	m_NotVirtual + [ EmployeeSuperClass ];
end;

constructor TManager.CreateWithSuperClass;
begin
	(* we initialize the base class BEFORE we initialize ourself
	so that all our member functions can rely on the superclass
	to exist. *)
	m_SuperClass	:=	SuperClass;
	(* the set is initialized empty so that if none of the
	destructors in the superclasses are invoked, they will be
	called *)
	m_Destructed := [ ];
	(* here goes all the code that customizes the superclass for
	use in the subclass, in this case every programmer get a fixed
	salary. *)
	(* now that we've created the superclass, we can initialize
	the members that is defined in this class *)
	m_PlansMade	:=	0;
	(* allow us to follow the order of creation *)
	writeln( 'DEBUG : Manager created.' );
end;

(* this destructor is capable of taking the entire object down.
all the superclasses will forward its destructors here. it can
safely be overridden( so that all destructors called through
superclasses will be calling the destructor of the new subclass)
as long as the new destructor calls this one to destruct the
superclasses properly *)
destructor TManager.Destroy;
begin
	(* the order of destruction is the opposite of the order of
	creation *)
	writeln( 'DEBUG : Manager destroyed.' );
	(* the superclasses need to be destroyed in opposite order of
	creation. we don't know if the object is destroyed through
	any of its superclasses and hence we don't know if any
	destructors are already called. therefore, we take down the
	superclasses through a regular method so that they can be
	destructed in proper order, and thereafter release the memory
	for the superclasses which it is needed *)
	if ( EmployeeSuperClass in m_NotVirtual ) then
	begin
		m_SuperClass.InheritedDestroy;
		(* we need to release the memory of those superclasses that
		has not yet have their destructor called (those superclasses
		which has not marked themselves as destructed) *)
		if not ( EmployeeSuperClass in m_Destructed ) then
			m_SuperClass.FreeInstance;
	end;
end;

(* if any of the virtual methods from the base class wasn't
intended to be overridden, then the default implmentation can
just call the implementation of the superclass, and it will seem
as it was never overridden (until some derived class does it) *)
function TManager.Wage;
begin
	(* we call the inherited version so that we get the
	implementation of the superclass. if we call the virtual
	version (i.e. Wage()) then it will be forwarded to this point
	again! *)
	Result	:=	m_SuperClass.InheritedWage;
end;

procedure TManager.Work;
begin
	(* first let the superclass have a shot at it *)
	m_SuperClass.InheritedWork( Hours );
	(* we want to do some extra work in this method *)
	inc( m_PlansMade, Hours * g_PlansPerHour );
end;

(* return the actual superclass part of this object, so that we
can pass this object when a reference to one of the superclasses
is required. this way, we still have polymorphism *)
function TManager.AsEmployee;
begin
	Result	:=	m_SuperClass;
end;

(* perform some work on the members of this class *)
function TManager.PlansMade;
begin
	Result	:=	m_PlansMade;
end;

end.
