package bmsi.tui;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.TextEvent;
import java.awt.peer.*;
import java.awt.image.*;
import java.io.*;

// Text component peers -

abstract class TUITextComponent extends TUIComponent
  implements TextComponentPeer
{
  /** txt is not null if and only if it contains a current copy of the
   text.  When the remote client modifies a TextComponent, it sends
   a SETMODIFIED message. */
  protected String txt = null;
  protected int selectBegin = 0;
  protected int selectEnd = 0;
  TUITextComponent(TextComponent t,TUIKit toolkit) { super(t,toolkit); }
  protected void initialize() {
    TextComponent c = (TextComponent)target;
    select(c.getSelectionStart(),c.getSelectionEnd());
    String txt = c.getText();
    if (txt != null)
      setText(txt);
    if (c.isEditable())
      setEditable(true);
    super.initialize();
  }

  public int getSelectionEnd() { return selectEnd; }
  public int getSelectionStart() { return selectBegin; }

  public void remoteMethod(int cmd) throws IOException {
    switch (cmd) {
    case MENUPICK:
      toolkit.theQueue.postEvent(
        //new Event(target,Event.ACTION_EVENT,getText())
	new ActionEvent(target,ActionEvent.ACTION_PERFORMED,
	getText())
      );
      return;
    case SETMODIFIED:
      // TextEvent introduced in JDK1.1b3
      toolkit.theQueue.postEvent(
	new TextEvent(target,TextEvent.TEXT_VALUE_CHANGED));
      txt = null;
      return;
    case SELECT:
      // SELECT arguments should really be 4 bytes, but keep in mind
      // that our current String format is restricted to 64K as well.
      selectBegin = toolkit.readUnsignedShort();
      selectEnd = toolkit.readUnsignedShort();
      return;
    }
    super.remoteMethod(cmd);
  }

  /** Get the current text value.  Use our local copy unless we have
   received a SETMODIFIED message. */
  synchronized public String getText() {
    if (txt == null) {
      toolkit.writeCmd(this,GETTEXT);
      txt = toolkit.getStringRet();
    }
    return txt;
  }

  public void select(int start,int end) {
    selectBegin = start;
    selectEnd = end;
    toolkit.writeCmd(this,SELECT,start,end);
  }
  public void setCaretPosition(int pos) {
    select(pos,pos);
  }
  public int getCaretPosition() { return getSelectionEnd(); }
  public void setEditable(boolean b) {
    toolkit.writeCmd(this,SETEDITABLE,b ? 1 : 0);
  }
  public void setText(String l) {
    if (l == null) l = "";
    txt = l;
    toolkit.writeCmd(this,SETTEXT,txt);
  }
}

class TUITextArea extends TUITextComponent implements TextAreaPeer {
  TUITextArea(TextArea t,TUIKit toolkit) { super(t,toolkit); }
  protected void create(TUIContainer parent) {
    winID = toolkit.createRemotePeer(parent,NEWTEXTAREA,this);
  }
  public void insert(String s,int pos) { replaceRange(s,pos,pos); }
  public void replaceRange(String s,int start,int end) {
    toolkit.writeCmd(this,REPLACETEXT,s,start,end);
    selectEnd = end + s.length();
    selectBegin = selectEnd;
    if (txt != null)
      txt = txt.substring(0,start) + s + txt.substring(end);
  }
  public Dimension getPreferredSize(int rows,int cols) {
    return toolkit.scale(new Dimension(cols,rows));	// room for scroll bars ?
  }
  public Dimension getMinimumSize(int rows,int cols) {
    Dimension dim = new Dimension(20,5);
    if (rows < dim.height)
      dim.height = rows;
    if (cols < dim.width)
      dim.width = cols;
    return toolkit.scale(dim);
  }

  /* jdk 1.0 compatibility */
  public final Dimension preferredSize(int rows,int cols) {
    return getPreferredSize(rows,cols);
  }
  public final Dimension minimumSize(int rows,int cols) {
    return getMinimumSize(rows,cols);
  }
  public final void insertText(String s,int pos) { insert(s,pos); }
  public final void replaceText(String s,int start,int end) {
    replaceRange(s,start,end);
  }
}

class TUITextField extends TUITextComponent implements TextFieldPeer {
  TUITextField(TextField t,TUIKit toolkit) { super(t,toolkit); }
  protected void create(TUIContainer parent) {
    winID = toolkit.createRemotePeer(parent,NEWTEXTFIELD,this);
  }
  protected void initialize() {
    TextField tf = (TextField)target;
    if (tf.echoCharIsSet())
      setEchoChar(tf.getEchoChar());
    super.initialize();
  }
  public void setEchoChar(char c) {
    toolkit.writeCmd(this,SETECHOCHAR,c);
  }
  public Dimension getPreferredSize(int cols) {
    return toolkit.scale(new Dimension(cols,1));
  }
  public Dimension getMinimumSize(int cols) {
    if (cols > 20)
      return toolkit.scale(new Dimension(20,1));
    return getPreferredSize(cols);
  }

  /* jdk 1.0 compatibility */
  public final void setEchoCharacter(char c) { setEchoChar(c); }
  public final Dimension preferredSize(int cols) {
    return getPreferredSize(cols);
  }
  public final Dimension minimumSize(int cols) {
    return getMinimumSize(cols);
  }
}
