package bmsi.tui;
import java.awt.*;
import java.awt.event.*;
import java.awt.peer.*;
import java.io.IOException;
import java.util.BitSet;

/** TUIChoice is straightforward.  We handle selection changes locally
  so that we can post the corresponding ItemEvent. */

class TUIChoice extends TUIComponent implements ChoicePeer {
  TUIChoice(Choice choice,TUIKit toolkit) { super(choice,toolkit); }
  protected void create(TUIContainer parent) {
    winID = toolkit.createRemotePeer(parent,NEWCHOICE,this);
  }
  public Dimension getMinimumSize() {
    int width = target.getSize().width;
    return toolkit.scale(new Dimension((width > 0) ? width : 4,1));
  }
  protected void initialize() {
    Choice choice = (Choice)target;
    int n = choice.getItemCount();
    for (int i = 0; i < n; ++i)
      add(choice.getItem(i),i);
    select(choice.getSelectedIndex());
    super.initialize();
  }
  public void add(String s,int i) { toolkit.writeCmd(this,ADDCHOICEITEM,s,i); }
  public void select(int idx) { toolkit.writeCmd(this,SELECTCHOICE,idx); }
  public void remove(int idx) { toolkit.writeCmd(this,REMOVECHOICE,idx); }
  public void remoteMethod(int cmd) throws IOException {
    if (cmd == SELECTCHOICE) {
      Choice choice = (Choice)target;
      int i = toolkit.readShort();
      choice.select(i);
      toolkit.theQueue.postEvent(
	new ItemEvent(choice,ItemEvent.ITEM_STATE_CHANGED,
	choice.getSelectedItem(),ItemEvent.SELECTED)
      );
      return;
    }
    super.remoteMethod(cmd);
  }
  /* jdk 1.0 compatibility */
  public final void addItem(String s,int i) { add(s,i); }
}

/** TUIList is mostly straightforward.  Selection changes are handled locally.
  FIXME: BitSet is *not* a good way to represent selections.  An ordered set
  of integers would be much better.  The Java General Library has
  a good OrderedSet class.  Java.awt just uses an int[], but it doesn't
  have to handle constant resizing!
 */

class TUIList extends TUIComponent implements ListPeer {
  private BitSet selected = new BitSet();
  private int lastsel = -1;
  private boolean multi = false;

  TUIList(List list,TUIKit toolkit) { super(list,toolkit); }
  protected void create(TUIContainer parent) {
    winID = toolkit.createRemotePeer(parent,NEWLIST,this);
  }
  public Dimension getMinimumSize() { return getMinimumSize(3); }
  public Dimension getPreferredSize(int n) {
    int width = target.getSize().width;
    return toolkit.scale(new Dimension((width > 0) ? width : 8,n));
  }
  public Dimension getMinimumSize(int n) {
    int width = target.getSize().width;
    return toolkit.scale(new Dimension((width > 0) ? width : 8,3));
  }
  protected void initialize() {
    List list = (List)target;
    int n = list.getItemCount();
    for (int i = 0; i < n; ++i)
      add(list.getItem(i),i);
    if (list.isMultipleMode())
      setMultipleMode(true);
    int [] idx = list.getSelectedIndexes();
    for (int i = 0; i < idx.length; ++i)
      select(idx[i]);
    super.initialize();
  }
  public void add(String s,int i) { toolkit.writeCmd(this,ADDCHOICEITEM,s,i); }
  private final void deselectAll() {
    int [] idx = getSelectedIndexes();
    for (int i = 0; i < idx.length; ++i)
      deselect(idx[i]);
  }
  public void select(int i) {
    if (!multi) deselectAll();
    toolkit.writeCmd(this,SELECTCHOICE,i);
    selected.set(i);
  }
  public void deselect(int idx) {
    toolkit.writeCmd(this,DESELECT,idx);
    selected.clear(idx);
  }
  public void delItems(int b,int e) { toolkit.writeCmd(this,DELITEMS,b,e); }
  public void makeVisible(int i) { toolkit.writeCmd(this,MAKEVISIBLE,i); }
  public int[] getSelectedIndexes() {
    int n = selected.size();
    int cnt = 0;
    for (int i = 0; i < n; ++i)
      if (selected.get(i)) ++cnt;
    int [] idx = new int[cnt];
    cnt = 0;
    for (int i = 0; i < n; ++i)
      if (selected.get(i))
	idx[cnt++] = i;
    return idx;
  }
  public void setMultipleMode(boolean m) {
    toolkit.writeCmd(this,SETMULTIPLE,m ? 1 : 0);
    multi = m;
  }
  public void removeAll() {
    toolkit.writeCmd(this,CLEARLIST);
    selected = new BitSet();
  }
  public void remoteMethod(int cmd) throws IOException {
    List list = (List)target;
    if (cmd == SELECTCHOICE) {
      int i = toolkit.readShort();
      if (selected.get(i)) {
	deselect(i);
	toolkit.theQueue.postEvent(
	  new ItemEvent(list,ItemEvent.ITEM_STATE_CHANGED,
	    new Integer(i),ItemEvent.DESELECTED)
	);
      }
      else {
	int [] idx = multi ? null : getSelectedIndexes();
	select(i);
	if (idx != null)
	  for (int j = 0; j < idx.length; ++j) {
	    toolkit.theQueue.postEvent(
	      new ItemEvent(list,ItemEvent.ITEM_STATE_CHANGED,
		new Integer(i),ItemEvent.DESELECTED)
	    );
	  }
	toolkit.theQueue.postEvent(
	  new ItemEvent(list,ItemEvent.ITEM_STATE_CHANGED,
	    new Integer(i),ItemEvent.SELECTED)
	);
      }
      return;
    }
    super.remoteMethod(cmd);
  }
  /* jdk 1.0 compatibility */
  public final void addItem(String s,int i) { add(s,i); }
  public final void clear() { removeAll(); }
  public final Dimension preferredSize(int n) { return getPreferredSize(n); }
  public final Dimension minimumSize(int n) { return getMinimumSize(n); }
  public final void setMultipleSelections(boolean b) { setMultipleMode(b); }
}
