/* $Log: TUIKit.java,v $
 * Revision 1.3  1997/04/19  20:15:53  stuart
 * Autoload tuipeer when bmsi.tuipeer property set.
 * Redirect errors to log window.
 *
 * Revision 1.2  1997/04/19  01:11:51  stuart
 * mini Applet viewer
 * buffer socket I/O (flushing now works)
 *
 * Revision 1.1  1997/04/02  17:30:11  stuart
 * new JDK1.1 methods
 * SETSCALE
 * handle IOException in writeShort and writeUTF
 *
 * Revision 1.0  1997/02/05  19:04:41  stuart
 * Initial revision
 *
 */
package bmsi.tui;

import bmsi.awt.TextLog;
import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;
import java.awt.peer.*;
import java.awt.image.*;
import java.awt.datatransfer.Clipboard;
import java.util.Properties;
import java.io.*;
import java.net.*;

/** TUIKit is an AWT Toolkit implementation for text terminals.
 * TUIKit communicates with a "client" program via a socket.  You could
 * also consider the "client" a "display server", but the "client" initiates
 * the connection.  
 * After construction, the conversation consists of method calls.  A method
 * call is an object id, a method index, optional parameter values, and a
 * possible return value.  Value types are object id, short, or string.
 */

public class TUIKit extends Toolkit implements Runnable, RemotePeer {
  private TextLog log;
  /** Used by TUIComponents for this toolkit to track FocusEvents */
  TUIComponent hasFocus;
  TUIWindow active;
  EventQueue theQueue;
  private boolean needFlush = false;
  private TUISync flushTarget = new TUISync(this);
  private Socket sock;
  private DataOutputStream out;
  private DataInputStream in;
  private Dimension screen;
  private Dimension scale = new Dimension(1,1);
  private static ServerSocket server;
  private static int serverPort = 9713;
  private Thread eventThread = new Thread(this,"TUIKitEvents");
  private java.util.Vector objs = new java.util.Vector();
  private Object retval = null;
  private TUIColorModel colorModel = new TUIColorModel();
  private String[] fontlist = { "Terminal" };

  /** Strings are converted to an 8-bit form easily understood by the remote
      peer.  Our TUI system uses the IBM PC codepage. */
  private String codepage = "Cp437";
  // NOTE, JDK1.1b3 uses String instead of Converter objects
  //       JDK1.0 must convert to ASCII7 or UTF8
  //private CharToByteConverter fromUnicode;
  //private ByteToCharConverter toUnicode;

  /** If setPort() is called with a non-zero port, this Toolkit waits for a
    connection from the "client" on that port.  Otherwise, this Toolkit
    assumes that this JVM was loaded via inetd and uses FileDescriptor.out
   */
  public static void setPort(int port) {
    if (server != null) {
      try { server.close(); }
      catch (IOException e) { }
      server = null;
    }
    serverPort = port;
  }

  /** Select the encoding used to exchange Strings with our
    remote peer. */
  public void setEncoding(String cpname) throws IOException {
    //CharToByteConverter from = CharToByteConverter.getConverter(cpname);
    //ByteToCharConverter to = ByteToCharConverter.getConverter(codepage);
    // if we haven't gotten an exception yet, it is safe to proceed
    codepage = cpname;
    //fromUnicode = from;
    //toUnicode = to;
  }

  /** Enable use of an extension package.  We don't support any extensions
      yet. */
  public void setExtension(String extname) { }

  public TUIKit() throws IOException {
    setEncoding(codepage);
    OutputStream bufout;
    Properties prop = System.getProperties();
    String tuipeer = prop.getProperty("bmsi.tuipeer");
    if (tuipeer == null && serverPort > 0 && server == null) {
      server = new ServerSocket(serverPort);
    }
    if (server != null) {
      debug("Waiting for connection...");
      sock = server.accept();
      debug("Accepting call from " + sock);
      bufout = new BufferedOutputStream(sock.getOutputStream());
      in = new DataInputStream(sock.getInputStream());
    }
    else {
      // check whether client autoload enabled
      if (tuipeer != null) {
	ServerSocket server = new ServerSocket(0);
	String cmd = tuipeer + " -p" + server.getLocalPort();
	Process prog = Runtime.getRuntime().exec(cmd);
	sock = server.accept();
	server.close();
	sock.setTcpNoDelay(true);
	bufout = new BufferedOutputStream(sock.getOutputStream());
	in = new DataInputStream(sock.getInputStream());
	log = new TextLog(20,60);
      }
      else {
	bufout = System.out;
	in = new DataInputStream(System.in);
      }
    }
    out = new DataOutputStream(bufout);
    objs.addElement(this);
    theQueue = new EventQueue();
    while (screen == null)
      readCmd();	// execute commands until Screen size is defined
    eventThread.start();
  }

  public void run() {
    if (log != null) {
      Frame f = new Frame("Java Error Log");
      f.add(log);
      f.pack();
      f.show();
      System.setErr(log.getPrintStream());
    }
    debug("Screen = " + screen);
    debug(" Scale = " + scale);
    try {
      for (;;)
	readCmd();
    }
    // remote peer has gone away
    catch (AWTError e) {
      debug(e);
      // exit if loaded via inetd 
      if (server == null)
	System.exit(0);
    }
  }

  public void debug(String ln) {
    System.err.println(ln);
  }

  public final void debug(Throwable e) {
    e.printStackTrace();
  }

  /** Scale a TUI dimension to defeat hardwired insets and sizes that
      are ridiculously large for a TUI interface. 
    @parm dim	The Dimension to scale in text positions.  
		The argument is modified in place.
    @return 	The Dimension scaled to simulated pixels.
   */
  final Dimension scale(Dimension dim) {
    dim.width *= scale.width;
    dim.height *= scale.height;
    return dim;
  }

  /** wait for an input event from our remote peer. */
  private void readCmd() {
    try {
      flush();
      int id = in.readShort();
      int cmd = in.readShort();
      //debug("readCmd: " + objs.elementAt(id) + "(" + id + "), " + cmd);
      RemotePeer peer = (RemotePeer)objs.elementAt(id);
      peer.remoteMethod(cmd);
    }
    catch (IOException e) {
      throw new AWTError(e.toString());
    }
  }

  public Object getTarget() { return null; }
  public int getID() { return 0; }

  /** Toolkit remote methods.  This Toolkit has a well known ID (0)
   * and supports some special methods.
   */
  public void remoteMethod(int cmd) throws IOException {
    switch (cmd) {
    case SETEXTENSION:
      setExtension(readUTF());
      return;
    case SETTEXT:	// select String encoding
      setEncoding(readUTF());
      return;
    case RETSHORT:	// return a 2 byte integer
      retObject(new Integer(in.readShort()));
      return;
    case RETSTRING:
      retObject(in.readUTF());
      return;
    case SETSCALE:
      scale.width = in.readShort();
      scale.height = in.readShort();
      return;
    case SCREENSIZE:
      int w = in.readShort();
      int h = in.readShort();
      screen = new Dimension(w,h);
      return;
    case RUNCLASS:
      try {
	String name = in.readUTF();
	Object app = Class.forName(name).newInstance();
	if (app instanceof Applet) {
	  // a mini applet-viewer
	  Applet a = (Applet)app;
	  Frame f = new Frame(name);
	  a.init();
	  a.start();
	  f.add("Center",a);
	  f.pack();
	  f.show();
	}
      }
      catch (Exception e) {
	e.printStackTrace();
      }
      return;
    }
  }

  private void writeShort(int val) {
    try {
      out.writeShort(val);
    }
    catch (IOException e) {
      throw new AWTError(e.toString());
    }
  }

  /** Send an encoded String to our remote peer.  The name is historical:
    UTF8 was originally the only encoding supported. */
  private void writeUTF(String s) {
    try {
      //byte[] buf = s.getBytes(fromUnicode);
      byte[] buf = s.getBytes(codepage);
      if (buf.length > 0xFFFF)
	throw new AWTError("String too long");
      out.writeShort(buf.length);
      out.write(buf);
    }
    catch (IOException e) {
      throw new AWTError(e.toString());
    }
  }

  /** Read an encoded String from our remote peer.  The name is historical:
    UTF8 was originally the only encoding supported. */
  synchronized String readUTF() throws IOException {
    byte[] buf = new byte[readShort() & 0xFFFF];
    in.readFully(buf);
    //return new String(buf,toUnicode);
    return new String(buf,codepage);
  }

  synchronized int readShort() throws IOException { return in.readShort(); }

  final int readUnsignedShort() throws IOException {
    return readShort() & 0xFFFF;
  }

  /** get the next returned integer value */
  final int getIntegerRet() {
    return ((Number)getObjectRet()).intValue();
  }

  /** get the next return String value */
  final String getStringRet() {
    return getObjectRet().toString();
  }

  /** get the next Object returned from our remote peer. */
  synchronized Object getObjectRet() {
    try {
      while (retval == null) {
	if (Thread.currentThread() == eventThread)
	  readCmd();
	else {
	  flush();
	  wait();
	}
      }
      Object val = retval;
      retval = null;
      notify();
      return val;
    }
    catch (InterruptedException e) {
      throw new AWTError("Interrupted waiting for a return value");
    }
  }

  /** post an object returned by our remote peer. */
  synchronized void retObject(Object val) throws IOException {
    try {
      while (retval != null)
	wait();
      retval = val;
      notify();
    }
    catch (InterruptedException e) {
      throw new IOException("Interrupted waiting to return a value");
    }
  }

  /** Send a simple command to our remote peer. */
  synchronized void writeCmd(RemotePeer peer,int cmd) {
    writeCmd(peer.getID(),cmd);
    if (!needFlush) {
      needFlush = true;
      flushTarget.queue();
    }
  }

  private synchronized void writeCmd(int id,int cmd) {
    //debug("CMD: " + objs.elementAt(id) + "(" + id + "), " + cmd);
    writeShort(id);
    writeShort(cmd);
  }

  /** Flush socket output to our remote peer.  */
  public synchronized void flush() {
    try {
      out.flush();
      needFlush = false;
      //debug("Flush");
    }
    catch (IOException e) {
      throw new AWTError(e.toString());
    }
  }

  // various remote commands with arguments follow -

  synchronized void writeCmd(RemotePeer id,int cmd,int val) {
    writeCmd(id,cmd);
    writeShort(val);
  }

  synchronized void writeCmd(RemotePeer id,int cmd,String s) {
    writeCmd(id,cmd);
    writeUTF(s);
  }

  synchronized void writeCmd(RemotePeer id,int cmd,int x,int y) {
    writeCmd(id,cmd);
    writeShort(x);
    writeShort(y);
  }

  synchronized void writeCmd(RemotePeer id,int cmd,String s,int x) {
    writeCmd(id,cmd);
    writeUTF(s);
    writeShort(x);
  }

  synchronized void writeCmd(RemotePeer id,int cmd,String s,int x,int y) {
    writeCmd(id,cmd);
    writeUTF(s);
    writeShort(x);
    writeShort(y);
  }

  synchronized void writeCmd(RemotePeer id,int cmd,int x,int y,int w,int h,
	int dx,int dy)
  {
    writeCmd(id,cmd);
    writeShort(x);
    writeShort(y);
    writeShort(w);
    writeShort(h);
    writeShort(dx);
    writeShort(dy);
  }

  synchronized void writeCmd(RemotePeer id,int cmd,int x,int y,int w,int h) {
    writeCmd(id,cmd);
    writeShort(x);
    writeShort(y);
    writeShort(w);
    writeShort(h);
  }

  public synchronized void close() throws IOException {
    try {
      out.flush();
    }
    finally {
      if (sock != null)
	sock.close();
    }
  }

  protected void finalize() throws IOException { close(); }

  // how to create various peer objects  -

  protected ButtonPeer createButton(Button b) { return new TUIButton(b,this); }
  protected TextFieldPeer createTextField(TextField t) {
    return new TUITextField(t,this);
  }
  protected LabelPeer createLabel(Label lbl) { return new TUILabel(lbl,this); }
  protected ListPeer createList(List lst) { return new TUIList(lst,this); }
  protected CheckboxPeer createCheckbox(Checkbox cb) {
    return new TUICheckbox(cb,this);
  }
  protected ScrollbarPeer createScrollbar(Scrollbar s) {
    return new TUIScrollbar(s,this);
  }
  protected TextAreaPeer createTextArea(TextArea t) {
    return new TUITextArea(t,this);
  }
  protected ChoicePeer createChoice(Choice c) { return new TUIChoice(c,this); }
  protected FramePeer createFrame(Frame f) { return new TUIFrame(f,this); }
  protected CanvasPeer createCanvas(Canvas c) { return new TUICanvas(c,this); }
  protected PanelPeer createPanel(Panel p) { return new TUIPanel(p,this); }
  protected WindowPeer createWindow(Window w) { return new TUIWindow(w,this); }
  protected DialogPeer createDialog(Dialog dlg) {
    return new TUIDialog(dlg,this);
  }
  protected MenuBarPeer createMenuBar(MenuBar mnu) {
    return new TUIMenuBar(mnu,this);
  }
  protected MenuPeer createMenu(Menu mnu) { return new TUIMenu(mnu,this); }
  protected MenuItemPeer createMenuItem(MenuItem mi) {
    return new TUIMenuItem(mi,this);
  }
  protected FileDialogPeer createFileDialog(FileDialog dlg) {
    return new TUIFileDialog(dlg);
  }
  protected CheckboxMenuItemPeer createCheckboxMenuItem(CheckboxMenuItem mi) {
    return new TUICheckboxMenuItem(mi,this);
  }

  /** Create a generic remote peer.  We generate and keep track of the remote
   object ids locally within Java so that we don't have to wait for the remote
   program to respond with a new id.  Turnaround on a remote socket is
   a performance killer.
   */
  synchronized int createRemotePeer(RemotePeer par,int type,RemotePeer peer) {
    if (par == null)
      par = this;
    /* Vector won't search for null, so we search for ourselves knowing
       that a real TUIKit reference will be in pos 0 only */
    int id = objs.indexOf(this,1);
    if (id < 0) {
      id = objs.size();
      objs.addElement(peer);
    }
    else
      objs.setElementAt(peer,id);
    writeCmd(par,type,id);
    return id;
  }

  /** Tell the remote program to remove an object. */
  synchronized void removePeer(int id) {
    if (id < 0) return;
    objs.setElementAt(this,id);
    writeCmd(id,DISPOSE);	// tell the remote peer to go away
    flush();
  }

  public Dimension getScreenSize() { return screen; }
  public int getScreenResolution() { return 10; } // "pixels"/inch
  public ColorModel getColorModel() { return colorModel; }
  public String[] getFontList() { return fontlist; }
  public FontMetrics getFontMetrics(Font f) {
    return new TUIFontMetrics(f,this);
  }

  public void sync() { writeCmd(0,SYNC); flush(); }

  // TUI's can't reasonably support Image.  Always fail -

  public Image getImage(String name) { return null; }
  public Image getImage(java.net.URL url) { return null; }
  public boolean prepareImage(Image img,int w,int h,ImageObserver obs) {
    return false;
  }
  public int checkImage(Image img,int w,int h,ImageObserver obs) {
    return ImageObserver.ERROR | ImageObserver.ABORT;
  }
  public Image createImage(ImageProducer p) { return null; }
  public Image createImage(byte[] data,int offset,int len) { return null; }

  // JDK 1.1 features -

  protected EventQueue getSystemEventQueueImpl() { return theQueue; }

  /** Anybody can beep! */
  public void beep() { writeCmd(0,BEEP); flush(); }

  /** This JDK1.1 feature has no documentation! This seems to be a placeholder
      that is not yet used for anything.
   */
  protected FontPeer getFontPeer(String name, int style) {
    //debug("FontPeer needed");
    return null;
  }

  /** TUI's have a clipboard, but we haven't finished this JDK1.1 feature. */
  public Clipboard getSystemClipboard() {
    debug("ClipBoard needed");
    return null;
  }

  /** TUI's have printers, but we haven't finished this JDK1.1 feature. */
  public PrintJob getPrintJob(Frame frame, String jobtitle, Properties props) {
    debug("PrintJob needed");
    return null;
  }

  /** TUI's have PopupMenus, but we haven't finished this JDK1.1 feature. */
  protected PopupMenuPeer createPopupMenu(PopupMenu target) {
    debug("PopupMenu needed");
    return null;
  }

  /** TUI's have ScrollPanes, but we haven't finished this JDK1.1 feature. */
  protected ScrollPanePeer createScrollPane(ScrollPane target) {
    debug("ScrollPane needed");
    return null;
  }

  /** Create a TUIKit on Filedescriptor.out and make it the default Toolkit.  
    Used when loading a Java VM from inetd. */
  public static void main(String[] args) {
    Properties props = System.getProperties();
    props.put("awt.toolkit","bmsi.tui.TUIKit");
    setPort(0);
    Toolkit.getDefaultToolkit();
  }
}
