--------------------TUIGraphics.java------------------
/* $Log: TUIGraphics.java,v $
 * Revision 1.4  1997/06/04  20:39:59  stuart
 * repaint pixels copied from outside clipRect
 *
 * Revision 1.3  1997/06/04  19:33:29  stuart
 * translate locally
 *
 */
package bmsi.tui;
import java.awt.*;
import java.awt.image.*;

/** TUIGraphics draws on the owning TUIComponent.
 It sends the drawing commands to the remote peer of the owning TUIComponent.
 To minimize message traffic, a top level server window stores its contents
 even when hidden, since the memory requirements are trivial for TUI.  
 Component redraw is required only when rearranging sub-components of
 a top level window.
 Warning: Since we do not synchronize on the target TUIComponent,
   multiple threads drawing on the same component will have
   strange results.
 */
class TUIGraphics extends Graphics {
  private TUIKit toolkit;
  private TUIComponent target;
  private int posx, posy;	// origin
  private Rectangle clipRect;
  private Font font;
  private Color color;
  private boolean XORmode;
  private static final int
    DRAWFONT = 150,
    DRAWCOLOR = 151,
    CLIPRECT = 152,
    TRANSLATE = 153,
    COPYAREA = 154,
    FILLRECT = 155,
    DRAWRECT = 156,
    CLEARRECT = 157,
    DRAWROUNDRECT = 158,
    FILLROUNDRECT = 159,
    DRAWSTRING = 160,
    DRAWLINE = 161,
    FILL3DRECT = 162,
    DRAW3DRECT = 163,
    INITGRAPHICS = 164;

  /** We send all graphics commands to the remote component. */
  TUIGraphics(TUIComponent comp) {
    target = comp;
    toolkit = (TUIKit)comp.getToolkit();
    posx = 0;
    posy = 0;
  }

  public Graphics create() {
    TUIGraphics g = new TUIGraphics(target);
    g.clipRect = clipRect;
    g.translate(posx,posy);
    return g;
  }

  public void translate(int x,int y) {
    posx += x; posy += y;
  }

  /* jdk 1.0 compatibility */
  public final Rectangle getClipRect() { return getClipBounds(); }

  public Rectangle getClipBounds() { return clipRect; }

  public void clipRect(int x,int y,int width,int height) {
    x += posx;
    y += posy;
    Rectangle newclip = new Rectangle(x,y,width,height);
    if (clipRect == null)
      setClip(newclip);
    else
      setClip(clipRect.intersection(newclip));
  }

  public void setClip(int x,int y,int width,int height) {
    setClip(new Rectangle(x,y,width,height));
  }

  public void setClip(Shape s) {
    clipRect = s.getBounds();
    if (target.currentGraphics == this)
      toolkit.writeCmd(target,CLIPRECT,
	clipRect.x,clipRect.y,clipRect.width,clipRect.height);
  }
  
  public Shape getClip() { return clipRect; }

  public Color getColor() { return color; }
  public void setColor(Color c) {
    color = c;
    if (c == null) return;
    if (target.currentGraphics == this)
      toolkit.writeCmd(target,DRAWCOLOR,target.pixelFromColor(color));
  }
  public void setPaintMode() { XORmode = false; }
  public void setXORMode(Color c) { XORmode = true; }
  public Font getFont() { return font; }
  public void setFont(Font f) {
    font = f;
    if (f == null) return;
    if (target.currentGraphics == this)
      toolkit.writeCmd(target,DRAWFONT,
	f.getName(),f.getSize(),f.getStyle());
  }
  public FontMetrics getFontMetrics(Font f) {
    return new TUIFontMetrics(f,toolkit);
  }

  /** Return the object of the remote peer to execute drawing commands.
   * Currently, we send all draw commands to the TUIComponents remote peer.
   * If we were not the last TUIGraphics to send drawing commands, we
   * send the local Graphics state information.
   */
  private RemotePeer getID() {
    if (target.currentGraphics != this) {
      toolkit.writeCmd(target,INITGRAPHICS);
      target.currentGraphics = this;
      if (font != null)
	setFont(font);
      // Should XORmode affect color?
      if (color != null)
	setColor(color);
      if (clipRect != null)
	setClip(clipRect);
    }
    return target;
  }

  /* Our TUI can do a lot of things like draw lines and boxes. 
     And don't forget text!  FIXME: for multiple TUIGraphics instances
     to work with multiple threads, each drawing command
     needs synchronized (target).
   */
  public void copyArea(int x,int y,int width,int height,int dx,int dy) {
    x += posx;
    y += posy;
    toolkit.writeCmd(getID(),COPYAREA,x,y,width,height,dx,dy);
    /* repaint areas copied from outside the component.  The remote client will
       handle obscured areas.  */
    Rectangle clip;
    if (clipRect == null) {
      Dimension sz = target.target.getSize();
      clip = new Rectangle(0,0,sz.width,sz.height);
    }
    else
      clip = clipRect;
    Rectangle r = new Rectangle(x,y,width,height);
    Rectangle src = r.intersection(clip);
    src.translate(dx,dy);
    r.translate(dx,dy);
    Rectangle dst = r.intersection(clip);
    /* dst - src is the area we need to repaint.  src is entirely
       contained within dst.
     */
    if (src.y > dst.y)
      target.repaint(0,dst.x,dst.y,dst.width,src.y - dst.y);
    int srctop = src.y + src.height;
    int dsttop = dst.y + dst.height;
    if (srctop < dsttop)
      target.repaint(0,dst.x,srctop,dst.width,dsttop - srctop);
    if (src.x > dst.x)
      target.repaint(0,dst.x,src.y,src.x - dst.x,src.height);
    int srcright = src.x + src.width;
    int dstright = dst.x + dst.width;
    if (srcright < dstright)
      target.repaint(0,srcright,src.y,dstright - srcright,src.height);
  }
  public void drawLine(int x1,int y1,int x2,int y2) {
    toolkit.writeCmd(getID(),DRAWLINE,x1+posx,y1+posy,x2+posx,y2+posy);
  }
  public void fillRect(int x,int y,int width,int height) {
    toolkit.writeCmd(getID(),FILLRECT,x+posx,y+posy,width,height);
  }
  public void fill3DRect(int x,int y,int width,int height,boolean raised) {
    toolkit.writeCmd(getID(),FILLRECT,x+posx,y+posy,width,height);
  }
  public void drawRect(int x,int y,int width,int height) {
    toolkit.writeCmd(getID(),DRAWRECT,x+posx,y+posy,width,height);
  }
  public void draw3DRect(int x,int y,int width,int height,boolean raised) {
    toolkit.writeCmd(getID(),DRAWRECT,x+posx,y+posy,width,height);
  }
  public void clearRect(int x,int y,int width,int height) {
    toolkit.writeCmd(getID(),CLEARRECT,x+posx,y+posy,width,height);
  }
  public void drawRoundRect(int x,int y,int w,int h,int arcw,int arch) {
    toolkit.writeCmd(getID(),DRAWROUNDRECT,x+posx,y+posy,w,h);
  }
  public void fillRoundRect(int x,int y,int w,int h,int arcw,int arch) {
    toolkit.writeCmd(getID(),FILLROUNDRECT,x+posx,y+posy,w,h);
  }
  public void drawString(String str,int x,int y) {
    toolkit.writeCmd(getID(),DRAWSTRING,str,x+posx,y+posy);
  }
  public void drawPolyline(int [] x,int [] y,int n) {
    for (int i = 1; i < n; ++i)
      drawLine(x[i-1],y[i-1],x[i],y[i]);
  }
  public void drawPolygon(int [] x,int [] y,int n) {
    drawPolyline(x,y,n);
    if (x[0] != x[n-1] || y[0] != y[n-1])
      drawLine(x[n-1],y[n-1],x[0],y[0]);
  }

  /* Real garbage gets thrown away fast! :-) */
  public void drawOval(int x,int y,int width,int height) { }
  public void fillOval(int x,int y,int width,int height) { }
  public void drawArc(int x,int y,int w,int h,int startang,int arcang) { }
  public void fillArc(int x,int y,int w,int h,int startang,int arcang) { }
  public void fillPolygon(int [] xpoints,int [] ypoints,int npoints) { }

  public boolean drawImage(Image img,int x,int y,
	ImageObserver obs) {
    obs.imageUpdate(img,obs.ERROR|obs.ABORT,x+posx,y+posy,0,0);
    return false;
  }
  public boolean drawImage(Image img,int x,int y,int w,int h,
	ImageObserver obs) {
    obs.imageUpdate(img,obs.ERROR|obs.ABORT,x+posx,y+posy,w,h);
    return false;
  }
  public boolean drawImage(Image img,int x,int y,Color c,ImageObserver obs) {
    obs.imageUpdate(img,obs.ERROR|obs.ABORT,x+posx,y+posy,0,0);
    return false;
  }
  public boolean drawImage(Image img,int x,int y,int w,int h,Color c,
	ImageObserver obs) {
    obs.imageUpdate(img,obs.ERROR|obs.ABORT,x+posx,y+posy,w,h);
    return false;
  }
  public boolean drawImage(Image img,int dx1,int dy1,int dx2,int dy2,
	int sx1,int sy1,int sx2,int sy2, ImageObserver obs) {
    obs.imageUpdate(img,obs.ERROR|obs.ABORT,sx1+posx,sy1+posy,sx2-sx1,sy2-sy1);
    return false;
  }
  public boolean drawImage(Image img,	int dx1,int dy1,int dx2,int dy2,
	int sx1,int sy1,int sx2,int sy2, Color c, ImageObserver obs) {
    obs.imageUpdate(img,obs.ERROR|obs.ABORT,sx1+posx,sy1+posy,sx2-sx1,sy2-sy1);
    return false;
  }

  /** null out references to speed GC */
  public void dispose() {
    if (target.currentGraphics == this)
      target.currentGraphics = null;
    target = null;
    toolkit = null;
  }
}



