// Data Block Quantizer Class Member Functions

#include <values.h>
#include "quant.hpp"
#include "tables.hpp"

#if !defined(TRUE) || !defined(FALSE)
#define FALSE 0
#define TRUE  1
#endif

// Class Constructor
Quantize::Quantize(void) {

  // Allocate memory for a luma quant table
  LumaQuantTablePtr = new int [64];

  // Move in the default table. Note BYTE to INT conversion
  for (int Index = 0; Index < 64; Index++)
    LumaQuantTablePtr[Index] = LumaQuantTable[Index];

  // Allocate memory for a chroma quant table
  ChromaQuantTablePtr = new int [64];

  // Move in the default table. Note BYTE to INT conversion
  for (Index = 0; Index < 64; Index++)
    ChromaQuantTablePtr[Index] = ChromaQuantTable[Index];
}


// Class Destructor
Quantize::~Quantize(void) {

  // Free up the memory for the two quant tables
  delete LumaQuantTablePtr;
  delete ChromaQuantTablePtr;
}


// The execution of this function modifies the quant tables maintained
// by this object. The specified quality factor must be in the range
// 1..100. With 1 causing the most quantization (and therefore compression)
// and 100 causing no quantization (least image compression). A QualityFactor
// of 50 uses the default tables as is without modification.
BOOL Quantize::SetQuality(int QualityFactor) {

  if ((QualityFactor > 100) || (QualityFactor < 0))
    return FALSE;

  // Move in the default table. Note BYTE to INT conversion
  for (int Index = 0; Index < 64; Index++)
    LumaQuantTablePtr[Index] = LumaQuantTable[Index];

  // Move in the default table. Note BYTE to INT conversion
  for (Index = 0; Index < 64; Index++)
    ChromaQuantTablePtr[Index] = ChromaQuantTable[Index];

  // If default quality factor specified, no need for further
  // action. Use tables as is.
  if (QualityFactor == 50)
    return TRUE;

  // Convert the quality factor specified into a scaling factor
  // to be applied to the quant tables.
  long ScaleFactor;

  if (QualityFactor < 50)
    ScaleFactor = 5000 / QualityFactor;
  else
    ScaleFactor = 200 - 2 * QualityFactor;

  // Now process the two quant tables with the new scale factor. Values
  // in quant tables must be in the range 1..MAXINT.
  long Value;
  for (Index = 0; Index < 64; Index++) {
    Value = (LumaQuantTablePtr[Index] * ScaleFactor) / 100L;
    Value = (Value > MAXINT) ? MAXINT:Value;
    Value = (Value <= 0)     ?      1:Value;
    LumaQuantTablePtr[Index] = (int) Value;
  }

  for (Index = 0; Index < 64; Index++) {
    Value = (ChromaQuantTablePtr[Index] * ScaleFactor) / 100L;
    Value = (Value > MAXINT) ? MAXINT:Value;
    Value = (Value <= 0)     ?      1:Value;
    ChromaQuantTablePtr[Index] = (int) Value;
  }

  return TRUE;
}


// Perform forward or reverse quantization on a block of image data.
// Forward quantization divides the block data by the quantization
// coefficients. Reverse quantizations multiples the block data by
// the quantization coefficients. When Type == TRUE, the luma table
// is used. When Type == FALSE, the chroma table is used.
// NOTE: quantization is applied after zigzag
void Quantize::QuantizeBlock(int *lpBlock, BOOL Type, BOOL Forward) {

  if (Type) {
    if (Forward) 
      for (register int Index = 0; Index < 64; Index++)
        lpBlock[Index] /= LumaQuantTablePtr[Index];  
    else
      for (register int Index = 0; Index < 64; Index++)
        lpBlock[Index] *= LumaQuantTablePtr[Index];  
  } else {
    if (Forward)
      for (register int Index = 0; Index < 64; Index++)
        lpBlock[Index] /= ChromaQuantTablePtr[Index];  
    else
      for (register int Index = 0; Index < 64; Index++)
        lpBlock[Index] *= ChromaQuantTablePtr[Index];
  }
}


