// Huffman Encoding/Decoding Class Member Functions

// This code borrows heavily in form and function from the huffman
// encoding/decoding code found in the Independent JPEG Group's software
// which is public domain and Copyright (C) 1991, Thomas G. Lane.

#include <mem.h>
#include "huffman.hpp"
#include "tables.hpp"

// Class Constructor
Huffman::Huffman(LPSTR FileName, BOOL Type) {

  // Setup the various encode/decode data structures. Although not
  // necessary, encoding and decoding tables are setup for each
  // instance of a Huffman object.

  // Setup encoding tables
  EncodeDCLumaTable.pCodeLengthTable = EncDCLumaCodeLength;
  EncodeDCLumaTable.pCodeTable = EncDCLumaCode;
  EncodeDCChromaTable.pCodeLengthTable = EncDCChromaCodeLength;
  EncodeDCChromaTable.pCodeTable = EncDCChromaCode;

  EncodeACLumaTable.pCodeLengthTable = EncACLumaCodeLength;
  EncodeACLumaTable.pCodeTable = EncACLumaCode;
  EncodeACChromaTable.pCodeLengthTable = EncACChromaCodeLength;
  EncodeACChromaTable.pCodeTable = EncACChromaCode;

  // Setup the decoding tables
  DecodeDCLumaTable.pMinCodeTable = DecDCLumaMinCodeIndex;
  DecodeDCLumaTable.pMaxCodeTable = DecDCLumaMaxCodeIndex;
  DecodeDCLumaTable.pCodeCategoryTable = DecDCLumaCodeCategory;
  DecodeDCLumaTable.pCodeTable = DecDCLumaCode;

  DecodeDCChromaTable.pMinCodeTable = DecDCChromaMinCodeIndex;
  DecodeDCChromaTable.pMaxCodeTable = DecDCChromaMaxCodeIndex;
  DecodeDCChromaTable.pCodeCategoryTable = DecDCChromaCodeCategory;
  DecodeDCChromaTable.pCodeTable = DecDCChromaCode;

  DecodeACLumaTable.pMinCodeTable = DecACLumaMinCodeIndex;
  DecodeACLumaTable.pMaxCodeTable = DecACLumaMaxCodeIndex;
  DecodeACLumaTable.pCodeCategoryTable = DecACLumaCodeCategory;
  DecodeACLumaTable.pCodeTable = DecACLumaCode;

  DecodeACChromaTable.pMinCodeTable = DecACChromaMinCodeIndex;
  DecodeACChromaTable.pMaxCodeTable = DecACChromaMaxCodeIndex;
  DecodeACChromaTable.pCodeCategoryTable = DecACChromaCodeCategory;
  DecodeACChromaTable.pCodeTable = DecACChromaCode;

  // Now open file according to type of access
  if (Type == HUFFMANENCODE) {    // This object will be encoding
    AccessType = HUFFMANENCODE;   // Set flag to indicate encoding
    FileObject.OpenWriteFile(FileName);
  } else {
    AccessType = HUFFMANDECODE;   // Set flag to indicate decoding
    FileObject.OpenReadFile(FileName);
  }

  // Initialized Huffman buffers and counts
  PutBuffer = 0;                  // Bit accumulation buffer
  PutBufferBitCount = 0;          // Number of bits accumulated

  GetBuffer = 0;                  // Bit extraction buffer
  GetBufferBitCount = 0;          // Number of unused bits in GetBuffer
};

// Class Destructor
Huffman::~Huffman(void) {

  FileObject.CloseFile();         // Close file if not already closed
}



// This function writes NumberOfBits bits of Code to the output
// file. NumberOfBits is never larger than 16 bits.
void Huffman::PutNBits(WORD Code, int NumberOfBits) {
  register long put_buffer = Code;
  register int put_bits = PutBufferBitCount;

  put_buffer &= (((long) 1) << NumberOfBits) - 1;// Mask off excess bits

  put_bits += NumberOfBits;		  // Number of bits in buffer now
  put_buffer <<= 24 - put_bits;   // Align incoming bits and merge
  put_buffer |= PutBuffer;        // with current contents of buffer
  
  while (put_bits >= 8) {         // If more than 8 bits in buffer, write them
	int c = (int) ((put_buffer >> 16) & 0xFF);
    
	FileObject.WriteByte(c);      // Write a byte to output file
	if (c == 0xFF) 		          // Stuff zero byte if need be
	  FileObject.WriteByte(0);
    
	put_buffer <<= 8;             // Adjust accumulator and count
	put_bits -= 8;
  }

  PutBuffer = put_buffer;	      // Save global variables
  PutBufferBitCount = put_bits;
}

void Huffman::FlushInputStream(void) {

  GetBuffer = 0;                  // Bit extraction buffer
  GetBufferBitCount = 0;          // Number of unused bits in GetBuffer
}


// This function flushes the huffman bit stream. It is called after
// each MCU is written to the output file.
void Huffman::FlushOutputStream(void) {

  PutNBits((WORD) 0x7F, 7);       // Fill up a partial byte with ones

  PutBuffer = 0;                  // Reset PutBuffer to empty
  PutBufferBitCount = 0;
}



// This function encodes a block of coefficients
void Huffman::EncodeABlock(int *Block, ENCODEHUFFMANTABLE *DCTable,
                                       ENCODEHUFFMANTABLE *ACTable) {
  register long Temp;
  register int NumberOfBits;
  register int Index, NumberOfZeros, Category;
  
  // Find the number of bits needed for the magnitude of DC coefficient
  Temp = Block[0];
  if (Temp < 0) Temp = -Temp;
  
  NumberOfBits = 0;
  while (Temp) {
    NumberOfBits++;
    Temp >>= 1;
  }
  
  // Emit the Huffman code for the number of bits
  PutNBits(DCTable->pCodeTable[NumberOfBits], DCTable->pCodeLengthTable[NumberOfBits]);
  
  // If DC coefficient is positive, emit NumberOfBits low order bits
  // If DC coefficient is negative, emit NumberOfBits low order bits of
  // value minus one.
  if ((Temp = Block[0]) < 0)
    Temp--;
  
  PutNBits((WORD) Temp, NumberOfBits);
  
  // Now encode the AC coefficients

  NumberOfZeros = 0;			  // Count of zeros initially zero
  
  for (Index = 1; Index < 64; Index++) {
	if ((Temp = Block[Index]) == 0) {  // If coefficient is zero
	  NumberOfZeros++;            // Inc count of zeros
	} else {
      // Run length of zero is > 15, emit special run-length-16 code
	  while (NumberOfZeros > 15) {
		PutNBits(ACTable->pCodeTable[0xF0], ACTable->pCodeLengthTable[0xF0]);
		NumberOfZeros -= 16;
	  }

	  // Find the number of bits needed for the magnitude of AC coefficient
	  if (Temp < 0) Temp = -Temp;

	  NumberOfBits = 1;
	  while (Temp >>= 1)
		NumberOfBits++;

	  // Emit Huffman code for category which is run length (MSN) and
      // number of bits (LSN)
	  Category = (NumberOfZeros << 4) + NumberOfBits;
	  PutNBits(ACTable->pCodeTable[Category],
               ACTable->pCodeLengthTable[Category]);

      // If coefficient is positive, emit NumberOfBits low order bits
      // If coefficient is negative, emit NumberOfBits low order bits
      // of value minus one.
	  if ((Temp = Block[Index]) < 0)
		Temp--;
      
	  PutNBits((WORD) Temp, NumberOfBits);
      
	  NumberOfZeros = 0;
    }
  }

  // If the last coefficients were zero, emit an EOB code
  if (NumberOfZeros > 0)
	PutNBits(ACTable->pCodeTable[0], ACTable->pCodeLengthTable[0]);
}



// This function return the next NumberOfBits from the huffman bit
// stream.
int Huffman::GetNBits(int NumberOfBits) {
  int InChar, Result;
  
  // If requested number of bits not currently in buffer, read a new
  // byte from the input file.
  while (NumberOfBits > GetBufferBitCount) {
	InChar = FileObject.ReadByte();
    
	GetBuffer = (GetBuffer << 8) + InChar;
	GetBufferBitCount += 8;
    // If byte just fetched is an 0xFF, discard zero byte which follows
	if (InChar == 0xFF)
	  FileObject.ReadByte();      // Discard zero byte
  }
  // Manipulate the buffers to return requested number of bits
  GetBufferBitCount -= NumberOfBits;
  Result = (int)((GetBuffer >> GetBufferBitCount) & ((1 << NumberOfBits) - 1));
  return Result;
}

// This function returns the next Huffman code from the input stream. Code
// is returned from the specified Huffman table (either a Luma or Chroma
// table).
int Huffman::GetCodeFromStream(DECODEHUFFMANTABLE *lpTable) {
  WORD Length, Index, IndexOfMinCodeThisLength;
  long Code;
  
  // Start with a code of length 2 because there are no length one codes
  Length = 2;
  Code = GetNBits(2);
  // While Code is greater than max code of specified length, loop
  while (Code > lpTable->pMaxCodeTable[Length]) {
	  Code = (Code << 1) + GetNBits(1);
	  Length++;
  }
  // Get index of min code of required length
  IndexOfMinCodeThisLength = lpTable->pMinCodeTable[Length].Index;

  // Index is the index into the category table for the decoded code
  Index = (WORD)(IndexOfMinCodeThisLength +
          (Code - lpTable->pMinCodeTable[Length].CodeValue));

  // Lookup category and return
  return lpTable->pCodeCategoryTable[Index];
}

// Given the code and its number of bits, this function sign extends
// and returns the appropriate integer value.
int Huffman::DecodeCode(int Code, int NumberOfBits) {

  // Determine if code represents a negative number. If not just return the
  // code. If negative process as shown below.
  if (Code < (1 << (NumberOfBits - 1)))
    return (Code + 1 + (-1 << NumberOfBits));
  else
    return Code;
}

// Decode a block of coefficients
void Huffman::DecodeABlock(int *Block, DECODEHUFFMANTABLE *DCTable,
									   DECODEHUFFMANTABLE *ACTable) {
  int NumberOfBits, Index, Code, NumberOfZeros;

  memset(Block, 0, 64 * sizeof(int));  // Zero block of coefficients
  
  // Decode the single DC coefficient. For DC coefficient category is
  // bit count.
  NumberOfBits = GetCodeFromStream(DCTable);// Get number of bits of code
  Code = GetNBits(NumberOfBits);            // Get code
  Block[0] = DecodeCode(Code, NumberOfBits);// Store DC coefficient in Block[0]
  
  // Now decode the 15 AC coefficients
  for (Index = 1; Index < 64; Index++) {    // For each AC coefficient
	Code = GetCodeFromStream(ACTable);      // Read category from stream
    
	NumberOfBits  = Code & 15;    // Isolate bit count
	NumberOfZeros = Code >> 4;    // Isolate number of zeros count
    
	if (NumberOfBits) {           // Non zero bit count is normal case
	  Index += NumberOfZeros;     // Advance storage index by number of zeros
	  Code = GetNBits(NumberOfBits);   // Get code from stream
	  Block[Index] = DecodeCode(Code, NumberOfBits);  // Store code at index
	} else {                      // Special case of bit count = 0
	  if (NumberOfZeros != 0x0F)  // If not special 16 zero code
		break;                    // then it must be EOB, exit
      Index += 15;                // Code was ZRL, advance index by 15
    }
  }
}


// Encode a block of Huffman encoded data and write it to a file
void Huffman::EncodeBlock(int *InBlock, BOOL Type) {

  // Determine which Huffman tables to use for block encoding
  if (Type == USELUMATABLE)
	EncodeABlock(InBlock, &EncodeDCLumaTable, &EncodeACLumaTable);
  else
	EncodeABlock(InBlock, &EncodeDCChromaTable, &EncodeACChromaTable);
}


// Read data from a file and decode a block of Huffman encoded data
void Huffman::DecodeBlock(int *OutBlock, BOOL Type) {

  // Determine which Huffman tables to use for block decoding
  if (Type == USELUMATABLE)
	DecodeABlock(OutBlock, &DecodeDCLumaTable, &DecodeACLumaTable);
  else
	DecodeABlock(OutBlock, &DecodeDCChromaTable, &DecodeACChromaTable);
}

