/*
* *********************************************************************** 
*
* BinDiff: diff algorithm on binary files.
*
* By #defining BDEXTR as 1, a reduced, decode-only version is created
* If BDEXTR is not defined or 0, a complete version is created.
*
* ***********************************************************************
*
* Identify compiler features & platform.
* BorlandC = Borland C++ 3.1 (MSDOS) or Borland C++ 1.0 (OS/2)
* MacC = Symantec Think C 7.0 (Mac) or MetroWerks CodeWarrior 1.0 (Mac/PPC)
* StdUnixC = Kernighan & Ritchie (Unix)
*
* On Macintosh and in Borland C++, you must explicitly activate the C++ compiler 
* because normally the C compiler will be invoked (extension .c).
*
*/

#ifndef BDEXTR
#define BDEXTR      0
#endif

/*
* Identify compiler
*/

#define BorlandC    0
#define MacC        1
#define StdUnixC    0

/*
* Derive compiler and platform capabilities
*/

#if BorlandC
#define IsANSI      1
#define IsCPlusPlus 1
#ifdef __OS2__
#define IsMSDOS     0
#define IsOS2       1
#else
#define IsMSDOS     1
#define IsOS2       0
#endif
#define IsMac       0
#define IsUnix      0
#endif

#if StdUnixC
#define IsANSI      0
#define IsCPlusPlus 0
#define IsMSDOS     0
#define IsOS2       0
#define IsMac       0
#define IsUnix      1
#endif

#if MacC
#define IsANSI      1
#define IsCPlusPlus 1
#define IsMSDOS     0
#define IsOS2       0
#define IsMac       1
#define IsUnix      0
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <string.h>

#if StdUnixC
#include <malloc.h>
#endif

#if BorlandC
#include <conio.h>
#endif

#ifndef TRUE
#define TRUE             1
#endif

#ifndef FALSE
#define FALSE            0
#endif

#if IsMac
/*
* Dialog parameters
*/
#define cProgressDialogID     10000
#define cOKBtn                1
#define cCancelBtn            2
#define cTitleItm             3
#define cMessage1Itm          4
#define cMessage2Itm          5
#define cSetOrgBtn            6
#define cOrgFilItm            7
#define cSubtractRadioBtn     8
#define cAddRadioBtn          9
#define cSet2ndBtn           10
#define c2ndFilItm           11
#define cProgressBarTop      75
#define cProgressBarLeft     10
#define cProgressBarWidth   450
#define cProgressBarHeight   10


/*
* Diff file type and creator
*/
#define cCreator             'B1dF'
#define cFileType            '.DIF'

#endif /* IsMac */

#define cMinMeanChunkLen     20.0
#define cMaxMeanChunkLen     80.0
#define cBigChunkLen        500.0
#define cFileHeader         "MDOSbinaDelta0001"

/*
* if cDropEOL is TRUE, cr and lf are not allowed to function as delimiters
*/
#define cDropEOL           FALSE

#if IsMSDOS
#define cBufSiz            128L
#else
#define cBufSiz            512L
#endif /* IsMSDOS */

#define cMinMatchLen         6
#define cMinEqualRunLen      (cMinMatchLen+4) /* Must be >= cMinMatchLen */

/*
* Tag values. A tag value is encoded in the 4 lowest bits of a tag byte.
* The 4 high bits of the tag byte are used for encoding a value 0-15.
*/

#define cTagSmallDiff            0
#define cTagMediumDiff           1
#define cTagLargeDiff            2
#define cTagSmallNearCopy        3
#define cTagMediumNearCopy       4
#define cTagLargeNearCopy        5
#define cTagSmallDistantCopy     6
#define cTagMediumDistantCopy    7
#define cTagLargeDistantCopy     8
#define cTagSmallFarCopy         9
#define cTagMediumFarCopy     0x0A
#define cTagLargeFarCopy      0x0B
/* Tags 0x0C,0x0D,0x0E unused. */
#define cTagEOF               0x0F

/*
* Maximum values encodable by different tags.
* 4-bit value (0-15)       is used to encode a value             1 - cSmallSize
* 12-bit value (0-4095)    is used to encode a value  cSmallSize+1 - cMediumSize
* 20-bit value (0-1048575) is used to encode a value cMediumSize+1 - cLargeSize
*/

#define cSmallSize            16L
#define cMediumSize           (4096L+cSmallSize)
#define cLargeSize            (1048576L+cMediumSize)

/*
* Maximum file positions encodable in 2 or 3 bytes
*/

#define cNearDistance         0xFFFFL
#define cDistantDistance      0xFFFFFFL

#define cMaxStrLen       255

/*
* Small, generic types
*/

typedef unsigned char TByte;
typedef TByte        *TpByte;
typedef char         *TpChar;
typedef char          TStr[cMaxStrLen+1];
typedef int           TBoolean;

/*
* Definition of TpFil: for MSDOS and KR just a pointer to a FILE; for Think C
* it is a specific structure
*/

#if ! IsMac
typedef FILE         *TpFil;
#else
struct TFil
 {
  TBoolean      isDirty;
  short         filRefNum;
  long          curPos;
  long          bufPos;
  long          bufLen;
  TByte         buf[cBufSiz];
 };
 
typedef TFil *TpFil;

/*
* Redefine stream I/O functions (routine names are in 
* uppercase to avoid clashes with Think C's ANSI library
* In this version, stream I/O can be used on data as well as
* on the resource fork of a file (see FOPEN below).
*/

#ifdef putc
#undef putc
#endif

#ifdef getc
#undef getc
#endif

#define fopen(x,y,z)   FOPEN(x,y,z)
#define ftell(x)       FTELL(x)
#define fread(x,y,z,t) FREAD(x,y,z,t)
#define fclose(x)      FCLOSE(x)
#define fputs(x,y)     FPUTS(x,y)
#define putc(x,y)      PUTC(x,y)
#define getc(x)        GETC(x)
#define fseek(x,y,z)   FSEEK(x,y,z)
#define fflush(x)      FFLUSH(x)

#endif /* IsMac */

#if ! BDEXTR
/* The following structures are not needed in extraction program */

/*
* Tree node: has 3 pointers: a pointer to lesser, and greater or equal nodes
* The third pointer is a linked list pointer: the root node of the tree is also
* the head of the linked list. The linked list is not built in a special order:
* nodes are added to it in order of occurence. The tree is a binary tree.
*/

typedef struct STreeNode
 {
  long       filPos;
  TByte      bytes[cMinMatchLen];
  struct STreeNode *pGE;  /* Greater or equal */
  struct STreeNode *pLT;  /* Less than */
 } TTreeNode;

/*
* Match block structure: for each match found between the two files, we
* encode the positions in the two files and the length of the match.
*/

typedef struct SMatchBlock
 {
  long         len;
  long         orgFilPos;
  long         derivedFilPos;
  long         distance; /* = orgFilPos - derivedFilPos */
  struct SMatchBlock *pNxt;
 } TMatchBlock;

typedef TTreeNode    *TpTreeNode;

typedef TMatchBlock  *TpMatchBlock;

typedef struct
 {
  unsigned long   sum;
  double sumSquares;
  double mean;
  double stdDev;
  /*
  * Last file position where a byte was encountered, for all bytes. Initialized
  * to -1 ('before' the first file position).
  */
  long lastPos;
  /*
  * Count of occurences
  */
  long cnt;
 } TByteAttribs;

typedef TByteAttribs *TpByteAttribs;

#endif /* ! BDEXTR */

#if IsMSDOS
#if ! BDEXTR
extern unsigned _stklen = 65000; /* Big stack for MSDOS */
#endif
#endif

/*
* Progress bar global values
*/

long gMinVal;
long gMaxVal;
int  gCurPos;

#if IsMac
/*
* Macintosh dialog pointer
*/
DialogPtr gpProgressDialog;
#endif

/* ***********************************************************************
*
* Error message display + terminate program
*/

#if IsMac 
extern void C2PasStr(TpByte pasStr, TpChar cStr); /* Forward declaration */
#endif

#if IsANSI
void FatalError(TpChar errMsg)
#else
int FatalError(errMsg)
TpChar errMsg;
#endif
 {
#if IsMSDOS || IsOS2 || IsUnix
  fprintf(stderr,"%s\n",errMsg);
  exit(-1);
#elif IsMac
  Str255 str;
  short  itmHit;
  short  itmType;
  Handle itmHnd;
  Rect   itmRect;
  
  GetDItem(gpProgressDialog,cMessage1Itm,&itmType,&itmHnd,&itmRect);
  SetIText(itmHnd,"\pError:");
  GetDItem(gpProgressDialog,cMessage2Itm,&itmType,&itmHnd,&itmRect);
  C2PasStr(str,errMsg);
  SetIText(itmHnd,str);
  
  HideDItem(gpProgressDialog,cSetOrgBtn);
  HideDItem(gpProgressDialog,cOrgFilItm);
  HideDItem(gpProgressDialog,cSubtractRadioBtn);
  HideDItem(gpProgressDialog,cAddRadioBtn);
  HideDItem(gpProgressDialog,cSet2ndBtn);
  HideDItem(gpProgressDialog,c2ndFilItm);

  /*
  * Enable OK button
  */
  ShowDItem(gpProgressDialog,cOKBtn);
  GetDItem(gpProgressDialog,cOKBtn,&itmType,&itmHnd,&itmRect);
  PenSize(3,3);
  InsetRect(&itmRect,-4,-4);
  FrameRoundRect(&itmRect,16,16);
  PenSize(1,1);
  
  ModalDialog(NULL,&itmHit);
  
  ExitToShell();
#endif /* IsMac */

 }

#if IsMac
/* *********************************************************************** 
*
* Fill memory with a value. Necessary because memset does not work well in
* MetroWerks 1.0a1
*/

void MemSet(void *s, int c, size_t n)
 {
  TpByte pByte;

  pByte = (TpByte) s;
  while (n-- > 0)
   {
    *(pByte++) = c;
   }
 }

#define memset(x,y,z) MemSet(x,y,z)
#endif /* IsMac */

/* *********************************************************************** 
*
* Calculate size of an open file
*/

#if IsANSI
long FileSize(TpFil pFil)
#else
long FileSize(pFil)
TpFil pFil;
#endif
 {
  long filSiz;

#if IsMac
  GetEOF(pFil->filRefNum,&filSiz);
#else
  long curPos;

  /*
  * Save file position
  */
  curPos = ftell(pFil);
  /*
  * Go to end of file and determine file size.
  */
  fseek(pFil,0L,SEEK_END);
  filSiz = ftell(pFil);
  /*
  * Restore previous position
  */
  fseek(pFil,curPos,SEEK_SET);
#endif

  return(filSiz);

 }

#if IsMac
/* **********************************************************************
* *********************************************************************** 
*
* Routines that are only needed on Macintosh:
* 
*/

/* *********************************************************************** 
*
* Be nice to other programs
*/

void TaskSwitch(void)
 {
  EventRecord evt;
  
  SystemTask();
  GetNextEvent(everyEvent,&evt);
 }

/* *********************************************************************** 
*
* Convert Pascal-str to C format. Follows StrCpy convention: to-from.
*/

void Pas2CStr(TpChar cStr,TpByte pasStr)
 {
  TpChar pFrom, pTo;
  int len;

  len = pasStr[0];
  pTo = cStr;
  pFrom = (TpChar) pasStr + 1;
  while (len > 0)
   {
    *(pTo++) = *(pFrom++);
    len--;
   }
   
  *pTo = '\0';
  
 }

/* *********************************************************************** 
*
* Convert C-str to Pascal format. Follows StrCpy convention: to-from .
*/

void C2PasStr(TpByte pasStr, TpChar cStr)
 {
  TpChar pFrom, pTo;
  int len;
  
  len = 0;
  pFrom = cStr;
  pTo = (TpChar) pasStr + 1;
  while (*pFrom != '\0' && len < cMaxStrLen)
   {
    *(pTo++) = *(pFrom++);
    len++;
   }

  *pasStr = len;
 }
 
/* *********************************************************************** 
*
* Emulate ftell for Macintosh
*/

long FTELL(TpFil pFil)
 {
  
  return(pFil->curPos);
 
 }

/* *********************************************************************** 
*
* Emulate fflush for Macintosh
*/

int FFLUSH(TpFil pFil)
 {
  long  curPos;
  OSErr err;
  
  if (pFil->isDirty)
   {
    curPos = pFil->curPos - pFil->bufPos;
    err = SetFPos(pFil->filRefNum,fsFromStart,curPos);
    if (err == noErr)
     {
      FSWrite(pFil->filRefNum,&pFil->bufLen,pFil->buf);
     }
    pFil->isDirty = FALSE;
   }
   
  return(err);
  
 } 
 
/* *********************************************************************** 
*
* Emulate fseek for Macintosh
*/

int FSEEK(TpFil pFil,long pos,int whence)
 {
  long newPos;
  
  if (whence == SEEK_SET)
   {
    newPos = pos;
   }
  else if (whence == SEEK_END)
   {
    newPos = FileSize(pFil) + pos;
   }
  else
   {
    newPos = pFil->curPos + pos;
   }
   
  pFil->curPos -= pFil->bufPos;
  pFil->bufPos = 0;
  
  if (pFil->curPos <= newPos && newPos < pFil->curPos + pFil->bufLen)
   {
    pFil->bufPos = newPos - pFil->curPos;
   }
  else
   {
    if (pFil->isDirty) fflush(pFil);
    pFil->bufPos = 0;
    pFil->bufLen = 0;
   }
  
  pFil->curPos = newPos;
   
  return(0);
 } 

/* *********************************************************************** 
*
* Emulate getc for Macintosh
*/

int GETC(TpFil pFil)
 {  
  int c;
  OSErr err;
  
  if (pFil->bufPos >= pFil->bufLen)
   {
    if (pFil->isDirty) fflush(pFil);
    err = SetFPos(pFil->filRefNum,fsFromStart,pFil->curPos);
    if (err == noErr)
     {
      pFil->bufLen = cBufSiz;
      err = FSRead(pFil->filRefNum,&pFil->bufLen,pFil->buf);
      pFil->bufPos = 0;
     }
    else
     {
      pFil->bufLen = 0;
      pFil->bufPos = 0;
     }
   }
   
  if (pFil->bufPos >= pFil->bufLen)
   {
    c = EOF;
   }
  else
   {
    c = pFil->buf[pFil->bufPos++];
    pFil->curPos++;
   }
  
  return(c);
 } 

/* *********************************************************************** 
*
* Emulate putc for Macintosh
*/

int PUTC(int c, TpFil pFil)
 {  
  
  if (c >= 0 && c < 256)
   {
    if (pFil->bufPos >= cBufSiz)
     {
      if (pFil->isDirty) fflush(pFil);
      pFil->bufPos = 0;
      pFil->bufLen = 0;
     }
   
    pFil->buf[pFil->bufPos++] = c;
    if (pFil->bufPos > pFil->bufLen) pFil->bufLen++;
    pFil->curPos++;
    pFil->isDirty = TRUE;
   }
  
  return(c);
 } 

/* *********************************************************************** 
*
* Emulate putc for Macintosh
*/

int FPUTS(TpChar pChr, TpFil pFil)
 {  
 
  while (*pChr != '\0')
   {  
    if (pFil->bufPos >= cBufSiz)
     {
      if (pFil->isDirty) fflush(pFil);
      pFil->bufPos = 0;
      pFil->bufLen = 0;
     }
   
    pFil->buf[pFil->bufPos++] = *pChr;
    if (pFil->bufPos > pFil->bufLen) pFil->bufLen++;
    pFil->curPos++;
    pFil->isDirty = TRUE;
    pChr++;
   }
   
  return(0);
  
 } 
 
/* *********************************************************************** 
*
* Emulate fclose for Macintosh
*/

int FCLOSE(TpFil pFil)
 {  
  OSErr err;
  
  if (pFil->isDirty) fflush(pFil);
  err = FSClose(pFil->filRefNum);
  DisposePtr((Ptr) pFil);
  
  return(err);
  
 } 
 
/* *********************************************************************** 
*
* Emulate fopen for Macintosh: open data fork or resource fork.
* Mode is also used to show data or resource fork ("R" or "D" after "r" or "w"). 
* Allowed modes: "rR", "rD", "wR", "wD"
*/

TpFil FOPEN(TpChar filNam, TpChar mode, short vRefNum)
 {  
  OSErr  err;
  TpFil  pFil;
  Str255 str;
  
  pFil = new TFil();
  if (pFil == NULL)
   {
    FatalError("Out of memory");
   }
  C2PasStr(str,filNam);
  if (mode[1] == 'R')
   {
    err = OpenRF(str,vRefNum,&pFil->filRefNum);
    if (mode[0] == 'w')
     {
      if (err == fnfErr)
       {
        err = Create(str,vRefNum,cCreator,cFileType);
        err = OpenRF(str,vRefNum,&pFil->filRefNum);
       }
      else if (err == noErr)
       {
        SetEOF(pFil->filRefNum,0);
       }
     }
   }
  else
   {
    err = FSOpen(str,vRefNum,&pFil->filRefNum);
    if (mode[0] == 'w')
     {
      if (err == fnfErr)
       {
        err = Create(str,vRefNum,cCreator,cFileType);
        err = FSOpen(str,vRefNum,&pFil->filRefNum);
       }
      else if (err == noErr)
       {
        SetEOF(pFil->filRefNum,0);
       }
     }
   }
   
  if (err != noErr)
   {
    delete pFil;
    pFil = NULL;
   }
  else
   {
    pFil->curPos = 0;
    pFil->bufPos = 0;
    pFil->bufLen = 0;
    pFil->isDirty = FALSE;
   }
   
  return(pFil);
 } 
 
/* *********************************************************************** 
*
* Emulate fread for Macintosh
*/

long FREAD(void *pBuf,long sze,long nmemb,TpFil pFil)
 {
  OSErr  err;
  long   bufLen;
  long   bytBufLen;
  long   saveBytBufLen;

  if (pFil->isDirty) fflush(pFil);
  pFil->bufLen = 0;
  pFil->bufPos = 0;

  bufLen = 0;
  err = SetFPos(pFil->filRefNum,fsFromStart,pFil->curPos);
  if (err == noErr)
   {
    saveBytBufLen = bytBufLen = sze*nmemb;
    err = FSRead(pFil->filRefNum,&bytBufLen,pBuf);
    pFil->curPos += bytBufLen;
    if (bytBufLen != saveBytBufLen)
     {
      bufLen = bytBufLen / sze;
     }
    else
     {
      bufLen = nmemb;
     }
   }
   
  return(bufLen);
  
 } 

/*
* End of Mac-only section
*
* *********************************************************************** 
* *********************************************************************** */

#endif  /* IsMac */

/* *********************************************************************** 
*
* Read n bytes and return in a long (n=1,2,3,4)
*/

#if IsANSI
long ReadLongNBytes(TpFil pFil,int n)
#else
long ReadLongNBytes(pFil,n)
TpFil pFil;
int n;
#endif
 {
  long x;
  int c;

  x = 0;
  while (n > 0)
   {
    c = getc(pFil);
    if (c == EOF)
     {
      n = 0;
     }
    else
     {
      x = (x << 8) | c;
     }
    n--;
   }
  
  return(x);
 }

#if ! IsUnix

/* *********************************************************************** 
*
* Initialise progress bar
*/

#if IsANSI
void InitProgressBar(long minVal, long maxVal, TpChar message)
#else
int InitProgressBar(minVal,maxVal,message)
long minVal;
long maxVal;
TpChar message;
#endif
 {

#if BorlandC
  cprintf("%s\r\n",message);
  cprintf("________________________________________\r");
#elif IsMac
  short  itmType;
  Handle itmHnd;
  Rect   itmRect;
  Str255 pasStr;

  GetDItem(gpProgressDialog,cMessage2Itm,&itmType,&itmHnd,&itmRect);
  C2PasStr(pasStr,message);
  SetIText(itmHnd,pasStr);

  SetRect(&itmRect,cProgressBarLeft,
                   cProgressBarTop,
                   cProgressBarLeft+cProgressBarWidth,
                   cProgressBarTop+cProgressBarHeight);
  FrameRect(&itmRect);
#endif

  if (maxVal == minVal)
   {
    maxVal = minVal + 1;
   }

  gMinVal = minVal;
  gMaxVal = maxVal;
  gCurPos = 0;

 }

/* ***********************************************************************
*
* Change progress bar length according to curVal between gMinVal and gMaxVal
*/

#if IsANSI
void AdjustProgressBar(long curVal)
#else
int AdjustProgressBar(curVal)
long curVal;
#endif
 {
  int pos;

#if BorlandC

  pos = 40*(curVal - gMinVal) / (gMaxVal - gMinVal);
  while (pos > gCurPos)
   {

    cprintf("#");
    gCurPos++;

    /*
    * Check for keyboard interrupts
    */
    if (kbhit())
     {
      if (getch() == 3 /* Ctrl-C */)
       {
        FatalError("Aborting");
       }
     }
   }

#elif IsMac
  Rect itmRect;

  TaskSwitch();

  pos = cProgressBarWidth*(curVal - gMinVal) / (gMaxVal - gMinVal);
    
  SetRect(&itmRect,cProgressBarLeft+gCurPos,
                   cProgressBarTop,
                   cProgressBarLeft+pos,
                   cProgressBarTop+cProgressBarHeight);
  FillRect(&itmRect,&qd.black);
  gCurPos = pos;
#endif

 }

/* *********************************************************************** 
*
* Close progress bar
*/

#if IsANSI
void CloseProgressBar(void)
#else
int CloseProgressBar()
#endif
 {
 
  AdjustProgressBar(gMaxVal);
#if BorlandC
  cprintf("\r\n");
#elif IsMac
  short  itmType;
  Handle itmHnd;
  Rect   itmRect;
   
  GetDItem(gpProgressDialog,cMessage2Itm,&itmType,&itmHnd,&itmRect);
  SetIText(itmHnd,"\p");
  
  SetRect(&itmRect,cProgressBarLeft,
                   cProgressBarTop,
                   cProgressBarLeft+cProgressBarWidth,
                   cProgressBarTop+cProgressBarHeight);
  EraseRect(&itmRect);
#endif
 }

#endif /* ! IsUnix */

#if ! BDEXTR
/* *********************************************************************** 
*
* Write n lowest bytes of a long
*/

#if IsANSI
void WriteLongNBytes(long x, TpFil pFil,int n)
#else
int WriteLongNBytes(x,pFil,n)
long x;
TpFil pFil;
int n;
#endif
 {

  while (n > 4)
   {
    putc(0,pFil);
    n--;
   }

  if (n == 4)
   {
    putc(x >> 24,pFil);
    n--;
   }

  if (n == 3)
   {
    putc((x >> 16) & 0xFF,pFil);
    n--;
   }

  if (n == 2)
   {
    putc((x >> 8) & 0xFF,pFil);
    n--;
   }

  if (n == 1)
   {
    putc(x & 0xFF,pFil);
   }
 }

/* *********************************************************************** 
*
* Scans file and for each byte 0-255, calculate mean distance and standard
* deviation of the distances between occurences of these bytes:
*
* E.g. look at byte b in the file:
*
* xxxxxxbxxxxxxxxxbxxxxxbxxxxxxxxxxxbxxxxxxbxxxxxbxxxx
* < d1  ><   d2   >< d3 ><     d4   >< d5  >< d6 >
* <  7  ><   10   ><  6 ><    12    ><  7  ><  6 >
* The mean and std.dev. are calculated on distances d1,d2,... for byte b.
*/

#if IsANSI
void ScanFile(TpFil pFil, TpByteAttribs byteTable)
#else
int ScanFile(pFil,byteTable)
TpFil pFil;
TpByteAttribs byteTable;
#endif
 {
  int           byte;
  unsigned long curPos;
  unsigned long dist;
  TpByteAttribs pByteAttribs;
  double        dDist;

  /*
  * Initialize tables
  */
  pByteAttribs = byteTable;
  for (byte = 0; byte < 256; byte++)
   {
    pByteAttribs->lastPos = -1L;
    pByteAttribs->sum = 0L;
    pByteAttribs->sumSquares = 0.0;
    pByteAttribs->mean = 0.0;
    pByteAttribs->stdDev = 0.0;
    pByteAttribs->cnt = 0L;
    pByteAttribs++;
   }

  /*
  * Scan through file
  */
  fseek(pFil,0L,SEEK_SET);
  curPos = 0L;
  do
   {
#if ! IsUnix
    /*
    * Adjust progress bar every 4 K bytes
    */
    if ((curPos & 0xFFF) == 0)
      AdjustProgressBar(curPos);
#endif

    byte = getc(pFil);
    if (byte != EOF)
     {
      pByteAttribs = &byteTable[byte];
      /* Calculate distance from last occurrence of this byte */
      dDist = dist = curPos - pByteAttribs->lastPos;
      /* Remember this byte's position */
      pByteAttribs->lastPos = curPos;
      pByteAttribs->sum += dist;      
      pByteAttribs->sumSquares += dDist * dDist;
      
      /* cnt contains the number of occurrences */
      pByteAttribs->cnt++;
     }
    curPos++;
   }
  while (byte != EOF);

  /*
  * Calculate mean and standard deviation for all bytes.
  */
  pByteAttribs = byteTable;
  for (byte = 0; byte < 256; byte++)
   {
    /*
    * Make byte 'occur' just after EOF
    */
    dDist = dist = curPos - pByteAttribs->lastPos;
    pByteAttribs->sum += dist;
    pByteAttribs->sumSquares += dDist*dDist;
    pByteAttribs->cnt++;

    /*
    * Calculate mean. Bytes that did not occur get mean equal to file size
    */
    pByteAttribs->mean =
       (double) pByteAttribs->sum / (double) pByteAttribs->cnt;

    /*
    * Calculate standard deviation. We could also use the variance
    * but I like the std. dev. more.
    */
    pByteAttribs->stdDev =
      sqrt(pByteAttribs->sumSquares / (double) pByteAttribs->cnt
	   - pByteAttribs->mean*pByteAttribs->mean);

    pByteAttribs++;
   }

 }

/* *********************************************************************** 
*
* Analyze open file and determine a suitable delimiter for chopping the file
* into chunks. This routine changes the current file position.
*/

#if IsANSI
int FindDelimiter(TpFil pFil, 
		  double minMeanChunkLen,
		  double maxMeanChunkLen)
#else
int FindDelimiter(pFil,minMeanChunkLen,maxMeanChunkLen)
TpFil pFil;
double minMeanChunkLen;
double maxMeanChunkLen;
#endif
 {
  int    byte;
  int    bestByte;
  TByteAttribs byteTable[256];
  long   filSiz;

  filSiz = FileSize(pFil);
#if ! IsUnix
  InitProgressBar(0L,filSiz,"Pass 1 of 3:");
#endif

  ScanFile(pFil,byteTable);

  /*
  * Determine best byte
  */
  bestByte = -1;
  while (bestByte == -1
         && maxMeanChunkLen < cBigChunkLen 
         && maxMeanChunkLen < filSiz)
   {
    TpByteAttribs pByteAttribs;
    TpByteAttribs pBestByteAttribs;

    pByteAttribs = byteTable;
    pBestByteAttribs = NULL;
    for (byte = 0; byte < 256; byte++)
     {
#if cDropEOL
      if (byte != '\015' && byte != '\012')
#endif
       {
        /*
        * Check if chunk length is between minMeanLen and maxMeanLen.
        */
        if (pByteAttribs->mean >= minMeanChunkLen &&  
            pByteAttribs->mean <= maxMeanChunkLen)
         {
	  if (bestByte == -1)
	   { 
	    bestByte = byte;
	    pBestByteAttribs = pByteAttribs;
	   }
	  else
	   {
	    /*
	    * Compare stddev: if it is lower, the byte is better
	    */
	    if (pBestByteAttribs->stdDev > pByteAttribs->stdDev)
	     {
	      bestByte = byte;
	      pBestByteAttribs = pByteAttribs;
	     }
	   }
         }
       }
      pByteAttribs++;
     }
    /*
    * Increase allowable chunk length for the case no acceptable delimiter was
    * found: we will loop then
    */
    maxMeanChunkLen += 50;
   }

#if ! IsUnix
  CloseProgressBar();
#endif

  return(bestByte);

 }

/* *********************************************************************** 
*
* NewTreeNode(): create a new tree node or reuse one of the free list
*/

TpTreeNode gLstFreeTreeNode = NULL;

#if IsANSI
TpTreeNode NewTreeNode(void)
#else
TpTreeNode NewTreeNode()
#endif
 {
  TpTreeNode pTreeNode;

  if (gLstFreeTreeNode == NULL)
   {
#if IsCPlusPlus
    pTreeNode = new TTreeNode();
#else
    pTreeNode = (TpTreeNode) malloc(sizeof(TTreeNode));
#endif
    if (pTreeNode == NULL)
     {
      FatalError("Out of memory");
     }
   }
  else
   {
    pTreeNode = gLstFreeTreeNode;
    gLstFreeTreeNode = gLstFreeTreeNode->pGE;
   }
  
  pTreeNode->filPos = 0L;
  pTreeNode->pGE = NULL;
  pTreeNode->pLT = NULL;

  return(pTreeNode);

 }

/* *********************************************************************** 
*
* Release a tree node: put on the free list
*/

#if IsANSI
void FreeTreeNode(TpTreeNode pTreeNode)
#else
int FreeTreeNode(pTreeNode)
TpTreeNode pTreeNode;
#endif
 {
  pTreeNode->pGE = gLstFreeTreeNode;
  gLstFreeTreeNode = pTreeNode;
 }
 
/* *********************************************************************** 
*
* Compare two tree nodes; return -1, 0, 1 if <, = , >.
* Every tree node has cMinMatchLen characters of the file
* buffered within the node: first compare these. If all these characters
* are equal, read characters from the associated files and compare these.
* If the nodes are different, set equalLen to the number of equal characters
* encountered.
* delim is the chopping character.
*/

#if IsANSI
int CmpNode(TpTreeNode pNode1,
            TpFil      pFil1,
            TpTreeNode pNode2,
            TpFil      pFil2,
            int        delim,
            long      *pEqualLen)
#else
int CmpNode(pNode1,pFil1,pNode2,pFil2,delim,pEqualLen)
TpTreeNode pNode1;
TpFil pFil1;
TpTreeNode pNode2;
TpFil pFil2;
int delim;
long *pEqualLen;
#endif
 {
  long   pos1;
  TByte  buf1[cBufSiz];
  TpByte p1;
  long   pos2;
  TByte  buf2[cBufSiz];
  TpByte p2;
  int    result;
  long   l;

  pos1 = pNode1->filPos;
  pos2 = pNode2->filPos;
  result = -2; /* -2 means: not yet defined */
  *pEqualLen = 0;

  /*
  * First compare the cMinMatchLen buffered bytes from both nodes
  */
  p1 = pNode1->bytes;
  p2 = pNode2->bytes;
  l = 0;
  while (l < cMinMatchLen && *p1 == *p2 && *p1 != delim /* && *p2 != delim */)
   {
    p1++;
    p2++;
    l++;
    (*pEqualLen)++;
   }

  if (l == cMinMatchLen)
   {
    /*
    * If no difference was found, we will have to compare both files from
    * cMinMatchLen bytes after pos1 and pos2.
    */
    pos1 += cMinMatchLen;
    pos2 += cMinMatchLen;
   }
  else if (*p1 == delim && *p2 != delim)
   {
    result = -1; /* node 1 < node 2 because node 1 is shorter */
   }
  else if (*p2 == delim && *p1 != delim)
   {
    result = 1; /* node 2 < node 1 because node 2 is shorter */
   }
  else if (*p1 == *p2 && *p1 == delim)
   {
    result = 0; /* node 1 == node 2: both end in a delimiter at the same time */
   }
  else if (*p1 < *p2)
   {
    result = -1; /* node 1 < node 2 because a different character found */
   }
  else if (*p2 < *p1)
   {
    result = 1; /* node 2 < node 1 because a different character found */
   }

  /*
  * If result is -2, no difference was found in the buffered bytes. Start 
  * reading bytes from the files in chuncks of cBufSiz bytes.
  */
  if (result == -2)
   {
    do
     {
      /*
      * Read a buffer from file 1. Pad file with delimiter after eof.
      */
      fseek(pFil1,pos1,SEEK_SET);
      l = fread(buf1,1L,cBufSiz,pFil1);
      if (l < cBufSiz) buf1[l] = delim;

      /*
      * Read a buffer from file 2. Pad file with delimiter after eof.
      */
      fseek(pFil2,pos2,SEEK_SET);
      l = fread(buf2,1L,cBufSiz,pFil2);
      if (l < cBufSiz) buf2[l] = delim;

      /*
      * Compare buffers
      */
      p1 = buf1;
      p2 = buf2;
      l = 0;
      while (l < cBufSiz && *p1 == *p2 && *p1 != delim /* && *p2 != delim */)
       {
	p1++;
        p2++;
	l++;
        (*pEqualLen)++;
       }

      /*
      * If no difference was found: set file positions to read new buffers
      */
      if (l == cBufSiz)
       {
	pos1 += cBufSiz;
        pos2 += cBufSiz;
       }
      else if (*p1 == delim && *p2 != delim)
       {
        result = -1; /* node 1 < node 2  */
       }
      else if (*p2 == delim && *p1 != delim)
       {
	result = 1; /* node 2 < node 1 */
       }
      else if (*p1 == *p2 && *p1 == delim)
       {
        result = 0; /* node 1 == node 2 */
       }
      else if (*p1 < *p2)
       {
        result = -1;
       }
      else if (*p2 < *p1)
       {
        result = 1;
       }
     }
    while (result == -2);
    
   }
  
  return(result);

 }

/* *********************************************************************** 
*
* From a tree with root node pOrgTreeRoot, find the node pOrgTreeNode that best
* matches pDerivedTreeNode. Also find length of match.
*/

#if IsANSI
void FindBestMatch(TpTreeNode  pOrgTreeRoot,
		   TpTreeNode *ppOrgTreeNode,
                   TpFil       pOrgFil,
		   TpTreeNode  pDerivedTreeNode,
                   TpFil       pDerivedFil,
                   int         delim,
		   long       *pMatchLen)
#else
int FindBestMatch(pOrgTreeRoot,ppOrgTreeNode,pOrgFil,
        	  pDerivedTreeNode,pDerivedFil,
                  delim,pMatchLen)
TpTreeNode  pOrgTreeRoot;
TpTreeNode  *ppOrgTreeNode;
TpFil       pOrgFil;
TpTreeNode  pDerivedTreeNode;
TpFil       pDerivedFil;
int         delim;
long       *pMatchLen;
#endif
 {
  int        direction;
  TpTreeNode pNode;
  long       equalLen;

  /*
  * Find best location from tree
  */
  *ppOrgTreeNode = NULL;
  pNode = pOrgTreeRoot;
  *pMatchLen = 0L;

  /*
  * Descend tree and remember node with longest match
  */
  while (pNode != NULL)
   {

    direction = CmpNode(pDerivedTreeNode,pDerivedFil,pNode,pOrgFil,delim,&equalLen);

    /*
    * Remember match if length is greater than previous best
    */
    if (equalLen > *pMatchLen)
     {
      *pMatchLen = equalLen;
      *ppOrgTreeNode = pNode;
     }

    if (direction == -1)
     {
      pNode = pNode->pLT;
     }
    else if (direction == 1)
     {
      pNode = pNode->pGE;
     }
    else /* Node is equal: stop search */
     {
      pNode = NULL;
     }
   }

 }

/* *********************************************************************** 
*
* Add new node to index tree and linked list.
*
*/

#if IsANSI
void AddTreeNode(TpTreeNode *ppTreeRoot, TpFil pFil, int delim, TpTreeNode pNewNode)
#else
int AddTreeNode(ppTreeRoot,pFil,delim,pNewNode)
TpTreeNode *ppTreeRoot;
TpFil pFil;
int delim;
TpTreeNode pNewNode;
#endif
 {
  TpTreeNode pNode;
  TpTreeNode pPrvNode;
  int        cmp;
  long       equalLen;  
        
  /*
  * Find location in tree
  */
  pNode = *ppTreeRoot;
  pPrvNode = NULL;
  while (pNode != NULL)
   {
    pPrvNode = pNode;
    cmp = CmpNode(pNewNode,pFil,pNode,pFil,delim,&equalLen);
    if (cmp == -1)
     {
      pNode = pNode->pLT;
     }
    else if (cmp == 1) 
     {
      pNode = pNode->pGE;
     }
    else /* There is an equal node: stop looking */
     {
      pNode = NULL;
     }
   }

  if (pPrvNode == NULL)
   {
    *ppTreeRoot = pNewNode;
   }
  else
   {
    if (cmp == -1)
     {
      pPrvNode->pLT = pNewNode;
     }
    else if (cmp == 1)
     {
      pPrvNode->pGE = pNewNode;
     }
    else
     {
      FreeTreeNode(pNewNode);
     }
   }
 }

/* *********************************************************************** 
*
* Build index tree: divide file in chunks by using the delimiter. We also
* create chunks of strings that contain cMinMatchLen or more equal bytes.
*
*/

#if IsANSI
TpTreeNode BuildTree(TpFil pFil, int delim, long *pOrgSum)
#else
TpTreeNode BuildTree(pFil,delim,pOrgSum)
TpFil pFil;
int delim;
long *pOrgSum;
#endif
 {
  TpTreeNode pTreeRoot;
  TpTreeNode pNewNode;
  TpTreeNode pEqualRunNode;
  int        byte;
  int        prevByte;
  long       equalByteCnt;
  long       curPos;
  long       prevPos;
  long       len;
  TpByte     pByte;

#if ! IsUnix
  InitProgressBar(0L,FileSize(pFil),"Pass 2 of 3:");
#endif

  pTreeRoot = NULL;

  curPos = 0;
  prevPos = 0;
  do
   {
#if ! IsUnix
    if (curPos - prevPos > 0x3FF) 
     {
      AdjustProgressBar(curPos);
      prevPos = curPos;
     }
#endif
     
    /*
    * Restore file position (because it is destroyed by CmpNode)
    */
    fseek(pFil,curPos,SEEK_SET);
    
    /*
    * Prepare new node
    */
    pNewNode = NewTreeNode();
    
    len = 0;
    pNewNode->filPos = curPos;
    pByte = pNewNode->bytes;
    byte = -1;
    equalByteCnt = 0;
    do
     {
      prevByte = byte;
      if ((byte = getc(pFil)) != EOF)
       {
        if (byte == prevByte)
         {
          equalByteCnt++;
         }
        else
         {
          /*
          * No need to check for cMinEqualRunLen or more equal bytes here; 
          * if they were here, they will be added to the tree anyhow. This 
          * loop only executes at most cMinMatchLen times.
          */
          equalByteCnt = 1;
         }
        (*pOrgSum) += byte;
        *(pByte++) = byte;
       }
      else
       {
        *(pByte++) = delim;
       }
      
      len++;
     }
    while (len < cMinMatchLen && byte != delim && byte != EOF);
    
    while (byte != delim && byte != EOF)
     {
      prevByte = byte;
      if ((byte = getc(pFil)) != EOF)
       {
        if (byte == prevByte)
         {
          equalByteCnt++;
         }
        else
         {
          /*
          * Check for long runs of equal bytes, and add these to 
          * the tree also.
          */
          if (equalByteCnt >= cMinEqualRunLen)
           {
            pEqualRunNode = NewTreeNode();
            /* Buffered characters consists of cMinMatchLen equal bytes */
            memset(pEqualRunNode->bytes,prevByte,cMinMatchLen);
            /* Calculate position of start of run */
            pEqualRunNode->filPos = curPos + len - equalByteCnt;
            AddTreeNode(&pTreeRoot,pFil,delim,pEqualRunNode);
            fseek(pFil,curPos+len+1,SEEK_SET); /* Restore file position */
           }
          equalByteCnt = 1;
         }
        (*pOrgSum) += byte;
       }
      len++;
     }
  
    AddTreeNode(&pTreeRoot,pFil,delim,pNewNode);
    curPos += len;
          
   }
  while (byte != EOF);

#if ! IsUnix
  CloseProgressBar();
#endif

  return(pTreeRoot);
 }

/* *********************************************************************** 
*
* Extend match in two directions, ignoring delimiters: if there is a match
* of length n at position p and p', it can be changed into a match of
* length n+m+q at position p - m and p' - m
*        <  m   ><   n     ><  q  >
*                p
*  ...aaazzzzzzzzxxxxxxxxxxxyyyyyyyccccccccccccccccc    (pOrgFil)
*  ...bbbzzzzzzzzxxxxxxxxxxxyyyyyyyddddddddddddddddd    (pDerivedFil)
*                p'
*/

#if IsANSI
void ExtendMatch(long *pOrgFilPos, TpFil pOrgFil,
		 long *pDerivedFilPos, TpFil pDerivedFil,
                 long *pMatchLen)
#else
int ExtendMatch(pOrgFilPos,pOrgFil,pDerivedFilPos,pDerivedFil,pMatchLen)
long *pOrgFilPos;
TpFil pOrgFil;
long *pDerivedFilPos;
TpFil pDerivedFil;
long *pMatchLen;
#endif
 {
  TByte buf1[cBufSiz];
  TByte buf2[cBufSiz];
  int l;
  long step;
  TBoolean isDone;
  TpByte p1, p2;
  long endPos1, endPos2;

  /*
  * First, try to read forward and extend match toward the end
  */
  isDone = FALSE;
  endPos1 = *pOrgFilPos + *pMatchLen;
  endPos2 = *pDerivedFilPos + *pMatchLen;
  while (! isDone)
   {
    step = cBufSiz;
    fseek(pOrgFil,endPos1,SEEK_SET);
    /* Eventually shrink step if pOrgFil is too short */
    step = fread(buf1,1L,step,pOrgFil);

    fseek(pDerivedFil,endPos2,SEEK_SET);
    /* Eventually shrink step if derivedFil is too short */
    step = fread(buf2,1L,step,pDerivedFil);
    
    /* step < cBufSiz if one of the files at eof: stop comparing */
    isDone = (step < cBufSiz);
    
    /* Prepare endPos1 and endPos2 for reading next buffer */
    endPos1 += step;
    endPos2 += step;
    
    p1 = buf1;
    p2 = buf2;
    while (*p1 == *p2 && step > 0)
     {
      p1++;
      p2++;
      step--;
      (*pMatchLen)++;
     }

    /* step > 0 if *p1 != *p2 found: stop comparing */
    isDone = isDone || step > 0;
    
   }
  
  /*
  * Second, try to read backwards and extend the match towards the file start.
  * Stop extending if orgFilPos or derivedFilPos equal 0.
  */
  isDone = FALSE;
  while (*pOrgFilPos > 0 && *pDerivedFilPos > 0 && ! isDone)
   {
    /*
    * Try to step cBufSiz bytes back. If orgFilPos or derivedFilPos is 
    * less than that, reduce the step size accordingly.
    */
    step = cBufSiz;
    if (*pOrgFilPos < step)
      step = *pOrgFilPos;
    if (*pDerivedFilPos < step)
      step = *pDerivedFilPos;

    /* Jump back 'step' bytes and read two buffers of 'step' bytes. */
    (*pOrgFilPos) -= step;
    fseek(pOrgFil,*pOrgFilPos,SEEK_SET);
    fread(buf1,1L,step,pOrgFil);

    (*pDerivedFilPos) -= step;
    fseek(pDerivedFil,*pDerivedFilPos,SEEK_SET);
    fread(buf2,1L,step,pDerivedFil);

    /* Put pointers at the end of the buffers */
    p1 = buf1 + step - 1;
    p2 = buf2 + step - 1;
    
    /* Run backwards until a difference found or at start of buffer */
    l = step;
    while (*p1 == *p2 && l > 0)
     {
      p1--;
      p2--;
      l--;
      /*
      * Adjust matchLen for each matched byte.
      */
      (*pMatchLen)++;
     }
     
    isDone = (l > 0);
    /*
    * Adjust orgFilPos and derivedFilPos: add length of unequal part of buffer
    */
    (*pOrgFilPos) += l;
    (*pDerivedFilPos) += l;
   }

 }

/* *********************************************************************** 
*
* NewMatchBlock(): create a new match block or reuse one of the free list
*/

TpMatchBlock gLstFreeMatchBlock = NULL;

#if IsANSI
TpMatchBlock NewMatchBlock(void)
#else
TpMatchBlock NewMatchBlock()
#endif
 {
  TpMatchBlock pMatchBlock;

  if (gLstFreeMatchBlock == NULL)
   {
#if IsCPlusPlus
    pMatchBlock = new TMatchBlock();
#else
    pMatchBlock = (TpMatchBlock) malloc(sizeof(TMatchBlock));
#endif
    if (pMatchBlock == NULL)
     {
      FatalError("Out of memory");
     }
   }
  else
   {
    pMatchBlock = gLstFreeMatchBlock;
    gLstFreeMatchBlock = gLstFreeMatchBlock->pNxt;
   }
  
  pMatchBlock->orgFilPos = 0L;
  pMatchBlock->len = 0L;
  pMatchBlock->derivedFilPos = 0L;
  pMatchBlock->pNxt = NULL;

  return(pMatchBlock);

 }

/* *********************************************************************** 
*
* Release a match block: put on the free list
*/

#if IsANSI
void FreeMatchBlock(TpMatchBlock pMatchBlock)
#else
int FreeMatchBlock(pMatchBlock)
TpMatchBlock pMatchBlock;
#endif
 {
  pMatchBlock->pNxt = gLstFreeMatchBlock;
  gLstFreeMatchBlock = pMatchBlock;
 }
 
/* *********************************************************************** 
*
* Add new match between pOrgTreeNode and pDerivedTreeNode, with length
* matchlen. If the match is long enough (after extending): add it to
* the list of matches. The list of matches is kept in order of increasing 
* starting position in derivedFil.
* If a new match is added that is completely part of an existing match, it
* is dropped.
* If a new match is added that encloses one or more existing matches, the
* enclosed matches are dropped, and the new one is added.
* The resulting list will only contain matches with different derivedFilPos 
* values: if there would be two equal derivedFilPos values, one of the two
* blocks would enclose the other and would have been dropped.
*/

#if IsANSI
void AddMatch(TpTreeNode pOrgTreeNode, TpFil pOrgFil,
              TpTreeNode pDerivedTreeNode, TpFil pDerivedFil,
              long matchLen,
              TpMatchBlock *ppMatchLst)
#else
int AddMatch(pOrgTreeNode,pOrgFil,pDerivedTreeNode,pDerivedFil,matchLen,ppMatchLst)
TpTreeNode pOrgTreeNode;
TpFil pOrgFil;
TpTreeNode pDerivedTreeNode;
TpFil pDerivedFil;
long matchLen;
TpMatchBlock *ppMatchLst;
#endif
 {
  long orgFilPos, derivedFilPos;
  long distance;
  TpMatchBlock pMatchBlock, pPrvMatchBlock, pNxtMatchBlock;
  TBoolean dropNewMatch;

  orgFilPos = pOrgTreeNode->filPos;
  derivedFilPos = pDerivedTreeNode->filPos;
  /*
  * Pass 1: check if there are matchblocks that enclose the new match
  * (relative to derivedFil), in a way that they are an expansion of this
  * new match block. This saves us expanding this block, because after
  * expansion, it would be the same as the overlapping block.
  * This is done by checking the distance between the positions for
  * orgFil and derivedFil: if the larger block and the smaller block have 
  * the same distance, and the smaller block is part of the larger 
  * one, the smaller will expand to be equal to the larger one.
  */

  distance = orgFilPos - derivedFilPos;

  pMatchBlock = *ppMatchLst;
  dropNewMatch = FALSE;
  while (pMatchBlock != NULL 
         && pMatchBlock->derivedFilPos <= derivedFilPos 
	 && ! dropNewMatch)
   {
    dropNewMatch =
    (
   /* pMatchBlock->derivedFilPos <= derivedFilPos 
     &&                  Already tested in while condition */
      derivedFilPos <= pMatchBlock->derivedFilPos + pMatchBlock->len
     &&
      pMatchBlock->distance == distance
    );
    pMatchBlock = pMatchBlock->pNxt;
   }
  
  if (! dropNewMatch)
   {
    ExtendMatch(&orgFilPos,pOrgFil,&derivedFilPos,pDerivedFil,&matchLen);

    if (matchLen >= cMinMatchLen)
     {
      /*
      * Pass 2: check if there are matchblocks that enclose the new match
      * (relative to derivedFil).
      */
      pMatchBlock = *ppMatchLst;
      while (pMatchBlock != NULL 
             && pMatchBlock->derivedFilPos <= derivedFilPos 
	     && ! dropNewMatch)
       {
	dropNewMatch =
	   (/*
             pMatchBlock->derivedFilPos <= derivedFilPos 
	    &&      Already tested in while condition */
             derivedFilPos+matchLen 
               <= 
             pMatchBlock->derivedFilPos + pMatchBlock->len 
           );
        pMatchBlock = pMatchBlock->pNxt;
       }
    
      if (! dropNewMatch)
       {
        /*
	* Pass 3: drop all matchblocks from list that are enclosed by the new one
        */
        pMatchBlock = *ppMatchLst;
	pPrvMatchBlock = NULL;
        while (pMatchBlock != NULL)
         {
	  pNxtMatchBlock = pMatchBlock->pNxt;
	  /* Check if pMatchBlock completely enclosed by new match. */
	  if
	   (
             derivedFilPos <= pMatchBlock->derivedFilPos
	    &&
             pMatchBlock->derivedFilPos + pMatchBlock->len 
              <= 
             derivedFilPos+matchLen
           )
           {
	    /* If completely enclosed: remove from list */
            if (pPrvMatchBlock == NULL)
             {
              *ppMatchLst = pNxtMatchBlock;
             }
            else
             {
              pPrvMatchBlock->pNxt = pNxtMatchBlock;
	     }
            FreeMatchBlock(pMatchBlock);
           }
	  else
           {
	    pPrvMatchBlock = pMatchBlock;
	   }
          pMatchBlock = pNxtMatchBlock;
	 }
         
        /*
	* Pass 4: Find location to add match to list; keep list sorted on
	* derivedFilPos.
        */
        pNxtMatchBlock = *ppMatchLst;
        pPrvMatchBlock = NULL;
        while (pNxtMatchBlock != NULL 
               && pNxtMatchBlock->derivedFilPos < derivedFilPos)
         {
          pPrvMatchBlock = pNxtMatchBlock;
	  pNxtMatchBlock = pNxtMatchBlock->pNxt;
         }
        
	/*
	* Add new match
	*/
	pMatchBlock = NewMatchBlock();
        pMatchBlock->orgFilPos = orgFilPos;
	pMatchBlock->derivedFilPos = derivedFilPos;
        pMatchBlock->distance = distance;
	pMatchBlock->len = matchLen;
        pMatchBlock->pNxt = pNxtMatchBlock;
        if (pPrvMatchBlock == NULL)
	 {
          *ppMatchLst = pMatchBlock;
         }
        else
         {
          pPrvMatchBlock->pNxt = pMatchBlock;
         }
       }
     }
   }
 }

/* *********************************************************************** 
*
* Clean up the list of matches: shrink overlapping matches
* and drop those that become too short
*/

#if IsANSI
void ShrinkMatchList(TpMatchBlock *ppMatchLst)
#else
int ShrinkMatchList(ppMatchLst)
TpMatchBlock *ppMatchLst;
#endif
 {
  TpMatchBlock pMatchBlock;
  TpMatchBlock pPrvMatchBlock;
  TpMatchBlock pNxtMatchBlock;
  long distance;
  
  pPrvMatchBlock = NULL;
  pMatchBlock = *ppMatchLst;
  while (pMatchBlock != NULL)
   {
    pNxtMatchBlock = pMatchBlock->pNxt;
    if (pNxtMatchBlock != NULL)
     {
      /* distance is maximal length of pMatchBlock without overlap */
      distance = pNxtMatchBlock->derivedFilPos - pMatchBlock->derivedFilPos;
      /* Shrink block if too long */
      if (distance < pMatchBlock->len)
       {
	pMatchBlock->len = distance;
       }
     }
    /*
    * Drop blocks that become too short.
    */
    if (pMatchBlock->len < cMinMatchLen)
     {
      if (pPrvMatchBlock == NULL)
       {
        *ppMatchLst = pNxtMatchBlock;
       }
      else
       {
	pPrvMatchBlock->pNxt = pNxtMatchBlock;
       }
      FreeMatchBlock(pMatchBlock);
     }
    else
     {
      pPrvMatchBlock = pMatchBlock;
     }
     
    pMatchBlock = pNxtMatchBlock;
   }
   
 }

/* *********************************************************************** 
*
* Compare chunks from file 2 with tree from file 1
*/

#if IsANSI
TpMatchBlock MatchFiles(TpTreeNode pOrgTreeRoot,
			TpFil      pOrgFil,
			TpFil      pDerivedFil,
			int        delim,
			long      *pDerivedSum)
#else
TpMatchBlock MatchFiles(pOrgTreeRoot,pOrgFil,pDerivedFil,delim,pDerivedSum)
TpTreeNode pOrgTreeRoot;
TpFil      pOrgFil;
TpFil      pDerivedFil;
int        delim;
long      *pDerivedSum;
#endif
 {
  TpTreeNode   pOrgTreeNode;
  long         matchLen;
  TpMatchBlock pMatchLst;
  int          byte;
  int          prevByte;
  long         curPos;
  long         prevPos;
  TTreeNode    treeNode;
  TTreeNode    equalRunNode;
  long         len;
  long         equalByteCnt;
  TpByte       pByte;

#if ! IsUnix
  InitProgressBar(0L,FileSize(pDerivedFil),"Pass 3 of 3:");
#endif

  pMatchLst = NULL;

  curPos = 0;
  prevPos = 0;
  do
   {
#if ! IsUnix
    if (curPos - prevPos > 0x3FF)
     {
      AdjustProgressBar(curPos);
      prevPos = curPos;
     }
#endif

    /*
    * Restore file position (destroyed by FindBestMatch below)
    */
    fseek(pDerivedFil,curPos,SEEK_SET);

    /*
    * Buffer some characters from the file.
    */
    len = 0;
    treeNode.filPos = curPos;
    pByte = treeNode.bytes;
    byte = -1;
    equalByteCnt = 0;
    do
     {
      prevByte = byte;
      if ((byte = getc(pDerivedFil)) != EOF) 
       {
        if (byte == prevByte)
         {
          equalByteCnt++;
         }
        else
         {
          /*
          * No need to check for cMinMatchLen or more equal bytes here; 
          * if they were here, they will be checked against the tree 
          * anyhow. This loop only executes at most cMinMatchLen times.
          */
          equalByteCnt = 1;
         }
        (*pDerivedSum) += byte;
        *(pByte++) = byte;
       }
      else
       {
        *(pByte++) = delim;
       }
      len++;
     }
    while (len < cMinMatchLen && byte != delim && byte != EOF);
    
    while (byte != delim && byte != EOF)
     {
      prevByte = byte;
      if ((byte = getc(pDerivedFil)) != EOF)
       {
        if (byte == prevByte)
         {
          equalByteCnt++;
         }
        else
         {
          /*
          * Check for long runs of equal bytes, and check these against 
          * the tree also.
          */
          if (equalByteCnt >= cMinEqualRunLen)
           {
            /* Buffered characters consists of cMinMatchLen equal bytes */
            memset(equalRunNode.bytes,prevByte,cMinMatchLen);
            /* Calculate position of start of run */
            equalRunNode.filPos = curPos + len - equalByteCnt;
            /*
            * Search best match in original tree
            */
            FindBestMatch(pOrgTreeRoot,&pOrgTreeNode,pOrgFil,
                          &equalRunNode,pDerivedFil,
                          delim,&matchLen);

            if (pOrgTreeNode != NULL) 
             {
              /*
              * Add match to list of matches
              */
              AddMatch(pOrgTreeNode,pOrgFil, 
                       &equalRunNode,pDerivedFil,
                       matchLen,&pMatchLst);
             }
            fseek(pDerivedFil,curPos+len+1,SEEK_SET); /* Restore file position */
           }
          /*
          * Reset equal byte count to 1.
          */
          equalByteCnt = 1;
         }
        (*pDerivedSum) += byte;
       }
      len++;
     }
    curPos += len;
    
    /*
    * Search best match in original tree
    */
    FindBestMatch(pOrgTreeRoot,&pOrgTreeNode,pOrgFil,
                  &treeNode,pDerivedFil,
                  delim,&matchLen);

    if (pOrgTreeNode != NULL) 
     {
      /*
      * Add match to list of matches
      */
      AddMatch(pOrgTreeNode,pOrgFil, 
               &treeNode,pDerivedFil,
               matchLen,&pMatchLst);
     }
     
   }
  while (byte != EOF);
  
  /*
  * Remove overlapping matches
  */
  ShrinkMatchList(&pMatchLst);

#if ! IsUnix
  CloseProgressBar();
#endif

  return(pMatchLst);

 }

/* *********************************************************************** 
*
* Write diff file
*/

#if IsANSI
void DumpDiff(TpMatchBlock pMatchLst, TpFil pDerivedFil, TpFil pDiffFil)
#else
int DumpDiff(pMatchLst,pDerivedFil,pDiffFil)
TpMatchBlock pMatchLst;
TpFil pDerivedFil;
TpFil pDiffFil;
#endif
 {
  TpMatchBlock pMatchBlock;
  long len, pos;
  long blockPos;
  long writeLen;
  long blockLen;
  long codeLen;
  long filSiz;
  long nextPos;

  filSiz = FileSize(pDerivedFil);
  
#if BorlandC
  cprintf("Dumping diff.\r\n");
#endif

  /*
  * Descend match block list. Resulting file is a series of matches and
  * non-matches between original file and derived file.
  */
  pMatchBlock = pMatchLst;
  len = filSiz;
  pos = 0;
  /*
  * Repeat until all bytes of derivedFil have been checked
  */
  while (len > 0)
   {
    /*
    * Get next matching position from match block. If there is none,
    * set nextPos beyond eof on derived file.
    * If there are unmatched bytes between the last position written
    * and the next position, write them to the diff file.
    * Possibly there are no unmatched bytes (nextPos == pos), in case
    * of consecutive match blocks.
    */
    nextPos = pMatchBlock != NULL ? pMatchBlock->derivedFilPos : filSiz;
    if (nextPos > pos)
     {
      /*
      * There are unmatched bytes: write one or more block of unmatched bytes 
      * from derivedFil
      */
      fseek(pDerivedFil,pos,SEEK_SET);
      writeLen = nextPos - pos;
      while (writeLen > 0)
       {
	/*
        * If remaining block is small, use tag for small 
        * blocks (1 tag/length byte)
	*/
        if (writeLen <= cSmallSize)
         {
          blockLen = writeLen;
          codeLen = blockLen - 1; /* Encode length 1-cSmallSize by subtr. 1 */
          /* codeLen: 4 bit value */
          /* bit 3-0 | Tag */
          putc((codeLen <<  4)          | cTagSmallDiff       ,pDiffFil);
         }
	/*
        * If remaining block is medium size use tag for 
	* medium blocks (2 tag/length bytes)
        */
        else if (writeLen <= cMediumSize)
         {
          blockLen = writeLen;
          /* Encode length cSmallSize+1 - cMediumSize by subtr. cSmallSize+1 */
	  codeLen = blockLen - cSmallSize - 1;
          /* codeLen: 12 bit value */
          /* bit 11-8 | Tag */
	  putc(((codeLen >>  4) & 0xF0) | cTagMediumDiff      ,pDiffFil);
          /* bit 7-0 */
          putc( (codeLen      ) & 0xFF                        ,pDiffFil);
         }
        else
         {
          /*
          * If remaining block is large: write a large block,
          * and then re-check the remaining length
	  */
          if (writeLen > cLargeSize)
	   {
            blockLen = cLargeSize;
           }
          else
           {
            blockLen = writeLen;
	   }
          /* Encode length cMediumSize+1 - cLargeSize by subtracting cMediumSize+1 */
          codeLen = blockLen - cMediumSize - 1;
	  /* codeLen: 20 bit value */
          /* bit 19-16 | Tag */
          putc(((codeLen >> 12) & 0xF0) | cTagLargeDiff       ,pDiffFil);
          /* bit 15-8 */
          putc( (codeLen >>  8) & 0xFF                        ,pDiffFil);
          /* bit 7-0 */
          putc( (codeLen      ) & 0xFF                        ,pDiffFil);
         }
         
	writeLen -= blockLen;
        len -= blockLen;

        while (blockLen > 0)
         {
          putc(getc(pDerivedFil),pDiffFil);
          blockLen--;
         }

       }
      pos = nextPos;
     }
     
    /*
    * Write block of matching bytes: encode them as a count and a file position
    * in the original file.
    */
    if (pMatchBlock != NULL)
     {
      blockPos = pMatchBlock->orgFilPos;
      writeLen = pMatchBlock->len;
      while (writeLen > 0)
       {
        if (writeLen <= cSmallSize)
         {
          blockLen = writeLen;
          codeLen = blockLen - 1;
          if (blockPos <= cNearDistance)
	   {
            /* codeLen: 4 bit value */
            /* bit 3-0 | Tag */
	    putc((codeLen <<  4)          | cTagSmallNearCopy   ,pDiffFil);

            /* blockPos: 16 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,2);
           }
          else if (blockPos <= cDistantDistance)
           {
            /* codeLen: 4 bit value */
            /* bit 3-0 | Tag */
	    putc((codeLen <<  4)          | cTagSmallDistantCopy,pDiffFil);

	    /* blockPos: 24 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,3);
           }
          else
           {
            /* codeLen: 4 bit value */
	    /* bit 3-0 | Tag */
            putc((codeLen <<  4)          | cTagSmallFarCopy    ,pDiffFil);

	    /* blockPos: 32 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,4);
           }
         }
        else if (writeLen <= cMediumSize)
         {
          blockLen = writeLen;
          codeLen = blockLen - cSmallSize - 1;
          if (blockPos <= cNearDistance)
	   {
            /* codeLen: 12 bit value */
	    /* bit 11-8 | Tag */
            putc(((codeLen >>  4) & 0xF0) | cTagMediumNearCopy  ,pDiffFil);
            /* bit 7-0 */
            putc( (codeLen      ) & 0xFF                        ,pDiffFil);

            /* blockPos: 16 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,2);
           }
          else if (blockPos <= cDistantDistance)
	   {
            /* codeLen: 12 bit value */
            /* bit 11-8 | Tag */
            putc(((codeLen >>  4) & 0xF0) |cTagMediumDistantCopy,pDiffFil);
            /* bit 7-0 */
            putc( (codeLen      ) & 0xFF                        ,pDiffFil);
            
            /* blockPos: 24 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,3);
	   }
          else
           {
            /* codeLen: 12 bit value */
            /* bit 11-8 | Tag */
            putc(((codeLen >>  4) & 0xF0) | cTagMediumFarCopy   ,pDiffFil);
            /* bit 7-0 */
            putc( (codeLen      ) & 0xFF                        ,pDiffFil);

            /* blockPos: 32 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,4);
	   }
         }
        else
         {
          if (writeLen > cLargeSize)
           {
            blockLen = cLargeSize;
           }
          else
	   {
            blockLen = writeLen;
           }
          codeLen = blockLen - cMediumSize - 1;
          if (blockPos <= cNearDistance)
           {
            /* codeLen: 20 bit value */
            /* bit 19-16 | Tag */
	    putc(((codeLen >> 12) & 0xF0) | cTagLargeNearCopy   ,pDiffFil);
            /* bit 15-8 */
            putc( (codeLen >>  8) & 0xFF                        ,pDiffFil);
	    /* bit 7-0 */
            putc( (codeLen      ) & 0xFF                        ,pDiffFil);
            
            /* blockPos: 16 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,2);
           }
          else if (blockPos <= cDistantDistance)
           {
            /* codeLen: 20 bit value */
	    /* bit 19-16 | Tag */
            putc(((codeLen >> 12) & 0xF0) | cTagLargeDistantCopy,pDiffFil);
            /* bit 15-8 */
            putc( (codeLen >>  8) & 0xFF                        ,pDiffFil);
            /* bit 7-0 */
            putc( (codeLen      ) & 0xFF                        ,pDiffFil);
            
            /* blockPos: 24 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,3);
           }
          else
	   {
            /* codeLen: 20 bit value */
            /* bit 19-16 | Tag */
            putc(((codeLen >> 12) & 0xF0) | cTagLargeFarCopy    ,pDiffFil);
            /* bit 15-8 */
            putc( (codeLen >>  8) & 0xFF                        ,pDiffFil);
            /* bit 7-0  */
            putc( (codeLen      ) & 0xFF                        ,pDiffFil);
            
	    /* blockPos: 32 bit value */
	    WriteLongNBytes(blockPos,pDiffFil,4);
           }
         }
        writeLen -= blockLen;
        len -= blockLen;
        pos += blockLen;
       }
      pMatchBlock = pMatchBlock->pNxt;
     }
   }
  putc(cTagEOF,pDiffFil);
 }

/* *********************************************************************** 
*
* Calculate diff file. BC & KR need only 3 filenames, Macintosh version
* needs file names + volume reference numbers.
*/

#if IsANSI
#if IsMac
void SubtractFiles(TpChar orgFilNam, short orgVRefNum,
                   TpChar derivedFilNam, short derivedVRefNum,
                   TpChar diffFilNam, short diffVRefNum)
#else
void SubtractFiles(TpChar orgFilNam, TpChar derivedFilNam, TpChar diffFilNam)
#endif /* IsMac */
#else
int SubtractFiles(orgFilNam,derivedFilNam,diffFilNam)
TpChar orgFilNam;
TpChar derivedFilNam;
TpChar diffFilNam;
#endif /* IsANSI */
 {
  TpFil        pOrgFil;
  TpFil        pDerivedFil;
  TpFil        pDiffFil;
  TpTreeNode   pOrgTreeRoot;
  int          delim;
  TpMatchBlock pMatchLst;
  long         size;
  long         orgSum;
  long         orgSize;
  long         derivedSum;
  long         derivedSize;
#if IsMac
  FInfo        fInfo;
  Str255       pasStr;
  int          i;
  TpChar       pChr;
#endif

#if IsMac
  pDiffFil = fopen(diffFilNam,"wD",diffVRefNum);
#else
  pDiffFil = fopen(diffFilNam,"wb");
#endif

  if (pDiffFil == NULL)
   {
    TStr errMsg;
    
    strcpy(errMsg,"Cannot open file ");
    strcat(errMsg,diffFilNam);
    FatalError(errMsg);
   }

#if IsMac
  /*
  * Insert creator & type for Macintosh, and version info
  */
  C2PasStr(pasStr,derivedFilNam);
  GetFInfo(pasStr,derivedVRefNum,&fInfo);
  pChr = (TpChar) &fInfo.fdCreator;
  for (i = 0; i < 4; i++)
   {
    putc(*pChr,pDiffFil);
    pChr++;
   }
  pChr = (TpChar) &fInfo.fdType;
  for (i = 0; i < 4; i++)
   {
    putc(*pChr,pDiffFil);
    pChr++;
   }
  fputs(cFileHeader + 8,pDiffFil);

  orgSize = 0;  
  derivedSize = 0;
  
#else
  pOrgFil = fopen(orgFilNam,"rb");
  if (pOrgFil == NULL)
   {
    TStr errMsg;
    
    strcpy(errMsg,"Cannot open file ");
    strcat(errMsg,orgFilNam);
    FatalError(errMsg);
   }

  pDerivedFil = fopen(derivedFilNam,"rb");
  if (pDerivedFil == NULL)
   {
    TStr errMsg;
    
    strcpy(errMsg,"Cannot open file ");
    strcat(errMsg,derivedFilNam);
    FatalError(errMsg);
   }

  fputs(cFileHeader,pDiffFil);

  orgSize = FileSize(pOrgFil);  
  size = derivedSize = FileSize(pDerivedFil);
#endif  

  /*
  * Write dummy check-data; gets rewritten at end of this procedure
  */
  WriteLongNBytes(0L,pDiffFil,4);
  WriteLongNBytes(0L,pDiffFil,4);
  WriteLongNBytes(0L,pDiffFil,4);
  WriteLongNBytes(0L,pDiffFil,4);  
  
  orgSum = 0;
  derivedSum = 0;

#if IsMac
  /*
  * Check 2 forks
  */
  for (i = 0; i < 2; i++)
   {
    short  itmType;
    Handle itmHnd;
    Rect   itmRect;
    
    if (i == 0)
     {
      pOrgFil = fopen(orgFilNam,"rD",orgVRefNum);
     }
    else
     {
      pOrgFil = fopen(orgFilNam,"rR",orgVRefNum);
     }
     
    if (pOrgFil == NULL)
     {
      TStr errMsg;
      
      strcpy(errMsg,"Cannot open file ");
      strcat(errMsg,orgFilNam);
      FatalError(errMsg);
     }
    else
     {
      orgSize += FileSize(pOrgFil);
     }
     
    if (i == 0)
     {
      pDerivedFil = fopen(derivedFilNam,"rD",derivedVRefNum);
     }
    else
     {
      pDerivedFil = fopen(derivedFilNam,"rR",derivedVRefNum);
     }
     
    if (pDerivedFil == NULL)
     {
      TStr errMsg;
    
      strcpy(errMsg,"Cannot open file ");
      strcat(errMsg,derivedFilNam);
      FatalError(errMsg);
     }
    else
     {
      size = FileSize(pDerivedFil);
      derivedSize += size;
     }
#else
   {  /* Dummy block on MSDOS & Unix */
#endif

    if (size == 0)	
     {
      int byte;
      
      /*
      * EOF on diff fil
      */
      putc(cTagEOF,pDiffFil);
      /*
      * Adjust checksum
      */
      do
       {
        if ((byte = getc(pOrgFil)) != EOF)
         {
          orgSum += byte;
         }
       }
      while (byte != EOF);
  
     }
    else
     {
#if IsMac
      GetDItem(gpProgressDialog,cMessage1Itm,&itmType,&itmHnd,&itmRect);
      if (i == 0)
       {
        SetIText(itmHnd,"\pData fork:");
       }
      else
       {
        SetIText(itmHnd,"\pResource fork:");
       }
#endif

      /*
      * Find suitable delimiter
      */
      delim = FindDelimiter(pOrgFil,cMinMeanChunkLen,cMaxMeanChunkLen);
      if (delim < 0) delim = 0;

      /*
      * Build indexed position tree
      */
      pOrgTreeRoot = BuildTree(pOrgFil,delim,&orgSum);
   
      /*
      * Match files
      */
      pMatchLst = MatchFiles(pOrgTreeRoot,pOrgFil,pDerivedFil,delim,&derivedSum);

      /*
      * Write diff file
      */
      DumpDiff(pMatchLst,pDerivedFil,pDiffFil);
     }
    fclose(pOrgFil);
    fclose(pDerivedFil);
    
#if IsMac
    GetDItem(gpProgressDialog,cMessage1Itm,&itmType,&itmHnd,&itmRect);
    SetIText(itmHnd,"\p");
#endif

   }
  
#if ! IsMac
  /*
  * MSDOS, Unix & OS/2 have no resource fork: encode extra EOF
  */
  putc(cTagEOF,pDiffFil);
#endif 

  /*
  * Adjust the check-data
  */
  fseek(pDiffFil,strlen(cFileHeader),SEEK_SET);
  
  WriteLongNBytes(orgSize,pDiffFil,4);
  WriteLongNBytes(orgSum,pDiffFil,4);
  WriteLongNBytes(derivedSize,pDiffFil,4);
  WriteLongNBytes(derivedSum,pDiffFil,4);

  fclose(pDiffFil);
 }

#endif /* ! BDEXTR */

/* *********************************************************************** 
*
* Copy characters between files
*/

#if IsANSI
void CopyFileChars(long count, TpFil inFil, TpFil outFil, long *pSum)
#else
int CopyFileChars(count,inFil,outFil,pSum)
long count;
TpFil inFil;
TpFil outFil;
long *pSum;
#endif
 {
  int c;

  c = 0;
  while (count > 0 && c != EOF)
   {
    c = getc(inFil);
    if (c != EOF)
     {
      (*pSum) += c;
      putc(c,outFil);
      count--;
     }
   }
 }

/* *********************************************************************** 
*
* Add diff file. MSDOS & Unix version needs only 3 filenames, Macintosh version
* needs file names + volume reference numbers.
*/

#if IsANSI
#if IsMac
void AddFiles(TpChar orgFilNam, short orgVRefNum,
              TpChar derivedFilNam, short derivedVRefNum,
              TpChar diffFilNam, short diffVRefNum)
#else
void AddFiles(TpChar orgFilNam, TpChar derivedFilNam, TpChar diffFilNam)
#endif /* ! IsMac */
#else
int AddFiles(orgFilNam,derivedFilNam,diffFilNam)
TpChar orgFilNam;
TpChar derivedFilNam;
TpChar diffFilNam;
#endif /* ! IsANSI */
 {
  TpFil        pOrgFil;
  TpFil        pDerivedFil;
  TpFil        pDiffFil;
  int          c;
  TStr         str;
  long         blockLen;
  long         blockPos;
  int          tag;
  long         derivedSize;
  long         derivedSum;
  long         orgSize;
  long         orgSum;
  long         checkDerivedSize;
  long         checkDerivedSum;
  long         checkOrgSize;
  long         checkOrgSum;
  long         prevDiffPos;
  long         diffPos;
  long         curPos;
#if IsMac
  long        *pType;
  long        *pCreator;
  int          i;
  FInfo        fInfo;
  Str255       pasStr;
  short        itmType;
  Handle       itmHnd;
  Rect         itmRect;
#endif

#if IsMac
  pDiffFil = fopen(diffFilNam,"rD",diffVRefNum);
#else
  pDiffFil = fopen(diffFilNam,"rb");
#endif

  if (pDiffFil == NULL)
   {
    TStr errMsg;
    
    strcpy(errMsg,"Cannot open file ");
    strcat(errMsg,diffFilNam);
    FatalError(errMsg);
   }

  strcpy(str,cFileHeader); /* Put '\0' in correct position in str */
  fread(str,1L,strlen(cFileHeader),pDiffFil);
  
  /*
  * Read sizes and checksums of original and updated file
  */
  checkOrgSize = ReadLongNBytes(pDiffFil,4);
  checkOrgSum = ReadLongNBytes(pDiffFil,4);
  orgSize = 0;
  orgSum = 0;
  checkDerivedSize = ReadLongNBytes(pDiffFil,4);
  checkDerivedSum = ReadLongNBytes(pDiffFil,4);
  derivedSize = 0;
  derivedSum = 0;
  
#if IsMac
  /*
  * Extract type and creator from file header
  */
  pCreator = (long *) str;
  pType = (long *) (str+4);
  
  /*
  * Check second part of file header (skip type and creator)
  */
  if (strcmp(str+8,cFileHeader+8) != 0)
   {
    TStr errMsg;
    
    strcpy(errMsg,diffFilNam);
    strcat(errMsg," is not a recognisable diff file.");
    FatalError(errMsg);
   }

  /*
  * Process 2 forks: 0 = Data, 1 = Resource
  */
  for (i = 0; i < 2; i++)
   {
    if (i == 0)
     {
      pOrgFil = fopen(orgFilNam,"rD",orgVRefNum);
     }
    else
     {
      pOrgFil = fopen(orgFilNam,"rR",orgVRefNum);
     }

    if (pOrgFil == NULL)
     {
      TStr errMsg;
    
      strcpy(errMsg,"Cannot open file ");
      strcat(errMsg,orgFilNam);
      FatalError(errMsg);
     }
    
    if (i == 0)
     {
      pDerivedFil = fopen(derivedFilNam,"wD",derivedVRefNum);
     }
    else
     {
      pDerivedFil = fopen(derivedFilNam,"wR",derivedVRefNum);
     }

    if (pDerivedFil == NULL)
     {
      TStr errMsg;
    
      strcpy(errMsg,"Cannot open file ");
      strcat(errMsg,derivedFilNam);
      FatalError(errMsg);
     }
     
    GetDItem(gpProgressDialog,cMessage1Itm,&itmType,&itmHnd,&itmRect);
    if (i == 0)
     {
      SetIText(itmHnd,"\pData fork:");
     }
    else
     {
      SetIText(itmHnd,"\pResource fork:");
     }
     
#else
   {
    /*
    * Check file header
    */
    if (strcmp(str+8,cFileHeader+8) != 0)
     {
      TStr errMsg;
      
      strcpy(errMsg,diffFilNam);
      strcat(errMsg," is not a recognisable diff file.");
      FatalError(errMsg);
     }

    pOrgFil = fopen(orgFilNam,"rb");
    if (pOrgFil == NULL)
     {
      TStr errMsg;
      
      strcpy(errMsg,"Cannot open file ");
      strcat(errMsg,orgFilNam);
      FatalError(errMsg);
     } 
  
    pDerivedFil = fopen(derivedFilNam,"wb");
    if (pDerivedFil == NULL)
     {
      TStr errMsg;
      
      strcpy(errMsg,"Cannot open file ");
      strcat(errMsg,derivedFilNam);
      FatalError(errMsg);
     }
#endif

    orgSize += FileSize(pOrgFil);
    
#if ! IsUnix
    InitProgressBar(0,FileSize(pOrgFil),"Checking:");
#endif
    curPos = 0;
    while ((c = getc(pOrgFil)) != EOF)
     {
#if ! IsUnix
      if ((curPos & 0xFFF) == 0) AdjustProgressBar(curPos);
#endif
      orgSum += c;
      curPos++;
     } 
    fseek(pOrgFil,0L,SEEK_SET);
#if ! IsUnix
    CloseProgressBar();
#endif
    
#if ! IsUnix
    InitProgressBar(0,FileSize(pDiffFil),"Apply diff:");
#endif
    tag = 0;
    prevDiffPos = 0;
    diffPos = 0;
    while (tag != cTagEOF && (c = getc(pDiffFil)) != EOF)
     {
      diffPos++;
#if ! IsUnix
      /*
      * Adjust bar every 256 bytes
      */
      if (diffPos - prevDiffPos > 0xFF) 
       {
        AdjustProgressBar(diffPos);
        prevDiffPos = diffPos;
       }
#endif
       
      tag = c & 0x0F;
      switch (tag)
       {  
        case cTagSmallDiff:
	  blockLen = (c >> 4) + 1;
	  CopyFileChars(blockLen,pDiffFil,pDerivedFil,&derivedSum);
	  diffPos += blockLen;
	  derivedSize += blockLen;
	  break;
        case cTagMediumDiff:
	  blockLen = (c >> 4);
	  c = getc(pDiffFil);
          diffPos++;
	  if (c != EOF)
	   {
	    blockLen = ((blockLen << 8) | c) + cSmallSize + 1;
	    CopyFileChars(blockLen,pDiffFil,pDerivedFil,&derivedSum);
	    diffPos += blockLen;
	    derivedSize += blockLen;
	   }
	  break;
        case cTagLargeDiff:
	  blockLen = (c >> 4);
	  c = getc(pDiffFil);
          diffPos++;
	  if (c != EOF)
	   {
	    blockLen = (blockLen << 8) | c;
	    c = getc(pDiffFil);
            diffPos++;
	    if (c != EOF)
	     {
	      blockLen = ((blockLen << 8) | c) + cMediumSize + 1;
	     }
	    CopyFileChars(blockLen,pDiffFil,pDerivedFil,&derivedSum);
            diffPos += blockLen;
	    derivedSize += blockLen;
	   }
	  break;
        case cTagSmallNearCopy:
          blockLen = (c >> 4) + 1;
	  blockPos = ReadLongNBytes(pDiffFil,2);
	  diffPos += 2;
	  fseek(pOrgFil,blockPos,SEEK_SET);
	  CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
	  derivedSize += blockLen;
          break;
        case cTagMediumNearCopy:
          blockLen = (c >> 4);
	  c = getc(pDiffFil);
          diffPos++;
	  if (c != EOF)
           {
            blockLen = ((blockLen << 8) | c) + cSmallSize + 1;
	    blockPos = ReadLongNBytes(pDiffFil,2);
	    diffPos += 2;
            fseek(pOrgFil,blockPos,SEEK_SET);
            CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
	    derivedSize += blockLen;
	   }
         break;
        case cTagLargeNearCopy:
	  blockLen = (c >> 4);
	  c = getc(pDiffFil);
          diffPos++;
          if (c != EOF)
           {
	    blockLen = (blockLen << 8) | c;
	    c = getc(pDiffFil);
            diffPos++;
            if (c != EOF)
             {
	      blockLen = ((blockLen << 8) | c) + cMediumSize + 1;
	      blockPos = ReadLongNBytes(pDiffFil,2);
	      diffPos += 2;
	      fseek(pOrgFil,blockPos,SEEK_SET);
	      CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
	      derivedSize += blockLen;
             }
           }
          break;
        case cTagSmallDistantCopy:
          blockLen = (c >> 4) + 1;
	  blockPos = ReadLongNBytes(pDiffFil,3);
	  diffPos += 3;
	  fseek(pOrgFil,blockPos,SEEK_SET);
          CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
	  derivedSize += blockLen;
	  break;
        case cTagMediumDistantCopy:
          blockLen = (c >> 4);
          c = getc(pDiffFil);
          diffPos++;
	  if (c != EOF)
	   {
            blockLen = ((blockLen << 8) | c) + cSmallSize + 1;
	    blockPos = ReadLongNBytes(pDiffFil,3);
	    diffPos += 3;
	    fseek(pOrgFil,blockPos,SEEK_SET);
            CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
	    derivedSize += blockLen;
	   }
	  break;
        case cTagLargeDistantCopy:
          blockLen = (c >> 4);
          c = getc(pDiffFil);
          diffPos++;
          if (c != EOF)
           {
	    blockLen = (blockLen << 8) | c;
	    c = getc(pDiffFil);
            diffPos++;
            if (c != EOF)
	     {
	      blockLen = ((blockLen << 8) | c) + cMediumSize + 1;
	      blockPos = ReadLongNBytes(pDiffFil,3);
	      diffPos += 3;
              fseek(pOrgFil,blockPos,SEEK_SET);
	      CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
	      derivedSize += blockLen;
	     }
           }
          break;
        case cTagSmallFarCopy:
          blockLen = (c >> 4) + 1;
	  blockPos = ReadLongNBytes(pDiffFil,4);
	  diffPos += 4;
	  fseek(pOrgFil,blockPos,SEEK_SET);
          CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
	  derivedSize += blockLen;
          break;
        case cTagMediumFarCopy:
          blockLen = (c >> 4);
          c = getc(pDiffFil);
          diffPos++;
	  if (c != EOF)
	   {
            blockLen = ((blockLen << 8) | c) + cSmallSize + 1;
	    blockPos = ReadLongNBytes(pDiffFil,4);
	    diffPos += 4;
	    fseek(pOrgFil,blockPos,SEEK_SET);
            CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
            derivedSize += blockLen;
           }
	  break;
        case cTagLargeFarCopy:
          blockLen = (c >> 4);
          c = getc(pDiffFil);
          diffPos++;
	  if (c != EOF)
           {
	    blockLen = (blockLen << 8) | c;
	    c = getc(pDiffFil);
            diffPos++;
            if (c != EOF)
             {
              blockLen = ((blockLen << 8) | c) + cMediumSize + 1;
	      blockPos = ReadLongNBytes(pDiffFil,4);
	      diffPos += 4;
              fseek(pOrgFil,blockPos,SEEK_SET);
	      CopyFileChars(blockLen,pOrgFil,pDerivedFil,&derivedSum);
	      derivedSize += blockLen;
	     }
	   }
	  break;
        case cTagEOF:
	  break;
        default:
          FatalError("Unknown tag in diff file - possibly created by a more recent BinDiff");
          break;
       }
     }
   
    fclose(pDerivedFil);
    fclose(pOrgFil);
#if ! IsUnix
    CloseProgressBar();
#endif

   } /* End of for loop on Macintosh; end of dummy block on MSDOS & Unix */


  if (orgSize != checkOrgSize || orgSum != checkOrgSum)
   {
    TStr errMsg;
    
    strcpy(errMsg,orgFilNam);
    strcat(errMsg," is not the file the diff was created with.");
    FatalError(errMsg);
   }
  
#if IsMac
  C2PasStr(pasStr,derivedFilNam);
  GetFInfo(pasStr,derivedVRefNum,&fInfo);
  fInfo.fdCreator = *pCreator;
  fInfo.fdType = *pType;
  SetFInfo(pasStr,derivedVRefNum,&fInfo);
#endif

  if (derivedSize != checkDerivedSize || derivedSum != checkDerivedSum)
   {
    TStr errMsg;
    
    strcpy(errMsg,derivedFilNam);
    strcat(errMsg," is corrupt.");
    FatalError(errMsg);
   }

  fclose(pDiffFil);
 }

#if IsMac
/* *********************************************************************** 
*
* Main program (Macintosh version)
*/

void main(void)
 {
  Handle     itmHnd;
  short      itmType;
  Rect       itmRect;
  short      itmHit;
  Point      p;
  SFReply    reply;
  GrafPtr    pSaveGfp;
  TStr       orgFilNam;
  short      orgVRefNum;
  TStr       diffFilNam;
  short      diffVRefNum;
  TStr       derivedFilNam;
  short      derivedVRefNum;
  Str255     pasStr;
  TBoolean   isSubtract;
  TBoolean   gotOrgFil;
  TBoolean   got2ndFil;
  CursHandle hWatchCursor;
  Cursor     watch;
 
  /*
  * Initialize toolbox
  */
  InitGraf(&qd.thePort);
  InitFonts();
  FlushEvents(everyEvent, 0);
  InitWindows();
  InitMenus();
  TEInit();
  InitDialogs(0L);
  InitCursor();
  MaxApplZone();

  hWatchCursor = GetCursor(watchCursor);
  HLock((Handle) hWatchCursor);
  watch = **hWatchCursor;
  HUnlock((Handle) hWatchCursor);
  
  GetPort(&pSaveGfp);

  /*
  * Open dialog & initialize
  */
  gpProgressDialog = GetNewDialog(cProgressDialogID,NULL,(WindowPtr) -1);
  SetPort(gpProgressDialog);    
  
  p.h = 100;
  p.v = 100;
  
  GetDItem(gpProgressDialog,cTitleItm,&itmType,&itmHnd,&itmRect);
#if BDEXTR
  SetIText(itmHnd,"\pBDExtr V1.1.1 for Macintosh - (c) 1994 Kris Coppieters");
#else
  SetIText(itmHnd,"\pBinDiff V1.1.1 for Macintosh - (c) 1994 Kris Coppieters");
#endif
  GetDItem(gpProgressDialog,cSubtractRadioBtn,&itmType,&itmHnd,&itmRect);

  gotOrgFil = FALSE;
  got2ndFil = FALSE;
#if BDEXTR
  isSubtract = FALSE;
  SetCtlValue((ControlHandle) itmHnd,0);
  GetDItem(gpProgressDialog,cAddRadioBtn,&itmType,&itmHnd,&itmRect);
  SetCtlValue((ControlHandle) itmHnd,1);
  GetDItem(gpProgressDialog,cSet2ndBtn,&itmType,&itmHnd,&itmRect);
  SetCTitle((ControlHandle) itmHnd,"\pSet Diff:");
#else
  isSubtract = TRUE;
  SetCtlValue((ControlHandle) itmHnd,1);
  GetDItem(gpProgressDialog,cAddRadioBtn,&itmType,&itmHnd,&itmRect);
  SetCtlValue((ControlHandle) itmHnd,0);
  GetDItem(gpProgressDialog,cSet2ndBtn,&itmType,&itmHnd,&itmRect);
  SetCTitle((ControlHandle) itmHnd,"\pSet Updated:");
#endif
  
#if BDEXTR
  /*
  * Remove unneeded buttons from dialog
  */
  HideDItem(gpProgressDialog,cAddRadioBtn);
  HideDItem(gpProgressDialog,cSubtractRadioBtn);
#endif

  /*
  * Handle dialog until OK or Cancel
  */
  do
   {
    
    GetDItem(gpProgressDialog,cOKBtn,&itmType,&itmHnd,&itmRect);
    PenSize(3,3);
    InsetRect(&itmRect,-4,-4);
    FrameRoundRect(&itmRect,16,16);
    PenSize(1,1);
    
    ModalDialog(NULL,&itmHit);
    
    switch (itmHit)
     {
      case cSubtractRadioBtn:
        isSubtract = TRUE;
        GetDItem(gpProgressDialog,cSubtractRadioBtn,&itmType,&itmHnd,&itmRect);
        SetCtlValue((ControlHandle) itmHnd,1);
        GetDItem(gpProgressDialog,cAddRadioBtn,&itmType,&itmHnd,&itmRect);
        SetCtlValue((ControlHandle) itmHnd,0);
        GetDItem(gpProgressDialog,c2ndFilItm,&itmType,&itmHnd,&itmRect);
        SetIText(itmHnd,"\p");
        got2ndFil = FALSE;
        GetDItem(gpProgressDialog,cSet2ndBtn,&itmType,&itmHnd,&itmRect);
        SetCTitle((ControlHandle) itmHnd,"\pSet Updated:");
        break;
      case cAddRadioBtn:
        isSubtract = FALSE;
        GetDItem(gpProgressDialog,cSubtractRadioBtn,&itmType,&itmHnd,&itmRect);
        SetCtlValue((ControlHandle) itmHnd,0);
        GetDItem(gpProgressDialog,cAddRadioBtn,&itmType,&itmHnd,&itmRect);
        SetCtlValue((ControlHandle) itmHnd,1);
        GetDItem(gpProgressDialog,c2ndFilItm,&itmType,&itmHnd,&itmRect);
        SetIText(itmHnd,"\p");
        got2ndFil = FALSE;
        GetDItem(gpProgressDialog,cSet2ndBtn,&itmType,&itmHnd,&itmRect);
        SetCTitle((ControlHandle) itmHnd,"\pSet Diff:");
        break;
      case cSetOrgBtn:
        SFGetFile(p,NULL,NULL,-1,NULL,NULL,&reply);
        if (reply.good)
         {
          Pas2CStr(orgFilNam,reply.fName);
          orgVRefNum = reply.vRefNum;
          GetDItem(gpProgressDialog,cOrgFilItm,&itmType,&itmHnd,&itmRect);
          SetIText(itmHnd,reply.fName);
          gotOrgFil = TRUE;
         }
        break;
      case cSet2ndBtn:
        if (isSubtract)
         {
          SFGetFile(p,NULL,NULL,-1,NULL,NULL,&reply);
          if (reply.good)
           {
            Pas2CStr(derivedFilNam,reply.fName);
          
            derivedVRefNum = reply.vRefNum;
            GetDItem(gpProgressDialog,c2ndFilItm,&itmType,&itmHnd,&itmRect);
            SetIText(itmHnd,reply.fName);
            got2ndFil = TRUE;
           }
         }
        else
         {
          SFTypeList tl;
          
          tl[0] = cFileType;
          tl[1] = 'TEXT'; /* SoftPC text */
          tl[2] = 'PCFA'; /* SoftPC binary */
          tl[3] = 'bina'; /* MSDOS floppy */
          
          SFGetFile(p,NULL,NULL,4,tl,NULL,&reply);
          if (reply.good)
           {
            Pas2CStr(diffFilNam,reply.fName);
          
            diffVRefNum = reply.vRefNum;
            GetDItem(gpProgressDialog,c2ndFilItm,&itmType,&itmHnd,&itmRect);
            SetIText(itmHnd,reply.fName);
            got2ndFil = TRUE;
          }
         }
        break;
     }
   }
  while (! (itmHit == cOKBtn && gotOrgFil && got2ndFil) && itmHit != cCancelBtn);
  
  /*
  * Abort on cancel
  */
  if (itmHit == cCancelBtn) ExitToShell();
  
  /*
  * Remove unneeded buttons from dialog
  */
  HideDItem(gpProgressDialog,cOKBtn);
  HideDItem(gpProgressDialog,cCancelBtn);
  
  /*
  * Add or subtract
  */
#if ! BDEXTR
  if (isSubtract)
   {
    strcpy(diffFilNam,orgFilNam);
    strcat(diffFilNam,".dif");
    C2PasStr(pasStr,diffFilNam);
    SFPutFile(p,"\pSave diff file as:",pasStr,NULL,&reply);
    if (reply.good)
     {
      Pas2CStr(diffFilNam,reply.fName);
      diffVRefNum = reply.vRefNum;
    
      DrawDialog(gpProgressDialog);

      SetCursor(&watch);
  
      SubtractFiles(orgFilNam,orgVRefNum,
                    derivedFilNam,derivedVRefNum,
                    diffFilNam,diffVRefNum);
     }
   }
  else
   {
#endif
    strcpy(derivedFilNam,orgFilNam);
    strcat(derivedFilNam,".upd");
    C2PasStr(pasStr,derivedFilNam);
    SFPutFile(p,"\pSave updated file as:",pasStr,NULL,&reply);
    if (reply.good)
     {
      Pas2CStr(derivedFilNam,reply.fName);
      derivedVRefNum = reply.vRefNum;

      DrawDialog(gpProgressDialog);
      
      SetCursor(&watch);
  
      AddFiles(orgFilNam,orgVRefNum,
               derivedFilNam,derivedVRefNum,
               diffFilNam,diffVRefNum);
     }
#if ! BDEXTR
   }
#endif

  SetCursor(&qd.arrow);
  SetPort(pSaveGfp);
 }
#else
/* *********************************************************************** 
*
* Main program (MSDOS, OS/2 & Unix version)
*
* Call with arguments:
*
*   diff = updated - original (= and - are arguments: 5 arguments)
*   updated = original + diff (= and + are arguments: 5 arguments)
*/

#if IsANSI
void main(int nArg, char **arg)
#else
int main(nArg,arg)
int nArg;
char **arg;
#endif
 {

#if BDEXTR
  if (nArg < 6 || strcmp(arg[2],"=") != 0 || strcmp(arg[4],"+") != 0)
   {
#if IsUnix
    fprintf(stderr,"BDExtr V1.1.1 for Unix - Usage:\n\n");
#elif IsOS2
    fprintf(stderr,"BDExtr V1.1.1 for OS/2 - Usage:\n\n");
#elif IsMSDOS
    fprintf(stderr,"BDExtr V1.1.1 for MSDOS - Usage:\n\n");
#endif

    fprintf(stderr,"Use a diff file and an original to get an updated file:\n\n");
    fprintf(stderr,"  bdextr updated = original + diff\n\n");
    fprintf(stderr,"+ and = need surrounding spaces on command line\n");
    fprintf(stderr,"Example:\n");
    fprintf(stderr,"bdextr TESTPRV12.C = TESTPRV11.C + TESTPROG.DIF\n");
    fprintf(stderr,"is ok; and\n");
    fprintf(stderr,"bdextr TESTPRV12.C=TESTPRV11.C+TESTPROG.DIF\n");
    fprintf(stderr,"is wrong.\n\n");
    fprintf(stderr,"BDExtr (c) 1994 Kris Coppieters\n");
    fprintf(stderr,"CompuServe: 100025,2724\n");
    fprintf(stderr,"Internet  : 100025.2724@compuserve.com\n\n");
    fprintf(stderr,"Use BinDiff to create diff files: GO IBMFF on CompuServe to find it\n");
    fprintf(stderr,"BDExtr can be copied freely.\n");
    exit(0);
   }
  else
   {
#if IsUnix
    fprintf(stderr,"BDExtr V1.1.1 for Unix - (c) 1994 Kris Coppieters\n");
#elif IsOS2   
    fprintf(stderr,"BDExtr V1.1.1 for OS/2 - (c) 1994 Kris Coppieters\n");
#elif IsMSDOS
    fprintf(stderr,"BDExtr V1.1.1 for MSDOS - (c) 1994 Kris Coppieters\n");
#endif
    AddFiles(arg[3],arg[1],arg[5]);
   }
  fprintf(stderr,"Done.\n");
#else
  if (nArg < 6 || strcmp(arg[2],"=") != 0 || (strcmp(arg[4],"+") != 0 && strcmp(arg[4],"-") != 0))
   {
#if IsUnix
    fprintf(stderr,"BinDiff V1.1.1 for Unix - Usages:\n\n");
#elif IsOS2
    fprintf(stderr,"BinDiff V1.1.1 for OS/2 - Usages:\n\n");
#elif IsMSDOS
    fprintf(stderr,"BinDiff V1.1.1 for MSDOS - Usages:\n\n");
#endif
    fprintf(stderr,"To CREATE a diff file:\n\n");
    fprintf(stderr,"  bindiff diff = original - updated\n\n");
    fprintf(stderr,"To USE a diff file and an original to get an updated file:\n\n");
    fprintf(stderr,"  bindiff updated = original + diff\n\n");
    fprintf(stderr,"+, =, and - need surrounding spaces on command line\n");
    fprintf(stderr,"Example:\n");
    fprintf(stderr,"bindiff TESTPROG.DIF = TESTPRV12.C - TESTPRV11.C\n");
    fprintf(stderr,"is ok; and\n");
    fprintf(stderr,"bindiff TESTPROG.DIF=TESTPRV12.C-TESTPRV11.C\n");
    fprintf(stderr,"is wrong.\n\n");
    fprintf(stderr,"BinDiff (c) 1994 Kris Coppieters\n");
    fprintf(stderr,"CompuServe: 100025,2724\n");
    fprintf(stderr,"Internet  : 100025.2724@compuserve.com\n\n");
    fprintf(stderr,"Please distribute with accompanying files: README.BDF and BDEXTR.COM\n");
    fprintf(stderr,"Free for non-commercial uses. Commercial uses: please contact the author.\n");
    exit(0);
   }
  else
   {
#if IsUnix
    fprintf(stderr,"BinDiff V1.1.1 for Unix - (c) 1994 Kris Coppieters\n");
#elif IsOS2
    fprintf(stderr,"BinDiff V1.1.1 for OS/2 - (c) 1994 Kris Coppieters\n");
#elif IsMSDOS
    fprintf(stderr,"BinDiff V1.1.1 for MSDOS - (c) 1994 Kris Coppieters\n");
#endif
    if (strcmp(arg[4],"-") == 0)
     {
      SubtractFiles(arg[5],arg[3],arg[1]);
     }
    else
     {
      AddFiles(arg[3],arg[1],arg[5]);
     }
   }
  fprintf(stderr,"Done.\n");
#endif
 }
#endif
