/* bobtalk.c: Contains the functions necessary to communicate with parent
 *
 * Copyright (c) 1994 Brett Dutton
 *
 *** Revision History:
 * 13-Dec-1994 Dutton      Changed so that BobTalkInit returns the com socket 
 * ....................... in the parent 
 * 12-Oct-1994 Dutton      Added function to block until either receives a 
 * ....................... Bob event or an event on the passed socket. 
 * ....................... This will not be neexed when ported to X Toolkit
 * 15-Sep-1994 Dutton      Set up extra opcode to differentiate between errors
 * ....................... and quitting. Rationialized the shutdown to make
 * ....................... sure that the child closes properly.
 *  7-Sep-1994 Dutton      Allowed arguments to be passed to the execute function
 *  6-Sep-1994 Dutton      Converted function descriptions to pre and 
 * ....................... post conditions
 *  1-Sep-1994 Dutton      Seperated out the function to read in a bob file 
 * ....................... so that it can be called by other functions
 * 15-Aug-1994 Dutton      Initial coding
 *
 */

/*
 * Description:
 * This program contains the functions necessary to communicate via 
 * socket communication. The initialization forks off the process and the 
 * child sits in an endless loop waiting for events to come in alon the 
 * communications socket. Both parent and child use the functions in the file
 * to communicate from theie respective sides of the socket.
 */

/*
 * includes
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <setjmp.h>
#include "bob.h"

/*
 * macros
 */

/*
 * typedefs
 */
typedef struct bobOpQueue_s BobOpQueue;
struct bobOpQueue_s {
    BobOpQueue  *next;          /* pointer to the next element in the list */
    BobOpQueue  *last;          /* Pointer to the last element in the list */
    int         op;             /* queued operation */
    int         len;            /* length of the buffer */
    char        buff[1];        /* buffer data */
};

typedef void (*BobOpCodeFunc)( int, VALUE *, VALUE * );
    
/*
 * prototypes
 */
static void BobEventLoop ( void );
static void BobOpLoad ( int argcnt, VALUE *args, VALUE *retval );
static void BobOpExec ( int argcnt, VALUE *args, VALUE *retval );
static void BobOpAddFunction ( int argcnt, VALUE *args, VALUE *retval );
static void BobOpQuit ( int argcnt, VALUE *args, VALUE *retval );
static void add_file( char *name, FILE *fp, VALUE *pval );
static void BobTalkProcessEvent ( int op, char *buff );
static void BobInvokeRemoteFunc ( char *func, char *err );
static int BobQueueOperation ( int op, char *buff, int len );
static int BobCheckQueue ( char *buff, int *len );
static void BobTalkShutdown ( int get_out );
static int BobWriteEventSock ( int op, GenBuffer buff, int len, int sk );
static int BobReadSockEvent ( GenBuffer buff, int *len, int sk );

/*
 * variables
 */
static int      sock = -1;      /* This is the socket used for comms */
static int      err_sock = -1;  /* Error socket */
static BobOpCodeFunc bobFunc[TALK_MAX];/* these are a ist of functions accociated to opcodes */
static char     *boblib = NULL; /* directory for the bob macro library */
static char     *bobuser = NULL; /* directory for the bob user home directory */
static BobOpQueue *bobOpQueue=NULL; /* operation queue */

char talkopcodes[TALK_MAX][10] = { 
    "TALK_NOOP",
    "TALK_LOAD",
    "TALK_EXEC",
    "TALK_ADDF",
    "TALK_REMO",
    "TALK_QUIT",
    "TALK_RERR"
};

extern jmp_buf  error_trap;     /* Error trap */
extern VALUE    symbols;        /* dictionary of valid symbols */
extern VALUE    stdin_iostream;
extern VALUE    stdout_iostream;
extern VALUE    stderr_iostream;

/*
 * functions
 */

/******************************************************************************
 * Function:
 * BobTalkInit -- Initializes the communications and processes
 * 
 * Description:
 * This function forks and returns to the parent. The child process
 * caklls a function that never returns from this function and sits in
 * an endless loop waiting for communications along the socket.
 * This function is called by the parent only ( before the fork)
 *
 * PRE:  Bob memory must correctly initialized
 * POST: returns socket id on parent and writes initialization opcode to child
 *       doesn't return from CHILD. If BOB fails to initialize then returns 
 *       -1 on the parent. The child is not started
 */
int BobTalkInit ( void )
{
    int         pid;            /* child process id after the fork */
    int         sockets[2];     /* socket pairs */
    int         err_sockets[2]; /* error socket pairs */
    int         op;             /* initialization opcode */
    
    /* set up a pair to talk to each other */
    if ( socketpair ( AF_UNIX, SOCK_STREAM, 0, sockets ) < 0 ) {
        error ( "Opening socket pair" );
        return ( -1 );
    }
    
    /* set up a pair of error sockets to talk to each other */
    if ( socketpair ( AF_UNIX, SOCK_STREAM, 0, err_sockets ) < 0 ) {
        fprintf ( stderr, "Opening socket pair\n" );
        exit ( 1 );
    }
    
    /* split off to another process */
    if ( ( pid = fork () ) < 0 ) {
        fprintf ( stderr, "Unable to start Bob server\n" );
        return ( -1 );
    }
    
    /* this is the parent */
    if ( pid ) {
        /* close the other end of the socket */
        close ( sockets[0] );
        sock = sockets[1];
        close ( err_sockets[0] );
        err_sock = err_sockets[1];
        
        /* send down initialization on the sockets */
        op = TALK_NOOP;
        write ( sock, &op, sizeof ( int ) );
        write ( err_sock, &op, sizeof ( int ) );
        
        /* send the socket back to the parent */
        return ( sock );
    }

    /* this is the child, close sown other end of socket and */
    /* set up out end of the socket */
    close ( sockets[1] );
    sock = sockets[0];
    close ( err_sockets[1] );
    err_sock = err_sockets[0];

    /* initialize */
    init_compiler();            /* initializer the compiler */
    boblib = getenv ( "BOBLIB" ); /* get the directory of files */
    bobuser = getenv ( "HOME" ); /* get the directory users home */
 
    /* setup the standard i/o streams */
    add_file("stdin",stdin,&stdin_iostream);
    add_file("stdout",stdout,&stdout_iostream);
    add_file("stderr",stderr,&stderr_iostream);

    /* set up the function interface */
    bobFunc[TALK_NOOP] = NULL;
    bobFunc[TALK_LOAD] = BobOpLoad;
    bobFunc[TALK_EXEC] = BobOpExec;
    bobFunc[TALK_ADDF] = BobOpAddFunction;
    bobFunc[TALK_REMO] = NULL;
    bobFunc[TALK_QUIT] = BobOpQuit;
    bobFunc[TALK_RERR] = NULL;

#ifdef DEBUG
    /* this is for debugging so I can trap the child */
    /* this is an environment variable so that I can */
    /* switch it without recompiling */
    if ( getenv ( "BOBWAIT" ) != NULL ) {
        info ( "Sleeping for 30 seconds" );
        sleep ( 30 );
    }
#endif

    /* read the initialization from the parent */
    read ( sock, &op, sizeof ( int ) );
    read ( err_sock, &op, sizeof ( int ) );
    
    /* go into an endless loop waiting for events */
    BobEventLoop ();
}
/******************************************************************************
 * Function:
 * BobEventLoop -- This is the main processing loop of bob
 * 
 * Description:
 * This function continuously waits for events from the socket and 
 * put them into an operation queue. They are not processed untill there
 * is no more info coming over the socket.
 * If there is a TALK_QUIT opcode then the socket finishes processing 
 * and gets out ***Maybe this should be immediate
 * If a -1 comes through then exit (system error)
 * This function is called by the child only
 *
 * PRE: Bob must be correctly set up.
 * POST: This function does not return, only exit.
 */
static void BobEventLoop ( void )
{
    VALUE   args[20];
    VALUE   retval;
    int     op;
    int     argcnt, len;
    GenBuffer   buff;
    int         wait;
    
    /* initially wait for events */
    wait = TRUE;
    
    /* main event loop for bob */
    for ( ; ; ) {

        /* set up the error trap default position */
        if ( setjmp ( error_trap ) ) continue;

        /* while events are on the socket then queue them */
        while ( ( op = BobReadEvent ( buff, &len, wait ) ) != TALK_NOOP ) {

            /* see if system error */
            if ( op < 0 ) BobTalkShutdown ( TRUE );
            
            /* make sure we don't wait for the next event */
            wait = FALSE;
        
            /* queue the event */
            BobQueueOperation ( op, buff, len );
        } 
        
        /* see if there is an event on the queue */
        if ( ( op = BobCheckQueue ( buff, &len ) ) == TALK_NOOP ) {
            /* if there are no events in the queue then should wait */
            wait = TRUE;
            continue;
        }

        /* processes the incoming event */
        BobTalkProcessEvent ( op, buff );
    }
}

/******************************************************************************
 * Function:
 * BobTalkShutdown -- Shuts down the sockets and gets out if specified
 * 
 * Description:
 *
 * PRE: Bob must be correctly set up.
 * POST: This function does not return, only exit.
 */
static void BobTalkShutdown ( int get_out )
{
    if ( sock >= 0 ) shutdown ( sock, 2 );
    if ( err_sock >= 0 ) shutdown ( err_sock, 2 );
    sock = err_sock = -1;
    if ( get_out ) exit ( 0 );
}

/******************************************************************************
 * Function:
 * BobTalkProcessEvent -- Processes the incoming event 
 * 
 * Description:
 * This function is passed the opcode and the buffer and from that processes 
 * the event
 *
 * PRE:  Bob correctly set up
 * POST: function associated (if there is one) is run with the passed buffer
 */
void BobTalkProcessEvent ( int op, char *buff )
{
    VALUE   args[20];
    VALUE   retval;
    int     argcnt;

    /* unpack the args from the read buffer */
    argcnt = BobUnpackBuffer ( buff, args, 20 );
    
    /* see if there is no event specified then get out */
    if ( bobFunc[op] == NULL ) return;
            
    /* call the external function */
    (*bobFunc[op]) ( argcnt, args, &retval );
}

/******************************************************************************
 * Function:
 * BobTalkTerm -- Terminates communications and processes
 * 
 * Description:
 * This functions shuts doen the parent and child sockets and
 * terminates the child process by sending request to child to shutdown
 * This is the only parent request that actually waits for an answer.
 * This function is called by the parent only
 *
 * PRE:  Bob correctly set up
 * POST: Returns TRUE on success and child process is shutdown
 */
int BobTalkTerm ( void )
{
    int         op=TALK_QUIT;   /* opcode */
    char        type=0;         /* type of the buffer */
    GenBuffer   buff;
    int         len;
    
    /* send a break and then a quit signal */
    BobWriteEventSock ( op, &type, sizeof ( int ), err_sock );
    BobWriteEvent ( op, &type, sizeof ( int ) );
    
    /* wait for a reply from the quit function */
    op = BobReadEvent ( buff, &len, TRUE );
    
    /* shutdown this end of the sockets */
    BobTalkShutdown ( FALSE );
    
    return ( TRUE );
}
    
/******************************************************************************
 * Function:
 * BobReadEvent -- Checks if there is an event outstanding
 * 
 * Description:
 * This function can be called by the parent or the child;
 * This function gets the next event from the socket. If there is not
 * an event on the socket then it waits. It assumes that the first 4
 * bytes is the opcode and the remaining is the data. The opcode is
 * returned and the data and length are passed back. If wait is TRUE
 * then the function will wait for the next event, otherwise it is a poll
 *
 * PRE:  Bob correctly set up
 * POST: returns The opcode associated with this packet, or TALK_NOOP is 
 *       there is no data available
 */
int BobReadEvent ( GenBuffer buff, int *len, int wait )
{
    fd_set      readfds;        /* read mask */
    struct timeval timeout;     /* timeout */
    struct timeval *tout;       /* timeout pointer */
    int         s;              /* result of select */

    /* make sure we have a socket */
    if ( sock < 0 || err_sock < 0 ) return ( -1 );
        
    /* initialise some stuff */
    if ( wait ) {               /* wait for event */
        tout = NULL;
    } else {                    /* poll, see if there is one there */
        timeout.tv_sec = timeout.tv_usec = 0;
        tout = &timeout;
    }

    /* check for reads and exceptions */
    FD_ZERO ( &readfds );
    FD_SET ( sock, &readfds );
    FD_SET ( err_sock, &readfds );

    /* do the select check if error */
    if ( ( s = select ( FD_SETSIZE, &readfds, NULL, NULL, tout ) ) < 0 ) {
        error ( "Unable to do a select on read socket %d", sock );
        return ( -1 );
    }
    
    /* get out if there is nothing there */
    if ( s == 0 ) return ( TALK_NOOP );
    
    /* check the read bitmask */
    if ( FD_ISSET ( sock, &readfds ) ) {

        /* return the result of the read */
        return ( BobReadSockEvent ( buff, len, sock ) );
    }
    
    /* check for exceptions */
    if ( BobCheckBreak () < 0 ) return ( -1 );
    
    /* should have not made it to here */
    return ( -1 );
}
/******************************************************************************
 * Function:
 * BobReadSockEvent -- Reads the data on the passed socket
 * 
 * Description:
 * This function can be called by the parent or the child;
 * This function gets the next event from the socket. 
 *
 * PRE:  Bob correctly set up, There MUST be data on the passed socket
 * POST: returns The opcode associated with this packet, or TALK_NOOP is 
 *       there is no data available. -1 is returned on error
 */
static int BobReadSockEvent ( GenBuffer buff, int *len, int sk )
{
    int         op;             /* opcode */

    /* get the opcode */
    if ( read ( sk, &op, sizeof ( int ) ) < 0 ) {
        error ( "Could not read from the socket" );
        return ( -1 );
    }
    
    /* get the length of the buffer */
    if ( read ( sk, len, sizeof ( int ) ) < 0 ) {
        error ( "Could not read from the socket" );
        return ( -1 );
    }
    
    /* get the data */
    if ( read ( sk, buff, *len ) < 0 ) {
        error ( "Could not read from the socket" );
        return ( -1 );
    }
    
    /* send back the opcode */
    return ( op ); 
}

/******************************************************************************
 * Function:
 * BobWriteEvent -- Writes the data back to the socket
 * 
 * Description:
 * This function can be called by the parent or the child;
 * This function writes the passed buffer to the socket. The opcode is
 * written first to the socket followed by the buffer
 *
 * PRE:  Bob correctly set up
 * POST: returns TRUE on success and data is written to the socket
 *       -1 if there is a socket eror
 */
int BobWriteEvent ( int op, GenBuffer buff, int len )
{
    /* make sure we have a socket */
    if ( sock < 0 ) return ( -1 );
    
    /* send to the socket */
    return ( BobWriteEventSock ( op, buff, len, sock ) );
}

/******************************************************************************
 * Function:
 * BobWriteEventSock -- Writes the data back to the passed socket
 * 
 * Description:
 * This function can be called by the parent or the child;
 * This function writes the passed buffer to the passed socket. The opcode is
 * written first to the socket followed by the buffer
 *
 * PRE:  Bob correctly set up
 * POST: returns TRUE on success and data is written to the socket
 *       -1 if there is a socket eror
 */
static int BobWriteEventSock ( int op, GenBuffer buff, int len, int sk )
{
    /* make sure we have a socket */
    if ( sk < 0 ) return ( -1 );
        
    /* write the opcode back first */
    if ( write ( sk, &op, sizeof ( int ) ) < 0 ) {
        error ( "Error writing on socket %d", sk );
        return ( -1 );
    }
    
    /* write the length to the socket */
    if ( write ( sk, &len, sizeof ( int ) ) < 0 ) {
        error ( "Error writing on socket %d", sk );
        return ( -1 );
    }
    
    /* write the data */
    if ( write ( sk, buff, len ) < 0 ) {
        error ( "Error writing on socket %d", sk );
        return ( -1 );
    }
    
    /* return TRUE */
    return ( TRUE );
}
/******************************************************************************
 * Function:
 * BobCheckBreak -- Checks if there is a break event
 * 
 * Description:
 * This function can be called by the child;
 * This function checks the error socket. Since the child cannot break
 * the parent it is not a problem if the parent check the break status
 * since the child should never write to the error socket
 *
 * PRE:  Bob correctly set up
 * POST: returns TRUE is there is a packet of the error socket
 *       FALSE if no error
 *       -1 on socket error
 */
int BobCheckBreak ( void )
{
    fd_set      readfds;        /* read mask */
    GenBuffer   buff;           /* buffer for message */
    struct timeval timeout;     /* 0 timeout (poll) */
    int         s;              /* result of select */
    int         op;             /* opcode */
    int         len;            /* length of the buffer */

    /* make sure the socket is set up */
    if ( err_sock < 0 ) return ( FALSE );
    
    /* initialise some stuff */
    timeout.tv_sec = timeout.tv_usec = 0;
    FD_ZERO ( &readfds );
    FD_SET ( err_sock, &readfds );
    
    /* do the select check if error */
    if ( ( s = select ( FD_SETSIZE, &readfds, NULL, NULL, &timeout ) ) < 0 ) {
        error ( "Unable to do a select on error socket %d", err_sock );
        return ( -1 );
    }
    
    /* get out if there is nothing there */
    if ( s == 0 ) return ( FALSE );
    
    /* check the bitmask */
    if ( FD_ISSET ( err_sock, &readfds ) ) {

        /* get the data from the socket */
        op = BobReadSockEvent ( buff, &len, err_sock );
        
        switch ( op ) {
        case TALK_RERR: BobInvokeRemoteFunc ( "error", buff ); break;
        case TALK_QUIT: longjmp ( error_trap, 1 ); break;
        default: break;
        }

        return ( TRUE );
    }
    
    /* should have not made it to here */
    return ( -1 );
}

/******************************************************************************
 * Function:
 * BobBreak -- Creates a break event
 * 
 * Description:
 * This function can be called by the parent;
 * This function creates a break on the socket for the child to
 * read. Since the parent can never be interrupted by the child then the
 * child should never cal this function
 *
 * PRE:  Bob correctly set up
 * POST: Returns TRUE and a message is written on the error socket
 */
int BobBreak ( char *fmt, ... )
{
    GenBuffer   buff;           /* buffer for message */
    va_list     ap;             /* arg list */
    int         op=TALK_RERR;   /* error opcode */
    
    /* make sure we have a socket */
    if ( err_sock < 0 ) return ( FALSE );
        
    va_start(ap,fmt);
    vsprintf(buff,fmt,ap);
    va_end(ap);
    
    BobWriteEventSock ( op, buff, strlen ( buff ) + 1, err_sock );

    /* do not wait for reply, just get out */
    return ( TRUE );
}
    
/******************************************************************************
 * Function:
 * BobLoadFilename -- Loads up the passed file name
 * 
 * Description:
 * This function loads the passed filename. It is called by the child.
 * As a future enhancement this function should check for compiled 
 * bob files. At the moment the Bob macro processor cannot compile 
 * files, so just check for .bob extensions
 * Search rules:
 * <filename>
 * getenv("HOME")/<filename>
 * getenv("BOBLIB")/<filename>
 * <filename>.bob
 * getenv("HOME")/<filename>.bob
 * getenv("BOBLIB")/<filename>.bob
 *
 * PRE:  Bob correctly set up
 * POST: returns TRUE on success and the filename is compiled into memory. 
 *       FALSE if the file name does not exist using the following rules
 */
int BobLoadFilename ( char *filename )
{
    FILE        *f=NULL;        /* pointer to the open file */
    GenBuffer   fname;          /* local copy of the filename */
    char        *ptr;           /* pointer into the string */
    int         i, j;              /* counter */
    char        *tryLst[6];     /* list to try */
    char        *sepLst[6];     /* list seperators */
    char        *suff;          /* suffixes */
    struct stat sbuf;
    GenBuffer   msg;            /* message to pass back */

    /* set up the directories and seperators */
    tryLst[0] = "";     sepLst[0] = "";  
    tryLst[1] = bobuser;sepLst[1] = "/"; 
    tryLst[2] = boblib; sepLst[2] = "/"; 
    
    /* loop the suffixes */
    for ( suff="", j=0; j<2 && f==NULL; j++, suff=".bob" ) {
        
        /* loop through the directories searching for the file */
        for ( i=0; i<3; i++ ) {
        
            /* create the file name */
            sprintf ( fname, "%s%s%s%s", tryLst[i], sepLst[i], filename, suff);
            
            /* check the status of the file */
            if ( stat ( fname, &sbuf ) != 0 ) continue;
            if ( S_ISDIR ( sbuf.st_mode ) ||
                S_ISSOCK ( sbuf.st_mode ) ||
                S_ISFIFO ( sbuf.st_mode ) ) continue;
                
            if ( ( f = fopen ( fname, "r" ) ) != NULL ) break;
        }
    }

    /* check if there was a file to open */
    if ( f == NULL ) return ( FALSE );

    /* if we have found a file then want to show that we are */
    /* compiling the file, so print it to the command line or */
    /* invoke the remote function "message" to display it */
    sprintf ( msg, "Compiling '%s'", fname );
#if 1
    printf ( "%s\n", msg );
#else
    BobInvokeRemoteFunc ( "message", msg );
#endif
    
    /* compile the file */
    compile_definitions(fgetc,f);
    fclose(f);

    /* get the name to initiaize */
    if ( ( ptr = strrchr ( fname, '/' ) ) != NULL ) {
        strcpy ( fname, ptr+1 );
    } else {
        strcpy ( fname, filename );
    }
    
    if ( ( ptr = strstr ( fname, ".bob" ) ) != NULL ) *ptr = '\0';
    for ( i=0; fname[i]; i++ ) {
        switch ( fname[i] ) {
        case '.': fname[i] = '_'; break;
        }
    }

    info ( "Initializing with function '%s'", fname );
    /* execute the initialization code */
    if ( execute ( fname, 0, NULL ) ) {
        del_function ( fname );
    }
    return ( TRUE );
}
/******************************************************************************
 * Function:
 * BobOpLoad -- Loads up the passed file name
 * 
 * Description:
 * This function loads the passed filename. It is called by the child after 
 * getting a LOAD opcode
 *
 * PRE:  Bob correctly set up
 * Returns:
 * Nothing
 */
static void BobOpLoad ( int argcnt, VALUE *args, VALUE *retval )
{
    GenBuffer   filename;
    
    /* check the arg count */
    if ( argcnt < 1 ) {
        set_nil ( retval );
        return;
    }

    /* get the file and try to load it */
    getcstring ( filename, sizeof ( filename ), args );
    if ( BobLoadFilename ( filename ) ) {
        set_integer ( retval, TRUE );
    } else {
        set_nil ( retval );
    }
}

/******************************************************************************
 * Function:
 * BobOpExec -- Executes the passed function name
 * 
 * Description:
 * This function executes the passed function. It is called by the child after 
 * getting an EXECUTE opcode
 *
 * PRE:  Bob correctly set up
 * POST: Function name id executed
 */
static void BobOpExec ( int argcnt, VALUE *args, VALUE *retval )
{
    char        funcname[TKNSIZE];

    /* check the arg count */
    if ( argcnt < 1 ) {
        set_nil ( retval );
        return;
    }
        
    /* set up the function name */
    getcstring ( funcname, sizeof ( funcname ), args );

    /* return value */
    set_integer ( retval, execute ( funcname, argcnt-1, &args[1] ) );
}

/******************************************************************************
 * Function:
 * BobOpAddFunction -- Adds the passed external function reference
 * 
 * Description:
 * This function adds the passed function name as an edternal reference
 * It is called by the child after receiving a ADDF opcode. 
 *
 * PRE:  Bob correctly set up
 * POST: Adds the external function name to the dictionary 
 */
static void BobOpAddFunction ( int argcnt, VALUE *args, VALUE *retval )
{
    DICT_ENTRY *sym;

    char        funcname[TKNSIZE];

    /* check the arg count */
    if ( argcnt < 1 ) {
        set_nil ( retval );
        return;
    }

    /* set up the function name */
    getcstring ( funcname, sizeof ( funcname ), args );

    /* set up the the external function */
    sym = addentry ( &symbols, funcname, ST_SFUNCTION );
    set_rcode ( &sym->de_value, makestring ( funcname ) );
    
    set_integer ( retval, TRUE );
}

/******************************************************************************
 * Function:
 * BobOpQuit -- Quits out of the child
 * 
 * Description:
 * This function quits from the child. It also write acknowledgement that
 * it is quitting before it terminates
 *
 * PRE:  Bob correctly set up
 * POST: Sends back the QUIT opcode and terminates the process
 */
static void BobOpQuit ( int argcnt, VALUE *args, VALUE *retval )
{
    int         op;             /* opcode */
    char        type;           /* type of the buffer */
    GenBuffer   buff;
    int         len;
    
    /* set up a small buffer */
    op = TALK_QUIT;
    type = 0;
    
    /* send a break and then a quit signal */
    BobWriteEvent ( op, &type, sizeof ( int ) );
    
    /* shutdown this end of the sockets TRUE to exit */
    BobTalkShutdown ( TRUE );
    
}

/******************************************************************************
 * Function:
 * BobRemoteFunction -- Executes a remote function call
 * 
 * Description:
 * This function is called by the child when the macro processor has to 
 * execute an external function. This is one of the few functions that sends 
 * a message to the parent. If this function can be called recursively and 
 * therefore can call bob recursively. 
 * This functions has a list of operations that cannot be performed 
 * recursively. This is currently only the LOAD operation. This could be added 
 * to if necessary.
 *
 * PRE:  Bob correctly set up
 * POST: Sends a request to parent for remote function
 */
void BobRemoteFunction ( VALUE *fcn, int argc )
{
    GenBuffer   buff;
    VALUE       *stk_args[20];
    VALUE       args[20];
    int         i;              /* counter */
    int         len;            /* length of the buffer read */
    int         op;             /* opcode of the packet */
    int         argcnt;         /* number of arguments */
    
    /* get the args and check, put them into another list */
    BobGetArgs ( stk_args, argc, -1, -1 );
    memcpy ( &args[0], fcn, sizeof ( VALUE ) );
    for ( i=0; i<argc; i++ ) {
        memcpy ( &args[i+1], stk_args[i], sizeof ( VALUE ) );
    }
    
    /* pack up the args */
    getcstring ( buff, sizeof ( buff ), fcn );

    info ( "Requesting remote function :- %s", buff );
    
    len = BobPackBuffer ( buff, args, argc+1 );
    BobWriteEvent ( TALK_REMO, buff, len );
    
    /* loop till we get the result */
    for ( op=TALK_NOOP; op==TALK_NOOP; ) {
        
        /* see if there is an event there */
        op = BobReadEvent ( buff, &len, TRUE );
        switch ( op ) {
        case TALK_REMO:         /* correct return */
            argcnt = BobUnpackBuffer ( buff, args, 20 );
            if ( argcnt == 0 ) BobFixStack ( argc, DT_NIL, FALSE );
            else BobFixStack ( argc, args[0].v_type, args[0].v.v_integer );
            break;
            
        case TALK_RERR:         /* error has occured */
            argcnt = BobUnpackBuffer ( buff, args, 20 );
            getcstring ( buff, sizeof ( buff ), &args[0] );
            BobInvokeRemoteFunc ( "error", buff );
            BobFixStack ( argc, DT_NIL, FALSE );
            break;
            
        case TALK_NOOP: break;  /* do nothing for NOOP */
            
        /* Cannot be re-entrant with loads while in remote function. */
        /* So cannot process a load while in a remote function */
        case TALK_LOAD:         /* cannot process a load while doing a remote function */
            BobQueueOperation ( op, buff, len );
            break;
            
        /* the default case is where there might be executing another command */   
        /* so this is the reentrancy situation */
        default:
            /* processes the incoming event */
            BobTalkProcessEvent ( op, buff );
            op = TALK_NOOP;
            break;
        }
    }
}

/******************************************************************************
 * Function:
 * BobUnpackBuffer -- Unpacks the buffer into the passed value list
 * 
 * Description:
 * Unpacks the buffer into the passed value list
 * Called by the parent or child
 *
 * PRE:  Bob correctly set up
 * POST: Returns the number of arguments in the buffer and
 *       the buffer args are unpacked into args
 */
int BobUnpackBuffer ( char *buff, VALUE *args, int argc )
{
    int         argcnt;         /* number of args */
    char        *bptr;          /* pointer into buffer */
    int         i;              /* counter */
    int         type;           /* type of the variable */
    int         ival;           /* integer value */
    
    /* get the number of args */
    bptr = buff;
    memcpy ( &argcnt, bptr, sizeof ( int ) ); bptr += sizeof ( int );
    
    /* get the values out */
    for ( i=0; i<argcnt && i<argc; i++ ) {
        /* make sure we don't overflow the buffer */
        if ( bptr - buff >= sizeof ( GenBuffer ) ) break;
        
        memcpy ( &type, bptr, sizeof ( int ) ); bptr += sizeof ( int );
        switch ( type ) {
        case DT_INTEGER:
            memcpy ( &ival, bptr, sizeof ( int ) ); bptr += sizeof ( int );
            set_integer ( &args[i], ival );
            break;
            
        case DT_NIL:
            set_nil ( &args[i] );
            break;
            
        case DT_RCODE:
        case DT_STRING:
            /* make sure the buffer is NULL terminated (incase of garbage) */
            memcpy ( &ival, bptr, sizeof ( int ) ); bptr += sizeof ( int );
            bptr[ival] = '\0';
            set_string ( &args[i], makestring ( bptr ) );
            bptr += ival + 1;
            break;
        }
    }
    return ( argcnt );
}
/******************************************************************************
 * Function:
 * BobPackBuffer -- Packs the args into the passed buffer
 * 
 * Description:
 * Packs the args into the passed buffer
 * Called by the parent or child
 *
 * PRE:  Bob correctly set up
 * POST: Returns the size of the buffer and packs the args into that buffer 
 */
int BobPackBuffer ( char *buff, VALUE *args, int argc )
{
    char        *bptr;          /* pointer into buffer */
    int         i;              /* counter */
    
    /* get the number of args */
    bptr = buff;
    memcpy ( bptr, &argc, sizeof ( int ) ); bptr += sizeof ( int );
    
    /* get the values out */
    for ( i=0; i<argc; i++ ) {
        
        /* make sure we don't overflow the buffer */
        if ( bptr - buff >= sizeof ( GenBuffer ) ) break;
        
        memcpy ( bptr, &args[i].v_type, sizeof ( int ) ); 
        bptr += sizeof ( int );
        
        switch ( args[i].v_type ) {
        case DT_INTEGER:
            memcpy ( bptr, &args[i].v.v_integer, sizeof ( int ) ); 
            bptr += sizeof ( int );
            break;

        case DT_NIL:
            /* do nothing */
            break;
            
        case DT_RCODE:
        case DT_STRING:
            memcpy ( bptr, &(strgetsize ( &args[i] ) ), sizeof ( int ) );
            bptr += sizeof ( int );
            memcpy ( bptr, strgetdata ( &args[i] ), strgetsize ( &args[i] ) );
            bptr += strgetsize ( &args[i] );
            *bptr++ = '\0';
            break;
        }
    }
    /* return the size */
    return ( bptr - buff );
}
/******************************************************************************
 * Function:
 * BobInvokeRemoteFunc -- sends a function name and arg string to parent
 * 
 * Description:
 * This functions sends a function request to the parent. The function name 
 * must exist in the remote function calls. This function will not handle 
 * reentrancy. So it cannot be called with any functions that requite user 
 * input
 *
 * PRE:  Bob correctly set up
 * POST: Sends a request to the parent to execure the remote function 
 */
static void BobInvokeRemoteFunc ( char *func, char *err )
{
    GenBuffer   buff;
    VALUE       args[2];        /* name and message */
    int         len;            /* length of buffer */
    
    set_string ( &args[0], makestring ( func ) );
    set_string ( &args[1], makestring ( err ) );
    
    len = BobPackBuffer ( buff, args, 2 );
    BobWriteEvent ( TALK_REMO, buff, len );
    
    /* wait for the returned event */
    BobReadEvent ( buff, &len, TRUE );
    
    /* if this was an error then stop processing */
    if ( strcmp ( func, "error" ) == 0 ) longjmp ( error_trap, 1 );

}

/* add_file - add a built-in file */
static void add_file(name,fp,pval)
  char *name; FILE *fp; VALUE *pval;
{
    extern IODISPATCH fileio;
    DICT_ENTRY *sym;            /* dictionary entry */
    
    sym = addentry(&symbols,name,ST_SDATA);
    set_iostream(&sym->de_value,newiostream(&fileio,fp));
    *pval = sym->de_value;
}

/******************************************************************************
 * Function:
 * BobQueueOperation -- Puts the passed operation on the Queue
 * 
 * Description:
 *
 * PRE:  Bob correctly set up
 * POST: The passed op and buffer are put in the queue to be processed later
 */
static int BobQueueOperation ( int op, char *buff, int len )
{
    BobOpQueue  *node;

    /* allocate some memory */
    if ( ( node = (BobOpQueue *)malloc ( sizeof ( BobOpQueue ) + len ) ) == NULL ) {
        return ( FALSE );
    }

    info ( "Queueing operation '%s'", talkopcodes[op] );
    
    /* initialize the node */
    node->next = NULL;
    node->op = op;
    node->len = len;
    memcpy ( node->buff, buff, len );
    
    /* put it on the end of the list */
    if ( bobOpQueue == NULL ) {
        bobOpQueue = node;
    } else {
        bobOpQueue->last->next = node;
    }
    bobOpQueue->last = node;
}
/******************************************************************************
 * Function:
 * BobCheckQueue - Checks if there is an op on the queue and removes it
 * 
 * Description:
 *
 * PRE:  Bob correctly set up
 * POST: Returns the opcode to process with the buffer.
 *       TALK_NOOP if there are no events to process
 */
static int BobCheckQueue ( char *buff, int *len )
{
    BobOpQueue  *node;
    int         op;

    /* see if there is an operation here */
    if ( bobOpQueue == NULL ) return ( TALK_NOOP );
    
    /* get the element off the queue */
    node = bobOpQueue;
    if ( ( bobOpQueue = bobOpQueue->next ) != NULL ) {
        bobOpQueue->last = node->last;
    }
    
    /* copy the data */
    memcpy ( buff, node->buff, node->len );
    *len = node->len;
    op = node->op;
    
    info ( "Removing queued operation '%s'", talkopcodes[op] );

    /* free up the memory */
    free ( node );
    
    return ( op );
}
/******************************************************************************
 * Function:
 * BobBlockWait - Blocks and waits for event from sockets or passed fd
 * 
 * Description:
 * This function efectively takes over the event wait in X. It is passed 
 * the X socket and checks for events coming through on it
 *
 * PRE:  Bob correctly set up
 * POST: Returns when there is an event on the socket or passed fd
 */
void BobBlockWait ( int fd )
{
    fd_set      readfds;        /* read mask */
    struct timeval timeout;     /* timeout */
    struct timeval *tout;       /* timeout pointer */
    int         s;              /* result of select */
    
    /* make sure we have a socket */
    if ( sock < 0 || err_sock < 0 || fd < 0 ) return;
    
    /* set to block till event */
    tout = NULL;
    
    /* check for reads and exceptions */
    FD_ZERO ( &readfds );
    FD_SET ( sock, &readfds );
    FD_SET ( err_sock, &readfds );
    FD_SET ( fd, &readfds );

    /* do the select wait for event */
    select ( FD_SETSIZE, &readfds, NULL, NULL, tout );
}
