/* bobface.c: This is the interface between bob and other libraries
 *
 * Copyright (c) 1994 Brett Dutton
 *
 *** Revision History:
 * 22-Sep-1994 Dutton    Changed BobExecute to check if the function being 
 * ..................... called to see if it can bypass the Bob processor
 * 14-Sep-1994 Dutton    Changed to use TALK_RERR (remote error) rather
 * ..................... than using the quit option
 *  7-Sep-1994 Dutton    Allowed BobExecute to have arguments passed
 *  6-Sep-1994 Dutton    Converted function descriptions to pre and 
 * ..................... post conditions
 * 31-Aug-1994 Dutton    Made modifications to allow this to be then
 * ..................... interface to all of the bob functions. This involved
 * ..................... calls to the socket comms interface
 * 9-Aug-1994 Dutton     Copied from bob.c and changed over
 *
 * Previous by D.Betz
 * 1.0 08/15/91        Initial version for DDJ article.
 * 1.1 08/20/91        Fixed do_for to allow null statements.
 * 1.2 08/28/91        Fixed problem with newobject().
 * 1.3 09/27/91        Fixed a bug in compact_vector().
 * 1.4 10/09/91        Reworked for Windows 3 - part 1.
 * 1.5 10/12/91        Various bug fixes.  Added gc().
 * 1.6 08/05/94        Fixed operator precedence bug in assignment.
 *
 */

/*
 * Description:
 */

/*
 * includes
 */
#include <setjmp.h>
#include "bob.h"

/*
 * macros
 */

/*
 * typedefs
 */

/*
 * prototypes
 */
static void compile_file(char *);

/*
 * variables
 */
/* global variables */
jmp_buf error_trap;
static VALUE extFunc;
static VALUE nil;

/*
 * functions
 */

/******************************************************************************
 * Function:
 * BobInitialize -- Initializes the Bob macro language
 * 
 * Description:
 * This function initializes the Bob macro language
 *
 * PRE:  TRUE
 * POST: External function dictionary is set up, memory is initialized
 *       Returns the result of BobTalkInit
 */

int BobInitialize ( void )
{
    /* initialize the dictionary for remote function calls */
    set_nil ( &extFunc );
    set_nil ( &nil );
    set_dictionary ( &extFunc, newdictionary ( &nil) );

    initialize ( SMAX );        /* initialize memory */

    /* initialize the processor by forking off */
    return ( BobTalkInit () );
}

/******************************************************************************
 * Function:
 * BobLoadFile -- Loads the passed bob filename 
 * 
 * Description:
 * This function loads the passed file name into memory. The filename is packed
 * up and passed to the chid. The child does the work. This function does 
 * not wait for a reply from the child because it will report any errors and fail
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE on success and write a load file request to the socket
 *       FALSE is the filename is zero length or null
 */
int BobLoadFile ( char *fname )
{
    int         len;            /* length of string */
    VALUE       val;            /* value to pack the buffer into */
    GenBuffer   buff;           /* buffer for comms */
    
    /* get out if null */
    if ( fname == NULL || fname[0] == '\0' ) return ( FALSE );

    /* set up the string and write the buffer */
    set_string ( &val, makestring ( fname ) );
    len = BobPackBuffer ( buff, &val, 1 );
    BobWriteEvent ( TALK_LOAD, buff, len );
    
    return ( TRUE );
}

/******************************************************************************
 * Function:
 * BobAddFunction -- Adds the passed function name and function id
 * 
 * Description:
 * Passes the function name to the child as a remote function. The name
 * and function pointer are stored in a local list. When a remote function 
 * is called by Bob then the arguments are unpacked and passed to the correct 
 * function
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE on success and write a add func  request to the socket
 *       also stores the name and pointer of the external function in dictionary
 *       FALSE is the function name is zero length or null
 */
int BobAddFunction ( char *fname, BobExternFunc func )
{
    int         len;            /* length of string */
    VALUE       val;            /* value to pack the buffer into */
    GenBuffer   buff;           /* buffer for comms */
    DICT_ENTRY  *sym;           /* dictionary entry */
    
    /* get out if null */
    if ( fname == NULL || fname[0] == '\0' ) return ( FALSE );

    /* set up the string and write the buffer */
    set_string ( &val, makestring ( fname ) );
    len = BobPackBuffer ( buff, &val, 1 );
    BobWriteEvent ( TALK_ADDF, buff, len );
    
    /* set up the function in local memory for remote calls */
    sym = addentry ( &extFunc, fname, ST_SFUNCTION );
    set_code ( &sym->de_value, (BobInternFunc)func );
    
    return ( TRUE );
}
/******************************************************************************
 * Function:
 * BobExecute -- Executes the passed function name
 * 
 * Description:
 * Puts the function name into buffer and passes it to the child. The child 
 * Then takes care of it. This function does not wait for a return from the 
 * child because it will report any errors and terminate. If this function 
 * is called with the name of an external function then the child processor 
 * is bypassed and the function executed directly.
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE on success and write a exec function request to the socket
 *       FALSE is the function name is zero length or null
 */
int BobExecute ( char *name, int cnt, ... )
{
    int         len;            /* length of string */
    VALUE       val[20];        /* value to pack the buffer into */
    GenBuffer   buff;           /* buffer for comms */
    va_list     ap;             /* list of values */
    int         i;              /* counter */
    int         type;           /* type of the arg */
    DICT_ENTRY  *sym;           /* dictionary entry */
    BobExternFunc func;         /* external function */
    
    /* get out if null */
    if ( name == NULL || name[0] =='\0' ) return ( FALSE );

    /* if the function is an external function and no args then */
    /* bypass the compiler */
    if ( cnt == 0 && ( sym = findentry ( &extFunc, name ) ) != NULL ) {

        info ( "Bypassing Bob for function:- %s", name );

        /* execute the remote function */
        func = (BobExternFunc)sym->de_value.v.v_code;
        (*func) ( 0, &val[0], &val[1] );
        
        return ( TRUE );
    }
        
    /* set up the ist of args */
    va_start ( ap, cnt );

    /* set up the string and write the buffer */
    set_string ( &val[0], makestring ( name ) );
    
    /* pack up the arguments into vaues */
    for ( i=1; i<=cnt; i++ ) {
        type = va_arg ( ap, int );
        switch ( type ) {
        case DT_INTEGER: 
            set_integer ( &val[i], va_arg ( ap, int ) ); 
            break;
        case DT_STRING:
            set_string ( &val[i], makestring ( va_arg ( ap, char * ) ) );
            break;
        case DT_NIL:
            set_nil ( &val[i] );
            break;
        }
    }

    /* pack it up and ship it out */
    len = BobPackBuffer ( buff, val, cnt + 1 );
    BobWriteEvent ( TALK_EXEC, buff, len );
    
    /* fix up the list */
    va_end ( ap );

    return ( TRUE );
}
/******************************************************************************
 * Function:
 * BobGetString -- Gets the string from the passed value
 * 
 * Description:
 *
 * PRE:  Bob is correctly set up
 * POST: Copied the string in the passed VALUE into the passed string buffer
 *       and returns the pointer to that string
 */
char * BobGetString ( char *buf, int len, VALUE *val )
{
    return ( getcstring ( buf, len, val ) );
}
/******************************************************************************
 * Function:
 * BobGetInteger -- Gets the integer from the passed value
 * 
 * Description:
 *
 * PRE:  Bob is correctly set up
 * POST: Returns the integer contained int the passed VALUE
 */
int BobGetInteger ( VALUE *val )
{
    return ( val->v.v_integer );
}

/******************************************************************************
 * Function:
 * BobCheckClient -- Checks if there is anything on the socket and acts on it
 * 
 * Description:
 * This function is called by the client and checks if there is any
 * messages from the Child (macro processor) and processes them. 
 *
 * PRE:  Bob is correctly set up
 * POST: If function request was waiting on the socket then get the 
 *       name and execute it, return the result to the socket
 */
int BobCheckClient ( void )
{
    VALUE       args[20];
    VALUE       retval;
    int         op;
    int         argcnt, len;
    GenBuffer   buff;
    GenBuffer   errstr;
    char        funcname[100];
    DICT_ENTRY  *sym;           /* dictionary entry */
    BobExternFunc func;         /* external function */
    int         res;            /* result of function */
    
    /* get the event, don't block */
    op = BobReadEvent ( buff, &len, FALSE );
    
    switch ( op ) {
    case TALK_REMO:
        /* unpack the args from the read buffer */
        argcnt = BobUnpackBuffer ( buff, args, 20 );
        getcstring ( funcname, sizeof ( funcname ), &args[0] );

        /* get the entry from the dictionary */
        sym = findentry ( &extFunc, funcname );
        func = (BobExternFunc)sym->de_value.v.v_code;
        
        info ( "Executing external function:- %s", funcname );
        
        /* execute the registered function */
        if ( ( res = (*func) ( argcnt-1, &args[1], &retval ) ) != TRUE ) {
            /* set up error string for return */
            getcstring ( buff, sizeof ( buff ), &retval );
            sprintf ( errstr, "Error: %s : %s", funcname, buff );
            set_string ( &retval, makestring ( errstr ) );
            op = TALK_RERR;
        }
        
        /* pack up the buffer */
        len = BobPackBuffer ( buff, &retval, 1 );
            
        /* send back the quit option on failure */
        BobWriteEvent ( op, buff, len );

        break;
        
    /* no action to be taken */
    default: return ( FALSE );
    }
    
    return ( TRUE );
}

/******************************************************************************
 * Function:
 * BobExtArgs -- Checks if the number of args agrees with the passed number
 * 
 * Description:
 * mn is the minimum number of args the function can have, mx is 
 * the maximum. If the mn, or mx is -1 then it is not checked. It 
 * sets up the error string and returns FALSE if there is a problem
 *
 * PRE:  Bob is correctly set up
 * POST: TRUE id the arg count is within passed limits
 *       FALSE otherwise. Also sets the error string to return
 */
int BobExtArgs ( int argc, int mn, int mx, VALUE *retval )
{
    char        *err=NULL;
    
    if ( (mn) >= 0 && argc < (mn) ) err = "Too few arguments";
    if ( (mx) >= 0 && argc > (mx) ) err = "Too many arguments"; 
    if ( err == NULL ) return ( TRUE );

    /* set up the return value */
    set_string ( retval, makestring ( err ) );
    return ( FALSE );
}
/******************************************************************************
 * Function:
 * BobExtCheckType -- Checks the type of the passed argument. 
 * 
 * Description:
 * Checks the type of the passed argument. If the arg is not of the passed 
 * type then sets up an error string of the expected error and returns FALSE
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE if the types of the arguments match passed types
 *       FALSE otherwise and the error string is set up in retval
 */
int BobExtCheckType ( VALUE *arg, int type, VALUE *retval )
{
    GenBuffer   buff;
    char        exp[20], rec[20];
    
    /* if the same type then get out */
    if ( arg->v_type == type ) return ( TRUE );
    
    /* set up an error string */
    strcpy ( exp, typename ( type ) );
    strcpy ( rec, typename ( arg->v_type ) );
    sprintf ( buff, "Argument type mismatch, received: %s, expected: %s",
             rec, exp );
    set_string ( retval, makestring ( buff ) );
    return ( FALSE );
}
/******************************************************************************
 * Function:
 * BobReturnValue -- Puts the passed value in retval 
 * 
 * Description:
 * This function takes the passed argument and sets up the returned value
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE and sets up the return value to send back to bob
 */
int BobReturnValue ( VALUE *retval, int type, ... )
{
    va_list ap;
    va_start(ap,type);
    switch ( type ) {
    case DT_INTEGER: 
        set_integer ( retval, va_arg ( ap, int ) ); 
        break;
        
    case DT_STRING: 
        set_string ( retval, makestring ( va_arg ( ap, char * ) ) );
        break;
        
    }
    retval->v_type = type;
    va_end(ap);
    return ( TRUE );
}

/* info - display progress information */
void info(char *fmt,...)
{
#ifdef DEBUG
    char buf[256];
    va_list ap;
    va_start(ap,fmt);
    vsprintf(buf,fmt,ap);
    va_end(ap);
    fprintf(stderr,"%s\n",buf);
#endif
}

/* error - print an error message and exit */
void error(char *fmt,...)
{
    char buf[256];
    va_list ap;
    va_start(ap,fmt);
    vsprintf(buf,fmt,ap);
    va_end(ap);
    fprintf(stderr,"Error: %s\n",buf);
    longjmp(error_trap,1);
}

void osputs(str)
  char *str;
{
    fputs(str,stderr);
}
