/* bob.h - bob definitions */
/*
 *      Copyright (c) 1991, by David Michael Betz
 *      All rights reserved
 *      
 *** Revision History
 * 14-Oct-1994 Dutton    Made the runtime stack 10000
 * 12-Oct-1994 Dutton    Added prototype for BobBlockWait.
 *  4-Oct-1994 Dutton    Put in the function descriptions for the API
 * 14-Sep-1994 Dutton    Added Remote error to the opcodes
 * 31-Aug-1994 Dutton    Changed the stack size to make it nice and big
 * 31-Aug-1994 Dutton    Added the rcode Remote function to the operations
 * 31-Aug-1994 Dutton    Put the version ID in here
 * 31-Aug-1994 Dutton    Added new prototype foe external interfaces and new types
 */

#ifndef BOB_HEADER
#define BOB_HEADER

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

/* limits */
#define TKNSIZE         50      /* maximum token size */
#define SMAX            10000   /* runtime stack size */

/* useful definitions */
#define TRUE            1
#define FALSE           0

#define BOBVERS "Bob v1.7 - Copyright (c) 1994, by David Betz, Modified by Brett Dutton."

/* argument check macros */
#define argcount(n,cnt) { if ((n) != (cnt)) wrongcnt(n,cnt); }

/* stack manipulation macros */
#define check(n)         { if (sp - (n) < stkbase) stackover(); }
#define chktype(o,t)     { if (sp[o].v_type != t) badtype(o,t); }
#define push(x,t,f)      (--sp, sp->v_type = (t), sp->v.f = (x))
#define push_integer(x)  push(x,DT_INTEGER,v_integer)
#define push_string(x)   push(x,DT_STRING,v_string)
#define push_class(x)    push(x,DT_CLASS,v_class)
#define push_object(x)   push(x,DT_OBJECT,v_object)
#define push_bytecode(x) push(x,DT_BYTECODE,v_vector)
#define push_var(x)      push(x,DT_VAR,v_var)
#define push_nil()       (--sp, sp->v_type = DT_NIL)

/* macros to set values */
#define set(s,x,t,f)        ((s)->v.f = (x), (s)->v_type = (t))
#define set_integer(s,x)    set(s,x,DT_INTEGER,v_integer)
#define set_class(s,x)      set(s,x,DT_CLASS,v_class)
#define set_object(s,x)     set(s,x,DT_OBJECT,v_object)
#define set_code(s,x)       set(s,x,DT_CODE,v_code)
#define set_rcode(s,x)      set(s,x,DT_RCODE,v_string)
#define set_bytecode(s,x)   set(s,x,DT_BYTECODE,v_vector)
#define set_dictionary(s,x) set(s,x,DT_DICTIONARY,v_dictionary)
#define set_var(s,x)        set(s,x,DT_VAR,v_var)
#define set_string(s,x)     set(s,x,DT_STRING,v_string)
#define set_vector(s,x)     set(s,x,DT_VECTOR,v_vector)
#define set_iostream(s,x)   set(s,x,DT_IOSTREAM,v_iostream)
#define set_nil(s)          ((s)->v_type = DT_NIL)

/* value field access macros */
#define valtype(x)              ((x)->v_type)
#define isnil(x)                ((x)->v_type == DT_NIL)

/* class field access macros */
#define claddr(x)               ((x)->v.v_class)
#define clgetname(x)            (&claddr(x)->cl_name)
#define clgetbase(x)            (&claddr(x)->cl_base)
#define clgetmembers(x)         (&claddr(x)->cl_members)
#define clgetfunctions(x)       (&claddr(x)->cl_functions)
#define clgetsize(x)            (claddr(x)->cl_size)

/* object field access macros */
#define objaddr(x)              ((x)->v.v_object)
#define objgetclass(x)          (&objaddr(x)->obj_class)
#define objgetmember(x,i)       (&objaddr(x)->obj_members[i])
#define objsetmember(x,i,v)     (objaddr(x)->obj_members[i] = (v))

/* vector field access macros */
#define vecaddr(x)              ((x)->v.v_vector)
#define vecgetsize(x)           (vecaddr(x)->vec_size)
#define vecgetelement(x,i)      (&vecaddr(x)->vec_data[i])
#define vecsetelement(x,i,v)    (vecaddr(x)->vec_data[i] = (v))

/* string field access macros */
#define straddr(x)              ((x)->v.v_string)
#define strgetsize(x)           (straddr(x)->str_size)
#define strgetdata(x)           (straddr(x)->str_data)

/* dictionary field access macros */
#define diaddr(x)               ((x)->v.v_dictionary)
#define digetclass(x)           (&diaddr(x)->di_class)
#define digetcontents(x)        (&diaddr(x)->di_contents)

/* dictionary entry field access macros */
#define deaddr(x)               ((x)->v.v_var)
#define degetdictionary(x)      (&deaddr(x)->de_dictionary)
#define degetkey(x)             (&deaddr(x)->de_key)
#define degetvalue(x)           (&deaddr(x)->de_value)
#define degetnext(x)            (&deaddr(x)->de_next)
#define degettype(x)            (deaddr(x)->de_type)

/* i/o stream access macros */
#define ios_t(x)                ((x)->v.v_iostream->ios_dispatch)
#define ios_d(x)                ((x)->v.v_iostream->ios_data)
#define iosclose(x)             ((*ios_t(x)->iod_close)(ios_d(x)))
#define iosgetc(x)              ((*ios_t(x)->iod_getc)(ios_d(x)))
#define iosputc(c,x)            ((*ios_t(x)->iod_putc)((c),ios_d(x)))
#define iosputs(s,x)            ((*ios_t(x)->iod_puts)((s),ios_d(x)))

/* value descriptor structure */
typedef struct value {
  int v_type;                           /* data type */
  union {                               /* value */
    struct class *v_class;              /*   class (in heap) */
    struct object *v_object;            /*   object (in heap) */
    struct vector *v_vector;            /*   vector (in heap) */
    struct string *v_string;            /*   string (in heap) */
    struct dictionary *v_dictionary;    /*   dictionary (in heap) */
    struct dict_entry *v_var;           /*   variable (in heap) */
    void (*v_code) ( int );     /*   code for built-in function */
    long v_integer;                     /*   integer */
    struct iostream *v_iostream;        /*   i/o stream (in heap) */
    struct hdr *v_hdr;                  /*   (used by garbage collector) */
    struct value *v_chain;              /*   (used by garbage collector) */
  } v;
} VALUE;

/* function pointer */
typedef int (*BobExternFunc)( int, VALUE *args, VALUE *retval );
typedef void (*BobInternFunc)( int );

typedef struct hdr {
    char hdr_type;
    char hdr_flags;
    VALUE *hdr_chain;
} HDR;

typedef struct class {
    HDR cl_hdr;
    VALUE cl_name;
    VALUE cl_base;
    VALUE cl_members;
    VALUE cl_functions;
    int cl_size;
} CLASS;

typedef struct object {
    HDR obj_hdr;
    VALUE obj_class;
    VALUE obj_members[1];
} OBJECT;

typedef struct vector {
    HDR vec_hdr;
    int vec_size;
    VALUE vec_data[1];
} VECTOR;

typedef struct string {
    HDR str_hdr;
    int str_size;
    char str_data[1];
} STRING;

typedef struct dictionary {
    HDR di_hdr;
    VALUE di_class;
    VALUE di_contents;
} DICTIONARY;

/* dictionary entry structure */
typedef struct dict_entry {
    HDR de_hdr;
    VALUE de_dictionary;        /* backpointer to dictionary */
    VALUE de_key;               /* symbol name */
    int de_type;                /* symbol type */
    VALUE de_value;             /* symbol value */
    VALUE de_next;              /* next entry */
} DICT_ENTRY;

typedef int (*CloseFunc) ( void *);
typedef int (*GetFunc) ( void *);
typedef int (*PutFunc) ( int, void *);
typedef int (*PutSFunc) ( char *, void *);
/* i/o stream dispatch structure */
typedef struct iodispatch {
    CloseFunc   iod_close;
    GetFunc     iod_getc;
    PutFunc     iod_putc;
    PutSFunc    iod_puts;
} IODISPATCH;

/* i/o stream structure */
typedef struct iostream {
    HDR ios_hdr;
    IODISPATCH *ios_dispatch;
    void *ios_data;
} IOSTREAM;

/* set up some general buffers */
typedef char GenBuffer[5120];

/* symbol types */
#define ST_CLASS        1       /* class definition */
#define ST_DATA         2       /* data member */
#define ST_SDATA        3       /* static data member */
#define ST_FUNCTION     4       /* function member */
#define ST_SFUNCTION    5       /* static function member */

/* data types */
#define _DTMIN          0
#define DT_NIL          0
#define DT_CLASS        1
#define DT_OBJECT       2
#define DT_VECTOR       3
#define DT_INTEGER      4
#define DT_STRING       5
#define DT_BYTECODE     6
#define DT_CODE         7
#define DT_DICTIONARY   8
#define DT_VAR          9
#define DT_IOSTREAM     10
#define DT_RCODE        11      /* remote function */
#define _DTMAX          11

/* opcodes */
#define OP_BRT          0x01    /* branch on true */
#define OP_BRF          0x02    /* branch on false */
#define OP_BR           0x03    /* branch unconditionally */
#define OP_NIL          0x04    /* load top of stack with nil */
#define OP_PUSH         0x05    /* push nil onto stack */
#define OP_NOT          0x06    /* logical negate top of stack */
#define OP_NEG          0x07    /* negate top of stack */
#define OP_ADD          0x08    /* add top two stack entries */
#define OP_SUB          0x09    /* subtract top two stack entries */
#define OP_MUL          0x0A    /* multiply top two stack entries */
#define OP_DIV          0x0B    /* divide top two stack entries */
#define OP_REM          0x0C    /* remainder of top two stack entries */
#define OP_BAND         0x0D    /* bitwise and of top two stack entries */
#define OP_BOR          0x0E    /* bitwise or of top two stack entries */
#define OP_XOR          0x0F    /* bitwise xor of top two stack entries */
#define OP_BNOT         0x10    /* bitwise not of top two stack entries */
#define OP_SHL          0x11    /* shift left top two stack entries */
#define OP_SHR          0x12    /* shift right top two stack entries */
#define OP_LT           0x13    /* less than */
#define OP_LE           0x14    /* less than or equal to */
#define OP_EQ           0x15    /* equal to */
#define OP_NE           0x16    /* not equal to */
#define OP_GE           0x17    /* greater than or equal to */
#define OP_GT           0x18    /* greater than */
#define OP_INC          0x19    /* increment */
#define OP_DEC          0x1A    /* decrement */
#define OP_LIT          0x1B    /* load literal */
#define OP_RETURN       0x1C    /* return from interpreter */
#define OP_CALL         0x1D    /* call a function */
#define OP_REF          0x1E    /* load a variable value */
#define OP_SET          0x1F    /* set the value of a variable */
#define OP_VREF         0x20    /* load a vector element */
#define OP_VSET         0x21    /* set a vector element */
#define OP_MREF         0x22    /* load a member variable value */
#define OP_MSET         0x23    /* set a member variable */
#define OP_AREF         0x24    /* load an argument value */
#define OP_ASET         0x25    /* set an argument value */
#define OP_TREF         0x26    /* load a temporary variable value */
#define OP_TSET         0x27    /* set a temporary variable */
#define OP_TSPACE       0x28    /* allocate temporary variable space */
#define OP_SEND         0x29    /* send a message to an object */
#define OP_DUP2         0x2A    /* duplicate top two elements on the stack */
#define OP_NEW          0x2B    /* create a new class object */

/* opcodes for communications */
#define TALK_NOOP 0             /* no op */
#define TALK_LOAD 1             /* load a file */
#define TALK_EXEC 2             /* execute a function */
#define TALK_ADDF 3             /* add a remote function */
#define TALK_REMO 4             /* executing a remore function */
#define TALK_QUIT 5             /* quit the system */
#define TALK_RERR 6             /* remote error */
#define TALK_MAX  7             /* maximum number of remote functions */

/* external variables */
extern VALUE *stkbase,*sp,*fp,*stktop;
extern VALUE nil;

/* external routines */

/* bobcom.c */
int init_compiler(void);
void mark_compiler(void);
int compile_definitions(int (*getcf)( FILE * ),void *getcd);

/* bob.c */
void info(char *fmt,...);
void error(char *fmt,...);
void osputs(char *str);

/* bobscn.c */
int init_scanner(int (*gf)(void *),void *gd);
int token(void);
int stoken(int tkn);
char *tkn_name(int tkn);
int parse_error(char *msg);

/* bobint.c */
int execute( char *name, int argc, VALUE *args );
int start_send(OBJECT *obj,char *selector);
int execute_call(int argc);
void badtype(int off,int type);
void stackover(void);
char *typename(int type);

/* bobdbg.c */
int decode_procedure(VALUE *code);
int decode_instruction(VALUE *code,int lc);

/* bobfcn.c */
void init_functions(void);
int del_function( char *name );
void add_function(char *name, BobInternFunc fcn );
int print1(VALUE *ios,int qflag,VALUE *val);
void wrongcnt(int n,int cnt);
void BobFixStack ( int argc, int type, ... );
void BobGetArgs ( VALUE **arg, int argc, int mn, int mx );
int BobCheckType ( VALUE *arg, int type );

/* bobmem.c */
int initialize(int smax);
DICT_ENTRY *addentry(VALUE *dict,char *key,int type);
DICT_ENTRY *findentry(VALUE *dict,char *key);
int delentry( VALUE *dict, char *key );
STRING *makestring(char *str);
char *getcstring(char *buf,int max,VALUE *str);
STRING *newstring(int n);
OBJECT *newobject(VALUE *class);
VECTOR *newvector(int n);
CLASS *newclass(char *name,VALUE *base);
DICTIONARY *newdictionary(VALUE *class);
IOSTREAM *newiostream(IODISPATCH *iod,void *data);
void gc(void);
void mark(VALUE *val);

/* bobtalk.c */
/******************************************************************************
 * Function:
 * BobTalkInit -- Initializes the communications and processes
 * 
 * Description:
 * This function forks and returns to the parent. The child process
 * caklls a function that never returns from this function and sits in
 * an endless loop waiting for communications along the socket.
 * This function is called by the parent only ( before the fork)
 *
 * PRE:  Bob memory must correctly initialized
 * POST: returns socket id on parent and writes initialization opcode to child
 *       doesn't return from CHILD. If BOB fails to initialize then returns 
 *       -1 on the parent. The child is not started
 */
int BobTalkInit ( void );

/******************************************************************************
 * Function:
 * BobTalkTerm -- Terminates communications and processes
 * 
 * Description:
 * This functions shuts doen the parent and child sockets and
 * terminates the child process by sending request to child to shutdown
 * This is the only parent request that actually waits for an answer.
 * This function is called by the parent only
 *
 * PRE:  Bob correctly set up
 * POST: Returns TRUE on success and child process is shutdown
 */
int BobTalkTerm ( void );

/******************************************************************************
 * Function:
 * BobReadEvent -- Checks if there is an event outstanding
 * 
 * Description:
 * This function can be called by the parent or the child;
 * This function gets the next event from the socket. If there is not
 * an event on the socket then it waits. It assumes that the first 4
 * bytes is the opcode and the remaining is the data. The opcode is
 * returned and the data and length are passed back. If wait is TRUE
 * then the function will wait for the next event, otherwise it is a poll
 *
 * PRE:  Bob correctly set up
 * POST: returns The opcode associated with this packet, or TALK_NOOP is 
 *       there is no data available
 */
int BobReadEvent ( GenBuffer buff, int *len, int wait );

/******************************************************************************
 * Function:
 * BobWriteEvent -- Writes the data back to the socket
 * 
 * Description:
 * This function can be called by the parent or the child;
 * This function writes the passed buffer to the socket. The opcode is
 * written first to the socket followed by the buffer
 *
 * PRE:  Bob correctly set up
 * POST: returns TRUE on success and data is written to the socket
 *       -1 if there is a socket eror
 */
int BobWriteEvent ( int op, GenBuffer buff, int len );

/******************************************************************************
 * Function:
 * BobCheckBreak -- Checks if there is a break event
 * 
 * Description:
 * This function can be called by the child;
 * This function checks the error socket. Since the child cannot break
 * the parent it is not a problem if the parent check the break status
 * since the child should never write to the error socket
 *
 * PRE:  Bob correctly set up
 * POST: returns TRUE is there is a packet of the error socket
 *       FALSE if no error
 *       -1 on socket error
 */
int BobCheckBreak ( void );

/******************************************************************************
 * Function:
 * BobBreak -- Creates a break event
 * 
 * Description:
 * This function can be called by the parent;
 * This function creates a break on the socket for the child to
 * read. Since the parent can never be interrupted by the child then the
 * child should never cal this function
 *
 * PRE:  Bob correctly set up
 * POST: Returns TRUE and a message is written on the error socket
 */
int BobBreak ( char *fmt, ... );

/******************************************************************************
 * Function:
 * BobRemoteFunction -- Executes a remote function call
 * 
 * Description:
 * This function is called by the child when the macro processor has to 
 * execute an external function. This is one of the few functions that sends 
 * a message to the parent. If this function can be called recursively and 
 * therefore can call bob recursively. 
 * This functions has a list of operations that cannot be performed 
 * recursively. This is currently only the LOAD operation. This could be added 
 * to if necessary.
 *
 * PRE:  Bob correctly set up
 * POST: Sends a request to parent for remote function
 */
void BobRemoteFunction ( VALUE *fcn, int argc );

/******************************************************************************
 * Function:
 * BobUnpackBuffer -- Unpacks the buffer into the passed value list
 * 
 * Description:
 * Unpacks the buffer into the passed value list
 * Called by the parent or child
 *
 * PRE:  Bob correctly set up
 * POST: Returns the number of arguments in the buffer and
 *       the buffer args are unpacked into args
 */
int BobUnpackBuffer ( char *buff, VALUE *args, int argc );

/******************************************************************************
 * Function:
 * BobPackBuffer -- Packs the args into the passed buffer
 * 
 * Description:
 * Packs the args into the passed buffer
 * Called by the parent or child
 *
 * PRE:  Bob correctly set up
 * POST: Returns the size of the buffer and packs the args into that buffer 
 */
int BobPackBuffer ( char *buff, VALUE *args, int argc );

/******************************************************************************
 * Function:
 * BobLoadFilename -- Loads up the passed file name
 * 
 * Description:
 * This function loads the passed filename. It is called by the child.
 * As a future enhancement this function should check for compiled 
 * bob files. At the moment the Bob macro processor cannot compile 
 * files, so just check for .bob extensions
 * Search rules:
 * <filename>
 * getenv("HOME")/<filename>
 * getenv("BOBLIB")/<filename>
 * <filename>.bob
 * getenv("HOME")/<filename>.bob
 * getenv("BOBLIB")/<filename>.bob
 *
 * PRE:  Bob correctly set up
 * POST: returns TRUE on success and the filename is compiled into memory. 
 *       FALSE if the file name does not exist using the following rules
 */
int BobLoadFilename ( char *filename ); /* called by child  */

/******************************************************************************
 * Function:
 * BobBlockWait - Blocks and waits for event from sockets or passed fd
 * 
 * Description:
 *
 * PRE:  Bob correctly set up
 * POST: Returns when there is an event on the socket or passed fd
 */
void BobBlockWait ( int fd );

/***********************************************************************
 * What follows is the library interface to the bob macro processor
 ***********************************************************************/

/* bobface.c */
/******************************************************************************
 * Function:
 * BobInitialize -- Initializes the Bob macro language
 * 
 * Description:
 * This function initializes the Bob macro language
 *
 * PRE:  TRUE
 * POST: External function dictionary is set up, memory is initialized
 *       Returns the result of BobTalkInit
 */
int BobInitialize ( void );

/******************************************************************************
 * Function:
 * BobLoadFile -- Loads the passed bob filename 
 * 
 * Description:
 * This function loads the passed file name into memory. The filename is packed
 * up and passed to the chid. The child does the work. This function does 
 * not wait for a reply from the child because it will report any errors and fail
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE on success and write a load file request to the socket
 *       FALSE is the filename is zero length or null
 */
int BobLoadFile ( char *fname );

/******************************************************************************
 * Function:
 * BobAddFunction -- Adds the passed function name and function id
 * 
 * Description:
 * Passes the function name to the child as a remote function. The name
 * and function pointer are stored in a local list. When a remote function 
 * is called by Bob then the arguments are unpacked and passed to the correct 
 * function
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE on success and write a add func  request to the socket
 *       also stores the name and pointer of the external function in dictionary
 *       FALSE is the function name is zero length or null
 */
int BobAddFunction ( char *fname, BobExternFunc func );

/******************************************************************************
 * Function:
 * BobExecute -- Executes the passed function name
 * 
 * Description:
 * Puts the function name into buffer and passes it to the child. The child 
 * Then takes care of it. This function does not wait for a return from the 
 * child because it will report any errors and terminate. If this function 
 * is called with the name of an external function then the child processor 
 * is bypassed and the function executed directly.
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE on success and write a exec function request to the socket
 *       FALSE is the function name is zero length or null
 */
int BobExecute ( char *name, int cnt, ... );

/******************************************************************************
 * Function:
 * BobGetString -- Gets the string from the passed value
 * 
 * Description:
 *
 * PRE:  Bob is correctly set up
 * POST: Copied the string in the passed VALUE into the passed string buffer
 *       and returns the pointer to that string
 */
char * BobGetString ( char *buf, int len, VALUE *val );

/******************************************************************************
 * Function:
 * BobGetInteger -- Gets the integer from the passed value
 * 
 * Description:
 *
 * PRE:  Bob is correctly set up
 * POST: Returns the integer contained int the passed VALUE
 */
int BobGetInteger ( VALUE *val );

/******************************************************************************
 * Function:
 * BobCheckClient -- Checks if there is anything on the socket and acts on it
 * 
 * Description:
 * This function is called by the client and checks if there is any
 * messages from the Child (macro processor) and processes them. 
 *
 * PRE:  Bob is correctly set up
 * POST: If function request was waiting on the socket then get the 
 *       name and execute it, return the result to the socket
 */
int BobCheckClient ( void );

/******************************************************************************
 * Function:
 * BobExtArgs -- Checks if the number of args agrees with the passed number
 * 
 * Description:
 * mn is the minimum number of args the function can have, mx is 
 * the maximum. If the mn, or mx is -1 then it is not checked. It 
 * sets up the error string and returns FALSE if there is a problem
 *
 * PRE:  Bob is correctly set up
 * POST: TRUE id the arg count is within passed limits
 *       FALSE otherwise. Also sets the error string to return
 */
int BobExtArgs ( int argc, int mn, int mx, VALUE *retval );

/******************************************************************************
 * Function:
 * BobExtCheckType -- Checks the type of the passed argument. 
 * 
 * Description:
 * Checks the type of the passed argument. If the arg is not of the passed 
 * type then sets up an error string of the expected error and returns FALSE
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE if the types of the arguments match passed types
 *       FALSE otherwise and the error string is set up in retval
 */
int BobExtCheckType ( VALUE *arg, int type, VALUE *retval );

/******************************************************************************
 * Function:
 * BobReturnValue -- Puts the passed value in retval 
 * 
 * Description:
 * This function takes the passed argument and sets up the returned value
 *
 * PRE:  Bob is correctly set up
 * POST: Returns TRUE and sets up the return value to send back to bob
 */
int BobReturnValue ( VALUE *retval, int type, ... );



#endif
