
/******************************************************************************
*
*       Recorder - Windows 95 macro recorder
*		
*		Copyright (c) 1995 Mark Russinovich and Bryce Cogswell
*
*		You have the right to take this code and use it in whatever 
*               way you
*		wish.
*
*    	PROGRAM: Record.c
*
*    	PURPOSE: Interface for recording keyboard/mouse macros.
*
******************************************************************************/

#include <windows.h>  
#include <stdio.h>
#include <process.h>
#include "resource.h"  
#include "record.h"    

// general globals
#define				VXD_NAME		"\\\\.\\RECORD.VXD"
static HANDLE		vxd_handle		= INVALID_HANDLE_VALUE;
static char			msgbuf[ 200 ];
HANDLE      		hEvent = 0;
OVERLAPPED  		ovlp = {0,0,0,0,0};

// status of macro recording
enum {
	STATUS_OFF,
	STATUS_STARTING,
	STATUS_RECORDING,
	STATUS_PLAYBACK,
} Status = STATUS_OFF;


// macros definitions
#define MAX_MACRO	4
#define MACRO_FMT	"macro.%d"
#define MAX_NAME	64
#define MAX_EVENTS	8192
#define EMPTY_MACRO	"(empty)"
#define NEW_MACRO	"new macro"

#define RESET_MOUSE	1

// hotkey types
enum {
	HK_FINISH	= 0,
	HK_BEGIN	= 100,
};

// virtual keyboard translation
#define VIRTUAL_KEY_CODE(f) ((f)+0x70)

// macro record definitions
#pragma pack(1)
struct REC_common {
	BYTE	rectype;			// either keyboard or mouse
	DWORD	timestamp;			// system time 
};
struct REC_kbd_entry {
	struct REC_common	common;
	BYTE			scancode;	// keyboard scancode
};
struct REC_mouse_entry {
	struct REC_common	common;		
	DWORD			deltax;		// mouse delta x	
	DWORD			deltay;		// mouse delta y	
	DWORD			mouseptr;	// pointer to mouse instance
	BYTE			button;		// button status
};
#pragma pack()


// macro structure
int	Key;
struct macro {
	char	Name[ MAX_NAME ];
	size_t	Event_Len;
	char	Events[ MAX_EVENTS ];
}		Macro[ MAX_MACRO ];
#define HDR_LEN	((char *)(((struct macro *)NULL)->Events) - (char *)NULL)

LRESULT CALLBACK MainDialog( HWND hDlg, UINT message, WPARAM uParam, 
			    LPARAM lParam );



/******************************************************************************
 *
 * 	Abort:
 *	
 *	Handles click on close button.
 *
 *****************************************************************************/
void Abort( HWND hWnd, char * Msg )
{
	MessageBox( hWnd, Msg, "Recorder", MB_OK );
	PostQuitMessage( 1 );
}


/****************************************************************************
 *
 *	WinMain:
 *
 *	Calls dialog box startup.
 *
 ****************************************************************************/
int CALLBACK WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,
        				LPSTR lpCmdLine, int nCmdShow )
{
	static char szAppName [] = "RECORD" ;
	HWND        	hwnd ;
	MSG         	msg ;
	WNDCLASSEX  	wndclass ;

	// create a dummy class
	if (!hPrevInstance) {
		wndclass.cbSize			= sizeof( WNDCLASSEX );
		wndclass.style          = CS_HREDRAW | CS_VREDRAW ;
 		wndclass.lpfnWndProc    = (WNDPROC) MainDialog ;
		wndclass.cbClsExtra     = 0 ;
		wndclass.cbWndExtra     = DLGWINDOWEXTRA ;
		wndclass.hInstance      = hInstance ;
		wndclass.hIcon          = LoadIcon (hInstance, "ICON") ;
		wndclass.hCursor        = LoadCursor (NULL, IDC_ARROW) ;
		wndclass.hbrBackground  = (HBRUSH) (COLOR_BTNFACE+1);
		wndclass.lpszMenuName   = NULL ;
		wndclass.lpszClassName  = szAppName ;
		wndclass.hIconSm		= LoadIcon (hInstance, "ICON");
		RegisterClassEx (&wndclass) ;
	}

	// create the dialog
	hwnd = CreateDialog (hInstance, "DIALOG", NULL, (DLGPROC)MainDialog) ;
	ShowWindow (hwnd, nCmdShow) ;
 
	while (GetMessage (&msg, NULL, 0, 0)) {
		if( !IsDialogMessage( hwnd, &msg )) {
          TranslateMessage (&msg) ;
          DispatchMessage (&msg) ;
		}
	}
	return msg.wParam ;
}


/****************************************************************************
 *
 *	centerDialog
 *
 * Centers the dialog on the screen.
 *
 ****************************************************************************/
VOID centerDialog( HWND hDlg )
{
	RECT            aRt;

	// center the dialog box
	GetWindowRect( hDlg, &aRt );
	OffsetRect( &aRt, -aRt.left, -aRt.top );
	MoveWindow( hDlg,
			((GetSystemMetrics( SM_CXSCREEN ) -
				aRt.right ) / 2 + 4) & ~7,
  			(GetSystemMetrics( SM_CYSCREEN ) -
				aRt.bottom) / 2,
			aRt.right, aRt.bottom, 0 );
}


/****************************************************************************
 *
 *	Center_Mouse:
 *
 *	Moves the mouse to the center of the screen.
 *
 ****************************************************************************/
void Center_Mouse( HWND hWnd )
{
#if RESET_MOUSE
	mouse_event( MOUSEEVENTF_ABSOLUTE | MOUSEEVENTF_MOVE, 32768, 
		    32768, 0, 0 );	
#endif
}


/****************************************************************************
 *
 *	Empty_Macro:
 *
 *	Zeroes out a macro that is deleted.
 *
 ****************************************************************************/
void Empty_Macro( struct macro * m )
{
	strcpy( m->Name, EMPTY_MACRO );
	m->Event_Len = 0;
	m->Events[0] = 0;
}


/****************************************************************************
 *
 *	Read_Macro_File:
 *
 *	Open a macro file and read the macro out of it and into a buffer.
 *
 ****************************************************************************/
void Read_Macro_File( HWND hDlg, int N )
{
	FILE * fp;

	/* Read previous macro info from file */
	wsprintf( msgbuf, MACRO_FMT, N );
	fp = fopen( msgbuf, "rb" );
	if ( fp == NULL  ||
		 fread( &Macro[N], HDR_LEN, 1, fp ) != 1  ||
		 fread( Macro[N].Events, sizeof Macro[N].Events[0], 
		       Macro[N].Event_Len, fp )
				!= Macro[N].Event_Len  ||
		 fclose( fp )  ||
		 Macro[N].Event_Len == 0  ||
		 Macro[N].Events[0] == 0 ) {
		/* No macro associated with name */
		Empty_Macro( &Macro[N] );
	}

	SetDlgItemText( hDlg, IDC_F1+N, Macro[N].Name );
}



/****************************************************************************
 *
 *	Write_Macro_File:
 *
 *	Saves a macro to a macro file in the current directory.
 *
 ****************************************************************************/ 
void Write_Macro_File( HWND hDlg, int N )
{
	FILE *	fp;
	size_t	len;

	len = (char *)&((struct macro *)NULL)->Events[Macro[N].Event_Len] - 
	  (char *)NULL;

	wsprintf( msgbuf, MACRO_FMT, N );
	fp = fopen( msgbuf, "wb" );
	if ( fp == NULL  ||  
		 fwrite( &Macro[N], len, 1, fp ) != 1  ||  
		 fclose( fp ) )	{
		wsprintf( msgbuf, "Error writing macro file" );
		MessageBox( hDlg, msgbuf, "Recorder", MB_OK );
	}
}


/****************************************************************************
 *
 *	Wait_callback:
 *
 *	This function is executed by a seperate thread. It blocks on a VxD 
 *  event that is signalled by the record VxD when a macro has completed 
 *  playback. It then plays a beep to let the user know that the macro
 *  is complete. 
 *
 ****************************************************************************/
VOID Wait_callback( VOID *dummy )
{
	DWORD cbRet;
	
	// wait for event to be signalled	
	GetOverlappedResult(vxd_handle, &ovlp, &cbRet, TRUE);

	// notify user that macro is done 
	Status = STATUS_OFF;
	if ( ! PlaySound( "BEEP", NULL, SND_RESOURCE|SND_ASYNC|SND_NOWAIT ) )
		MessageBeep( MB_ICONEXCLAMATION	);
	// end thread implied 
}


/****************************************************************************
 *
 *	MainDialog:
 *
 *	Dialog message handling routine.
 *
 ****************************************************************************/
LONG APIENTRY MainDialog( HWND hDlg, UINT message, WPARAM uParam, 
			    LPARAM lParam )
{
	int	j;
	int hotkey;
	static int ignore = -1;

    switch ( message ) {
    case WM_INITDIALOG:

        // open the handle to the VXD allowing overlapped results
		vxd_handle = CreateFile( VXD_NAME, 0, 0, NULL,
							0, FILE_FLAG_OVERLAPPED|
							FILE_FLAG_DELETE_ON_CLOSE,
							NULL );
		if ( vxd_handle == INVALID_HANDLE_VALUE )  {
	       	wsprintf( msgbuf, "Opening %s: error %d", VXD_NAME, 
						GetLastError( ) );
	       	Abort( hDlg, msgbuf );
		}

		// Create event which will be used by VxD to indicate operation is
		// complete
		if ( !(hEvent = CreateEvent(0, TRUE, 0, NULL)) ) {
	       	wsprintf( msgbuf, "CreateEvent failed!" );
	       	Abort( hDlg, msgbuf);
		}
		ovlp.hEvent = hEvent;

		// Register hot keys
		for ( j = 0; j < MAX_MACRO; ++j )  {
	       	if ( ! RegisterHotKey( hDlg, HK_BEGIN+j, 0, 
					VIRTUAL_KEY_CODE(j) ) )
		 		Abort( hDlg, "Can't set hotkey" );
	       		Read_Macro_File( hDlg, j );
		}

		// center ourselves on screen
		centerDialog( hDlg );
		break;

    case WM_HOTKEY:
		hotkey = uParam - HK_BEGIN;

		// ignore the replay of a macro terminating hot-key - so we don't 
        // infinitly loop
		if(ignore == hotkey) {
			ignore = -1;
	        break;
		}

        // What we do with the hotkey depends on what we are currently doing
		switch ( Status )  {

		case STATUS_STARTING:

			// start recording macro
			Status = STATUS_RECORDING;

	 		Key = hotkey;
			if ( ! CheckRadioButton( hDlg, IDC_RADIO1, 
                	IDC_RADIO1+MAX_MACRO, IDC_RADIO1+Key ) )
		    	Abort( hDlg, "Can't check radio button" );

	 		if ( ! DeviceIoControl(	vxd_handle, VXD_RECORDON,
					&hDlg, sizeof hDlg, NULL, 0, NULL, NULL ) ) {
				Abort( hDlg, "Can't start recording" );
			}
						
			// Put mouse in a standard location for starting macro 
			Center_Mouse( hDlg );
			break;

		case STATUS_RECORDING:

			// we are recording so see if we should stop
			if ( hotkey != Key )
				break;

			// stop this macro
			Status = STATUS_OFF;
	 		if ( ! DeviceIoControl(	vxd_handle, VXD_RECORDOFF,
					NULL, 0, Macro[Key].Events, 
					sizeof Macro[Key].Events, 
					&Macro[Key].Event_Len, NULL ) ) {
				Abort( hDlg, "Can't stop recording" );
			}
			Macro[Key].Event_Len /= sizeof Macro[Key].Events[0];
			if ( strcmp( Macro[Key].Name, EMPTY_MACRO ) == 0 )  {
				wsprintf( Macro[Key].Name, NEW_MACRO );
				SetDlgItemText( hDlg, IDC_F1+Key, Macro[Key].Name );
			}
			EnableWindow( GetDlgItem( hDlg, IDC_RECORD ), TRUE );
			break;

		case STATUS_PLAYBACK:
			// if playing back ignore all hotkeys
			break;

		case STATUS_OFF:

			// start a playback if one is defined for the hotkey
			if ( Macro[hotkey].Event_Len == 0 )  {
				wsprintf( msgbuf, "No macro associated with key F%d",
                                 hotkey+1 );
				MessageBox( hDlg, msgbuf, "Recorder", MB_OK );
				break;
			}	

			Key = hotkey;
			if ( ! CheckRadioButton( hDlg, IDC_RADIO1,
                	IDC_RADIO1+MAX_MACRO, IDC_RADIO1+Key ) )
				Abort( hDlg, "Can't check radio button" );

			Center_Mouse( hDlg );
			Status = STATUS_PLAYBACK;
			ignore = hotkey;
	 		if ( ! DeviceIoControl(	vxd_handle, VXD_PLAYBACK,
					Macro[Key].Events, 
				    Macro[Key].Event_Len, 
					NULL, 0, NULL, &ovlp ) ) {
				Abort( hDlg, "Can't put buffer" );
			}
					
			// Thread will block until playback done 
			_beginthread( Wait_callback, 0, NULL );
			break;
		}
		break;

    case WM_COMMAND:
		if ( Status != STATUS_OFF )
			break;

    	switch( LOWORD( uParam ) ) {

		case IDOK:
		case IDCANCEL:   
	    	EndDialog( hDlg, TRUE );        // Exit the dialog
			PostQuitMessage( 0 );
	        return TRUE;
	        break;

		case IDC_RECORD:

			// Open a dialog box to get the function key 
			EnableWindow( GetDlgItem( hDlg, IDC_RECORD ), FALSE );
			switch ( MessageBox( hDlg, 
            	"Press a function key (F1-F4) to begin recording, "
    		   	"and press a second time to terminate recording", 
		       	"Recorder", MB_OKCANCEL ) ) {
			case IDOK:
		  		Status = STATUS_STARTING;
		  		break;
			case IDCANCEL:
				EnableWindow( GetDlgItem( hDlg, IDC_RECORD ), TRUE );
				break;
			}
			break;

		case IDC_SAVE:  {

			// save all recorded macros to files
			int j;
			for ( j = 0; j < MAX_MACRO; ++j )  {
				// Write new one 
				if ( Macro[j].Event_Len > 0 )
				Write_Macro_File( hDlg, j );
			}
		}
		break;

		case IDC_DELETE:  {
			// clear the selected macro
			wsprintf( msgbuf, MACRO_FMT, Key );
			if ( remove( msgbuf ) )  {
				wsprintf( msgbuf, "Error deleting macro file" );
				MessageBox( hDlg, msgbuf, "Recorder", MB_OK );
			}	
			Empty_Macro( &Macro[Key] );
			SetDlgItemText( hDlg, IDC_F1+Key, Macro[Key].Name );
		}
		break;

		case IDC_F1:
		case IDC_F2:
		case IDC_F3:
		case IDC_F4:

			// select the macro entry listbox indicated by the function key
			Key = LOWORD(uParam) - IDC_F1;
			if ( ! CheckRadioButton( hDlg, IDC_RADIO1, 
           			IDC_RADIO1+MAX_MACRO, IDC_RADIO1+Key ) )
				Abort( hDlg, "Can't check radio button" );
			GetDlgItemText( hDlg, LOWORD(uParam), Macro[Key].Name, 
             			sizeof Macro[Key].Name );
		break;

		case IDC_RADIO1:
		case IDC_RADIO2:
		case IDC_RADIO3:
		case IDC_RADIO4:
			Key = uParam - IDC_RADIO1;
			break;
		}
		break;
	}
	return DefWindowProc ( hDlg, message, uParam, lParam); 
}
