
/************************************************************************
 *																								*
 *		D51 8051/52 Disassembler - Copyright (C) 1995 by						*
 *		Jeffery L. Post																	*
 *		22726 Benner Ave.																	*
 *		Torrance, CA  90505																*
 *																								*
 *		D51P2.C - Pass two of disassembly											*
 *																								*
 *		Rescan data array and generate output file. Generate label if		*
 *		location flagged as referenced by some other opcode. If un-			*
 *		initialized data is encountered, ignore it but set skip flag so	*
 *		that an ORG statement can be generated when the next initialized	*
 *		data is found. This file also contains pass 3, which outputs		*
 *		equates to the assembly source file.										*
 *																								*
 *  Version 2.1 - 10/21/95																*
 *																								*
 *	This program is free software; you can redistribute it and/or modify	*
 *	it under the terms of the GNU General Public License as published by	*
 *	the Free Software Foundation; either version 2 of the License, or		*
 *	(at your option) any later version.												*
 *																								*
 *	This program is distributed in the hope that it will be useful,		*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.									*
 *																								*
 *	You should have received a copy of the GNU General Public License		*
 *	along with this program; if not, write to the Free Software				*
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.				*
 *																								*
 ************************************************************************/

#include	<stdio.h>
#include	<dos.h>
#include	<ctype.h>
#include	<alloc.h>
#include	"d51.h"

/*********************
 *							*
 *		Prototypes		*
 *							*
 *********************/

void	pass2(void);
void	pass3(void);
void	doopcode(char *mnem);
void	dodir(FILE *fp, word dir);
void	dobit(FILE *fp, word bit);
void	chk_ref(word i);
void	splitcheck(int i);
void	puthex(word j);
int	ascii(int i);
int	is_ascii(byte data);

extern char	*find_entry(word val, struct sym *ptr);
extern void	dump_ascii(word adrs);
extern void	dump_bytes(word adrs);

#ifdef	DEBUG

extern void	dump_sym(void);
extern void	dump_lab(void);

#endif

/************************
 *								*
 *		Global variables	*
 *								*
 ************************/

int	newline;								/* just output newline flag	*/

extern char	src[32], dst[32];		   /* file name buffers				*/
extern struct date	date;				/* disassembly date				*/
extern struct time	time;				/* disassembly time				*/
extern FILE	*fp;							/* dos file struct				*/
extern int	hexflag;						/* append hex flag				*/
extern int	kcnt;							/* output char counter			*/
extern word	himark;						/* highest data adrs				*/
extern word	offset;						/* program offset					*/
extern byte	far *pmem;					/* program data pointer			*/
extern byte	far *pflg;					/* pointer to program flags	*/
extern char	string[ASCLIMIT];			/* ascii data for defb			*/
extern word	asccount;					/* count for string data		*/
extern byte	byte_data[BYTELIMIT];	/* binary data for defb			*/
extern word	byte_cnt;					/* count for binary data		*/
extern byte	dump;							/* dump just done flag			*/
extern byte	ascii_flag;					/* use ascii string flag		*/
extern char	defbstr[8];					/* string for defined bytes	*/
extern char	defwstr[8];					/* string for defined words	*/
extern byte	dirregs[32];				/* access flags for dir reg	*/
extern struct sym	*sym_tab;			/* symbol table pointer			*/
extern struct sym	*lab_tab;			/* label table pointer			*/

extern struct mnementry mnemtbl[256];
extern byte	opttbl[256];
extern struct sfrent sfr[256];
extern struct sfrent sfrbits[128];
extern struct sfrent rbname[32];

/************************************************************************

	Pass two of disassembly. Rescan data array and generate output file.
	Generate label if location flagged as referenced by some other opcode.
	If un-initialized data is encountered, ignore it but set skip flag so
	that an ORG statement can be generated when the next initialized data
	is found.

*************************************************************************/

void pass2(void)
{
	char	c, *cptr;
	int	skip = 1;
	word	i, l, pc, q, temp;
	byte	j, k;

	k = 0;
	j = 0xff;
	dump = 0;
	byte_cnt = 0;
	asccount = 0;
	newline = 0;

	printf("\nPass 2 0000");

	if ((fp = fopen(dst, "w")) == NULL)			/* open output source file */
	{
		printf("\n* Can't open file %s *\n", dst);
		exit(FILE_ERROR);
	}
	fprintf(fp, ";\n;  8051 Disassembly of %s", src);	/* output header */
	getdate(&date);
	gettime(&time);
	fprintf(fp, "\n;  %d/%d/%d %d:%02d", date.da_mon, date.da_day,
		date.da_year, time.ti_hour, time.ti_min);

	if (ascii_flag)					/* if user specified A on command line... */
	{
		fprintf(fp, "\n;\nascii\tmacro\tx");		/* generate macro for text */
		fprintf(fp, "\n\t%s\t'\#x\#'", defbstr);
		fprintf(fp, "\n\tendm");
	}

/*  Generate output file */

	for (i=offset; i<himark; )
	{
		l = i & 0xff;
		if (pflg[i] == PF_INIT)					/* ignore un-initialized data */
		{
			if (byte_cnt)				/* if binary or ascii data in buffers... */
				dump_bytes(i);			/* output them now */
			if (asccount)
				dump_ascii(i);
			if (dump)					/* if we had to flush buffers...	*/
			{								/* do a newline, since we will	*/
				dump = 0;				/* be doing an org (or end)		*/
				fprintf(fp, "\n;");
				newline++;
			}
			i++;										/* next program location */
			skip = 1;								/* flag skip for ORG statement */
			j = -1;
		}
		else if (pflg[i] & (PF_ADRS | PF_WORD | PF_BYTE | PF_ASCII))
		{												/* if not executable code */
			switch (pflg[i] & ~PF_REF)			/* ignore reference bit */
			{
				case PF_ASCII:						/* if ascii text... */
					if (byte_cnt)					/* dump any pending binary */
						dump_bytes(i);
					if (skip)						/* insert ORG statement */
					{
						if (!newline)
							fprintf(fp, "\n;");
						fprintf(fp, "\n\torg\t");
						puthex(i);
						fprintf(fp, "\n;");
						newline++;
						skip = 0;					/* reset skip flag */
					}
					if (is_ascii(pmem[i]))		/* if it really is ascii... */
					{
						string[asccount] = pmem[i];
						asccount++;
						if (asccount >= ASCLIMIT)
							dump_ascii(i);
					}
					else								/* else treat it as binary data */
					{
						pflg[i] &= ~PF_ASCII;
						pflg[i] |= PF_BYTE;
						if (asccount)				/* dump any accumulated ascii */
							dump_ascii(i);
						byte_data[byte_cnt] = pmem[i];	/* add data to buffer */
						byte_cnt++;
					}
					break;

				case PF_WORD:					/* if word data... */
					if (byte_cnt)			/* dump any binary or ascii in buffers */
						dump_bytes(i);
					if (asccount)
						dump_ascii(i);
					if (skip)					/* insert ORG statement */
					{
						if (!newline)
							fprintf(fp, "\n;");
						fprintf(fp, "\n\torg\t");
						puthex(i);
						fprintf(fp, "\n;");
						newline++;
						skip = 0;				/* reset skip flag */
					}
					chk_ref(i);							/* add label if referenced */
					q = ((word) pmem[i]) << 8;		/* get word value */
					temp = pmem[i + 1] & 0xff;
					q |= temp;
					fprintf(fp, "%s\t", defwstr);
					cptr = find_entry(q, sym_tab);	/* see if symbol exists */
					if (cptr == NULL)						/* if not, do hex value */
						puthex(q);
					else
						fprintf(fp, "%s", cptr);		/* else output symbol */
					if (hexflag)							/* add comment field */
						fprintf(fp, "\t\t; %04x   %02x %02x      %c%c",
							i, pmem[i], pmem[i+1],
							ascii(pmem[i]), ascii(pmem[i+1]));
					i++;
					if (pflg[i + 2] != PF_ADRS && pflg[i + 2] != PF_WORD)
					{
						fprintf(fp, "\n;");
						newline++;
					}
					break;

				case PF_ADRS:					/* if address data... */
					if (byte_cnt)				/* output any pending binary or */
						dump_bytes(i);			/* ascii data from buffers */
					if (asccount)
						dump_ascii(i);
					if (skip)					/* insert ORG statement */
					{
						if (!newline)
							fprintf(fp, "\n;");
						fprintf(fp, "\n\torg\t");
						puthex(i);
						fprintf(fp, "\n;");
						newline++;
						skip = 0;				/* reset skip flag */
					}
					chk_ref(i);					/* add label if referenced */
					q = ((word) pmem[i]) << 8;	/* get address value */
					temp = pmem[i + 1] & 0xff;
					q |= temp;
					fprintf(fp, "%s\t", defwstr);
					cptr = find_entry(q, lab_tab);	/* see if label exists */
					if (cptr == NULL)						/* if not, output hex */
					{
						cptr = find_entry(q, sym_tab);
						if (cptr == NULL)
							fprintf(fp, "X%04x", q);
						else
							fprintf(fp, "%s", cptr);
					}
					else
						fprintf(fp, "%s", cptr);		/* else output label text */
					if (hexflag)							/* do comment field */
						fprintf(fp, "\t\t; %04x   %02x %02x      %c%c",
							i, pmem[i], pmem[i+1],
							ascii(pmem[i]), ascii(pmem[i+1]));
					i++;
					if (pflg[i + 2] != PF_ADRS)
					{
						fprintf(fp, "\n;");
						newline++;
					}
					break;

				default:							/* default = binary data... */
					if (asccount)				/* output any pending ascii data */
						dump_ascii(i);
					if (skip)					/* insert ORG statement */
					{
						if (!newline)
							fprintf(fp, "\n;");
						fprintf(fp, "\n\torg\t");
						puthex(i);
						fprintf(fp, "\n;");
						newline++;
						skip = 0;				/* reset skip flag */
					}
					byte_data[byte_cnt] = pmem[i];	/* add data to buffer */
					byte_cnt++;
					if (byte_cnt >= BYTELIMIT)			/* if end of buffer...	*/
						dump_bytes(i);						/* dump accumulated data */
			}
			i++;									/* next program location */
			if (pflg[i] != PF_INIT && pflg[i] & PF_ASCII)
			{										/* if next byte is flagged as	*/
				if (!is_ascii(pmem[i]))		/* ascii, but is not...			*/
				{
					pflg[i] |= PF_BYTE;		/* then flag it as				*/
					pflg[i] &= ~PF_ASCII;	/* binary data						*/
				}
			}
		}

/*	If previous data was an unconditional transfer, AND current
	location is not referenced by some other opcode, AND current
	byte is 0 or 0FFH, then treat this byte as un-initialized data.
*/

		else if ((j & 0x80) && (!(pflg[i] & PF_REF)) &&
				((pmem[i] == 0) || (pmem[i] == NO_DATA)))
		{
			if (byte_cnt)				/* since we're going to skip some */
				dump_bytes(i);			/* data, output any pending ascii */
			if (asccount)				/* or binary data remaining in buffers */
				dump_ascii(i);
			if (dump)					/* if ascii or binary output was done, */
			{								/* stick in a newline */
				fprintf(fp, "\n;");
				dump = 0;
				newline++;
			}
			pflg[i] = PF_INIT;		/* flag as uninitialized data */
			i++;
			skip = 1;
			byte_cnt = 0;
		}

/* If previous code was 0 or 0ffH, AND current location is not
	referenced, AND current opcode is 0 or 0ffH, un-initialize it.
	This avoids disassembling line after line after line of NOP or
	MOV R7,A.
*/

		else if ((k == 0 || k == 0xff) && (!(pflg[i] & PF_REF)) &&
			(pmem[i] == 0 || pmem[i] == 0xff) && (pflg[i] != PF_CLREF))
		{
			pflg[i] = '\xff';
			i++;
			skip = 1;
			fprintf(fp, "\t; data truncated");
			j = -1;
		}

		else								/**** IT'S EXECUTABLE CODE! ****/
		{
			if (byte_cnt)				/* if any ascii or binary data remains */
				dump_bytes(i);			/* in the buffers, output them now */
			if (asccount)
				dump_ascii(i);
			if (dump)
			{
				fprintf(fp, "\n;");
				dump = 0;
				newline++;
			}
			byte_cnt = 0;
			if (skip)							/* insert ORG statement */
			{
				if (!newline)
					fprintf(fp, "\n;");
				fprintf(fp, "\n\torg\t");
				puthex(i);
				fprintf(fp, "\n;");
				newline++;
				skip = 0;						/* reset skip flag */
			}
			k = pmem[i];						/* get opcode */
			if (k == 0xa5 && !newline)		/* if invalid opcode */
			{
				fprintf(fp, "\n;");
				newline++;
			}
			chk_ref(i);							/* add label if referenced */
			kcnt = 8;
			j = opttbl[k];						/* get options */
			doopcode(mnemtbl[k].mcode);	/* output mnemonic */

		/* Generate operands */

			switch (j & OPT_MASK)
			{
				case OPT_NONE:					/* single byte - no options */
					break;

				case OPT_IMM2:					/* 2 byte immediate data */
					q = (word) pmem[i + 1] & 0xff;
					cptr = find_entry(q, sym_tab);
					if (cptr == NULL)
						puthex(pmem[i + 1]);
					else
						kcnt += fprintf(fp, "%s", cptr);
					splitcheck(i + 1);
					break;

				case OPT_LREF:					/* 3 byte immediate data */
					q = ((pmem[i+1] & 0xff) << 8) | (pmem[i+2] & 0xff);
					cptr = find_entry(q, lab_tab);
					if (cptr == NULL)
					{
						cptr = find_entry(q, sym_tab);
						if (cptr == NULL)
							kcnt += fprintf(fp, "X%04x", q);
					}
					if (cptr != NULL)
						kcnt += fprintf(fp, "%s", cptr);
					splitcheck(i+1);
					splitcheck(i+2);
					break;

				case OPT_DIR2:					/* 2 byte direct addressing */
					dodir(fp, pmem[i+1]);
					if ((k & 0xf) == 2 || k == 0xf5)
						kcnt += fprintf(fp, ",a");
					else if ((k & 0xf0) == 0x80)
					{
						kcnt += fprintf(fp, ",");
						if (k < 0x88)
							kcnt +=  fprintf(fp, "@r%d", k&1);
						else
							kcnt += fprintf(fp, "r%d", k & 7);
					}
					splitcheck(i+1);
					break;

				case OPT_DIR3:					/* mov dir,dir */
					dodir(fp, pmem[i+2]);
					kcnt += fprintf(fp, ",");
					dodir(fp, pmem[i+1]);
					splitcheck(i+1);
					splitcheck(i+2);
					break;

				case OPT_DMM3:		/* 3 byte direct and immediate addressing */
					dodir(fp, pmem[i+1]);
					kcnt += fprintf(fp, ",#");
					q = (word) pmem[i + 2] & 0xff;
					cptr = find_entry(q, sym_tab);
					if (cptr == NULL)
						puthex(pmem[i + 2]);
					else
						kcnt += fprintf(fp, "%s", cptr);
					splitcheck(i+1);
					splitcheck(i+2);
					break;

				case OPT_BIT2:					/* 2 byte bit addressing */
					dobit(fp, pmem[i+1]);
					if (k == 0x92)
						kcnt += fprintf(fp, ",c");
					splitcheck(i+1);
					break;

				case OPT_REL2:					/* 2 byte relative addressing */
					q = pmem[i+1] & 0xff;
					q = (q > 0x7f) ? q | 0xff00 : q;
					q = i + 2 + q;
					cptr = find_entry(q, lab_tab);
					if (cptr == NULL)
						kcnt += fprintf(fp, "X%04x", q);
					else
						kcnt += fprintf(fp, "%s", cptr);
					splitcheck(i+1);
					break;

				case OPT_IR3:		/* 3 byte immediate and relative addressing */
					q = (word) pmem[i + 1] & 0xff;
					cptr = find_entry(q, sym_tab);
					if (cptr == NULL)
						puthex(pmem[i + 1]);
					else
						kcnt += fprintf(fp, "%s", cptr);
					q = pmem[i+2] & 0xff;
					q = (q > 0x7f) ? q | 0xff00 : q;
					q = i + 3 + q;
					cptr = find_entry(q, lab_tab);
					if (cptr == NULL)
						kcnt += fprintf(fp, ",X%04x", q);
					else
						kcnt += fprintf(fp, ",%s", cptr);
					splitcheck(i+1);
					splitcheck(i+2);
					break;

				case OPT_DR3:		/* 3 byte direct and relative addressing */
					dodir(fp, pmem[i+1]);
					q = pmem[i+2] & 0xff;
					q = (q > 0x7f) ? q | 0xff00 : q;
					q = i + 3 + q;
					cptr = find_entry(q, lab_tab);
					if (cptr == NULL)
						kcnt += fprintf(fp, ",X%04x", q);
					else
						kcnt += fprintf(fp, ",%s", cptr);
					splitcheck(i+1);
					splitcheck(i+2);
					break;

				case OPT_RELB:			/* 3 byte bit and relative addressing */
					dobit(fp, pmem[i+1]);
					q = pmem[i+2] & 0xff;
					q = (q > 0x7f) ? q | 0xff00 : q;
					q = i + 3 + q;
					cptr = find_entry(q, lab_tab);
					if (cptr == NULL)
						kcnt += fprintf(fp, ",X%04x", q);
					else
						kcnt += fprintf(fp, ",%s", cptr);
					splitcheck(i+1);
					splitcheck(i+2);
					break;

				case OPT_112:			/* 2 byte 11 bit adrs (ajmp & acall) */
					q = ((k & 0xe0) << 3) | ((i + 2) & 0xf800);
					q = q | (pmem[i+1] & 0xff);
					cptr = find_entry(q, lab_tab);
					if (cptr == NULL)
						kcnt += fprintf(fp, "X%04x", q);
					else
						kcnt += fprintf(fp, "%s", cptr);
					splitcheck(i+1);
					break;
			}

			if (k == 0xa5)				/* if invalid opcode */
				puthex(k);				/* put hex defb in file */

			if (hexflag)				/* do comment field */
			{
				while (kcnt < tstop)
				{
					fprintf(fp, "\t");
					kcnt = (kcnt + 8) & 0x78;
				}
				fprintf(fp,"; %04x   %02x", i, pmem[i] & 0xff);
				switch (opttbl[k] & OPT_SIZE)			/* additional data bytes */
				{
					case 0:
						fprintf(fp, "	  ");
						break;
					case 1:
						fprintf(fp, " %02x   ", pmem[i+1] & 0xff);
						break;
					case 2:
						fprintf(fp, " %02x %02x",
							pmem[i+1] & 0xff, pmem[i+2] & 0xff);
				}
				fprintf(fp, "   %c", ascii(pmem[i]));
				switch (opttbl[k] & OPT_SIZE)			/* additional ascii */
				{
					case 1:
						fprintf(fp, "%c", ascii(pmem[i+1]));
						break;
					case 2:
						fprintf(fp, "%c%c",
							ascii(pmem[i+1]), ascii(pmem[i+2]));
				}
			}

			newline = 0;
			i = i + (opttbl[k] & OPT_SIZE) + 1;	/* update location counter */

			if (j & OPT_XFER || k == 0xa5)	/* if unconditional transfer or */
			{											/* invalid code, add a newline */
				fprintf(fp, "\n;");
				newline++;
			}
		}
		if ((i & 0xff) < l)
			printf("\rPass 2 %04x", i & 0xff00);
	}
	if (byte_cnt)					/* if any remaining ascii or binary, */
		dump_bytes(i);				/* output it now */
	if (asccount)
		dump_ascii(i);
	printf("\rPass 2 - Source generation complete");

/* add equates for register names */

	j = 0;
	for (i=0; i<32; i++)
	{
		if (dirregs[i])
		{
			if (!j)
			{
				if (!newline || dump)
					fprintf(fp, "\n;");
				fprintf(fp, "\n;\tregister equates\n;");
				newline = 0;
			}
			fprintf(fp, "\n%s\tequ\t", &rbname[i].dent[0]);
			puthex(i);
			j = 1;
		}
	}

#ifdef	DEBUG

	dump_lab();
	dump_sym();

#endif
}													/*  End of Pass 2 */

/************************************************************************
 *																								*
 *		Check if data is printable ascii other than the string delimiter	*
 *																								*
 ************************************************************************/

int is_ascii(byte data)
{
	if (data < ' ' || data > 0x7e || data == '\'')
		return(0);
	return(1);
}

/*********************************************************************
 *																							*
 *	add label to output line if current location marked as referenced	*
 *																							*
 *********************************************************************/

void chk_ref(word i)
{
	char	*cptr;

	if (pflg[i] & PF_REF)
	{
		cptr = find_entry(i, lab_tab);		/* see if label exists */
		if (cptr == NULL)
			fprintf(fp, "\nX%04x:\t", i);		/* if not, output hex value */
		else
			fprintf(fp, "\n%s:\t", cptr);		/* else output label text */
	}
	else
		fprintf(fp, "\n\t");
}

/*********************************************************************
 *																							*
 *						Pass three of disassembly									*
 *	Search for references to un-initialized data or split references	*
 *	and, if found, generate EQU statements for them.						*
 *																							*
 *********************************************************************/

void pass3(void)
{
	int	i, j, l;
	word	val;
	struct sym	*ptr;

	printf("\nPass 3 0000");
	j = 1;
	for (i=0; ; )
	{
		l = i & 0xfff;
		if (pflg[i] & PF_REF)	/* if is un-initialized or is referenced */
		{
			if (!(pflg[i] & PF_CLREF) && (pflg[i] & (PF_SPLIT | PF_NOINIT)))
			{					/* if referenced and (split opcode or not init)... */
				if (j)			/* do header if first one */
				{
					j = 0;
					if (!newline || dump)
						fprintf(fp, "\n;");
					fprintf(fp, "\n;\tmiscellaneous equates\n;");
					newline = 0;
				}
				fprintf(fp, "\nX%04x\tequ\t", i);	/* do EQU statement */
				puthex(i);
			}
		}
		i++;
		if (!i)
			break;
		if ((i & 0xfff) < l)
			printf("\rPass 3 %04x", i);
	}

	/* now do equates for symbol table */

	ptr = sym_tab;						/* start at beginning of symbol table */
	j = 1;
	while (ptr != NULL)				/* not at the end yet */
	{
		if (j)							/* do header if first one */
		{
			j = 0;
			if (!newline || dump)
				fprintf(fp, "\n;");
			fprintf(fp, "\n;\tsymbol equates\n;");
		}
		fprintf(fp, "\n%s\tequ\t", ptr->label);
		val = ptr->adrs;
		puthex(val);
		ptr = ptr->next;				/* get the next entry */
		newline = 0;
	}

	printf("\rPass 3 - Equate generation complete");
	if (!newline || dump)
		fprintf(fp, "\n");
	fprintf(fp, ";\n\tend\n;\n");
	fflush(fp);
	close(fp);
}										/*  End of Pass 3   */

/***************************************
 *													*
 *		Output opcode for current code	*
 *													*
 ***************************************/

void doopcode(char *mnem)
{
	char	c;

	while (c = *mnem++)			/* output text from opcode table */
	{
		if (c == ' ')				/* convert spaces to tabs */
		{
			putc('\t', fp);
			kcnt = (kcnt + 8) & 0x78;
		}
		else
		{
			putc(c, fp);
			kcnt++;
		}
	}
}

/*********************************************
 *															*
 *		Output operand for direct addressing	*
 *															*
 *********************************************/

void dodir(FILE *fp, word dir)
{
	dir &= 0xff;
	if (dir < 0x20)
	{
		kcnt += fprintf(fp, rbname[dir].dent);
		dirregs[dir] = 1;
	}
	else if (dir < 0x80)
		kcnt += fprintf(fp, "%02xh", dir);
	else
		kcnt += fprintf(fp, "%s", sfr[dir & 0x7f].dent);
}

/***************************
 *									*
 *		Output sfr bit name	*
 *									*
 ***************************/

void dobit(FILE *fp, word bit)
{
	int	i;

	bit &= 0xff;
	if (bit < 0x80)
	{
		i = (bit >> 3) + 0x20;
		kcnt += fprintf(fp, "%02xh.%d", i, bit & 7);
	}
	else
		kcnt += fprintf(fp, "%s", sfrbits[bit & 0x7f].dent);
}

/***************************************************
 *																	*
 *		Check for reference to address in middle of	*
 *		code and flag split reference if true			*
 *																	*
 ***************************************************/

void splitcheck(int i)
{
	if (!(pflg[i] & PF_CLREF))		/* ignore if not referenced */
		pflg[i] |= PF_SPLIT;			/* else flag split ref */
}

/************************************
 *												*
 *		Output hexadecimal operand		*
 *												*
 ************************************/

void puthex(word j)
{
	if (j < 10)
		kcnt += fprintf(fp, "%x", j);
	else if (j < 16)
		kcnt += fprintf(fp, "0%xh", j);
	else if (j < 0xa0)
		kcnt += fprintf(fp, "%xh", j);
	else if (j < 0x100)
		kcnt += fprintf(fp, "0%xh", j);
	else if (j < 0xa00)
		kcnt += fprintf(fp, "%xh", j);
	else if (j < 0x1000)
		kcnt += fprintf(fp, "0%xh", j);
	else if (j < 0xa000)
		kcnt += fprintf(fp, "%xh", j);
	else
		kcnt += fprintf(fp, "0%xh", j);
}

/******************************************
 *														*
 *		Convert code to printable ascii		*
 *														*
 ******************************************/

int ascii(int i)
{
	i = i & 0x7f;
	if (i == 0x7f)
		return ('.');
	else if (i < 0x20)
		return ('.');
	else
		return (i);
}

/* End of d51p2.c */

