
/************************************************************************
 *																								*
 *		D51 8051/52 Disassembler - Copyright (C) 1995 by						*
 *		Jeffery L. Post																	*
 *		22726 Benner Ave.																	*
 *		Torrance, CA  90505																*
 *																								*
 *		D51.C - Main File																	*
 *																								*
 *  Version 2.0 - 10/15/95																*
 *																								*
 *	This program is free software; you can redistribute it and/or modify	*
 *	it under the terms of the GNU General Public License as published by	*
 *	the Free Software Foundation; either version 2 of the License, or		*
 *	(at your option) any later version.												*
 *																								*
 *	This program is distributed in the hope that it will be useful,		*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.									*
 *																								*
 *	You should have received a copy of the GNU General Public License		*
 *	along with this program; if not, write to the Free Software				*
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.				*
 *																								*
 ************************************************************************/

#include	<stdio.h>
#include	<dos.h>
#include	<ctype.h>
#include	<string.h>
#include	<alloc.h>
#include	"d51.h"
#include	"d51.tbl"			/* get global tables */

#define	EITHERFILE	0
#define	HEXFILE		1
#define	BINFILE		2

/*********************
 *							*
 *		Prototypes		*
 *							*
 *********************/

void	usage(void);
void	main(int argc, char *argv[]);
word	atox(char *str);
void	getcode(char *from, byte far *loc);
char	*find_entry(word val, struct sym *ptr);
void	add_entry(word val, char *symbol, int type);
struct sym	*get_smem(struct sym *ptr, int type);

#ifdef	DEBUG

void	dump_sym(void);		/* diagnostic routines */
void	dump_lab(void);		/* do not include in final program */

#endif

/***************************
 *									*
 *		Global variables		*
 *									*
 ***************************/

char	src[32], dst[32];		   /* file name buffers				*/
char	ctl[32];						/* control file name				*/
struct date	date;					/* disassembly date				*/
struct time	time;					/* disassembly time				*/
char	linebuffer[128];			/* input line buffer				*/
FILE	*fp;							/* dos file struct				*/
int	hexflag = 0;				/* append hex flag				*/
int	fileflag = 0;				/* file type flag					*/
int	kcnt;							/* output char counter			*/
word	pc;							/* current program counter		*/
word	himark;						/* highest data adrs				*/
word	offset;						/* program offset					*/
byte	far *pmem;					/* program data pointer			*/
byte	far *pflg;					/* pointer to program flags	*/
char	string[ASCLIMIT];			/* ascii data for defb			*/
word	asccount;					/* count for string data		*/
byte	byte_data[BYTELIMIT];	/* binary data for defb			*/
word	byte_cnt;					/* count for binary data		*/
word	word_data[WORDLIMIT];	/* binary data for defw			*/
word	word_cnt;					/* count for word data			*/
byte	dump;							/* dump just done flag			*/
byte	ascii_flag;					/* use ascii string flag		*/
struct sym	*sym_tab;			/* symbol table pointer			*/
struct sym	*lab_tab;			/* label table pointer			*/
char	defbstr[8];					/* string for defined bytes	*/
char	defwstr[8];					/* string for defined words	*/
char	ascistr[8];					/* string for defined ascii	*/
byte	dirregs[32];				/* access flags for dir reg	*/

/***************
 *					*
 *		Code		*
 *					*
 ***************/

void usage(void)
{
	printf("\nUsage: d51 <filename> [options]\n"
		"Options may be entered Unix style (-d) or DOS style (/b)\n"
		"\t-a use ascii macro instead of db/defb for text.\n"
		"\t-d include address and data in the comment field.\n"
		"\t-b force .bin extension on input file.\n"
		"\t-h force .hex extension on input file.\n"
		"\t   If neither 'b' nor 'h' is specified, D51 will first search\n"
		"\t   for a .hex file, and if not found, then a .bin file\n"
		"\t-s use 'defb' and 'defw' instead of 'db' "
		"and 'dw' for binary data.\n"
		"\t-x add a hexadecimal offset to file addresses.\n\n"
		"Options may be entered in a freeform fashion as long "
		"as a dash (-) or\n"
		"a slash (/) preceeds any option that preceeds the filename."
		"\nExamples:\n"
		"\td51 filename bd\n"
		"\td51 -d filename x100\n"
		"\td51 /h filename d -x100\n");
	exit(GOOD_EXIT);
}

/*********************************
 *											*
 *			The Main Program			*
 *											*
 *********************************/

void main(int argc, char *argv[])
{
	char	c;
	word	count;
	word	i, l;
	char	*inp;
	int	line;

	printf("\nD51 8051/8052 Disassembler V %d.%d"
			 "\nCopyright (C) 1995 by J. L. Post\n", verno, revno);
	if (argc < 2)
		usage();

/*******************************
	find filename in command line
 *******************************/

	for (line=1; line<argc; line++)	/* find first non option string */
	{
		inp = argv[line];
		c = *inp;
		if (c == '?')						/* '?' without preceeding '-' or '/' */
			usage();
		if (c == '-' || c == '/')
		{
			if (*(++inp) == '?')			/* '?' following '-' or '/' */
				usage();
		}
		else
		{										/* assume first found is file name */
			strcpy(src, argv[line]);	/* init filenames */
			strcat(src, ".hex");			/* assume search for .hex first */
			strcpy(dst, argv[line]);
			strcat(dst, ".d51");
			strcpy(ctl, argv[line]);
			strcat(ctl, ".ctl");
			break;
		}
	}
	strcpy(defbstr, "db");				/* init define byte and word strings */
	strcpy(defwstr, "dw");
	strcpy(ascistr, "db");				/* init define ascii string */

	sym_tab = NULL;						/* no symbols or labels yet */
	lab_tab = NULL;
	fileflag = EITHERFILE;				/* assume search for either file type */
	fp = NULL;
	offset = 0;								/* default start at address 0 */
	ascii_flag = 0;

/******************************
	process command line options
 ******************************/

	if (argc > 2)
	{
		for (count=1; count<argc; count++)
		{
			inp = argv[count];			/* to avoid modifying pointer in argv */
			while (c = toupper(*inp++))
			{
				if (c == '-' || c == '/')
					c = toupper(*inp++);		/* skip over option specifier */

				if (c == '?')
					usage();

				if (count == line)			/* skip if already identified */
					break;						/* as the file name */

				if (c == 'X')					/* add hex offset to program */
				{
					offset = atox(inp);
					break;
				}

				else if (c == 'D')			/* add data in comment field */
					hexflag = 1;

				else if (c == 'B')			/* binary instead of hex file */
				{
					fileflag = BINFILE;
					strcpy(src, argv[line]);
					strcat(src, ".bin");
				}

				else if (c == 'H')			/* force search for hex file */
				{
					fileflag = HEXFILE;
					strcpy(src, argv[line]);
					strcat(src, ".hex");
				}

				else if (c == 'S')			/* change db/dw strings */
				{
					strcpy(defbstr, "defb");
					strcpy(defwstr, "defw");
				}

				else if (c == 'A')			/* use ascii macro */
				{
					strcpy(ascistr, "ascii");
					ascii_flag = 1;
				}
			}
		}
	}

/******************
	open source file
 ******************/

	switch (fileflag)
	{
		case EITHERFILE:				/* if no type specified... */
			fp = fopen(src, "r");	/* search for hex file first */
			if (fp == NULL)			/* if not found, search for bin file */
			{
				fileflag = BINFILE;
				strcpy(src, argv[line]);
				strcat(src, ".bin");
				fp = fopen(src, "rb");
				if (fp == NULL)
				{
					printf("* Can't open either '%s.hex' nor '%s.bin' *\n",
						argv[line], argv[line]);
					exit(FILE_ERROR);
				}
				else
					fileflag = BINFILE;
			}
			break;

		case HEXFILE:					/* force hex file */
			fp = fopen(src, "r");
			break;

		case BINFILE:					/* force bin file */
			fp = fopen(src, "rb");
			break;
	}
	if (fp == NULL)					/* if file not found... */
	{
		printf("* Can't open file '%s' *\n", src);
		exit(FILE_ERROR);
	}

/*****************************************
	allocate memory for program and flags
 *****************************************/

	if ((pmem = (byte far *) farmalloc(pmemsize)) == NULL)
	{
		printf("INTERNAL ERROR! - Can't allocate program space!\n");
		exit(MEM_ERROR);
	}

	if ((pflg = (byte far *) farmalloc(pmemsize)) == NULL)
	{
		printf("INTERNAL ERROR! - Can't allocate flag space!\n");
		exit(MEM_ERROR);
	}

	printf("Initializing program spaces...");
	for (count=0xffff; count; --count)			/* fill code space with	*/
	{
		pmem[count] = NO_DATA;						/* invalidate data	*/
		pflg[count] = PF_INIT;						/* invalidate flags	*/
	}
	pmem[0] = NO_DATA;								/* include first location */
	pflg[0] = PF_INIT;

	for (count=0; count<32; count++)		/* no register references yet */
		dirregs[count] = 0;

/***************************************
	read input file and set up data array
 ***************************************/

	himark = 0;
	line = 0;
	pc = offset;
	printf("\nreading %s\n", src);

	if (fileflag == BINFILE)				/* if binary file... */
	{
		while (!feof(fp))							/* until end of file... */
		{
			i = fread(linebuffer, 1, 128, fp);	/* read a block of data */
			for (l=0; l<i; l++)
			{
				pmem[pc] = linebuffer[l];		/* copy to program space */
				pflg[pc] = PF_DATA;
				pc++;
				if ((pc & 0xff) == 0)
					printf("\r%04x", pc);		/* show progress */
			}
			himark = pc;							/* flag highest location */
		}
	}

	else											/* else hex file... */
	{
		while (!feof(fp))							/* until end of file... */
		{
			*linebuffer = '\0';					/* clear previous line */
			fgets(linebuffer, 127, fp);		/* read one line */
			line++;
			inp = linebuffer;						/* local copy of pointer */
			if (sscanf(inp, "%*c%2x%4x", &i, &pc) != EOF)
														/* get count and address */
			{
				pc += offset;						/* add offset to address */
				if (i == 0)
					break;							/* done if end of hex record */
				if (i > 64)							/* oops! line too long */
				{
					printf("invalid count (%d) in line %d:\n", i, line);
					printf("%s", linebuffer);
					exit(FILE_ERROR);
				}
				for (l=0; l<i ; l++)				/* now move data to program area */
				{
					getcode(inp+9+l*2, &pmem[pc]);	/* code to program space */
					pflg[pc] = PF_DATA;					/* flag valid data */
					pc++;
					if ((pc & 0xff) == 0)				/* show progress */
						printf("\r%04x", pc);
				}
			}
			himark = (himark > pc) ? himark : pc;	/* update last address */
		}
	}
	close(fp);										/* done reading input file */
	if (!himark)									/* if wrapped around to 0... */
		himark = 0xffff;							/* is 64K program (binary?) */
	printf("\rHighest location = %04x\n", himark);	/* show last location */

/********************************************************
	Got the program in data array, now let's go to work...
 ********************************************************/

	pass0();								/* read control file					*/
	pass1();								/* find internal references		*/
	pass2();								/* disassemble to output file		*/
	pass3();								/* generate equ's for references	*/
	printf("\nDone");					/* go bye-bye							*/
	exit(GOOD_EXIT);
}
				/*****  End of Main  *****/

/***************************************************
 *																	*
 *	Convert ascii hex to hexadecimal for X option	*
 *																	*
 ***************************************************/

word atox(char *str)
{
	char	c;
	word	i;

	i = 0;
	while (c = toupper(*str++))
	{
		if (isxdigit(c))
		{
			c = (c > '9') ? c - 0x37 : c & 0xf;
			i = (i << 4) | c;
		}
		else
			break;
	}
	return(i);
}

/*********************************************
 *															*
 *	Put ascii hex data into binary far array	*
 *															*
 *********************************************/

void getcode(char *from, byte far *loc)
{
	byte	c, i;

	c = *from++ - 0x30;
	c = (c > 10) ? c - 7 : c;
	i = c << 4;
	c = *from++ - 0x30;
	c = (c > 10) ? c - 7 : c;
	*loc = i | c;
}

/******************************
 *										*
 *		Find symbol or label		*
 *										*
 ******************************/

char *find_entry(word val, struct sym *ptr)
{
	while (ptr != NULL)				/* not at the end yet */
	{
		if (val == ptr->adrs)		/* if we found it... */
			return(ptr->label);		/* return text pointer */
		ptr = ptr->next;				/* else get the next entry */
	}
	return(NULL);						/* symbol or label not found */
}

/************************************************************
 *																				*
 *		Allocate new entry in symbol table or label table		*
 *																				*
 ************************************************************/

struct sym *get_smem(struct sym *prev, int type)
{
	struct sym	*ptr;

	ptr = malloc(sizeof(struct sym));		/* get memory from OS */
	if (ptr == NULL)								/* what? out of memory?... */
	{
		printf("\nINTERNAL ERROR! - No memory for ");
		if (type)
			printf("label");
		else
			printf("symbol");
		printf(" table!\n");
		exit(MEM_ERROR);
	}
	ptr->adrs = 0;						/* initialize the new entry */
	ptr->prev = prev;
	ptr->next = NULL;
	return(ptr);						/* then give caller the address */
}

/***************************************
 *													*
 *		Add symbol or label to table		*
 *  Symbol if type == 0, else label.	*
 *													*
 ***************************************/

void add_entry(word val, char *symbol, int type)
{
	struct sym	*sptr, *nptr, *tbl_ptr;
	char			*cptr;
	int			i;
	char			tbl_name[8];

	cptr = symbol;				/* ensure that input string is null terminated */
	for (i=0; i<6; i++, cptr++)
	{
		if (!isalnum(*cptr))
			break;
	}
	*cptr = '\0';

	if (type)
	{
		tbl_ptr = lab_tab;				/* get pointer to correct table */
		strcpy(tbl_name, "label");
	}
	else
	{
		tbl_ptr = sym_tab;
		strcpy(tbl_name, "symbol");
	}

	if (tbl_ptr == NULL)					/* if first symbol or label... */
	{
		sptr = get_smem(NULL, type);	/* allocate first entry */
		tbl_ptr = sptr;
		if (type)							/* initialize head pointer */
			lab_tab = sptr;
		else
			sym_tab = sptr;
		sptr->prev = NULL;				/* first one, so no previous pointer */
	}
	else
	{
		if (cptr = find_entry(val, tbl_ptr))	/* check for duplicate value */
		{
			if (strcmpi(cptr, symbol))				/* if not same name */
			{
				printf("\nAttempted redefinition of value 0x%x"
						 ", '%s', as '%s' in %s table\n",
					val, cptr, symbol, tbl_name);
				exit(USER_ERROR);
			}
		}
		sptr = tbl_ptr;								/* if not found... */
		while (1)										/* check for duplicate name */
		{
			if (!strcmpi(sptr->label, symbol))	/* if duplicate name */
			{
				if (val != sptr->adrs)				/* and not same address */
				{
					printf("\nAttempted redefinition of '%s', value 0x%x"
							 ", as value 0x%x in %s table\n",
						symbol, sptr->adrs, val, tbl_name);
					exit(USER_ERROR);
				}
			}
			nptr = sptr->next;			/* search for end of table */
			if (nptr == NULL)				/* when end of table found... */
			{
				nptr = get_smem(sptr, type);	/* allocate space for new entry */
				sptr->next = nptr;			/* update links */
				nptr->prev = sptr;
				sptr = nptr;
				break;
			}
			else								/* else get next entry */
				sptr = nptr;
		}
	}
	sptr->adrs = val;						/* set value in new entry */
	strcpy(sptr->label, symbol);		/* and copy text */
	sptr->next = NULL;					/* last entry, so no next pointer */
}

/******************************
 *										*
 *		Diagnostic Routines		*
 *										*
 ******************************/

#ifdef	DEBUG

/*	Diagnostic routine - output symbol table */

void dump_sym(void)
{
	struct sym	*ptr;

	printf("\nSymbol table: (sym_tab = %04x)", sym_tab);
	ptr = sym_tab;
	while (ptr)
	{
		printf("\naddress = 0x%04x, label = %s",
			ptr->adrs, ptr->label);
		printf("\nprev = %04x, next = %04x",
			ptr->prev, ptr->next);
		ptr = ptr->next;
	}
	printf("\nend of symbols\n");
}

/*	Diagnostic routine - output label table */

void dump_lab(void)
{
	struct sym	*ptr;

	printf("\nLabel table: (lab_tab = %04x)", lab_tab);
	ptr = lab_tab;
	while (ptr)
	{
		printf("\naddress = 0x%04x, label = %s",
			ptr->adrs, ptr->label);
		printf("\nprev = %04x, next = %04x",
			ptr->prev, ptr->next);
		ptr = ptr->next;
	}
	printf("\nend of labels\n");
}

#endif

/* end of d51.c */

