/*
  This is a combination of a stripped down verison of DBWIN and DBMON.

  AST 2/26/96
  
    3/4/96      1.0 - first release
    3/21/96     1.1 - added read/write state, fixed \r\n handling, handled WM_ENDSESSION better
    5/96        1.2 beta - added Win95 support, fixed PID display, added options
                to toggle window read-only and PID display
*/

#define STRICT
#include <windows.h>            // required for all Windows applications
#include <windowsx.h>
#include <process.h>
#include <limits.h>
#include <assert.h>

#include "dbwin32.h"
#include "resource.h"


// Globals
HINSTANCE   hinstDBWin = NULL;
char szDBWinClass[] = "DBWin32";
HWND        hwndDBWin = NULL;
HWND        hwndClient = NULL;
HFONT       hfontClient = NULL;
LOGFONT    logfontClient;
HACCEL      haccelDBWin = NULL;
HANDLE      hEventThread;           // handle to our worker thread
BOOL        fTopmost = FALSE;       // Stay on top or not
BOOL        fOutput = TRUE;         // Log output to window or not
BOOL        fReadOnly = TRUE;       // Output window is read-only
BOOL        fDisplayPID = TRUE;     // Display PID in output strings



/*
    main entry point
*/
int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;

    if ( !DBWin32Init(hInstance, hPrevInstance, lpCmdLine, nCmdShow) )
        return FALSE;
    

    // Acquire and dispatch messages until a WM_QUIT message is received.
    while (GetMessage(&msg, NULL, 0, 0))
    {
        if (!TranslateAccelerator(hwndDBWin, haccelDBWin, &msg))
        {
          TranslateMessage(&msg);
          DispatchMessage(&msg); 
        }
    }

    DBWin32Terminate(FALSE);

    return msg.wParam;
}


/*
    Initialize and create resources
*/
BOOL DBWin32Init(HINSTANCE hinst, HINSTANCE hinstPrev, LPSTR szCmdLine, int showCmd)
{
    WNDCLASS cls;

    // hinstPrev is always NULL in Win32
    // Find the first instance main window, and post a message
    // to it to tell it to activate itself.
    //
    HWND hwnd = FindWindow(szDBWinClass, NULL);
    if (hwnd)
    {
        PostMessage(hwnd, WM_ACTIVATEFIRST, 0, 0L);
        return FALSE;
    }

    // store our instance handle in a global
    hinstDBWin = hinst;

    cls.hCursor        = LoadCursor(NULL, IDC_ARROW);
    cls.hIcon          = LoadIcon(hinst, MAKEINTRESOURCE(IDI_DBWIN32));
    cls.lpszMenuName   = MAKEINTRESOURCE(IDM_MAINMENU);
    cls.lpszClassName  = szDBWinClass;
    cls.hbrBackground  = (HBRUSH)(COLOR_WINDOW + 1);
    cls.hInstance      = hinstDBWin;
    cls.style          = CS_BYTEALIGNCLIENT;
    cls.lpfnWndProc    = DBWin32WndProc;
    cls.cbWndExtra     = 0;
    cls.cbClsExtra     = 0;

    if (!RegisterClass(&cls))
        return FALSE;

    haccelDBWin = LoadAccelerators(hinstDBWin, MAKEINTRESOURCE(IDR_FASTKEYS));
    if (!haccelDBWin)
        return FALSE;

    // Create the main window

    hwndDBWin = CreateWindow(szDBWinClass,           // Class name
                            "Debug Messages (WIN32)",// Caption
                            WS_OVERLAPPEDWINDOW,    // Style bits
                            0, 0, 0, 0,
                            (HWND)NULL,             // Parent window (no parent)
                            (HMENU)NULL,            // use class menu
                            (HINSTANCE)hinstDBWin,  // handle to window instance
                            (LPSTR)NULL             // no params to pass on
                           );
    if (!hwndDBWin)
        return FALSE;

    hwndClient = CreateWindow("EDIT", NULL,
            WS_CHILD | WS_BORDER | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL |
            ES_AUTOHSCROLL | ES_AUTOVSCROLL | ES_MULTILINE | ES_READONLY, 
            0, 0, 0, 0,
            hwndDBWin, (HMENU)0, hinstDBWin, NULL);

    if (!hwndClient)
        return FALSE;

    // Use the small icon title font
    SystemParametersInfo(SPI_GETICONTITLELOGFONT, sizeof(LOGFONT), &logfontClient, FALSE);
    hfontClient = CreateFontIndirect(&logfontClient);
    if (hfontClient)
        SendMessage(hwndClient, WM_SETFONT, (WPARAM)(HFONT)hfontClient, 1);

    // set the text limit to the maximum
    SendMessage(hwndClient, EM_LIMITTEXT, 0, 0);

    ShowWindow(hwndClient, SW_SHOWNORMAL);

    /* restore the saved state if there is one */
    ReadDBWinState(hwndDBWin);

    /* launch the worker thread that sends us the debug text */
    hEventThread = (HANDLE)_beginthread(EventThreadRoutine, 0, NULL);
    if ( hEventThread == (HANDLE)-1 )
        return FALSE;


    return TRUE;
}

/*

    Write out the state, destroy our resources, and close the thread    

*/
void DBWin32Terminate(BOOL fEndSession)
{

   /* write the state ofo the application */
   WriteDBWinState(hwndDBWin);

   /* terminate the worker thread */
   CloseHandle(hEventThread);

   if ( !fEndSession )
   {
       /* destroy the main window */
       if (hwndDBWin)
       {
          DestroyWindow(hwndDBWin);
          hwndDBWin = NULL;
       }

       /* destroy the font object */
       if (hfontClient)
       {
          DeleteFont(hfontClient);
          hfontClient = NULL;
       }
   }

}


/*
    Main Window Function
*/
LRESULT CALLBACK DBWin32WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{

    LPSTR lpszDebugText;

    switch (msg)
    {
    case WM_COMMAND:
        HANDLE_WM_COMMAND(hwnd, wParam, lParam, OnCommand);
        return 0L;

    case WM_ENDSESSION:
        DBWin32Terminate(TRUE);
        break;

    case WM_CLOSE:
        PostQuitMessage(0);
        break;

    case WM_ACTIVATE:
        if (wParam)
            SetFocus(hwndClient);
        break;

    case WM_SIZE:
        /* Make sure the edit control always occupies the entire
         * client area.
         */
        if (hwndClient)
        {
            RECT rc;

            GetClientRect(hwnd, &rc);
            //
            // Outset border
            //
            InflateRect(&rc, 1, 1);
            MoveWindow(hwndClient, rc.top, rc.left, rc.right - rc.left, rc.bottom - rc.top, TRUE);
        }
        break;

    case WM_INITMENU:
       break;

    case WM_ACTIVATEFIRST:
        // The user tried to run a second instance of this app.
        // The second instance posted this message to us and exited.
        //
        // If we're iconic, restore ourself, otherwise just make
        // ourself the active window.
        //
        if (IsIconic(hwnd))
            ShowWindow(hwnd, SW_RESTORE);
        else
            SetActiveWindow(hwnd);
        break;


    case WM_DEBUGTEXT:

        //
        // This is the message from our worker thread which has the
        // text we are supposed to append in wParam.  It is our 
        // responsibility to free it.

        lpszDebugText = (LPSTR)wParam;

        // append the text to the end
        Edit_SetSel(hwndClient, INT_MAX, INT_MAX);
        Edit_ReplaceSel(hwndClient, lpszDebugText);
        Edit_SetSel(hwndClient, INT_MAX, INT_MAX);
    
        // it's our responsibility to delete the memory
        free(lpszDebugText);
        break;


    default:
        return DefWindowProc(hwnd, msg, wParam, lParam);
    }

    return 0L;
}


/*
    Handle WM_COMMAND messages
*/
void OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify) 
{

    switch( id )
    {

        case ID_FILE_SAVEBUFFERAS:
        {
            BOOL bOK = SaveBuffer(hwndClient);
            break;
        }

        case ID_FILE_EXIT:
            PostQuitMessage(0);
            break;

        case ID_EDIT_SELECTALL:
            Edit_SetSel(hwndClient, 0, INT_MAX);
            break;

        case ID_EDIT_CLEARBUFFER:
            Edit_SetSel(hwndClient, 0, INT_MAX);
            Edit_ReplaceSel(hwndClient, "");
            break;

        case ID_EDIT_COPY_SELECTION:
            PostMessage(hwndClient, WM_COPY, 0, 0L);
            break;

        case ID_OPTIONS_NOOUTPUT:
            fOutput = FALSE;
            SetOutputMenuItems();
            break;

        case ID_OPTIONS_OUTPUTTOWINDOW:
            fOutput = TRUE;
            SetOutputMenuItems();
            break;

        case ID_OPTIONS_ALWAYSONTOP:
            fTopmost = !fTopmost;
            SetOnTopMenuItem();
            break;

        case ID_OPTIONS_DISPLAYPID:
            fDisplayPID = !fDisplayPID;
            SetDisplayPIDMenuItem();
            break;

        case ID_OPTIONS_READONLY:
            fReadOnly = !fReadOnly;
            SetReadOnlyMenuItem();
            break;

        case ID_HELP_ABOUT:
            MessageBox(hwnd, "DBWin32 v" DBWIN32_VERSION " - A Debugging Aid for Windows NT/95\n\nWritten by Andrew Tucker (ast@halcyon.com), 1996", "About DBWin32", MB_OK);
            break;

    }

     return;
}



/*
    The main gist of this routine comes from the MSDN DBMON example.
*/
void EventThreadRoutine(void *pvParam)
{
    HANDLE AckEvent;
    HANDLE ReadyEvent;
    HANDLE SharedFile;
    LPVOID SharedMem;
    LPSTR  String;
    LPSTR  lpszDebugText;
    DWORD  ret;
    DWORD  LastPid;
    LPDWORD pThisPid;
    BOOL   fDidCR = TRUE;


    AckEvent = CreateEvent(NULL, FALSE, FALSE, "DBWIN_BUFFER_READY");
    if (!AckEvent) 
    {
        MessageBox(NULL, "DBWIN32: Unable to create synchronization object DBWIN_BUFFER_READY", "Error", MB_OK);
        return;
    }

    if (GetLastError() == ERROR_ALREADY_EXISTS) 
    {
        MessageBox(NULL, "DBWIN32: already running", "Error", MB_OK);
        return;
    }

    ReadyEvent = CreateEvent(NULL, FALSE, FALSE, "DBWIN_DATA_READY");
    if (!ReadyEvent) 
    {
        MessageBox(NULL, "DBWIN32: Unable to create synchronization object DBWIN_DATA_READY", "Error", MB_OK);
        return;
    }

    SharedFile = CreateFileMapping((HANDLE)-1, NULL, PAGE_READWRITE, 0, 4096, "DBWIN_BUFFER");
    if (!SharedFile) 
    {
        MessageBox(NULL, "DBWIN32: Unable to create file mapping object DBWIN_BUFFER", "Error", MB_OK);
        return;
    }

    SharedMem = MapViewOfFile(SharedFile, FILE_MAP_READ, 0, 0, 512);
    if (!SharedMem) 
    {
        MessageBox(NULL, "DBWIN32: Unable to map shared memory", "Error", MB_OK);
        return;
    }

    String = (LPSTR)SharedMem + sizeof(DWORD);
    pThisPid = SharedMem;
    LastPid = 0xffffffff;
    SetEvent(AckEvent);

    for (;;) {

        ret = WaitForSingleObject(ReadyEvent, INFINITE);

        if (ret != WAIT_OBJECT_0) 
        {

            MessageBox(NULL, "DBWIN32: wait failed", "Error", MB_OK);
            return;
        } 
        else 
        {
            // only output if it is requested
            if ( fOutput )
            {

                LPSTR lpszScratch;
                BOOL fWantNewLine = FALSE;

                if (LastPid != *pThisPid) 
                {
                    LastPid = *pThisPid;
                    if ( !fDidCR )
                    {
                        fWantNewLine = TRUE;
                        fDidCR = TRUE;
                    }
                }

                // fix up the newlines
                lpszScratch = FixNewLines(String);

                // allocate display buffer
                lpszDebugText = malloc(strlen(lpszScratch) + 20);
                if ( !lpszDebugText )
                {
                    MessageBox(NULL, "DBWIN32: malloc failed", "Error", MB_OK);
                    continue;
                }

                // make it empty or a newline    
                if ( fWantNewLine )
                    strcpy(lpszDebugText, "\r\n");
                else
                    *lpszDebugText = 0;

                // prepend the PID necessary
                if ( fDisplayPID && fDidCR )
                {
                    // avoid overwriting the \r\n if it exists
                    int iLen = strlen(lpszDebugText);
                    wsprintf(&lpszDebugText[iLen], "%3lu: ", LastPid);
                }

                // build display buffer
                strcat(lpszDebugText, lpszScratch);

                // record newline status
                fDidCR = (*lpszScratch && (lpszScratch[strlen(lpszScratch) - 1] == '\n'));

                // cleanup from FixNewLines
                free(lpszScratch);

                // send the display string to the window
                PostMessage(hwndDBWin, WM_DEBUGTEXT, (WPARAM)lpszDebugText, 0);
            }

            // ready for new event
            SetEvent(AckEvent);
        }
    }

    return;
}

/*
    Make sure that any CRLF pairs are \r\n and not \n\r
    and that any solo \n chars are \r\n pairs so they display 
    correctly in the EDIT box

    This routine returns a pointer to a buffer which
    the caller must free or NULL on error
*/
LPSTR FixNewLines( LPCSTR lpcszText )
{
    LPSTR lpszScratch;
    LPSTR lpszNewLine;

    if ( !lpcszText )
    {
        assert( FALSE );
        return NULL;
    }

    /*
        allocate a buffer twice as big in case we need to 
        add any \r's to solo \n's 
    */
    lpszScratch = malloc( (strlen(lpcszText) * 2) + 1 );
    if ( !lpszScratch )
        return NULL;

    /* make a copy of the passed text */
    strcpy(lpszScratch, lpcszText);

    /* replace \n\r with \r\n */
    lpszNewLine = strstr(lpszScratch, "\n\r");
    while ( lpszNewLine )
    {
        *lpszNewLine = '\r';
        *(lpszNewLine + 1) = '\n';

        // skip our changes    
        lpszNewLine += 2;
        
        lpszNewLine = strstr(lpszNewLine, "\n\r");
    }

    /* replace solo \n's with \r\n */
    lpszNewLine = lpszScratch;
    while ( *lpszNewLine )
    {
        if ( *lpszNewLine == '\n' )
        {
            /* handle the case of \n being the 1st char */
            if ( (lpszNewLine == lpszScratch) )
            {
                MoveMemory(lpszScratch + 1, lpszScratch, strlen(lpszScratch) + 1);
                *lpszScratch = '\r';
                lpszNewLine++;
            }
            else if ( *(lpszNewLine - 1) != '\r' )
            {
                MoveMemory(lpszNewLine + 1, lpszNewLine, strlen(lpszNewLine) + 1);
                *lpszNewLine = '\r';
                lpszNewLine++;
            }
        }

        lpszNewLine++;
    }

    return lpszScratch;
}    



/*
  Save all the text in the buffer to a file.

  Stolen from DBWIN.C
*/

BOOL SaveBuffer(HWND hwnd)
{
    char szFilename[256];
    OPENFILENAME ofn;           /* passed to the File Open/save APIs */
    BOOL fSuccess = FALSE;

    szFilename[0] = 0;
    ofn.lStructSize = sizeof(OPENFILENAME);
    ofn.hwndOwner = hwnd;
    ofn.hInstance = hinstDBWin;
    ofn.lpstrFilter = (LPSTR)"All\0*.*\0";
    ofn.lpstrCustomFilter = NULL;
    ofn.nMaxCustFilter = 0;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = (LPSTR)szFilename;
    ofn.nMaxFile = sizeof(szFilename);
    ofn.lpstrFileTitle = NULL;
    ofn.nMaxFileTitle = 0;
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrTitle = NULL;
    ofn.Flags = OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY | OFN_PATHMUSTEXIST;
    ofn.nFileOffset = 0;
    ofn.nFileExtension = 0;
    ofn.lpstrDefExt = (LPSTR)"";
    ofn.lCustData = 0L;
    ofn.lpfnHook = NULL;
    ofn.lpTemplateName = NULL;

    if (GetSaveFileName(&ofn))
    {
        OFSTRUCT of;
        int cch;
        char* pch;
        HFILE hfile;
        HLOCAL h;

        hfile = OpenFile(szFilename, &of, OF_CREATE | OF_WRITE);

        if (hfile != HFILE_ERROR)
        {
            h = Edit_GetHandle(hwndClient);
            cch = Edit_GetTextLength(hwndClient);
            pch = LocalLock(h);
            if ((int)_lwrite(hfile, pch, cch) == cch)
                fSuccess = TRUE;

            LocalUnlock(h);
            _lclose(hfile);
        }
        return fSuccess;
    }
}


/*
    Convert the passed string to an integer up to the first non-digit or -
*/
int IntFromString(LPSTR FAR* lplpsz)
{
    LPSTR lpsz = *lplpsz;
    int i = 0;
    char ch;
    BOOL fNeg;

    while (*lpsz == ' ')
        lpsz++;

    fNeg = FALSE;
    while (ch = *lpsz++)
    {
        if (ch == '-')
        {
            fNeg = !fNeg;
            continue;
        }

        if (ch < '0' || ch > '9')
            break;
        i = (i * 10) + (ch - '0');
    }
    *lplpsz = lpsz;

    return (fNeg ? -i : i);
}


/*
    Restore the saved state of the application
*/
void ReadDBWinState(HWND hwndMain)
{
    WINDOWPLACEMENT wpl;
    char ach[128];
    LPSTR lpsz;
    int cch;

    lpsz = ach;
    cch = GetProfileString("DBWin32", "State", "", ach, sizeof(ach));

    if (cch == 0 || IntFromString(&lpsz) != 14)
    {
        int cxScreen, cyScreen;
        int x, y, cx, cy;

        // default position window along right edge of screen
        //
        cxScreen = GetSystemMetrics(SM_CXSCREEN);
        cyScreen = GetSystemMetrics(SM_CYSCREEN);

        cx = cxScreen / 2;
        x = cxScreen - cx;
        cy = cyScreen - GetSystemMetrics(SM_CYICONSPACING);
        y = 0;

        wpl.length = sizeof(wpl);
        GetWindowPlacement(hwndMain, &wpl);
        wpl.rcNormalPosition.left = x;
        wpl.rcNormalPosition.top = y;
        wpl.rcNormalPosition.right = x + cx;
        wpl.rcNormalPosition.bottom = y + cy;

        fOutput = TRUE;
        SetOutputMenuItems();

        fTopmost = FALSE;
        SetOnTopMenuItem();

        fReadOnly = TRUE;
        SetReadOnlyMenuItem();

        fDisplayPID = TRUE;
        SetDisplayPIDMenuItem();
        
        SetWindowPlacement(hwndMain, &wpl);
        ShowWindow(hwndMain, SW_SHOWNORMAL);
    }
    else
    {
        wpl.length = sizeof(wpl);
        wpl.flags = (UINT)IntFromString(&lpsz);
        wpl.showCmd = (UINT)IntFromString(&lpsz);
        wpl.ptMinPosition.x = IntFromString(&lpsz);
        wpl.ptMinPosition.y = IntFromString(&lpsz);
        wpl.ptMaxPosition.x = IntFromString(&lpsz);
        wpl.ptMaxPosition.y = IntFromString(&lpsz);
        wpl.rcNormalPosition.left = IntFromString(&lpsz);
        wpl.rcNormalPosition.top = IntFromString(&lpsz);
        wpl.rcNormalPosition.right = IntFromString(&lpsz);
        wpl.rcNormalPosition.bottom = IntFromString(&lpsz);

        fOutput = (BOOL)IntFromString(&lpsz);
        SetOutputMenuItems();

        fTopmost = (BOOL)IntFromString(&lpsz);
        SetOnTopMenuItem();

        fReadOnly = (BOOL)IntFromString(&lpsz);
        SetReadOnlyMenuItem();

        fDisplayPID = (BOOL)IntFromString(&lpsz);
        SetDisplayPIDMenuItem();

        SetWindowPlacement(hwndMain, &wpl);
    }
}

/*
    Save the State of the application so we can restore on startup
*/
void WriteDBWinState(HWND hwndMain)
{
    WINDOWPLACEMENT wpl;

    char ach[128];

    wpl.length = sizeof(wpl);
    GetWindowPlacement(hwndMain, &wpl);

    wsprintf(ach, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
        14,
        wpl.flags,
        wpl.showCmd,
        wpl.ptMinPosition.x,
        wpl.ptMinPosition.y,
        wpl.ptMaxPosition.x,
        wpl.ptMaxPosition.y,
        wpl.rcNormalPosition.left,
        wpl.rcNormalPosition.top,
        wpl.rcNormalPosition.right,
        wpl.rcNormalPosition.bottom,
        fOutput,
        fTopmost,
        fReadOnly,
        fDisplayPID);

    WriteProfileString("DBWin32", "State", ach);
}


/*
    Update the state of the Output menu item
*/
void SetOutputMenuItems( void )
{
    if ( !fOutput )
    {        
        CheckMenuItem(GetMenu(hwndDBWin), ID_OPTIONS_NOOUTPUT, MF_CHECKED);
        CheckMenuItem(GetMenu(hwndDBWin), ID_OPTIONS_OUTPUTTOWINDOW, MF_UNCHECKED);
    }
    else
    {
        CheckMenuItem(GetMenu(hwndDBWin), ID_OPTIONS_NOOUTPUT, MF_UNCHECKED);
        CheckMenuItem(GetMenu(hwndDBWin), ID_OPTIONS_OUTPUTTOWINDOW, MF_CHECKED);
    }
}


/*
    Update the "Always On Top" menu item and window property
*/
void SetOnTopMenuItem( void )
{
    SetWindowPos(hwndDBWin, (fTopmost ? HWND_TOPMOST : HWND_NOTOPMOST), 0, 0, 0, 0, SWP_NOACTIVATE | SWP_NOSIZE | SWP_NOMOVE);
    CheckMenuItem(GetMenu(hwndDBWin), ID_OPTIONS_ALWAYSONTOP, (fTopmost) ? MF_CHECKED : MF_UNCHECKED);
}

/*
    Update the "Read Only" menu item and window property
*/
void SetReadOnlyMenuItem( void )
{
    SendMessage(hwndClient, EM_SETREADONLY, fReadOnly, 0);
    CheckMenuItem(GetMenu(hwndDBWin), ID_OPTIONS_READONLY, (fReadOnly) ? MF_CHECKED : MF_UNCHECKED);
}

/*
    Update the "Display PID" menu item and window property
*/
void SetDisplayPIDMenuItem( void )
{
    CheckMenuItem(GetMenu(hwndDBWin), ID_OPTIONS_DISPLAYPID, (fDisplayPID) ? MF_CHECKED : MF_UNCHECKED);
}

