// getcmt.cpp - Get C/C++ comments
// Greg Messer

// Demonstration of the use of the
// Chronograph class.

/*
 --------------------------------------------
 Filter to extract comments from
 C and C++ source files.

 This program reads source code from cin,
 and writes comments to cout.

 The output comments are indented the same as
 they were in the source file.

 Use the following command to build with
 Borland C++:

      BCC getcmt.cpp chrono.cpp
 --------------------------------------------
 */

#include <iostream.h>
#include <string.h>
#include <ctype.h>

// include the Chronograph class header

#include "chrono.h"

// long input lines are allowed
#define MAXLINE 1024

void getcmt(istream &in);
void usage(void);

// Global program name, for messages.

char *pgm = "GETCMT";

// ------------------------------------------
main(int argc, char *argv[])
{
   int    i;
   double elapsed_time;

   // One Chronograph to time the entire
   // program.
   // A Chronograph starts running as soon as
   // it is instantiated.

   Chronograph chrono;

   // ---------------------------------------

   for(i = 1; i < argc; i++)
   {
      if(argv[i][0] == '/' ||
         argv[i][0] == '-')
      {
         // an option switch is on the
         // commandline

         switch(argv[i][1])
         {
            case '?':
            case 'h':
            case 'H':
               usage();
               return 1;
            default:
               break;
         }
      }
   }

   getcmt(cin);

   elapsed_time = chrono.elapsed();

   // Show the total time this program was
   // running.
   cout << endl;
   cout << pgm << " main()";
   cout << ": Total elapsed time: ";
   cout.precision(2);
   cout <<  elapsed_time;
   cout << " seconds." << endl;
   cout << endl;

   return 0;
}

// ------------------------------------------
void getcmt(istream &in)
{
   int  i, j, len;
   long linecount = 0l;

   // Variables in which to store the
   // elapsed hours, minutes and seconds.

   double hours, minutes, seconds;

   // Flag to indicate if we are in
   // a comment and what type it is.
   // 0 = not in a comment
   // 1 = in a C comment
   // 2 = in a C++ comment

   int incomment = 0;

   // Flag to indicate if there was
   // a comment on the current line.
   // If so, we output the comment.
   // 0 = no comment on this line
   // 1 = comment on this line

   int commentline = 0;

   // I/O buffer

   char inbuff[MAXLINE + 1];

   double elapsed_time;
   double elapsed_time_inside;
   double elapsed_time_outside;

   // One Chronograph to count the time
   // spent in this function.

   Chronograph chrono;

   // Two Chronographs, to count the time
   // spent inside and outside of comments.

   Chronograph inside;
   Chronograph outside;

   // ---------------------------------------

   // reset the "inside comment" timer to
   // zero and stop it

   inside.reset();

   cout << endl;

   in.getline(inbuff, MAXLINE);

   while(in.eof() == 0)
   {
      linecount++;

      if(linecount % 100 == 0)
      {
         cout << pgm;
         cout << " getcmt()";
         cout << ": 100 lines in ";
         cout.precision(2);
         cout << chrono.lap() << " seconds.";
         cout << endl;
      }

      len = strlen(inbuff);
      i = 0;

      // while chars are left on the line...
      while(i < len)
      {
         // process depending on if we are
         // in a comment and what type it is
         switch(incomment)
         {
            case 0:
               // not inside a comment
               if(inbuff[i] == '/' &&
                  inbuff[i+1] == '*')
               {
                  outside.stop();
                  inside.start();

                  incomment = 1;
                  commentline = 1;

                  i += 2;
               }
               else if(inbuff[i] == '/' &&
                       inbuff[i+1] == '/')
               {
                  outside.stop();
                  inside.start();

                  incomment = 2;
                  commentline = 1;

                  i += 2;
               }
               else
               {
                  if(!isspace(inbuff[i]))
                      inbuff[i] = ' ';

                  i++;
               }

               break;

            case 1:
               // inside a C comment
               commentline = 1;
               if(inbuff[i] == '/' &&
                  inbuff[i+1] == '/')
               {
                  // "//" is allowed inside
                  // a C comment

                  i += 2;
               }
               else if(inbuff[i] == '/' &&
                       inbuff[i+1] == '*')
               {
                  i += 2;
               }
               else if(inbuff[i] == '*' &&
                       inbuff[i+1] == '/')
               {
                  inside.stop();
                  outside.start();

                  incomment = 0;

                  i += 2;
               }
               else
                  i++;

               break;

            case 2:
               // inside a C++ comment
               commentline = 1;
               if(inbuff[i] == '/' &&
                  inbuff[i+1] == '/')
               {
                  // "//" is allowed inside
                  // a C++ comment

                  i += 2;
               }
               else if(i + 1 == len)
               {
                  inside.stop();
                  outside.start();

                  incomment = 0;

                  i++;
               }
               else
                  i++;

               break;

            default:
               // error, bad "incomment"
               // shouldn't happen
               cerr << "Error!" << endl;

            return;

         }
         // end of switch()
      }
      // end of while()

      if(commentline == 1)
         cout << inbuff << endl;

      if(incomment == 1)
         commentline = 1;
      else
         commentline = 0;

      in.getline(inbuff, MAXLINE);
   }

   // stop all three of the timers
   // and get the elapsed times
   elapsed_time         = chrono.stop();
   elapsed_time_inside  = inside.stop();
   elapsed_time_outside = outside.stop();

   // show the time spent outside comments
   cout << endl;
   cout << pgm << " getcmt(): ";
   cout.precision(2);
   cout << elapsed_time_outside;
   cout << " seconds outside of comments.";
   cout << endl;

   // show the time spent inside comments
   cout << pgm << " getcmt(): ";
   cout.precision(2);
   cout << elapsed_time_inside;
   cout << " seconds inside of comments.";
   cout << endl;

   // show the total time spent
   cout << pgm << " getcmt(): ";
   cout.precision(2);
   cout << elapsed_time;
   cout << " seconds total.";
   cout << endl;

   // get the total time spent broken out
   // into hours, minutes, and seconds
   chrono.elapsedHMS(hours,
                     minutes,
                     seconds);

   // show the total hours, minutes, seconds
   cout << pgm << " getcmt(): ";
   cout.precision(0);
   cout << hours << " hours, ";
   cout.precision(0);
   cout << minutes << " minutes, ";
   cout.precision(2);
   cout << seconds << " seconds.";
   cout << endl;

   // show the total lines processed
   cout << pgm << " getcmt(): ";
   cout << linecount << " lines.";
   cout << endl;

   // show the lines per second processed
   cout << pgm << " getcmt(): ";
   cout.precision(2);
   cout << linecount / chrono.elapsed();
   cout << " lines per second.";
   cout << endl;

   return;
}

// ------------------------------------------
void usage()
{
   // show a usage message
   cout << endl;
   cout << pgm << " is a filter program ";
   cout << "that extracts comments from ";
   cout << "C and C++ source files.";
   cout << endl;
   cout << endl;
   cout << "Usage:  " << pgm;
   cout << " < infile ";
   cout << "[> outfile/device] [/?]";
   cout << endl;
   cout << endl;
}

// ------------------------------------------
// end of getcmt.cpp

