///////////////////////////////////////////////////////////////////////////
// Copyright (c) Panos Kougiouris 1997
///////////////////////////////////////////////////////////////////////////

#include <kfThread.h>
#include <kfReferenceCount.h>
#include <pthread.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>

#include "kfMutexPosix.h"

// Each condition has the number of threads waiting and
// an Event. Waiting threads wait on the Event. Signal
// and broadcast PulseEvent the Event. The number of
// waiting threads is a hint used in broadcast() to know
// how many times to Pulse the Event
class CKFMTConditionImplementation : public CKFReferenceCount {
  public:
    pthread_cond_t m_condition;
    pthread_condattr_t    cv_attr;
    
    CKFMTConditionImplementation(const char* name)
    {
        pthread_condattr_init(&cv_attr);
        pthread_condattr_setpshared(&cv_attr, PTHREAD_PROCESS_PRIVATE);
        int ret = pthread_cond_init(&m_condition, &cv_attr);
        assert(ret == 0);
    }

  private:
    ~CKFMTConditionImplementation()
    {
        pthread_condattr_destroy(&cv_attr);
        int ret = pthread_cond_destroy(&m_condition);
        assert(ret == 0);
    }
};


//--------------------------------------------------------

CKFMT::CCondition::CCondition(const char* a_name)
{
    m_impl = new CKFMTConditionImplementation(a_name);   
}

//--------------------------------------------------------

CKFMT::CCondition::CCondition(const CKFMT::CCondition& a_m)
{
    m_impl = a_m.m_impl;
    m_impl->increment();
}

//--------------------------------------------------------

CKFMT::CCondition& 
CKFMT::CCondition::operator= (const CKFMT::CCondition& a_m)
{ 
    m_impl->decrement();
    m_impl = a_m.m_impl;
    m_impl->increment();

    return *this;
}

//--------------------------------------------------------

CKFMT::CCondition::~CCondition()
{
    m_impl->decrement();
}

//--------------------------------------------------------


CKFMT::RetValue
CKFMT::CCondition::wait(CKFMT::CMutex a_mut, unsigned a_msecs)
{
    int fret;
 
    if (a_msecs == Infinity) {
        fret = pthread_cond_wait(&m_impl->m_condition, &a_mut.m_impl->m_mutex);
    } else {
        // The timed wait requires an absolute time specification, so we
	// first have to figure out what time it is now.
	struct timeval curtime;
	gettimeofday(&curtime, 0);

	// Now add in the amount of time to sleep.
	timespec_t abstime;
	abstime.tv_sec = curtime.tv_sec + (a_msecs / 1000);
	abstime.tv_nsec = (curtime.tv_usec * 1000) + (a_msecs % 1000) * 1000000;
	if (abstime.tv_nsec >= 1000000000) {
	    abstime.tv_nsec -= 1000000000;
	    abstime.tv_sec++;
	}
        fret = pthread_cond_timedwait(&m_impl->m_condition, 
                                 &a_mut.m_impl->m_mutex, &abstime);
    }   

    if (fret == 0) {
	return Ok;
    } else if (fret == ETIMEDOUT) {
	return Timeout;
    }

    return Error;
}


//--------------------------------------------------------

CKFMT::RetValue
CKFMT::CCondition::signal()
{
    int ret = pthread_cond_signal(&m_impl->m_condition);
    if (ret != 0) {
        return Error;
    }

    return Ok;
}

//--------------------------------------------------------

CKFMT::RetValue
CKFMT::CCondition::broadcast()
{
    int ret = pthread_cond_broadcast(&m_impl->m_condition);
    if (ret != 0) {
        return Error;
    }

    return Ok;
}
