//: DIRWALK.CPP
#if defined(_MSC_VER)
 #include <cstring>
#else
    #include <string.h>
#endif
#include <iostream>
#include <stdexcept>
#include <windows.h>
#include "DirWalk.h"

#if defined(__BORLANDC__)
 using namespace std;
 #if defined(__MT__)
  #if !defined(_MT)
   #define _MT
  #endif
 #endif
#endif

/**
 * Returns the short file name for the current file. The way the 
 * WIN32_FIND_DATA structure is filled makes these contortions
 * necessary. The cFileName member is always filled with the file
 * name. If the cFileName member is a standard 8.3 file name, then
 * cAlternateFileName does not get filled at all. If the cFileName
 * member is a long file name, then only in that case does
 * cAlternateFileName get filled.
 **/
const char* const DirWalk::ShortFilename() throw() {
 strncpy(mShortFileName,mFindData.cAlternateFileName,
      ShortFileNameLength);
 if(strlen(mShortFileName) <= 0) { // anything there?
  // No. The short filename=long filename
  strncpy(mShortFileName,mFindData.cFileName,
       ShortFileNameLength);
 }
 return mShortFileName;
}

/**
 * Initiates the journey by doing some setup and calling Recurse().
 **/
void DirWalk::Walk() throw(runtime_error) {
 char prevDir[_MAX_PATH];
 // save current directory
 DWORD dwResult=GetCurrentDirectory(_MAX_PATH,prevDir);
 if((dwResult==0) || (dwResult>_MAX_PATH))
  throw runtime_error(
  "DirWalk::Walk().Problem with GetCurrentDirectory.");
 if(SetCurrentDirectory(mStartingDirectory) == FALSE)
  throw runtime_error(
  "DirWalk::Walk().Problem with SetCurrentDirectory.");
#if defined(_MT)
 dwResult=GetCurrentDirectory(_MAX_PATH,mCurrentDirectory);
 if((dwResult==0) || (dwResult>_MAX_PATH))
  throw runtime_error(
  "DirWalk::Walk().Problem with GetCurrentDirectory.");
#endif
 // Start!
 Recurse();
 if(SetCurrentDirectory(prevDir) == FALSE)
  throw runtime_error(
  "DirWalk::Walk().Problem resetting current directory.");
}

/**
 * Performs some initializations common to all the constructors
 **/
void DirWalk::ConstructorHelper(const BOOL RecurseSubDirs,
        const BOOL ListSubDirs) throw() {
    mDepth=-1;              // Starting recursion depth
    mFoundAnother=FALSE;    // Haven't found any files at all...yet
                            // There's not yet a current file, so...
    mIsDir=FALSE;           // Current file is not a directory
    mSize=0;                // Current file has zero size
 mSizeHigh=0;
    mRecurse=RecurseSubDirs;// whether to go into subdirectories
    mListSubDirs=ListSubDirs;//whether to include the subdirectory
                             //names in FoundFile() hits
}

/**
 * Constructor flavor #1.
 **/
DirWalk::DirWalk(const BOOL RecurseSubDirs,
     const BOOL ListSubDirs) throw() {
 ConstructorHelper(RecurseSubDirs,ListSubDirs);
 // no criteria was specified, so look at all files
 strncpy(mSearchSpec,"*",_MAX_PATH);
 // user didn't specify, so start in the current directory
 strncpy(mStartingDirectory,".",_MAX_PATH);
}

/**
 * Constructor flavor #2.
 **/
DirWalk::DirWalk(const char* StartingDirectory,
     const BOOL RecurseSubDirs,
     const BOOL ListSubDirs) throw() {
 ConstructorHelper(RecurseSubDirs,ListSubDirs);
 // no criteria was specified, so look at all files
 strncpy(mSearchSpec,"*",_MAX_PATH);
 // use starting directory as specified by user
 strncpy(mStartingDirectory,StartingDirectory,_MAX_PATH);
}

/**
 * Constructor flavor #3.
 **/
DirWalk::DirWalk(const char* StartingDirectory,
     const char* SearchSpec,const BOOL RecurseSubDirs,
     const BOOL ListSubDirs) throw() {
 ConstructorHelper(RecurseSubDirs,ListSubDirs);
 // use search criteria specified as specified
 strncpy(mSearchSpec,SearchSpec,_MAX_PATH);
 // use starting directory as specified by user
 strncpy(mStartingDirectory,StartingDirectory,_MAX_PATH);
}

/**
 * Called whenever a file meeting the criteria is found. Even though
 * this is a pure virtual function, some code is included here that
 * can be called by a derived class. It is handy to call this while
 * debugging a derived class just to see what files are being found.
 * Normally, the overriden version of this function will form the
 * heart of the derived class and will key other actions by such a
 * class.
 **/
void DirWalk::FoundFile() {
 // indent according to recursion depth
 for(int i=0;i < mDepth;i++) cout << "   ";
 if(mFA.IsDirectory()) {
  cout << "[" << Filename() << "]";// bracket it if it's a dir
 } else {
        cout << Filename();           // just the filename otherwise
 }
}

/**
 * This is where the real work of the class is done. We just trapse
 * through the current directory, calling FoundFile() for each file
 * that meets the search specification. Then, if we are to go down
 * into subdirectories, we call Recurse() again for each such
 * directory. Note that only one HANDLE is created on the stack for
 * each recursion, which is very efficient storage-wise. This type
 * of recursion lends itself well to object-implementation, since
 * one copy of all of the working variables can be placed in the
 * object rather than having new copies created on the stack for
 * every new instance of Recurse().
 **/
void DirWalk::Recurse() throw(runtime_error) {
 HANDLE hFind;
#if defined(_MT)
 char* ThisDirectory=new char[strlen(mCurrentDirectory)+1];
 strcpy(ThisDirectory,mCurrentDirectory);
#endif
 mDepth++;// if this has been called, then we're one level lower
 
#if defined(_MT)
 strcpy(mCurrentDirectorySearch,mCurrentDirectory);
 // The root directory and possibly certain UNC path names have a
 // trailing backslash already, so do we need to add one?
    if(mCurrentDirectorySearch[strlen(mCurrentDirectorySearch)-1]
       !='\\')
 { strcat(mCurrentDirectorySearch,"\\"); }
 // the search has to have the dir and the wildcard...
 strcat(mCurrentDirectorySearch,mSearchSpec);
 hFind=FindFirstFile(mCurrentDirectorySearch,&mFindData);
#else
 hFind=FindFirstFile(mSearchSpec,&mFindData);
#endif

 mFoundAnother=(hFind!=INVALID_HANDLE_VALUE);
 while(mFoundAnother) {
  mSize = mFindData.nFileSizeLow; // set the file size
  mSizeHigh = mFindData.nFileSizeHigh;
  mFA   = mFindData.dwFileAttributes;
  mIsDir= mFA.IsDirectory();
  // set up the time objects for this file
  mCreationTime.NewTime(&mFindData.ftCreationTime);
  mLastAccessTime.NewTime(&mFindData.ftLastAccessTime);
  mLastWriteTime.NewTime(&mFindData.ftLastWriteTime);
  if(!mIsDir) {
   FoundFile();     // announce each file (but not dirs, yet)
  }
  mFoundAnother=FindNextFile(hFind, &mFindData);    // next one
 }
 if(hFind != INVALID_HANDLE_VALUE) {
  if(FindClose(hFind) == FALSE) throw runtime_error(
   "DirWalk::Recurse().Problem closing hFind.");
 }

 // OK, we've listed all the files of interest.
 // Now, we go into subdirectories and list those if we need to.
 if(mRecurse) {
  hFind=FindFirstChildDir();
  mFoundAnother=(hFind != INVALID_HANDLE_VALUE);
  while(mFoundAnother) {
   mFA=mFindData.dwFileAttributes;// same game as above
   mCreationTime.NewTime(&mFindData.ftCreationTime);
   mLastAccessTime.NewTime(&mFindData.ftLastAccessTime);
   mLastWriteTime.NewTime(&mFindData.ftLastWriteTime);
   mSize=mFindData.nFileSizeLow;
   mSizeHigh=mFindData.nFileSizeHigh;
#if defined(_MT) // multithreaded version
   if(mCurrentDirectory[strlen(mCurrentDirectory)-1]!='\\') {
    strcat(mCurrentDirectory,"\\");
   }
   // add the new subdirectory to the current dir
   strcat(mCurrentDirectory,mFindData.cFileName);
            if(mListSubDirs) FoundFile();   //announce, if necessary
   Recurse();
            strcpy(mCurrentDirectory,ThisDirectory);//reset dir
#else   // singlethreaded version
   // go into subdirectory
   if(SetCurrentDirectory(mFindData.cFileName) == TRUE) {
    if(mListSubDirs) FoundFile();// announce subdir name
                Recurse();      //list files in the subdirectory now
    if(SetCurrentDirectory("..") == FALSE)
     throw runtime_error(
              "DirWalk::Recurse().Problem changing to parent dir.");
   }
#endif
            mFoundAnother=FindNextChildDir(hFind);//next subdir
  }
  if(hFind != INVALID_HANDLE_VALUE) {
   if(FindClose(hFind) == FALSE)
    throw runtime_error(
    "DirWalk::Recurse().Problem closing hFind(2).");
  }
 }
#if defined(_MT)
 delete []ThisDirectory;
#endif
    mDepth--;           // we're returning to the parent's depth now
} //void DirWalk::Recurse()

/**
 * TRUE if the current file in mFindData is a child of the current
 * directory. FALSE otherwise.
 **/
BOOL DirWalk::IsChildDir() const throw() {
    return((mFindData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)&&
     (mFindData.cFileName[0] != '.'));
}

/**
 * Continue search through current directory until we find a
 * subdirectory. TRUE if one is found. FALSE otherwise. 
 **/
BOOL DirWalk::FindNextChildDir(HANDLE hFindFile) throw() {
 BOOL fFound=FALSE;
 do {
  fFound=FindNextFile(hFindFile,&mFindData);
 } while(fFound && !IsChildDir());
 return fFound;
}

/**
 * Returns the handle to the first subdirectory under the current
 * directory
 **/
HANDLE DirWalk::FindFirstChildDir() throw() {
 BOOL fFound;
#if defined(_MT)
 strcpy(mCurrentDirectorySearch,mCurrentDirectory);
 // The root directory and possibly certain UNC path names have a
 // trailing backslash already, so we don't need to add one.
 if(mCurrentDirectorySearch[strlen(mCurrentDirectorySearch)-1]
  !='\\') {
  strcat(mCurrentDirectorySearch,"\\");
 }
 strcat(mCurrentDirectorySearch,"*");
 // notice that the search spec for files doesn't apply to dirs
 HANDLE hFindFile=FindFirstFile(mCurrentDirectorySearch,
                                   &mFindData);
#else
 HANDLE hFindFile=FindFirstFile("*",&mFindData);
#endif
 if(hFindFile != INVALID_HANDLE_VALUE) {
  fFound=IsChildDir();
  if(!fFound) fFound=FindNextChildDir(hFindFile);
  if(!fFound) {
   FindClose(hFindFile);
   hFindFile=INVALID_HANDLE_VALUE;
  }
 }
 return hFindFile;
}

/**
 * Returns the full long pathname of the current file.
 * This is only necessary in multithreaded apps where
 * the GetFullPathName() API function won't work.
 **/
#if defined(_MT)
const char* const DirWalk::FullPathName() throw() { 
 strcpy(mFullPathName,mCurrentDirectory);
 // The root directory and possibly certain UNC path names have a
 // trailing backslash already, so we don't need to add one.
 if(mFullPathName[strlen(mFullPathName)-1]!='\\') {
  strcat(mFullPathName,"\\");
 }
    //add the current file to the current directory of this instance
 strcat(mFullPathName,mFindData.cFileName);
 return mFullPathName;
}
#endif
