/* tstcompl.cpp / test of complaint printer */

/* here's the scoop: we write a dummy complaint dictionary file, then go */
/* out and read it to ensure that we can obtain values from it. */

#include <stdio.h>
#include <stdlib.h>
#include "utils.hpp"                    /* skipblanks() etc. */
#include "errmgr.hpp"                   /* ASSERT() */
#include "c_failur.hpp"                 /* counting_failure_handler */
#include "complain.hpp"                 /* what we're testing */

/* message "foo" is a test of the method used to get a backslash */
/* (continuation character) at the end of a message. */

static const char *msgs1[] = {          /* table of good stuff */
    "\n",                               /* blank line */
    "# comment line\n",                 /* as it says */
    "a: testing\n",                     /* single-line message */
    "b:testing two\n",                  /* no space after colon */
    "c : testing three\n",              /* space before colon */
    "d: testing four\\\ntesting five\n",  /* two line message */
    "e: testing six\\\ntesting seven\\\ntesting eight\n",  /* three lines */
    "foo: backslash at end of line\\\\\n\n",  /* as it says */
    "123: numeric message key\n",       /* keys are alphanumeric */
    0
};

static const char *msgnames1[] = {      /* names of what we expect to find */
    "",                                 /* nothing for this line */
    "",                                 /* ditto */
    "a",                                /* first valid line */
    "b",
    "c",
    "d",
    "e",
    "foo",
    "123",
    0
};

static const char *msgs2[] = {          /* table full of errors */
    "a: valid key\n",                   /* test for duplicate key */
    "a: duplicate key\n",
    "^&*^*&: non-alphanumeric key\n",
    "abcdef missing ':'\n",
    0
};

static int write_complaint_file(const char *filename,
                                const char **text);
static int compare_results(complaint_dict *dict,char *line,int linelen,
                           const char *msgname,const char *testline);
static int compare_strings(const char *dict_str,const char *test_str);


int main(int argc,char *argv[])
{
    char line[512];
    int i,all_ok = 1;
    complaint_dict *dict;
    char tempname[32];
    counting_failure_handler count;     /* self-installing */

    /* write a file of keyed error messages */

    tmpnam(tempname);
    if (write_complaint_file(tempname,msgs1))
        return EXIT_FAILURE;

    /* now read them into a dictionary. */

    dict = new complaint_dict(tempname);

    /* next, ensure they got there properly. */

    for (i = 0; msgs1[i]; ++i)          /* compare each message */
        if (!compare_results(dict,line,sizeof(line),msgnames1[i],msgs1[i]))
            all_ok = 0;
    delete dict;

    if (count.errors_logged() || count.warns_logged())
        all_ok = 0;

    /* now test the error handling code. we can only count the number */
    /* of warnings. there should be three. */

    remove(tempname);
    if (write_complaint_file(tempname,msgs2))
        return EXIT_FAILURE;
    dict = new complaint_dict(tempname);
    if (count.warns_logged() != 3L) {
        err_mgr.warn("Invalid number of warnings: expected 3, got %ld\n",
                     count.warns_logged());
        all_ok = 0;
    }

    delete dict;
    remove(tempname);
    if (all_ok)
        fprintf(stdout,"all tests succeeded\n");
    return all_ok ? EXIT_SUCCESS : EXIT_FAILURE;

}  /* end of main() */

static int write_complaint_file(const char *filename,
                                const char **msgs)
{
    /* write the text to the named file. return 1 on failure, 0 on success. */

    int i;
    FILE *tempfile;

    if ((tempfile = fopen(filename,"w")) == NULL) {
        fprintf(stderr,"Unable to create temporary file\n");
        return 1;
    }
    for (i = 0; msgs[i]; ++i)
        fputs(msgs[i],tempfile);
    fclose(tempfile);
    return 0;

}  /* end of write_complaint_file() */

static int compare_results(complaint_dict *dict,char *line,int linelen,
                           const char *msgname,const char *testline)
{
    /* compare the text of testline with what's in the dictionary. skip */
    /* the message name in the front (we use msgname for the lookup), then */
    /* compare what's left with what comes back from the dictionary. ignore */
    /* any backslash that is right before a newline in testline. */

    if (!*msgname)
        return 1;                       /* assumed to be OK */
    if (!dict->key_defined(msgname)) {
        fprintf(stderr,"key_defined() failed for key %s\n",msgname);
        return 0;
    }
    if (!dict->complaint_text(msgname,line,linelen)) {
        fprintf(stderr,"lookup failed for key %s\n",msgname);
        return 0;
    }

    testline += skip_ident(testline);
    testline += skipblanks(testline);
    ASSERT(*testline == ':');
    ++testline;
    testline += skipblanks(testline);

    return compare_strings(line,testline) == 0;

}  /* end of compare_results() */

static int compare_strings(const char *dict_str,const char *test_str)
{
    /* compare the two strings, ignoring any backslash in test_str that's */
    /* right in front of a newline (the dictionary will strip it out when */
    /* it combines lines in the file). */

    /* returns negative, 0, or positive (just like strcmp()). */

    /* there's no good way to write this; the exit has to be in the */
    /* middle of the loop, after skipping '\\' and before incrementing. */

    for (;;) {                          /* exit from within */
        if (*test_str == '\\' && *(test_str + 1) == '\n')
            ++test_str;
        if (!*dict_str || !*test_str || *dict_str != *test_str)
            return *dict_str - *test_str;
        ++dict_str;
        ++test_str;
    }  /* end of for(ever) */

    /* NOTREACHED */

}  /* end of compare_strings() */
