// textprnt.cpp

#include <ctype.h>
#include <string.h>
#pragma hdrstop
#include <textprnt.h>

/*****************************************************************************
	Constructor 1 (default)
	1) Create a port to the system`s default printer.
	2) Set default CPI value to 17 chars per inch, and 6 lines per inch
	3) Set default tab width to 8 characters
*****************************************************************************/
TTextPrinter::TTextPrinter(DWORD cpi, int tabWidth, COLORREF aColor)
	:	dwCPI(cpi),
		row(0),
		column(0),
		page(0),
		iTabWidth(tabWidth),
		isAlias(FALSE),
		color(aColor),
		dwCharExtent(0),
		bLineWrapping(TRUE),
		documentON(FALSE)
	{
	memset(&pd, 0, sizeof(PRINTDLG));
	pd.lStructSize = sizeof(PRINTDLG);

	memset(&lf, 0, sizeof(LOGFONT));
	strcpy(lf.lfFaceName, "Courier");
	lf.lfWidth = lf.lfHeight = 1;

	InitDefaultPrinter();
	}

/*****************************************************************************
	Constructor 2 (Alias)
	1) Construct the object with an already initialized DC handle
	2) Set default CPI value to 17 chars per inch, and 6 lines per inch
	3) Set default tab width to 8 characters
*****************************************************************************/
TTextPrinter::TTextPrinter(HDC dc, DWORD cpi, int tabWidth, COLORREF aColor)
	:	dwCPI(cpi),
		row(0),
		column(0),
		page(0),
		iTabWidth(tabWidth),
		isAlias(TRUE),
		color(aColor),
		dwCharExtent(0),
		bLineWrapping(TRUE),
		documentON(FALSE)
	{
	memset(&pd, 0, sizeof(PRINTDLG));
	pd.lStructSize = sizeof(PRINTDLG);
	pd.hDC = dc;

	memset(&lf, 0, sizeof(LOGFONT));
	strcpy(lf.lfFaceName, "Courier");
	lf.lfWidth = lf.lfHeight = 1;
	}

/*****************************************************************************
	Destructor
	End current document and close printer
*****************************************************************************/
TTextPrinter::~TTextPrinter()
	{
	ClosePrinter();
	}

/*****************************************************************************
	Initiliaze default printer and document
*****************************************************************************/
void TTextPrinter::InitDefaultPrinter()
	{
	if( !isAlias )
		{
		// Get a DC handle to the system`s default printer. using
		// the PD_RETURNDEFAULT flag ensures that no dialog is opened.
		pd.Flags = PD_RETURNDC | PD_RETURNDEFAULT | PD_NOSELECTION;
		PrintDlg(&pd);
		}
	}

/*****************************************************************************
	Close printer
*****************************************************************************/
void TTextPrinter::ClosePrinter()
	{
	if( pd.hDC )
		{
		if( documentON )
			EndDocument();
		if( pd.hDevMode )
			GlobalFree(pd.hDevMode);
		if( pd.hDevNames )
			GlobalFree(pd.hDevNames);
		if( !isAlias )
			DeleteDC(pd.hDC);
		}
	}


/*****************************************************************************
	Write an ASCIIZ string to the printer.
	This method is declared as virtual, to allow enhancements like
	syntax-highlighting parsing etc.
*****************************************************************************/
void TTextPrinter::WriteString(const char* sz)
	{
	for(const char* p = sz; *p; p++)
		WriteChar(*p);
	}

/*****************************************************************************
	Write a single character to the printer. This function does all the
	TTY processing, like line-feed, carriage-return, form-feed, tabs, etc.
*****************************************************************************/
void TTextPrinter::WriteChar(char ch)
	{
	if( !Ok() )
		return;

	if( !documentON )
		StartDocument();

	switch( ch )
		{
		case '\n' :				// Line-feed
			{
			NextLine();
			break;
			}

		case '\f' :				// Form-feed
			{
			NextPage();
			break;
			}

		case '\r' :				// Carriage-return
			{
			column = 0;
			break;
			}

		case '\t' :				// Tab
			{
			column += iTabWidth - (column % iTabWidth);
			if( bLineWrapping && column >= LOWORD(dwCharExtent) )
				NextLine();
			break;
			}

		default :
			{
			if( isprint(ch) )
				{
				HFONT hOldFont = SetFont();
				PrepareDC();
				TextOut(pd.hDC, column++, row, &ch, 1);
				DeleteObject(SelectObject(pd.hDC, hOldFont));

				// Wrap line
				if( bLineWrapping && column >= LOWORD(dwCharExtent) )
					NextLine();
				}
			break;
			}
		} // End switch
	}


/*****************************************************************************
	Prepare the printer DC for output. This function uses the specified CPI
	values to scale the DC, using MM_ANISOTROPIC mapping mode.
*****************************************************************************/
void TTextPrinter::PrepareDC()
	{
	int lpxlX = GetDeviceCaps(pd.hDC, LOGPIXELSX);
	int lpxlY = GetDeviceCaps(pd.hDC, LOGPIXELSY);
	SetMapMode(pd.hDC, MM_ANISOTROPIC);
	SetViewportExtEx(pd.hDC, lpxlX, lpxlY, NULL);
	SetWindowExtEx(pd.hDC, LOWORD(dwCPI), HIWORD(dwCPI), NULL);
	SetViewportOrgEx(pd.hDC, 0, 0, NULL);

	SetBkMode(pd.hDC, TRANSPARENT);
	SetTextColor(pd.hDC, color);
	}


/*****************************************************************************
	Select a fixed font into the printer DC. Since the DC is scaled for rows
	and columns, we set the font size to 1x1 always.
	This function returns a handle to the previous selected font.
*****************************************************************************/
HFONT TTextPrinter::SetFont()
	{
	// Create font
	lf.lfWidth = lf.lfHeight = 1;
	return (HFONT)SelectObject(pd.hDC, CreateFontIndirect(&lf));
	}


/*****************************************************************************
	Open a printer-setup dialog to allow changing the printer
*****************************************************************************/
BOOL TTextPrinter::Setup(HWND hwndParent)
	{
	BOOL rc = FALSE;
	HDC hdcPrevPrinter = pd.hDC; // Save current printer`s DC

	pd.Flags = PD_RETURNDC | PD_PRINTSETUP | PD_NOSELECTION;
	pd.hwndOwner = hwndParent ? hwndParent : GetActiveWindow();

	if( PrintDlg(&pd) && pd.hDC )
		{
		// Before starting with a new settings, we have to close the current document
		if( documentON )
			{
			HDC hdcNewPrinter = pd.hDC;
			pd.hDC = hdcPrevPrinter;
			EndDocument();
			pd.hDC = hdcNewPrinter;
			StartDocument();
			}
		rc = TRUE;
		}
	return rc;
	}


/*****************************************************************************
	Start document
*****************************************************************************/
void TTextPrinter::StartDocument()
	{
	if( pd.hDC )
		{
		// If there`s still an open document, we have to close it first
		if( documentON )
			EndDocument();

		// Re-calculate page extent in characters
		dwCharExtent = MAKELONG(
			((GetDeviceCaps(pd.hDC, HORZRES) / GetDeviceCaps(pd.hDC, LOGPIXELSX)) * LOWORD(dwCPI)),
			((GetDeviceCaps(pd.hDC, VERTRES) / GetDeviceCaps(pd.hDC, LOGPIXELSY)) * HIWORD(dwCPI))
			);

		// Open a new document and start at its first page
		DOCINFO di = { sizeof(DOCINFO), "TTextPrinter", NULL };
		StartDoc(pd.hDC, &di);
		StartPage(pd.hDC);
		row = column = page = 0;
		PageHeader(page+1);

		documentON = TRUE;
		}
	}


/*****************************************************************************
	End document
*****************************************************************************/
void TTextPrinter::EndDocument()
	{
	EndPage(pd.hDC);
	EndDoc(pd.hDC);
	documentON = FALSE;
	}


/*****************************************************************************
	Print formatted strings, printf style
*****************************************************************************/
int _cdecl TTextPrinter::pprintf(const char* fmt, ...)
	{
	static const int _bufsize = 4098;
	va_list argptr;
	int cnt;
	char* buffer = new char[_bufsize];

	va_start(argptr, fmt);
	cnt = vsprintf(buffer, fmt, argptr);
	va_end(argptr);

	WriteString(buffer);

	delete [] buffer;

	return cnt;
	}


/*****************************************************************************
	Advance by one line. I crossed page boundary, go to next page
*****************************************************************************/
void TTextPrinter::NextLine()
	{
	row++;
	column = 0;
	if( row == HIWORD(dwCharExtent) )
		NextPage();
	}


/*****************************************************************************
	Go to next page
*****************************************************************************/
void TTextPrinter::NextPage()
	{
	EndPage(pd.hDC);
	StartPage(pd.hDC);
	column = row = 0;
	page++;
	PageHeader(page+1);
	}


/*****************************************************************************
	Open a standrad fonts dialog and let the user select a font. The selection
	is limited to fixed fonts only.
*****************************************************************************/
BOOL TTextPrinter::SelectFont(HWND hwndParent)
	{
	CHOOSEFONT cf;

	memset(&cf, 0, sizeof(CHOOSEFONT));
	cf.lStructSize = sizeof(CHOOSEFONT);
	cf.hwndOwner = hwndParent ? hwndParent : GetActiveWindow();
	cf.lpLogFont = &lf;
	cf.Flags = CF_SCREENFONTS | CF_FORCEFONTEXIST | CF_FIXEDPITCHONLY | CF_INITTOLOGFONTSTRUCT | CF_EFFECTS;
	cf.nFontType = SCREEN_FONTTYPE;

	BOOL rc = ChooseFont(&cf);

	if( rc )
		color = cf.rgbColors;

	return rc;
	}


/*****************************************************************************
	Assign new CPI and LPI values
*****************************************************************************/
void TTextPrinter::SetCPI(DWORD cpi)
	{
	if( documentON )
		EndDocument();
	dwCPI = cpi;
	}


