//File: Printers.cpp
// (c) David Tamashiro, CBBG Consulting, 1996
// Compiles with BC++ 5.0
//  DEBUG CONTROL =============================
//Comment out the next line for ASSERT debuging
//#define NDEBUG 1
#include <windows.h>
#include <iostream.h>
#include "printers.hpp"
#include <assert.h>





//==============================================
//	Printer Class Implementation
//==============================================

void
BasePrinter::
EnumPrinters(BOOL Display, char* ReturnDefPrinter)
{
BYTE* pbuf;
DWORD dwSize, dwPrinters;

// Find out how much bytes is needed to
// hold PRINTER_INFO_5 array
::EnumPrinters(PRINTER_ENUM_LOCAL,NULL,5,NULL,0,
	&dwSize,
   &dwPrinters);

pbuf = new BYTE[dwSize];

//Read in the printer array
::EnumPrinters(PRINTER_ENUM_LOCAL,NULL,5,	pbuf,
	dwSize, &dwSize, &dwPrinters);

if (dwPrinters != 0) {
	int i;
	PRINTER_INFO_5 * pPrnInfo =
   	(PRINTER_INFO_5 *) pbuf;
   for (i=0; i < dwPrinters; i++,pPrnInfo++){
   	if (Display)
   		cout << "Printer Device: \"" <<
      		pPrnInfo->pPrinterName << "\"" ;
      if ((pPrnInfo->Attributes) &
      	PRINTER_ATTRIBUTE_DEFAULT) {
         	// Default Printer Name found
         	int len;

            // clip printer name length
            len = strlen(pPrnInfo->pPrinterName);
            if (len > MaxPrnNameLen)
            	len = MaxPrnNameLen;

            if (ReturnDefPrinter != NULL) {
            	strncpy(ReturnDefPrinter,
            		pPrnInfo->pPrinterName,
            		len);
               *(ReturnDefPrinter + len) = '\0';
               } // if
            if (Display)
      			cout << " ** Default Printer **\n";
           } // if
       else {
       	if (Display)
        		cout << "\n";
         } // else
      }  // for
	} // if

delete[] pbuf;
}

//===============================================
//===== DirectPrinter Class Implementation ======
//===============================================

DirectPrinter::
DirectPrinter(char* Name)
{
fPrinter =fopen(Name,"wb");

//if assert fails, it means that the printer
// name does not exist (or is mis-spelled).
assert(fPrinter!=NULL);

}

DirectPrinter::
~DirectPrinter()
{
fclose(fPrinter);
}

BasePrinter::PrinterRetCode
DirectPrinter::
PrintLine(char* Line)
{
fprintf(fPrinter,"%s\r\n",Line);
return(DirectPrinter::SUCCESS);
};

void
DirectPrinter::
EjectPaper(void)
{
fprintf(fPrinter,"\f");
}


//==============================================
//	WinGDIPrinter Class Implementation
//==============================================

WinDirPrinter::
WinDirPrinter(char* Name)
{

DOC_INFO_1 DocAttrib;
BOOL Retval;
DWORD Dret;

//Use default printer if Name is NULL
if (Name == NULL) {
	//Find and store the default printer
	EnumPrinters(FALSE,PrinterName);
   }
else {
   strcpy(PrinterName,	Name);
	}

Retval = OpenPrinter(
	PrinterName,
	&hPrinter,
	NULL);
if (Retval== FALSE)  {
   // Couldn't find printer!
	ErrorCode = GetLastError();
   assert(0);
	}

DocAttrib.pDocName = "Dave's Document";
DocAttrib.pOutputFile=NULL;

// Very Important! Must be "RAW" or else
// StarDocPrinter() will return
// Invalid Datatype!!
DocAttrib.pDatatype= "RAW";


Dret = StartDocPrinter(
	hPrinter,
   1,        // Specify DOC_INFO_1 data structure
   (LPBYTE) &DocAttrib);

if (Dret == 0 ) {
   // StartDocPrinter Fail, probalby invalid
   // Datatype!
	ErrorCode = GetLastError();
   assert(0);
	}

// Nothing to print yet, so don't open page
PageInitialized = FALSE;
}


WinDirPrinter::
~WinDirPrinter()
{
BOOL Retval;

if (PageInitialized == TRUE) {
	Retval = EndPagePrinter(hPrinter);
	assert(Retval == TRUE);
	}

Retval = EndDocPrinter(hPrinter);
assert(Retval == TRUE);

Retval = ClosePrinter(hPrinter);
assert(Retval == TRUE);

}





BasePrinter::PrinterRetCode
WinDirPrinter::
PrintLine(char *Line)
{
BOOL Retval;
DWORD Count;  // Count of printed bytes
int len= strlen(Line);

if (PageInitialized == FALSE) {
	PageInitialized = TRUE;
	Retval = StartPagePrinter(hPrinter);
	if (Retval== FALSE)  {
  	 	// Couldn't start page!
		ErrorCode = GetLastError();
   	assert(0);
		}
    }

Retval= WritePrinter(hPrinter, Line, len, &Count);
if ((Count != len) || (Retval != TRUE))
	return(FAIL);
Retval= WritePrinter(hPrinter, "\r\n", 2, &Count);
if ((Count != 2) || (Retval != TRUE))
	return(FAIL);
return(SUCCESS);
}

void
WinDirPrinter::
EjectPaper(void)
{
BOOL Retval;
DWORD Count;  // Count of printed bytes

if (PageInitialized == FALSE) {
	PageInitialized = TRUE;
	Retval = StartPagePrinter(hPrinter);
	if (Retval== FALSE)  {
  	 	// Couldn't start page!
		ErrorCode = GetLastError();
   	assert(0);
		}
    }

WritePrinter(hPrinter, "\f", 1, &Count);
// Inform Printer Spooler that End of page
// has been reached.
Retval = EndPagePrinter(hPrinter);
assert(Retval == TRUE);
PageInitialized = FALSE;
}



//==============================================
//	WinGDIPrinter Class Implementation
//==============================================

WinGDIPrinter::
WinGDIPrinter(char* Name)
{

//Use default printer if Name is NULL
if (Name == NULL) {
	//Find and store the default printer
	EnumPrinters(FALSE,PrinterName);
   }
else {
   strcpy(PrinterName,	Name);
	}

//  Get Device Context for printer
hDC = CreateDC(NULL,PrinterName,NULL,NULL);
if (hDC == NULL)  {
   // Couldn't find printer!
   assert(0);
	}

HorizDevRes = GetDeviceCaps(hDC, HORZRES);
VertDevRes = GetDeviceCaps(hDC, VERTRES);
VertPixelsPerInch =GetDeviceCaps(hDC, LOGPIXELSY);

// Nothing to print yet, so don't initialize the
// page
PageInitialized = FALSE;
::ZeroMemory(&di,sizeof(DOCINFO));
di.cbSize = sizeof(DOCINFO);
di.lpszDocName= "- WinGDIPrinter -";

DocValid = StartDoc(hDC,&di);
if (!DocValid) {
	// Could not start print job
   assert(0);
	}

// Load in the default Font
::ZeroMemory(&FontAttrib,sizeof(FontAttrib));
GetObject(GetCurrentObject(hDC, OBJ_FONT),
	sizeof (LOGFONT), (PSTR) &FontAttrib);

hFont = CreateFontIndirect(&FontAttrib);
//Load the "new" font.
// Save the old font, just in case...
hStartFont = SelectObject(hDC,hFont);
CalcLinesPerPage();
CurYOffset=0;
}


WinGDIPrinter::
~WinGDIPrinter()
{

// Probably not neccessary, but
// put the original font back anyway
SelectObject(hDC, hStartFont);

if (hFont != NULL)
	DeleteObject(hFont);

if (PageInitialized && !EndPage(hDC)) {
   	DocValid = 0;
	}

if (DocValid)
	EndDoc(hDC);

if (hDC != NULL)
	DeleteDC(hDC);
}





void
WinGDIPrinter::CalcLinesPerPage(void)
{
TEXTMETRIC tm;

if (hDC == NULL)
	return;

if (GetTextMetrics(hDC,&tm) == FALSE) {
	// Could not retrieve Font info
	assert(0);
   }

FontYSize = tm.tmHeight + tm.tmExternalLeading;
MaxYOffset = VertDevRes - FontYSize;
}



BasePrinter::PrinterRetCode
 WinGDIPrinter::ChooseFont(void)
{
CHOOSEFONT NewFont;
PrinterRetCode retval=FUNC_NOT_SUPPORTED;


memset(&NewFont,0, sizeof(CHOOSEFONT));
NewFont.lStructSize = sizeof(CHOOSEFONT);
NewFont.hwndOwner=NULL;
NewFont.hDC=hDC;
NewFont.lpLogFont=&FontAttrib;
NewFont.Flags= CF_INITTOLOGFONTSTRUCT |
		CF_WYSIWYG | CF_BOTH |
	 	CF_SCALABLEONLY | CF_EFFECTS;
NewFont.lCustData = NULL;
NewFont.lpfnHook = NULL;
NewFont.lpTemplateName = NULL;
NewFont.lpszStyle = NULL;

if (::ChooseFont(&NewFont) == TRUE) {
	// New Font Selected
	SetFontPointSize(NewFont.iPointSize/10);
	}

return(retval);
}


BasePrinter::PrinterRetCode
WinGDIPrinter::
SetFontPointSize(unsigned PointSize)
{
// This function changes the font size of the
// currently selected Font.  It may also change
// the current font if a new font is
// described by the variable FontAttrib.

	// New Font Selected
	HFONT hNewFont;
   // Calculate new point size.
   FontAttrib.lfHeight =
   	-MulDiv(PointSize,
      	VertPixelsPerInch, 72);

   hNewFont  = CreateFontIndirect(&FontAttrib);
   SelectObject(hDC,hNewFont);

	// Delete the old font
   if (hFont != NULL)
   	DeleteObject(hFont);

	// Set the new current Font
   hFont = hNewFont;
   CalcLinesPerPage();
return(SUCCESS);
}


BasePrinter::PrinterRetCode
WinGDIPrinter::
PrintLine(char *Line)
{
int len= strlen(Line);

if (!DocValid)
	return(FAIL);

if (PageInitialized==FALSE) {
	if (StartPage(hDC) > 0) {
		SelectObject(hDC, hFont);
 	  	PageInitialized = TRUE;
		}
   else {// Page initialization FAILED!
   	DocValid = 0;
      return(FAIL);
   	}
	}


TextOut(hDC,0,CurYOffset,Line,len);

CurYOffset += FontYSize;
if (CurYOffset > MaxYOffset)
   EjectPaper();

return(SUCCESS);
}

void
WinGDIPrinter::
EjectPaper(void)
{

if (!DocValid)
	return;

//Make sure page has been initialized first!
if (PageInitialized==FALSE) {
	if (StartPage(hDC)) {
 	  	PageInitialized = TRUE;
		}
   else {// Page initialization FAILED!
   	DocValid = 0;
   	}
	}


if (PageInitialized && !EndPage(hDC)) {
   	DocValid = 0;
	}

PageInitialized = FALSE;
CurYOffset=0;
}


