// ****************************************************************************
//
// Module:  htmlhelp.h
// Author:  Dick Lam
//
// Purpose: C++ class header file for cgiHTMLHelper
//
// Notes:  This is a base class. It contains helper methods for generating
//         HTML documents. Some custom iostream manipulators are also included
//         for dynamically constructing HTML using the appropriate tags. You
//         can use the manipulators with any stream as follows:
//
//            cout << htmltitle("A Sample Title") << endl
//                 << htmlhead("A Sample Heading") << endl
//                 << htmlh1("Heading level 1") << endl
//                 << htmlpara << "This is a link to the " <<
//                 << htmlbeginlink("http://www.ibm.com")
//                 << "IBM home page"
//                 << htmlendlink
//                 << "." << endl;
//
//         Which will result in the following output:
//
//            <title>A Sample Title</title>
//            <head>A Sample Heading</head>
//            <h1>Heading level 1</h1>
//            <p>This is a link to the
//            <a href="http://www.ibm.com">IBM home page</a>.
//
// ****************************************************************************

#ifndef MODULE_cgiHTMLHelperh
#define MODULE_cgiHTMLHelperh

#include <iostream.h>
#include <iomanip.h>
#include <stl.h>

// iostream manipulators for HTML output
omanip<const char *> htmltitle(const char *title);          // TITLE
omanip<const char *> htmlhead(const char *head);            // HEAD
omanip<const char *> htmlh1(const char *heading);           // H1-H6
omanip<const char *> htmlh2(const char *heading);
omanip<const char *> htmlh3(const char *heading);
omanip<const char *> htmlh4(const char *heading);
omanip<const char *> htmlh5(const char *heading);
omanip<const char *> htmlh6(const char *heading);

ostream& htmlbreak(ostream& os);                            // line BReak
ostream& htmlpara(ostream& os);                             // Paragraph
ostream& htmlrule(ostream& os);                             // Horizontal Rule

omanip<const char *> htmlbeginlink(const char *link);       // Anchor
omanip<const char *> htmlnamedlink(const char *link);       // named anchor
ostream& htmlendlink(ostream& os);                          // end anchor

omanip<const char *> htmllistitem(const char *listitem);    // List Item
ostream& htmllist(ostream& os);                             // Unordered List
ostream& htmlendlist(ostream& os);
ostream& htmlordered(ostream& os);                          // Ordered List
ostream& htmlendordered(ostream& os);
omanip<const char *> htmllistitem(const char *listitem);    // List Item

ostream& htmldeflist(ostream& os);                          // Definition List
ostream& htmlenddeflist(ostream& os);
omanip<const char *> htmlterm(const char *term);            // Term
omanip<const char *> htmldefinedas(const char *definition); // Definition

ostream& htmlpreformatted(ostream& os);                     // PREformatted
ostream& htmlendpreformatted(ostream& os);

omanip<const char *> htmlblockquote(const char *quote);     // BLOCKQUOTE
omanip<const char *> htmladdress(const char *address);      // ADDRESS

ostream& htmlbold(ostream& os);                             // text styles
ostream& htmlendbold(ostream& os);
ostream& htmlitalics(ostream& os);
ostream& htmlenditalics(ostream& os);
ostream& htmlcode(ostream& os);
ostream& htmlendcode(ostream& os);

ostream& htmllessthan(ostream& os);                         // special chars
ostream& htmlgreaterthan(ostream& os);
ostream& htmlampersand(ostream& os);
ostream& htmldoublequote(ostream& os);

omanip<const char *> htmlimage(const char *image);          // in-line IMaGe

omanip<const char *> htmlformget(const char *link);   // FORM METHOD="GET"
omanip<const char *> htmlformpost(const char *link);  // FORM METHOD="POST"
ostream& htmlendform(ostream& os);

// class declaration
class cgiHTMLHelper {

public:

   // constructor and destructor
   cgiHTMLHelper(ostream& os);
   virtual ~cgiHTMLHelper();

   // outputs the MIME type for the document (required for all CGI programs)
   void contentType(const char *type = "text/html");

   // outputs recommended header and trailer for HTML documents
   virtual void header(const char *title = "");
   virtual void trailer();

   // form support - writes HTML tags for different form controls

   // writes a name, value pair
   void hidden(const char *name, const char *value);

   // creates an entryfield (the field text defaults to empty and has
   // a default length - if maxlength is 0, there is no limit to the number
   // of input characters - otherwise the number is limited to the value of
   // maxlength)
   void entryfield(const char *name, const char *value = "", int len = 20,
                   int maxlength = 0);

   // creates an entryfield for entering a password
   void password(const char *name, const char *value = "", int len = 10,
                 int maxlength = 8);

   // creates a checkbox (defaults to not checked) - if the box is checked,
   // its value is set to the input label for the box
   void checkbox(const char *name, const char *label, bool checked = false);

   // creates a radiobutton (defaults to not selected) - use the same name
   // to create a group of radiobuttons - if a button is selected, its value
   // is set to the input label for the button
   void radiobutton(const char *name, const char *label, bool selected = false);

   // creates submit and reset pushbuttons (submit packages up the form
   // and submits it for processing - reset clears all form contents to
   // their default settings)
   void submitbutton(const char *label = "Submit");
   void resetbutton(const char *label = "Clear");

   // creates a multiline edit box
   void texteditor(const char *name, int rows, int cols, const char *text = "");

   // creates a listbox for selecting one or multiple items
   void listbox(const char *name, int size = 1, bool multiselect = false);
   void listoption(const char *option, bool selected = false);
   void endlistbox();

   // creates an imagemap
   void imagemap(const char *name, const char *link);

protected:

   ostream& output;     // output stream

private:

   // private copy constructor and operator= (define these and make them
   // public to enable copy and assignment of the class)
   cgiHTMLHelper(const cgiHTMLHelper&);
   cgiHTMLHelper& operator=(const cgiHTMLHelper&);

};

#endif

// ****************************************************************************

// end of htmlhelp.h
