// ****************************************************************************
//
// Module:  htmlhelp.C
// Author:  Dick lam
//
// Purpose: C++ class source file for cgiHTMLHelper
//
// Notes:  This is a base class. It contains helper methods for generating
//         HTML documents. Some custom iostream manipulators are also included
//         for dynamically constructing HTML using the appropriate tags. You
//         can use the manipulators with any stream as follows:
//
//            cout << htmltitle("A Sample Title") << endl
//                 << htmlhead("A Sample Heading") << endl
//                 << htmlh1("Heading level 1") << endl
//                 << htmlpara << "This is a link to the " <<
//                 << htmlbeginlink("http://www.ibm.com")
//                 << "IBM home page"
//                 << htmlendlink
//                 << "." << endl;
//
//         Which will result in the following output:
//
//            <title>A Sample Title</title>
//            <head>A Sample Heading</head>
//            <h1>Heading level 1</h1>
//            <p>This is a link to the
//            <a href="http://www.ibm.com">IBM home page</a>.
//
// ****************************************************************************

#include <cgi/htmlhelp.h>

// ****************************************************************************

// iostream manipulators for HTML output

// TITLE
static ostream& htitle(ostream& os, const char *title)
   { return os << "<title>" << title << "</title>" << endl; }
omanip<const char *> htmltitle(const char *title)
   { return omanip<const char *>(htitle, title); }

// HEAD
static ostream& hhead(ostream& os, const char *head)
   { return os << "<head>" << head << "</head>" << endl; }
omanip<const char *> htmlhead(const char *head)
   { return omanip<const char *>(hhead, head); }

// H1
static ostream& h1(ostream& os, const char *heading)
   { return os << "<h1>" << heading << "</h1>" << endl; }
omanip<const char *> htmlh1(const char *heading)
   { return omanip<const char *>(h1, heading); }

// H2
static ostream& h2(ostream& os, const char *heading)
   { return os << "<h2>" << heading << "</h2>" << endl; }
omanip<const char *> htmlh2(const char *heading)
   { return omanip<const char *>(h2, heading); }

// H3
static ostream& h3(ostream& os, const char *heading)
   { return os << "<h3>" << heading << "</h3>" << endl; }
omanip<const char *> htmlh3(const char *heading)
   { return omanip<const char *>(h3, heading); }

// H4
static ostream& h4(ostream& os, const char *heading)
   { return os << "<h4>" << heading << "</h4>" << endl; }
omanip<const char *> htmlh4(const char *heading)
   { return omanip<const char *>(h4, heading); }

// H5
static ostream& h5(ostream& os, const char *heading)
   { return os << "<h5>" << heading << "</h5>" << endl; }
omanip<const char *> htmlh5(const char *heading)
   { return omanip<const char *>(h5, heading); }

// H6
static ostream& h6(ostream& os, const char *heading)
   { return os << "<h6>" << heading << "</h6>" << endl; }
omanip<const char *> htmlh6(const char *heading)
   { return omanip<const char *>(h6, heading); }

// line BReak
ostream& htmlbreak(ostream& os) { return os << "<br>" << endl; }

// P (paragraph)
ostream& htmlpara(ostream& os) { return os << "<p>"; }

// HR (horizontal rule)
ostream& htmlrule(ostream& os) { return os << "<hr>" << endl; }

// Anchor
static ostream& hanchor(ostream& os, const char *link)
   { return os << "<a href=\"" << link << "\">"; }
omanip<const char *> htmlbeginlink(const char *link)
   { return omanip<const char *>(hanchor, link); }

// Named link
static ostream& hnamed(ostream& os, const char *link)
   { return os << "<a name=\"" << link << "\">"; }
omanip<const char *> htmlnamedlink(const char *link)
   { return omanip<const char *>(hnamed, link); }

// Anchor and Named link end
ostream& htmlendlink(ostream& os) { return os << "</a>"; }

// Bulleted list
ostream& htmllist(ostream& os) { return os << "<ul>" << endl; }
ostream& htmlendlist(ostream& os) { return os << "</ul>" << endl; }

// Ordered list
ostream& htmlordered(ostream& os) { return os << "<ol>" << endl; }
ostream& htmlendordered(ostream& os) { return os << "</ol>" << endl; }

// List item
static ostream& hitem(ostream& os, const char *listitem)
   { return os << "<li>" << listitem << endl; }
omanip<const char *> htmllistitem(const char *listitem)
   { return omanip<const char *>(hitem, listitem); }

// Definition list
ostream& htmldeflist(ostream& os) { return os << "<dl>" << endl; }
ostream& htmlenddeflist(ostream& os) { return os << "</dl>" << endl; }

// Definition list term
static ostream& hterm(ostream& os, const char *term)
   { return os << "<dt>" << term << endl; }
omanip<const char *> htmlterm(const char *term)
   { return omanip<const char *>(hterm, term); }

// Definition list definition
static ostream& hdefine(ostream& os, const char *definition)
   { return os << "<dd>" << definition << endl; }
omanip<const char *> htmldefinedas(const char *definition)
   { return omanip<const char *>(hdefine, definition); }

   // Preformatted
ostream& htmlpreformatted(ostream& os) { return os << "<pre>" << endl; }
ostream& htmlendpreformatted(ostream& os) { return os << "</pre>" << endl; }

// BLOCKQUOTE
static ostream& hbq(ostream& os, const char *quote)
   { return os << "<blockquote>" << quote << "</blockquote>" << endl; }
omanip<const char *> htmlblockquote(const char *quote)
   { return omanip<const char *>(hbq, quote); }

// ADDRESS
static ostream& haddr(ostream& os, const char *address)
   { return os << "<address>" << address << "</address>" << endl; }
omanip<const char *> htmladdress(const char *address)
   { return omanip<const char *>(haddr, address); }

// special characters
ostream& htmllessthan(ostream& os) { return os << "&lt;"; }
ostream& htmlgreaterthan(ostream& os) { return os << "&gt;"; }
ostream& htmlampersand(ostream& os) { return os << "&amp;"; }
ostream& htmldoublequote(ostream& os) { return os << "&quot;"; }

// text styles
ostream& htmlbold(ostream& os) { return os << "<B>"; }
ostream& htmlendbold(ostream& os) { return os << "</B>"; }

ostream& htmlitalics(ostream& os) { return os << "<I>"; }
ostream& htmlenditalics(ostream& os) { return os << "</I>"; }

ostream& htmlcode(ostream& os) { return os << "<CODE>"; }
ostream& htmlendcode(ostream& os) { return os << "</CODE>"; }

// in line IMaGe
static ostream& himg(ostream& os, const char *image)
   { return os << "<img src=\"" << image << "\">"; }
omanip<const char *> htmlimage(const char *image)
   { return omanip<const char *>(himg, image); }

// FORM
static ostream& hformget(ostream& os, const char *link)
   { return os << "<FORM METHOD=\"GET\" ACTION=\"" << link << "\">" << endl; }
omanip<const char *> htmlformget(const char *link)
   { return omanip<const char *>(hformget, link); }

static ostream& hformpost(ostream& os, const char *link)
   { return os << "<FORM METHOD=\"POST\" ACTION=\"" << link << "\">" << endl; }
omanip<const char *> htmlformpost(const char *link)
   { return omanip<const char *>(hformpost, link); }

ostream& htmlendform(ostream& os) { return os << "</FORM>"; }


// ****************************************************************************

// cgiHTMLHelper - constructor

cgiHTMLHelper::cgiHTMLHelper(ostream& os)
   : output(os)

{
   // init instance variables
}

// ----------------------------------------------------------------------------

// ~cgiHTMLHelper - destructor

cgiHTMLHelper::~cgiHTMLHelper()

{
}

// ----------------------------------------------------------------------------

// contentType - outputs the content type

void cgiHTMLHelper::contentType(const char *type)

{
   output << "Content-type: " << type << endl << endl;
}

// ----------------------------------------------------------------------------

// header - writes recommended header

void cgiHTMLHelper::header(const char *title)

{
   output << "<html>" << endl
          << htmlhead(title)
          << htmltitle(title)
          << "<body>" << endl;
}

// ----------------------------------------------------------------------------

// trailer - writes recommended trailer

void cgiHTMLHelper::trailer()

{
   output << "</body>" << endl
          << "</html>" << endl;
}

// ----------------------------------------------------------------------------

// hidden - writes a name, value pair

void cgiHTMLHelper::hidden(const char *name, const char *value)

{
   // make sure input is ok
   if (!name || !value)
      return;

   output << "<input type=\"hidden\" name=\"" << name
          << "\" value=\"" << value << "\">" << endl;
}

// ----------------------------------------------------------------------------

// entryfield - creates an input text field

void cgiHTMLHelper::entryfield(const char *name, const char *value, int len,
                               int maxlength)

{
   // make sure input is ok
   if ( !name || !value || (len < 1) || (maxlength < 0) )
      return;

   output << "<input type=\"text\" name=\"" << name
          << "\" value=\"" << value
          << "\" size=" << len;

   if (maxlength == 0)
      output << " maxlength=" << maxlength << ">" << endl;
   else
      output << ">" << endl;
}

// ----------------------------------------------------------------------------

// password - creates an input text field for a password

void cgiHTMLHelper::password(const char *name, const char *value, int len,
                             int maxlength)

{
   // make sure input is ok
   if ( !name || !value || (len < 1) || (maxlength < 0) )
      return;

   output << "<input type=\"password\" name=\"" << name
          << "\" value=\"" << value
          << "\" size=" << len;

   if (maxlength == 0)
      output << " maxlength=" << maxlength << ">" << endl;
   else
      output << ">" << endl;
}

// ----------------------------------------------------------------------------

// checkbox - creates a checkbox

void cgiHTMLHelper::checkbox(const char *name, const char *label, bool checked)

{
   // make sure input is ok
   if (!name || !label)
      return;

   output << "<input type=\"checkbox\" name=\"" << name
          << "\" value=\"" << label
          << "\"";

   if (checked)
      output << " checked> " << label << endl;
   else
      output << "> " << label << endl;
}

// ----------------------------------------------------------------------------

// radiobutton - creates a radiobutton

void cgiHTMLHelper::radiobutton(const char *name, const char *label,
                                bool selected)

{
   // make sure input is ok
   if (!name || !label)
      return;

   output << "<input type=\"radio\" name=\"" << name
          << "\" value=\"" << label
          << "\"";

   if (selected)
      output << " checked> " << label << endl;
   else
      output << "> " << label << endl;
}

// ----------------------------------------------------------------------------

// submitbutton - creates a submit pushbutton

void cgiHTMLHelper::submitbutton(const char *label)

{
   // make sure input is ok
   if (!label)
      return;

   output << "<input type=\"submit\" value=\"" << label << "\">" << endl;
}

// ----------------------------------------------------------------------------

// resetbutton - creates a reset pushbutton

void cgiHTMLHelper::resetbutton(const char *label)

{
   // make sure input is ok
   if (!label)
      return;

   output << "<input type=\"reset\" value=\"" << label << "\">" << endl;
}

// ----------------------------------------------------------------------------

// texteditor - creates a multi-line entryfield

void cgiHTMLHelper::texteditor(const char *name, int rows, int cols,
                               const char *text)

{
   // make sure input is ok
   if (!name || !text)
      return;

   output << "<textarea name=\"" << name
          << "\" rows=" << rows
          << " cols=" << cols << ">" << endl
          << text << endl
          << "</textarea>" << endl;
}

// ----------------------------------------------------------------------------

// listbox - creates a new listbox

void cgiHTMLHelper::listbox(const char *name, int size, bool multiselect)

{
   // make sure input is ok
   if ( !name || (size < 0) )
      return;

   output << "<select name=\"" << name
          << "\" size=" << size;

   if (multiselect)
      output << " multiple>" << endl;
   else
      output << ">" << endl;
}

// ----------------------------------------------------------------------------

// listoption - adds an option to the listbox

void cgiHTMLHelper::listoption(const char *option, bool selected)

{
   // make sure input is ok
   if (!option)
      return;

   if (selected)
      output << "<option selected> " << option << endl;
   else
      output << "<option> " << option << endl;
}

// ----------------------------------------------------------------------------

// endlistbox - ends the definition of a listbox

void cgiHTMLHelper::endlistbox()

{
   output << "</select>" << endl;
}

// ----------------------------------------------------------------------------

// imagemap - creates an imagemap

void cgiHTMLHelper::imagemap(const char *name, const char *link)

{
   // make sure input is ok
   if (!name || !link)
      return;

   output << "<input type=\"image\" name=\"" << name
          << "\" src=\"" << link << "\">" << endl;
}

// ****************************************************************************

// end of htmlhelp.C
