/********************  EMS.C  ---  Listing 1 *****************
*
*    This listing contains a module of functions that may be called by a C
*    program to manage expanded memory. To compile under Microsoft C, 
*    define the constant MC on the compiler command line (use the /DMC flag).
*    To compile under Turbo C, define TC (-DTC flag).
*
*    Copyright (C) Michael J. Young, 1989.  All rights reserved.
*    May be used freely for non-commercial purposes.
**************************************************************/

#include <DOS.H>
#include <STDDEF.H>

#include "EMS.H"

static int Allocated = 0;          /* Flag indicates EMS memory allocated.  */
static int Handle;                 /* The EMM handle.                       */

char far *EmsAlloc (int Pages)
/*
     This function allocates the number of logical EMS pages specified by
     the parameter 'Pages'.  If successful, it returns a far pointer to the
     base of the EMS page frame;  if an error occurs, it returns NULL.
*/
{
     char far *FarPtrCh;                /* Temporarily holds return value.  */
     union REGS Reg;

     if (!EmsInstalled ())              /* First test if EMM installed.     */
          return (NULL);

     if (Allocated)                     /* Make sure that pages have not    */
          return (NULL);                /* already been allocated.          */

     Reg.x.bx = Pages;                  /* BX specifies number of pages.    */
     Reg.h.ah = 0x43;                   /* EMM function to allocate pages.  */
     int86 (0x67,&Reg,&Reg);

     if (Reg.h.ah)                      /* Test for error.                  */
          return (NULL);                /* NULL return indicates error.     */

     Handle = Reg.x.dx;                 /* Store EMM handle from DX.        */
     Allocated = 1;                     /* Set flag to indicate that EMS    */
                                        /* memory has been allocated.       */

     Reg.h.ah = 0x41;                   /* EMM function to return segment   */
     int86 (0x67,&Reg,&Reg);            /* address of EMS page frame.       */

     if (Reg.h.ah)                      /* Test for error.                  */
          return (NULL);                /* NULL return indicates an error.  */

#ifdef MC
     FP_SEG (FarPtrCh) = Reg.x.bx;      /* Segment address is in BX.        */
     FP_OFF (FarPtrCh) = 0;
#endif
#ifdef TC
     FarPtrCh = MK_FP (Reg.x.bx,0);
#endif

     return (FarPtrCh);
}    /* end EmsAlloc */

int EmsFree (void)
/*
     This function frees all pages of EMS memory that have been allocated by
     a prior call to 'EmsAlloc'.  If successful, it returns zero;  if an error
     occurs, it returns -1.
*/
{
     union REGS Reg;

     if (!EmsInstalled ())              /* First test if EMM installed.     */
          return -1;

     if (!Allocated)                    /* Test that EMS memory has been    */
          return -1;                    /* successfully allocated.          */

     Reg.x.dx = Handle;                 /* EMM handle goes in DX.           */
     Reg.h.ah = 0x45;                   /* EMM function to free pages.      */
     int86 (0x67,&Reg,&Reg);

     if (Reg.h.ah)                      /* Test for error.                  */
          return (-1);                  /* Return error code.               */

     Allocated = 0;                     /* Function successful.             */
     return (0);

}    /* end EmsFree */

int EmsInstalled (void)
/*
     This function returns a non-zero value if the expanded memory manager
     is installed, or zero if it is not installed.
*/
{
     static char *EmmName = "EMMXXXX0"; /* EMM code in device name field.   */
     char far *EmmPtr;                  /* Pointer to device header.        */

#ifdef MC
     EmmPtr = (char far *)_dos_getvect (0x67);  /* Address of device header.*/
#endif
#ifdef TC
     EmmPtr = (char far *)getvect (0x67);
#endif

     FP_OFF (EmmPtr) = 0x000a;                  /* Adjust offset to name.   */

     while (*EmmName)                   /* Test if name field matches the   */
          if (*EmmName++ != *EmmPtr++)  /* EMM code.                        */
               return (0);              /* Mismatch:  return false.         */

     return (1);                        /* All characters matched.          */

}    /* end EmsInstalled */

int EmsMap (int Page0,int Page1,int Page2,int Page3)
/*
     This function maps the four logical pages specified by the parameters
     onto the four physical pages. If a parameter has the value -1,
     the mapping of the corresponding page is left unaltered. If successful,
     the function returns zero; if an error occurs, it returns -1.
*/
{
     register int i;
     register int Page;
     union REGS Reg;

     if (!EmsInstalled ())              /* First test if EMM installed.     */
          return (-1);

     if (!Allocated)                    /* Test that EMS memory has been    */
          return (-1);                  /* successfully allocated.          */

     for (i = 0; i < 4; ++i)            /* Four pages to map.               */
          {
          if (i == 0)                   /* Select one parameter with each   */
               Page = Page0;            /* iteration of the loop.           */
          else if (i == 1)
               Page = Page1;
          else if (i == 2)
               Page = Page2;
          else if (i == 3)
               Page = Page3;
          if (Page == -1)
               continue;

          Reg.x.bx = Page;              /* Logical page number.             */
          Reg.x.dx = Handle;            /* EMM handle.                      */
          Reg.h.al = (unsigned char) i; /* Physical page number.            */
          Reg.h.ah = 0x44;              /* EMM function to map logical      */
          int86 (0x67,&Reg,&Reg);       /* pages onto physical pages.       */

          if (Reg.h.ah)                 /* Test for error.                  */
               return (-1);             /* Return actual error code.        */
          }

     return (0);

} /* end EmsMap */

int EmsPagesAvail (void)
/*
     This function returns the number of free logical pages of EMS memory
     that can be allocated.  If an error occurs, it returns -1.
*/
{
     union REGS Reg;

     if (!EmsInstalled ())              /* First test if EMM installed.     */
          return (-1);

     Reg.h.ah = 0x42;                   /* EMM function to return number of */
     int86 (0x67,&Reg,&Reg);            /* free pages.                      */

     if (Reg.h.ah)                      /* Test for error.                  */
          return (-1);                  /* -1 return indicates an error.    */

     return (Reg.x.bx);                 /* BX contains the number of pages. */

} /* end EmsPagesAvail */
