/*
 *  SADDRESS.C
 *
 *  This module contains functions associated with the address dialog box
 *  in SAMPLE.EXE.
 *
 *  Copyright (C) 1991 by Daytris.  All rights reserved.
 */

#include <windows.h>
#include <stdlib.h>
#include <string.h>
#ifndef ZORTECH
#include <memory.h>
#endif
#include "dbmgr.h"
#include "sampledb.h"
#include "sample.h"


/************************************************
 * Local data
 ************************************************/

static ADDRESS address;

static enum 
    {
    MODE_ADD,
    MODE_UPDATE,
    MODE_DELETE
    } eMode;


/************************************************
 * Function Declarations
 ************************************************/

BOOL AddAddressDlg( HWND hWnd);
BOOL UpdateAddressDlg( HWND hWnd);
BOOL DeleteAddressDlg( HWND hWnd);
static BOOL GetSelectedAddress( HWND hWnd, ADDRESS *pAddress, short *pIndex);
BOOL FAR PASCAL AddressProc( HWND hDlg, unsigned iMessage, WORD wParam,
    LONG lParam);
static void SetAddressFields( HWND hDlg);
static BOOL GetAddressFields( HWND hDlg);


/***************************************************************************
 * Function : AddAddressDlg
 *
 * Purpose  : This function drives the Add Address dialog box.  The
 *            database is not updated, only the address listbox.
 *
 * Returns  : TRUE  - address added
 *            FALSE - address not added
 ***************************************************************************/
BOOL AddAddressDlg( HWND hWnd)
    {
    short nStatus;
    FARPROC lpfnAddressProc;

    eMode = MODE_ADD;

    /* Initialize the address structure */
    memset( &address, 0, sizeof( ADDRESS));

    /* Create an instance and open the Address window */
    lpfnAddressProc = MakeProcInstance( AddressProc, hInst);
    nStatus = DialogBox( hInst, "address", hWnd, lpfnAddressProc);
    FreeProcInstance( lpfnAddressProc);

    /* User selected OK */
    if( nStatus == IDOK)
        {
        /* Add the address to listbox in the Client dialog */
        if( ! AddToAddressListBox( hWnd, &address))
            return FALSE;
        }

    return TRUE;
    }


/***************************************************************************
 * Function : UpdateAddressDlg
 *
 * Purpose  : This function drives the Update Address dialog box.  The
 *            database is not updated, only the address listbox.
 *
 * Returns  : TRUE  - address updated
 *            FALSE - address not updated
 ***************************************************************************/
BOOL UpdateAddressDlg( HWND hWnd)
    {
    short nStatus, nIndex;
    FARPROC lpfnAddressProc;

    eMode = MODE_UPDATE;

    /* Retrieve the selected address */
    if( ! GetSelectedAddress( hWnd, &address, &nIndex))
        {
        MessageBeep( 0);
        return FALSE;
        }

    /* Create an instance and open the Address window */
    lpfnAddressProc = MakeProcInstance( AddressProc, hInst);
    nStatus = DialogBox( hInst, "address", hWnd, lpfnAddressProc);
    FreeProcInstance( lpfnAddressProc);

    /* User selected OK */
    if( nStatus == IDOK)
        {
        /* Update the address */
        if( ! DeleteFromAddressListBox( hWnd, nIndex))
            return FALSE;
        if( ! AddToAddressListBox( hWnd, &address))
            return FALSE;
        }

    return TRUE;
    }


/***************************************************************************
 * Function : DeleteAddressDlg
 *
 * Purpose  : This functions queries about deleting the selected address
 *            record.  The database is not updated, only the address
 *            listbox.
 *
 * Returns  : TRUE  - address deleted
 *            FALSE - address not deleted
 ***************************************************************************/
BOOL DeleteAddressDlg( HWND hWnd)
    {
    short nStatus, nIndex;

    /* Retrieve the selected address */
    if( ! GetSelectedAddress( hWnd, &address, &nIndex))
        {
        MessageBeep( 0);
        return FALSE;
        }

    /* Ask user if they're sure */
    nStatus = MessageBox( hWnd, "Are you sure?", "Delete Address",
        MB_ICONQUESTION | MB_YESNO);

    /* User selected YES */
    if( nStatus == IDYES)
        {
        /* Remove the address from the listbox */
        if( ! DeleteFromAddressListBox( hWnd, nIndex))
            return FALSE;
        }

    return TRUE;
    }


/***************************************************************************
 * Function : GetSelectedAddress
 *
 * Purpose  : This function retrieves the selected address record from
 *            memory.  The handle to the address is stored in the listbox
 *            string.  The selected index is also returned.
 *
 * Returns  : TRUE  - address retrieved
 *            FALSE - error
 ***************************************************************************/
static BOOL GetSelectedAddress( HWND hWnd, ADDRESS *pAddress, short *pIndex)
    {
    HANDLE hAddress;
    ADDRESS FAR *lpAddress;

    /* Get the listbox selection */
    if( (*pIndex = (short)SendMessage( GetDlgItem( hWnd, IDC_ADDR_LISTBOX),
        LB_GETCURSEL, 0, 0L)) == LB_ERR)
        {
        return FALSE;
        }
    hAddress = GetAddressHandle( hWnd, *pIndex);

    /* Copy the Address info */
    if( (lpAddress = (ADDRESS FAR *)GlobalLock( hAddress)) == NULL)
        {
        MessageBox( hWnd, "Memory: GlobalLock", "Fatal Error",
            MB_ICONEXCLAMATION | MB_OK);
        return FALSE;
        }
    _fmemcpy( pAddress, lpAddress, sizeof( ADDRESS));
    GlobalUnlock( hAddress);

    return TRUE;
    }


/***************************************************************************
 * Function : AddressProc
 *
 * Purpose  : This function is the window procedure for 'add' and 'update'
 *            address.
 *
 * Returns  : TRUE  - message processed
 *            FALSE - message not processed
 ***************************************************************************/
BOOL FAR PASCAL AddressProc( HWND hDlg, unsigned iMessage, WORD wParam,
    LONG lParam)
    {
    switch( iMessage)
        {
        case WM_INITDIALOG:
            if( eMode == MODE_ADD)
                SetWindowText( hDlg, "Add Address");
            else
                SetWindowText( hDlg, "Update Address");

            SetAddressFields( hDlg);

            /* Set focus to Street field */
            SetFocus( GetDlgItem( hDlg, IDC_EDIT_STREET));
            return TRUE;

        case WM_COMMAND:
            switch( wParam)
                {
                case IDOK:
                    if( ! GetAddressFields( hDlg))
                        break;
                    EndDialog( hDlg, IDOK);
                    break;

                case IDCANCEL:
                    EndDialog( hDlg, IDCANCEL);
                    break;
                }
            return TRUE;
        }

    return FALSE;
    }


/***************************************************************************
 * Function : SetAddressFields
 *
 * Purpose  : This function initializes the address dialog fields.  It
 *            uses the static 'address' structure.
 *
 * Returns  : n/a
 ***************************************************************************/
static void SetAddressFields( HWND hDlg)
    {
    SetDlgItemText( hDlg, IDC_EDIT_STREET, address.szStreet);
    SetDlgItemText( hDlg, IDC_EDIT_CITY, address.szCity);
    SetDlgItemText( hDlg, IDC_EDIT_STATE, address.szState);
    SetDlgItemText( hDlg, IDC_EDIT_ZIP, address.szZip);
    SetDlgItemText( hDlg, IDC_EDIT_TEL, address.szTelephone);
    SetDlgItemText( hDlg, IDC_EDIT_FAX, address.szFax);
    }


/***************************************************************************
 * Function : GetAddressFields
 *
 * Purpose  : This function retrieves the address dialog fields.  It
 *            also performs minor field error checking.  Address fields
 *            are stored in the static 'address' structure.
 *
 * Returns  : TRUE  - fields ok
 *            FALSE - error in field (focus is set)
 ***************************************************************************/
static BOOL GetAddressFields( HWND hDlg)
    {
    GetDlgItemText( hDlg, IDC_EDIT_STREET, address.szStreet,
        sizeof( address.szStreet));
    if( ! *address.szStreet)
        {
        MessageBox( hDlg, "Street required", "Invalid Input",
            MB_APPLMODAL | MB_ICONEXCLAMATION | MB_OK);
        SetFocus( GetDlgItem( hDlg, IDC_EDIT_STREET));
        return FALSE;
        }
    GetDlgItemText( hDlg, IDC_EDIT_CITY, address.szCity,
        sizeof( address.szCity));
    if( ! *address.szCity)
        {
        MessageBox( hDlg, "City required", "Invalid Input",
            MB_APPLMODAL | MB_ICONEXCLAMATION | MB_OK);
        SetFocus( GetDlgItem( hDlg, IDC_EDIT_CITY));
        return FALSE;
        }
    GetDlgItemText( hDlg, IDC_EDIT_STATE, address.szState,
        sizeof( address.szState));
    if( ! *address.szState)
        {
        MessageBox( hDlg, "State required", "Invalid Input",
            MB_APPLMODAL | MB_ICONEXCLAMATION | MB_OK);
        SetFocus( GetDlgItem( hDlg, IDC_EDIT_STATE));
        return FALSE;
        }
    GetDlgItemText( hDlg, IDC_EDIT_ZIP, address.szZip,
        sizeof( address.szZip));
    if( ! *address.szZip)
        {
        MessageBox( hDlg, "Zip Code required", "Invalid Input",
            MB_APPLMODAL | MB_ICONEXCLAMATION | MB_OK);
        SetFocus( GetDlgItem( hDlg, IDC_EDIT_ZIP));
        return FALSE;
        }
    GetDlgItemText( hDlg, IDC_EDIT_TEL, address.szTelephone,
        sizeof( address.szTelephone));
    GetDlgItemText( hDlg, IDC_EDIT_FAX, address.szFax,
        sizeof( address.szFax));

    return TRUE;
    }
