/*  $Id$
 *  
 *  File	latex.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	LaTeX utility
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	05/10/93  (Created)
 *  		15/10/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"

char *		UTILITY_NAME = "LaTeX utility";
char *		UTILITY_VERSION = "1.1.0";

void		helpUtility(FILE *fd);

static void	latexGame(Game g, FILE *fd, int);
static void	startLatexGame(Game, TextBuffer);
static void	endLatexGame(Game, TextBuffer);



/*------------------------------------------------------------
 *  Verify utility
 *------------------------------------------------------------*/

void
latexUtility(Option opt)
{ Game g = newGame();
  CBase cb = opt->database;
  long from = opt->from;
  long to = (opt->to < getNoGamesCBase(cb) ? opt->to : getNoGamesCBase(cb));
  long n;

  reportCBase(cb, stderr);

  n = to-from+1;
  if (n < 0)
    return;
  
  for (n=from; n<=to; n++)
  { environmentError(cb, g, n);
    initialiseGame(g, n, cb);
    if (foundError())
    { reportError(stderr);
      continue;
    }
    latexGame(g, opt->output, opt->notation);
    if (foundError())
      reportError(stderr);
  }

  freeGame(g);
}


/*------------------------------------------------------------
 *  Printing games in informant style
 *------------------------------------------------------------*/

static void
latexGame(Game g, FILE *fd, int notation)
{ TextBuffer tb;
  Position pos;

  tb = newTextBuffer(fd, 75);

  startLatexGame(g, tb);

  pos = newPosition();
  if (!fullGameP(g))
  { copyPosition(pos, g->position);
    diagramPosition(pos, tb);
  }

  printMovesGame(g, getFirstMoveGame(g), pos, 0, 0, notation, tb);

  endLatexGame(g, tb);

  freeTextBuffer(tb);
}


/*------------------------------------------------------------
 *  Printing the header and the trailer of an informant game
 *------------------------------------------------------------
 *
 *  This prints the header of a game in the style of the Chess
 *  Informant.  It assumes LaTeX macros called ``informantgame'' and
 *  ``informantmainline'' have been defined.
 */

static void
startLatexGame(Game g, TextBuffer tb)
{ char *eco = getEcoGame(g);
  char *white = getWhiteGame(g);
  char *black = getBlackGame(g);
  char *source = getSourceGame(g);
  int year = getYearGame(g);
  char *result;
  char *annotator = "";
  char *reference = "";

  if (eco == NULL) eco = "";
  if (white == NULL) white = "White (NN)";
  if (black == NULL) black = "Black (NN)";
  switch (getResultGame(g))
  { case WHITE_WINS:  result = chessSymbol(WHITE_WINS); break;
    case DRAW:        result = chessSymbol(DRAW); break;
    case BLACK_WINS:  result = chessSymbol(BLACK_WINS); break;
    default:	      result = "Line";
  }

  formatRawTextBuffer(tb, "\n\\informantgame{%s}{%s}{%s}{%s}{%s %d}",
		      eco, reference, white, black, source, year);
  formatRawTextBuffer(tb, "\\begin{informantmainline}{%s}{%s}",
		      result, annotator);
}


static void
endLatexGame(Game g, TextBuffer tb)
{ formatRawTextBuffer(tb, "\\end{informantmainline}");
}


/*------------------------------------------------------------
 *  Main
 *------------------------------------------------------------*/

int
main(int argc, char *argv[])
{ int i;
  Option options = newOption();

  initChessSymbols();

  for (i=1; i<argc; i++)
  {
    if (strhead(argv[i], "-insert"))	/* -insert */
    { FILE *fd;

      ++i;
      fd = fileArgument(argv[i], "-insert", argc, i, "r");
      { char line[MAX_LINE_SIZE+1];

	while (fgets(line, MAX_LINE_SIZE, fd))
	  fputs(line, options->output);
      }
      fclose(fd);
      continue;
    }

    if (strhead(argv[i], "-"))
    { int n;

      n = genericOption(options, argv, argc, i);
      if (n == 0)
      { fprintf(stderr, "Fatal: Unknown command %s\n", argv[i]);
	fprintf(stderr, "Do ``%s -help'' or see the documentation\n", argv[0]);
	exit(1);
      }
      i = n;
      continue;
    }

    setCurrentCBase(argv[i], "-database", argc, i);
    options->database = CurrentBase;
    latexUtility(options);
    freeCBase(options->database);
    options->database = (CBase) NULL;
  }

  if (options->database)
  { latexUtility(options);
    freeCBase(options->database);
  }

  exit(0);
}


/*------------------------------------------------------------
 *  Help
 *------------------------------------------------------------*/

void
helpUtility(FILE *fd)
{ helpCBUFF(fd);
  fprintf(fd, "%s options:\n", UTILITY_NAME);
  fprintf(fd, "-insert file  Insert contents of file here\n");
}
