/*  $Id: cbuff.h,v 1.1.1.1 1993/06/21 11:11:59 anjo Exp $
 *  
 *  File	cbuff.h
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Horst Aurisch, aurisch@informatik.uni-bonn.de
 *              Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	Definitions
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	08/06/93  (Created)
 *  		10/11/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <setjmp.h>
#include <assert.h>
#include <ctype.h>
#include <stdarg.h>

#include "machine.h"


/*------------------------------------------------------------
 *  Compatibility issues
 *------------------------------------------------------------*/

/*  ``Move'' is already defined by THINK_C on the Macintosh.
 *  (Reported by Rolf Exner).
 */

#ifdef THINK_C
#	define Move	MMove
#endif


/*------------------------------------------------------------
 *  Constants
 *------------------------------------------------------------*/

#ifndef TRUE
#	define TRUE	1
#	define FALSE	0
#endif

#define NO_RESULT	0
#define NO_ROUND	0

#define REQUEST_DIAGRAM	'\004'

#define LONG_ALGEBRAIC	1
#define SHORT_ALGEBRAIC	2

extern char *	NullString;		/* Equals "" */


/*------------------------------------------------------------
 *  Global data
 *------------------------------------------------------------*/

extern unsigned short UserId;		/* ChessBase uid; option.c */


/*------------------------------------------------------------
 *  Type definitions
 *------------------------------------------------------------*/

typedef int			bool;		/* TRUE or FALSE */
typedef struct cbase *		CBase;		/* ChessBase database */
typedef struct game *		Game;		/* Chess game */
typedef struct move *		Move;		/* Chess move */
typedef struct cbheader *	CbHeader;	/* ChessBase game header */
typedef struct cbgame *		CbGame;		/* ChessBase game */
typedef struct option *		Option;		/* Command line options */
typedef struct position *	Position;	/* Chess position */
typedef unsigned char		Square;		/* Square (0..63) */
typedef unsigned char		Piece;		/* Piece in position */
typedef int			Colour;		/* WHITE or BLACK */
typedef struct textbuffer *	TextBuffer;     /* To print formatted text */
typedef int			Result;		/* Result of a game */


/*------------------------------------------------------------
 *  Chess pieces
 *------------------------------------------------------------*/

#define NO_PIECE	0

#define WHITE_MASK	0
#define BLACK_MASK	8
#define PIECE_MASK	7

#define KING		1
#define QUEEN		2
#define KNIGHT		3
#define BISHOP		4
#define ROOK		5
#define PAWN		6


#define WHITE_KING	(KING | WHITE_MASK)
#define WHITE_QUEEN	(QUEEN | WHITE_MASK)
#define WHITE_KNIGHT	(KNIGHT | WHITE_MASK)
#define WHITE_BISHOP	(BISHOP | WHITE_MASK)
#define WHITE_ROOK	(ROOK | WHITE_MASK)
#define WHITE_PAWN	(PAWN | WHITE_MASK)

#define BLACK_KING	(KING | BLACK_MASK)
#define BLACK_QUEEN	(QUEEN | BLACK_MASK)
#define BLACK_KNIGHT	(KNIGHT | BLACK_MASK)
#define BLACK_BISHOP	(BISHOP | BLACK_MASK)
#define BLACK_ROOK	(ROOK | BLACK_MASK)
#define BLACK_PAWN	(PAWN | BLACK_MASK)

#define isPieceP(x)		((x) & PIECE_MASK)
#define isWhitePieceP(x)	((x) >= WHITE_KING && (x) <= WHITE_PAWN)
#define isBlackPieceP(x)	((x) >= BLACK_KING && (x) <= BLACK_PAWN)
#define isKingP(x)		(((x) & PIECE_MASK) == KING)
#define isQueenP(x)		(((x) & PIECE_MASK) == QUEEN)
#define isKnightP(x)		(((x) & PIECE_MASK) == KNIGHT)
#define isBishopP(x)		(((x) & PIECE_MASK) == BISHOP)
#define isRookP(x)		(((x) & PIECE_MASK) == ROOK)
#define isPawnP(x)		(((x) & PIECE_MASK) == PAWN)


/*----------------------------------------------
 *  Chess squares
 *----------------------------------------------
 *
 *  Below are the definitions related to squares on the chess board.
 *  There are not many design considerations about squares, the most
 *  obvious representation is as a number.  Given that an ordered
 *  collection of squares is a board, squares are numbered from 0 (a1)
 *  to 63 (h8).
 *
 *  Representation.  A square is represented as a small integer, where
 *  the three lowest bits represented the file (a..h) and the next
 *  three bits the rank (1..8).
 *
 *  Notes.
 *  (1) Always refer to a number representing a square as of type
 *      Square for reasons of clarity.
 *  (2) Symbolic names for all the squares are defined below.  They
 *      are of the form A1 (where A is the character for the column
 *      and 1 the number for the row).
 */


/*----------------------------------------------
 *  Square macro's
 *----------------------------------------------*/

#define newSquare(file, rank)	(((rank) - '1') | (((file) - 'a') << 3))
#define makeSquare(file, rank)	(((file)<<3) | (rank))

#define rankSquare(sq)		((sq) & 0x7)
#define fileSquare(sq)		(((sq)>>3) & 0x7)
#define isFileSquareP(sq, file)	(fileSquare(sq) == (file))
#define isRankSquareP(sq, rank)	(rankSquare(sq) == (rank))

#define for_squares(sq)		for (sq=H8+1; --sq;)


/*----------------------------------------------
 *  Ranks and files
 *----------------------------------------------*/

#define RANK_1		0
#define RANK_2		1
#define RANK_3		2
#define RANK_4		3
#define RANK_5		4
#define RANK_6		5
#define RANK_7		6
#define RANK_8		7

#define A_FILE		0
#define B_FILE		1
#define C_FILE		2
#define D_FILE		3
#define E_FILE		4
#define F_FILE		5
#define G_FILE		6
#define H_FILE		7


/*----------------------------------------------
 *  Square names (rather boring)
 *----------------------------------------------*/

#define A1	newSquare('a','1')
#define A2	newSquare('a','2')
#define A3	newSquare('a','3')
#define A4	newSquare('a','4')
#define A5	newSquare('a','5')
#define A6	newSquare('a','6')
#define A7	newSquare('a','7')
#define A8	newSquare('a','8')

#define B1	newSquare('b','1')
#define B2	newSquare('b','2')
#define B3	newSquare('b','3')
#define B4	newSquare('b','4')
#define B5	newSquare('b','5')
#define B6	newSquare('b','6')
#define B7	newSquare('b','7')
#define B8	newSquare('b','8')

#define C1	newSquare('c','1')
#define C2	newSquare('c','2')
#define C3	newSquare('c','3')
#define C4	newSquare('c','4')
#define C5	newSquare('c','5')
#define C6	newSquare('c','6')
#define C7	newSquare('c','7')
#define C8	newSquare('c','8')

#define D1	newSquare('d','1')
#define D2	newSquare('d','2')
#define D3	newSquare('d','3')
#define D4	newSquare('d','4')
#define D5	newSquare('d','5')
#define D6	newSquare('d','6')
#define D7	newSquare('d','7')
#define D8	newSquare('d','8')

#define E1	newSquare('e','1')
#define E2	newSquare('e','2')
#define E3	newSquare('e','3')
#define E4	newSquare('e','4')
#define E5	newSquare('e','5')
#define E6	newSquare('e','6')
#define E7	newSquare('e','7')
#define E8	newSquare('e','8')

#define F1	newSquare('f','1')
#define F2	newSquare('f','2')
#define F3	newSquare('f','3')
#define F4	newSquare('f','4')
#define F5	newSquare('f','5')
#define F6	newSquare('f','6')
#define F7	newSquare('f','7')
#define F8	newSquare('f','8')

#define G1	newSquare('g','1')
#define G2	newSquare('g','2')
#define G3	newSquare('g','3')
#define G4	newSquare('g','4')
#define G5	newSquare('g','5')
#define G6	newSquare('g','6')
#define G7	newSquare('g','7')
#define G8	newSquare('g','8')

#define H1	newSquare('h','1')
#define H2	newSquare('h','2')
#define H3	newSquare('h','3')
#define H4	newSquare('h','4')
#define H5	newSquare('h','5')
#define H6	newSquare('h','6')
#define H7	newSquare('h','7')
#define H8	newSquare('h','8')


/*------------------------------------------------------------
 *  File alloc.c
 *------------------------------------------------------------*/

void *		alloc(unsigned long);
void		unalloc(void *);
char *		allocCharp(char *);
void		unallocCharp(char *);
void		printMemoryAvailable(FILE *);


/*------------------------------------------------------------
 *  File cbase.c
 *------------------------------------------------------------*/

#define	START_VARIATION		255	/* Indicates start of a variation */
#define END_VARIATION		128	/* Indicates end of a variation */

#define PHYSICALLY_DELETED	(unsigned long) 0x80000000/* Value for index */

#define READ_MODE	1		/* Base opened for reading */
#define WRITE_MODE	2		/* Base opened for writing */


struct cbase
{ char *	name;			/* Name of base */
  FILE *	cbf;			/* ChessBase .cbf file */
  FILE *	cbi;			/* ChessBase .cbi file */
  long		noGames;		/* Number of games */
  unsigned long *index;			/* Start position of each game */
  unsigned long	position;		/* Next index position for write */
  int		mode;			/* READ_MODE, WRITE_MODE */
};


CBase		newCBase(char *, char *);
void		freeCBase(CBase);
long		getNoGamesCBase(CBase);
void		reportCBase(CBase, FILE *);
void		deleteGameCBase(CBase, long);
bool		deletedGameCBaseP(CBase, long);
void		exportGameCBase(CBase, CBase, long);
void		exportManyCBase(CBase, CBase, long, long);
unsigned long	getIndexGameCBase(CBase, long);


/*------------------------------------------------------------
 *  File cbgame.c
 *------------------------------------------------------------*/

struct cbgame
{ CbHeader	header;			/* Game header */
  char *	text;			/* Player and source fields */
  unsigned char *moves;			/* Moves */
  unsigned char *comments;		/* Comments */
  char *	position;		/* Position */
  int		textLength;		/* Bytes in text field */
  int		movesLength;		/* Bytes in moves field */
  int		commentLength;		/* Bytes in comments field */
					/* Remaining fields: partial games */
  Piece		board[64];		/* Starting position */
  short		nextMove;		/* Next move number */
};


CbGame		newCbGame(void);
void		freeCbGame(CbGame);
void		resetCbGame(CbGame);
long		readCbGame(CbGame, CBase, long);
bool		extractHeaderCbGame(CbGame, Game);
bool		extractMovesCbGame(CbGame, Game);
void		dumpCommentsCbGame(CbGame, FILE *);


/*------------------------------------------------------------
 *  File cbheader.c
 *------------------------------------------------------------*/

/*  Field: year
 *
 *  The year the game was played.  The real year is the value found
 *  in the field + 1900.
 */

#define YEAR_UNKNOWN		127


/*  Field: result
 *
 *  Encodes the result of the game.
 */

#define RESULT_MASK		(3<<0)	/* 0: 0-1; 1: 1/2; 2: 1-0; 3: below */
#define EVALUATION_MASK		(0xf<<2)/* If RESULT==3 */
#define RESULT_BIT6_MASK	(1<<6)	/* UNUSED (CBM 34) */
#define RESULT_BIT7_MASK	(1<<7)	/* UNUSED (CBM 34) */


/*  Field: playersLength
 *
 *  The number of characters used for the players names.  The unused
 *  bits are for the ECO code.
 */

#define PLAYERS_MASK		(0x3f<<0)
#define ECO_PART1_MASK		(3<<6)	/* Part 1 of ECO */


/*  Field: sourceLength
 *
 *  The number of characters used for the source.  The unused
 *  bits are for the ECO code.
 */

#define SOURCE_MASK		(0x3f<<0)
#define ECO_PART2_MASK		(3<<6)	/* Part 2 of ECO */


/*  Field: flags1
 *
 *  This field either encodes the castling rights (if the game does
 *  not start from the initial position) or part of the ECO code.
 */

#define FULL_GAME_MASK		(1<<0)	/* 0=game, 1=position */
#define COLOUR_TO_MOVE_MASK	(1<<1)	/* 0=white, 1=black */
#define WHITE_CASTLE_LONG_MASK	(1<<2)	/* For positions */
#define WHITE_CASTLE_SHORT_MASK	(1<<3)	/* For positions */
#define BLACK_CASTLE_LONG_MASK	(1<<4)	/* For positions */
#define BLACK_CASTLE_SHORT_MASK	(1<<5)	/* For positions */
#define GAME_MARKED_MASK	(1<<6)	/* 0=normal, 1=marked */
#define GAME_DELETED_MASK	(1<<7)	/* 0=not deleted, 1=deleted */
#define ECO_PART3_MASK		(0x1f<<1) /* Part 3 of ECO */


/*  Field: flags2
 *
 *  This field contains the ECO-2 (full games) or information
 *  about en passant capability (partial games).
 */

#define FLAGS2_BIT6_MASK	(1<<6)	/* UNKNOWN; VERSION ??? */
#define ECO2_PART1_MASK		(0x1f<<0) /* Part 1 of ECO2 */
#define ECO2_PART2_MASK		(1<<7)	/* Part 2 of ECO2 */
#define EN_PASSANT_MASK		(0xf<<0)/* 1...8: A-H */


/*  Field: moves
 *
 *  Contains the number of moves in the game.
 */

#define NO_MOVES_MASK		(0xff<0)

/*  Field: checksum
 *
 *  This field is a checksum.
 */

#define CORRECT_CHECKSUM	0x01
#define INCORRECT_CHECKSUM	0x00

struct cbheader
{ char		year;
  unsigned char	result;
  short		movesLength;
  unsigned char	playersLength;
  unsigned char	sourceLength;
  short		commentLength;
  unsigned char	whiteElo;
  unsigned char blackElo;
  unsigned char flags1;
  unsigned char	flags2;
  unsigned char	moves;
  unsigned char checksum;
};


CbHeader	newCbHeader(void);
void		freeCbHeader(CbHeader);
void		resetCbHeader(CbHeader);
void		initCbHeader(CbHeader, unsigned char *);
void		dumpCbHeader(CbHeader, FILE *);

bool		fullGameCbHeaderP(CbHeader);
bool		deletedCbHeaderP(CbHeader);
bool		markedCbHeaderP(CbHeader);
int		yearCbHeader(CbHeader);
int		whiteEloCbHeader(CbHeader);
int		blackEloCbHeader(CbHeader);
int		movesLengthCbHeader(CbHeader);
int		playersLengthCbHeader(CbHeader);
int		sourceLengthCbHeader(CbHeader);
int		commentLengthCbHeader(CbHeader);
int		movesCbHeader(CbHeader);
char *		ecoCbHeader(CbHeader);
bool		flags2bit6CbHeader(CbHeader);

void		setYearCbHeader(CbHeader, int);
void		setWhiteEloCbHeader(CbHeader, int);
void		setBlackEloCbHeader(CbHeader, int);

void		decodeCbHeader(CbHeader);


/*------------------------------------------------------------
 *  File cbuff.c
 *------------------------------------------------------------*/

extern char *	UTILITY_NAME;		/* Different for each utility */
extern char *	UTILITY_VERSION;	/* Different for each utility */

void		helpUtility(FILE *);	/* Different for each utility */

extern CBase	CurrentBase;

void		versionCBUFF(FILE *);
void		helpCBUFF(FILE *);

void		setCurrentCBase(char *, char *, int, int);
void		checkCurrentBase(CBase cb, char *);
long		longArgument(char *, char *, int, int);
int		intArgument(char *, char *, int, int);
CBase		databaseArgument(char *, char *, int, int);
CBase		createBaseArgument(char *, char *, int, int);
CBase		appendBaseArgument(char *, char *, int, int);
char *		stringArgument(char *, char *, int, int);
FILE *		fileArgument(char *, char *, int, int, char *);

bool		strhead(char *, char *);


/*------------------------------------------------------------
 *  File file.c
 *------------------------------------------------------------*/

FILE *		fopenExtension(char *, char *, char *, bool);
FILE *		fopenCbuffFile(char *, char *);
long		readLong(FILE *);
void		writeLong(long, FILE *);
char *		getCbuffDirectory(void);


/*------------------------------------------------------------
 *  File game.c
 *------------------------------------------------------------*/

#define	GAME_DELETED		(1 << 0)
#define	GAME_MARKED		(1 << 1)
#define	GAME_INIT_OK		(1 << 2)
#define	GAME_HEADER_OK		(1 << 3)
#define	GAME_MOVES_OK		(1 << 4)
#define	GAME_FLAGS2_BIT6	(1 << 5)
#define GAME_INITIAL_POSITION	(1 << 6)

struct game
{ long		number;			/* Game number relative to base: 1.. */
  CBase		database;		/* ChessBase data-base */
  CbGame	cbGame;			/* ChessBase game representation */
  short		flags;			/* Flags and status information */
  char		players[65];		/* Names of the players */
  char		source[65];		/* Source for game */
  char *	place;			/* Place where game was played */
  char *	event;			/* Event of which game is part */
  char *	annotator;		/* Person who wrote the comments */
  short		year;			/* Year game was played */
  Result	result;			/* Result */
  short		eloWhite;		/* ELO of White */
  short		eloBlack;		/* ELO of Black */
  char		eco[7];			/* Format A00/00 */
  short		moves;			/* Number of full moves */
  short		plies;			/* Number of plies */
  Position	position;		/* Initial position */
  Move		firstMove;		/* Pointer to first move */
};


Game		newGame(void);
void		freeGame(Game);
void		resetGame(Game);
bool		initialiseGame(Game, long, CBase);
unsigned long	bytesGame(Game);

bool		flags2Bit6GameP(Game);
bool		fullGameP(Game);
bool		deletedGameP(Game);
bool		markedGameP(Game);
bool		containsCommentsGameP(Game);
bool		containsVariationsGameP(Game);
bool		containsPlayerNamesGameP(Game);
char *		getPlayersGame(Game);
char *		getSourceGame(Game);
char *		getPlaceGame(Game);
char *		getEventGame(Game);
char *		getAnnotatorGame(Game);
char *		getWhiteGame(Game);
char *		getBlackGame(Game);
long		getNumberGame(Game);
short		getYearGame(Game);
char *		getEcoGame(Game);
char *		getEcoMajorGame(Game);
Move		getFirstMoveGame(Game);
Result		getResultGame(Game);
short		getRoundGame(Game);
short		getMovesGame(Game);
short		getEloWhiteGame(Game);
short		getEloBlackGame(Game);

bool		checkHeaderGame(Game);

void		printGame(Game, FILE *);
void		printMovesGame(Game, Move, Position, int, int, int, TextBuffer);
void		pgnGame(Game, FILE *, int);
void		pgnMovesGame(Game, Move, Position, int, int, TextBuffer);


/*------------------------------------------------------------
 *  File error.c
 *------------------------------------------------------------*/

#define ERR_INCORRECT_CHECKSUM		1
#define ERR_COULD_NOT_CREATE_FILE	2
#define ERR_COULD_NOT_OVERWRITE_FILE	3
#define ERR_COULD_NOT_OPEN_WRITE	4
#define ERR_COULD_NOT_OPEN_READ		5
#define ERR_GAME_NUMBER_OUT_OF_RANGE	6
#define ERR_CANNOT_SEEK              	7
#define ERR_CANNOT_READ_HEADER       	8
#define ERR_CANNOT_READ_TEXT         	9
#define ERR_CANNOT_READ_MOVES        	10
#define ERR_CANNOT_READ_COMMENTS     	11
#define ERR_CANNOT_READ_POSITION       	12
#define ERR_CANNOT_READ_MOVE_NUMBER     13
#define ERR_UNEXPECTED_EOF              14
#define ERR_ILLEGAL_RESULT              15
#define ERR_VARIATION_SEEN		16
#define ERR_CANNOT_INTERPRET_MOVE	17
#define ERR_MOVE_COUNT_INCORRECT	18
#define ERR_COMMENT_NOT_TERMINATED	19
#define ERR_FORMAT_ERROR_ANNOTATIONS	20
#define ERR_UNKNOWN_MOVE_EVALUATION	21
#define ERR_UNKNOWN_POSITION_EVALUATION	22
#define ERR_UNKNOWN_EXTRA_EVALUATION	23
#define ERR_UNKNOWN_CHARACTER		24
#define ERR_UNKNOWN_PIECE		25
#define ERR_UNKNOWN_SYMBOL		26


void		clearError(void);
void		environmentError(CBase, Game, long);
void		setNameError(char *);
void		setIntError(int);
void		setError(int);
bool		foundError(void);
void      	reportError(FILE *);


/*------------------------------------------------------------
 *  File language.c
 *------------------------------------------------------------*/

void		setLanguage(char *);


/*------------------------------------------------------------
 *  File list.c
 *------------------------------------------------------------*/

void		listUtility(Option);
void		listGame(Game g, FILE *fd);


/*------------------------------------------------------------
 *  File move.c
 *------------------------------------------------------------*/

struct move
{ Square       	from;			/* From square */
  Square        to;			/* To square */
  unsigned char piece;			/* Piece promoted to */
  unsigned char moveEvaluation;		/* Evaluation of the move */
  unsigned char	positionEvaluation;	/* Evaluation of the position */
  unsigned char	extraEvaluation;	/* Additional evaluation */
  int		commentLength;		/* Length of comments */
  unsigned char *comments;		/* Comments (annotations) */
  Move		next;			/* Next move */
  Move		alternative;		/* Alternative move */
};


Move		newMove(void);
void		freeMove(Move);
bool		diagramMoveP(Move);
bool		containsCommentsMoveP(Move);
bool		containsVariationsMoveP(Move);
void		outputMove(Move, TextBuffer, Position, int);
void		outputCommentsMove(Move, TextBuffer, Position);
unsigned char *	commentMove(Move, unsigned char *);


/*------------------------------------------------------------
 *  File option.c
 *------------------------------------------------------------*/

struct option
{ long		from;			/* From this game number */
  long		to;			/* To this game number */
  CBase		database;		/* Database to operate on */
  CBase		dump;			/* Output database */
  FILE *	output;			/* Output file */
  bool		verbose;		/* Additional printing */
  int		notation;		/* SHORT_ALGEBRAIC, LONG_ALGEBRAIC */
};


Option		newOption(void);
void		freeOption(Option);
int		genericOption(Option, char **, int, int);


/*------------------------------------------------------------
 *  File position.c
 *------------------------------------------------------------*/

#define WHITE		0
#define BLACK		1

#define WHITE_SHORT	(1 << 0)
#define WHITE_LONG	(1 << 1)
#define BLACK_SHORT	(1 << 2)
#define BLACK_LONG	(1 << 3)

struct position
{ Piece         board[64];		/* Pieces on the board */
  int		enPassant;		/* File for en passant, a=1 */
  Colour	toMove;			/* WHITE or BLACK */
  short		castling;		/* Castling rights */
  short		nextMove;		/* Next move to play */
};


Position	newPosition(void);
void		freePosition(Position);
void		clearPosition(Position);
void		copyPosition(Position dst, Position src);
void		doMovePosition(Position, Move);
bool		ambiguousMovePositionP(Position, Move, Square *);
bool		checkMovePositionP(Position, Move);
bool		enPassantMovePositionP(Position, Move);
bool		castlingMovePositionP(Position, Move);
bool		attacksSquarePositionP(Position, Piece, Square, Square);
void		diagramPosition(Position, TextBuffer);
bool		getMovePosition(Position, int, Move);
char *		getStringMovePosition(Position, Move, int);


/*------------------------------------------------------------
 *  File string.c
 *------------------------------------------------------------*/

char *		normalisePlayer(char *);
char *		normalisePlace(char *);

bool		headStringP(char *s, char *head);
bool		tailStringP(char *s, char *tail);
void		replaceString(char *s, char *in, char *out);
void		chopString(char *s, char *specials);
void		stripString(char *s);


/*------------------------------------------------------------
 *  File strtable.c
 *------------------------------------------------------------*/

typedef struct string_table *	StringTable;

struct string_table
{ long		count;			/* Number of entries present */
  long		allocated;		/* Number of entries allocated */
  char **	entries;		/* Entries themselves */
  void **	values;			/* Values associated with entries */
};


StringTable	newStringTable(long entries);
void		freeStringTable(StringTable);
bool		containsStringTableP(StringTable, char *);
void *		findStringTable(StringTable, char *);
void		editStringTable(StringTable, char *, void *);
char *		addStringTable(StringTable, char *, void *);
void		printStringTable(StringTable, char *, FILE *);
void		incStringTable(StringTable, char *);


/*------------------------------------------------------------
 *  File symbols.c
 *------------------------------------------------------------*/

#define ETC			1
#define KING_SYMBOL		2
#define QUEEN_SYMBOL		3
#define ROOK_SYMBOL		4
#define BISHOP_SYMBOL		5
#define KNIGHT_SYMBOL		6
#define PAWN_SYMBOL		7
#define QUARTER			8
#define HALF			9
#define THREE_QUARTERS		10
#define WHITE_BETTER		11
#define WHITE_ADVANTAGE		12
#define WHITE_WINNING		13
#define BLACK_BETTER		14
#define BLACK_ADVANTAGE		15
#define BLACK_WINNING		16
#define WHITE_COMPENSATION	17
#define BLACK_COMPENSATION	18
#define UNCLEAR			19
#define BISHOP_PAIR		20
#define OPPOSITE_BISHOPS	21
#define SAME_BISHOPS		22
#define DOUBLE_PAWN		23
#define PASSED_PAWN		24
#define ISOLATED_PAWN		25
#define CONNECTED_PAWN		26
#define WORSE_IS		27
#define BETTER_IS		28
#define SPACE_ADVANTAGE		29
#define ONLY_MOVE		30
#define WITH_IDEA		31
#define QUEENS_SIDE		32
#define KINGS_SIDE		33
#define WITH			34
#define WITHOUT			35
#define WITH_INITIATIVE		36
#define WITH_ATTACK		37
#define WITH_COUNTERPLAY	38
#define DIAGONAL		39
#define CENTRE			40
#define LINE_FILE		41
#define ENDGAME			42
#define ZUGZWANG		43
#define TIME_TROUBLE		44
#define DEVELOPMENT_ADVANTAGE	45
#define WHITE_WINS		46
#define BLACK_WINS		47
#define DRAW			48
#define EQUALITY		49
#define NOVELTY			50
#define COMPENSATION		51
#define WEAK_POINT		52
#define PAWN_EVALUATION		53
#define PAWN_CONNECTION		54
#define UMLAUT_u		55
#define UMLAUT_a		56
#define UMLAUT_U		57
#define UMLAUT_o		58
#define GERMAN_S		58
#define NEXT_LINE		59
#define EDITORIAL_COMMENT	60
#define GOOD_MOVE		61
#define BAD_MOVE		62
#define INTERESTING_MOVE	63
#define DUBIOUS_MOVE		64
#define BRILLIANT_MOVE		65
#define BLUNDER			66
#define MATE			67
#define CEDILLE_C		68
#define ACUTE_e			69
#define CIRCUMFLEX_a		70
#define GRAVE_a			71
#define DOT_a			72
#define CEDILLE_c		73
#define CIRCUMFLEX_e		74
#define UMLAUT_e		75
#define GRAVE_e			76
#define UMLAUT_i		77
#define CIRCUMFLEX_i		78
#define GRAVE_i			79
#define UMLAUT_A		80
#define DOT_A			81
#define ACUTE_E			82
#define SCANDI_ae		83
#define SCANDI_AE		84
#define CIRCUMFLEX_o		85
#define GRAVE_o			86
#define CIRCUMFLEX_u		87
#define GRAVE_u			88
#define UMLAUT_y		89
#define UMLAUT_O		90
#define ACUTE_a			91
#define ACUTE_i			92
#define ACUTE_o			93
#define ACUTE_u			94
#define TILDE_n			95
#define TILDE_N			96
#define CASTLING_SHORT		97
#define CASTLING_LONG		98
#define CAPTURE_SYMBOL		99
#define INCHECK_SYMBOL		100
#define MOVE_HYPHEN		101
#define PROMOTION_SYMBOL	102
#define WHITE_MOVE_SYMBOL	103
#define BLACK_MOVE_SYMBOL	104
#define START_MAJOR_ALTERNATIVE	105
#define END_MAJOR_ALTERNATIVE	106
#define START_ALTERNATIVE	107
#define END_ALTERNATIVE		108
#define START_COMMENT		109
#define END_COMMENT		110
#define START_SUB_ALTERNATIVE	111
#define START_DIAGRAM		112
#define END_DIAGRAM		113
#define START_DIAGRAM_RANK	114
#define END_DIAGRAM_RANK	115
#define DIAGRAM_WK		116
#define DIAGRAM_WQ		117
#define DIAGRAM_WR		118
#define DIAGRAM_WB		119
#define DIAGRAM_WN		120
#define DIAGRAM_WP		121
#define DIAGRAM_BK		122
#define DIAGRAM_BQ		123
#define DIAGRAM_BR		124
#define DIAGRAM_BB		125
#define DIAGRAM_BN		126
#define DIAGRAM_BP		127
#define DIAGRAM_WHITE_SQUARE	128
#define DIAGRAM_BLACK_SQUARE	129

#define LAST_SYMBOL		130


extern char *	SymbolNames[LAST_SYMBOL];
extern char *	AsciiSymbols[LAST_SYMBOL];


void		initChessSymbols(void);
void		loadChessSymbols(FILE *);
void		printChessSymbols(FILE *);
char *		chessSymbol(int);
char *		mapChessSymbol(unsigned char);
char *		stringChessSymbol(char *, char *);


/*------------------------------------------------------------
 *  File text.c
 *------------------------------------------------------------*/

struct textbuffer
{ FILE *	file;			/* File to which text is written */
  int		columns;		/* Columns per line */
  int		maxColumns;		/* For extremely long lines */
  char *	line;			/* Current line of text */
  int		count;			/* Count of characters in line */
  char *	current;		/* Current position in buffer */
  int		lastSpace;		/* Index of last space */
};


TextBuffer	newTextBuffer(FILE *, int);
void		freeTextBuffer(TextBuffer);
void		flushTextBuffer(TextBuffer);

void		stringTextBuffer(TextBuffer, char *);
void		stringSpaceTextBuffer(TextBuffer, char *);
void		stringNewlineTextBuffer(TextBuffer, char *);
void		stringRawTextBuffer(TextBuffer, char *);
void		formatTextBuffer(TextBuffer, char *, ...);
void		formatSpaceTextBuffer(TextBuffer, char *, ...);
void		formatNewlineTextBuffer(TextBuffer, char *, ...);
void		formatRawTextBuffer(TextBuffer, char *, ...);
