/* EMACS CLASS BROWSER FOR C++.
   Copyright (C) 1993 Gerd Moellmann. All rights reserved.
   Altenbergstr. 6, D-40235 Duesseldorf, Germany
   CompuServe ID 100025,3303

   $Id: parse.c,v 3.1 1995/02/17 18:20:24 mmann Exp $

   This file may be made part of GNU Emacs at the option of the FSF.

   This code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.  No author or distributor
   accepts responsibility to anyone for the consequences of using it
   or for whether it serves any particular purpose or works at all,
   unless he says so in writing.  Refer to the GNU Emacs General Public
   License for full details.

   Everyone is granted permission to copy, modify and redistribute
   this code, but only under the conditions described in the
   GNU Emacs General Public License.   A copy of this license is
   supposed to have been given to you along with GNU Emacs so you
   can know your rights and responsibilities.  It should be in a
   file named COPYING.  Among other things, the copyright notice
   and this notice must be preserved on all copies. */

/* REMAINING PROBLEMS
   ------------------
   * Handling of nested classes is incorrect. Declarations of the form
   `class X : public Base::X' will lead to a circular reference from
   `X' to `X' as subclass.  I have chosen to ignore the relationship
   in these cases.

   * member
   What about the case `class X& some_member();' or `struct {} a;'. 
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "lex.h"

unsigned tk = ~0;

#define match()         (tk = yylex ())
#define lookahead()     (tk == ~0u ? (tk = (unsigned) yylex ()) : tk)
#define looking_at(t)   (tk == (t))
#define current_tk()    tk

#ifdef PROTOTYPES
static unsigned skip_to (unsigned t);
static void skip_matching (void);
static void member (sym_t* cls, unsigned vis);
static void class_body (sym_t* cls);
static void class_definition (sym_t* anonymous, int is_struct, int nested);
static void declaration (void);
static long parm_list (int* is_const, int* is_pure);
static void operator_name (char* id, int* sc);
#endif

void
re_init_parse ()
{
  tk = -1;
}

static unsigned
skip_to (t)
     unsigned t;
{
  while (!looking_at (YYEOF) && !looking_at (t))
    match ();

  return current_tk ();
}

static void
skip_matching ()
{
  unsigned open, close, n;

  switch (open = lookahead ())
    {
    case '{':
      close = '}';
      break;
    case '(':
      close = ')';
      break;
    case '<':
      close = '>';
      break;
    case '[':
      close = ']';
      break;
    }

  for (n = 0;;)
    {
      if (looking_at (open))
        ++n;
      else if (looking_at (close))
        --n;
      else if (looking_at (YYEOF))
        break;

      match ();

      if (0 == n)
        break;
    }
}

static long
parm_list (is_const, is_pure)
     int* is_const;
     int* is_pure;
{
  unsigned long hash = 0;
  int type_seen = 0;
  unsigned long ident_type_hash = 0;
  unsigned long prev_ident_type_hash = 0;

  *is_const = *is_pure = 0;

  while (!looking_at (YYEOF) && !looking_at (')'))
    {
      switch (lookahead ())
        {
        case '(':
          skip_matching ();
          break;

        case ',':
          match ();
          type_seen = 0;
          break;

	  /* Ignore the scope part of types, if any. This is because
	     some types need scopes when defined outside of a class body,
	     and don't need them inside the class body.  This means that
	     we have to look for the last IDENT in a sequence of
	     IDENT::IDENT::... */

        case IDENT:
	  if (!type_seen)
            {
	      char* s;
	      prev_ident_type_hash = ident_type_hash;
	      ident_type_hash = 0;
              for (s = yytext; *s; ++s)
                ident_type_hash = (ident_type_hash << 1) ^ *s;
            }
          match ();
          break;

        case VOID:
          /* This distinction is made to make `func (void)' equivalent
             to `func ()'. */
          type_seen = 1;
          match ();
          if (!looking_at (')'))
            hash = (hash << 1) ^ VOID;
          break;

        case ELLIPSIS:  case CHAR:      case CLASS:     case CONST:
        case DOUBLE:    case ENUM:      case FLOAT:     case INT:
        case LONG:      case SHORT:     case SIGNED:    case STRUCT:
        case UNION:     case UNSIGNED:  case VOLATILE:
          type_seen = 1;
          hash = (hash << 1) ^ lookahead ();
          match ();
          break;

        case '*':       case '&':       case '[':       case ']':
          hash = (hash << 1) ^ lookahead ();
          match ();
          break;

        default:
          match ();
          break;
        }
    }

  if (!type_seen)
    hash = (hash << 1) ^ prev_ident_type_hash;

  /* We can overload the same function on `const' */

  if (looking_at (')'))
    {
      match ();
      if (looking_at (CONST))
        {
          hash = (hash << 1) ^ CONST;
          *is_const = 1;
        }

      if (looking_at ('='))
        {
          match ();
          if (looking_at (CINT) && long_val == 0)
            {
              match ();
              *is_pure = 1;
            }
        }
    }

  return hash;
}

/* Note that `cls' my be NULL due to options given on the command
   line. */

static void
member (cls, vis)
     sym_t* cls;
     unsigned vis;
{
  member_t* m;
  char id[MAX_ID_LENGTH];
  int sc = SC_MEMBER;
  char* regexp;
  unsigned pos;
  long hash;
  int is_constructor;
  int anonymous = 0;
  int is_struct;
  int is_virtual = 0;
  int is_inline = 0;
  int is_const, is_pure;
  int type_seen = 0;
  int paren_seen = 0;
  *id = 0;

  while (!looking_at (';') && !looking_at ('{') && !looking_at (YYEOF))
    {
      switch (lookahead ())
        {
        default:
          match ();
          break;

        case '[':
          skip_matching ();
          break;

        case ENUM:
          sc = SC_TYPE;
	  type_seen = 1;
          match ();
          break;

        case TYPEDEF:
          sc = SC_TYPE;
	  type_seen = 1;
          match ();
          break;

        case '~':
          *id = '~';
          *(id + 1) = 0;
          match ();
          break;

        case FRIEND:
          sc = SC_FRIEND;
	  type_seen = 1;
          match ();
          break;

        case INLINE:
          is_inline = 1;
	  type_seen = 1;
          match ();
          break;

        case VIRTUAL:
          is_virtual = 1;
	  type_seen = 1;
          match ();
          break;

        case STATIC:
          sc = SC_STATIC;
	  type_seen = 1;
          match ();
          break;

        case IDENT:
          /* Remember IDENTS seen so far. Among these will be the member
             name. */
          if (*id == '~')
            strcat (id, yytext);
          else
            strcpy (id, yytext);

	  assert (strlen (id) < sizeof id);
          match ();
          break;

        case OPERATOR:
          operator_name (id, &sc);
	  assert (strlen (id) < sizeof id);
          break;

        case '(':
          /* Most probably the beginning of a parameter list. */
          match ();
	  paren_seen = 1;

          if (*id && cls)
            {
              if (!(is_constructor = !strcmp (id, cls->name)))
                regexp = dupstr (line_start ());
            }
          else
            is_constructor = 0;

          pos = buffer_pos ();
          hash = parm_list (&is_const, &is_pure);

          if (is_constructor)
            regexp = dupstr (line_start ());

          if (*id && cls != NULL)
	    add_member_declaration (cls, id, regexp, pos, hash, 0,
				    sc, vis, is_virtual, is_inline,
				    is_const, is_pure);

          while (!looking_at (';') && !looking_at ('{')
                 && !looking_at (YYEOF))
            match ();

          if (looking_at ('{') && *id && cls)
            add_member_definition (cls, id, regexp, pos, hash,
                                   0, sc, 1, is_const);

          *id = 0;
          sc = SC_MEMBER;
          break;

        case STRUCT: case UNION: case CLASS:
          /* Nested class */
          is_struct = lookahead () != CLASS;
	  type_seen = 1;
          match ();
          anonymous = 1;

          /* More than one ident here to allow for MS-DOS specialties
             like `_export class' etc.  The last IDENT seen counts
             as the class name. */

          while (looking_at (IDENT))
            {
              anonymous = 0;
              match ();
            }

          if (looking_at (':') || looking_at ('{'))
            class_definition (anonymous ? NULL : cls, is_struct, 1);
          else
	    skip_to (';');

	  break;

	case INT: 	case CHAR: 	case LONG: 	case UNSIGNED:
	case SIGNED: 	case CONST: 	case DOUBLE: 	case VOID:
	case SHORT:	case VOLATILE:
	  type_seen = 1;
	  match ();
	  break;
        }
    }

  if (looking_at (';'))
    {
      /* The end of member variable, a friend declaration or an access
	 declaration. We don't want to add friend classes as members. */

      if (*id && sc != SC_FRIEND && cls)
        {
          regexp = dupstr (line_start ());
          pos = buffer_pos ();
	  
	  if (cls != NULL)
	    {
	      if (type_seen || !paren_seen)
		add_member_declaration (cls, id, regexp, pos, 0, 1,
					sc, vis, 0, 0, 0, 0);
	      else
		add_member_declaration (cls, id, regexp, pos, hash, 0,
					sc, vis, 0, 0, 0, 0);
	    }
        }
      match ();
    }
  else if (looking_at ('{'))
    {
      /* A named enum */
      if (sc == SC_TYPE && *id && cls)
        {
          regexp = dupstr (line_start ());
          pos = buffer_pos ();

	  if (cls != NULL)
	    {
	      add_member_declaration (cls, id, regexp, pos, 0, 1, sc, vis,
				      0, 0, 0, 0);
	      add_member_definition (cls, id, regexp, pos, 0, 1, sc, 0, 0);
	    }
        }

      skip_matching ();
    }
}

static void
class_body (cls)
     sym_t* cls;
{
  int t;
  unsigned vis = PRIVATE;
  unsigned temp;

  while (!looking_at (YYEOF) && !looking_at ('}'))
    {
      switch (lookahead ())
        {
        case PRIVATE: case PROTECTED: case PUBLIC:
          temp = lookahead ();
          match ();

          if (looking_at (':'))
            {
              vis = temp;
              match ();
            }
          else
            {
              /* Probably conditional compilation for inheritance list.
                 We don't known whether there comes more of this.
                 This is only a crude fix that works most of the time. */
              do
                {
                  match ();
                }
              while (looking_at (IDENT) || looking_at (',')
                     || looking_at (PUBLIC) || looking_at (PROTECTED)
                     || looking_at (PRIVATE));
            }
          break;

          /* Try to synchronize */
        case CHAR:      case CLASS:     case CONST:
        case DOUBLE:    case ENUM:      case FLOAT:     case INT:
        case LONG:      case SHORT:     case SIGNED:    case STRUCT:
        case UNION:     case UNSIGNED:  case VOID:      case VOLATILE:
        case TYPEDEF:   case STATIC:    case INLINE:    case FRIEND:
        case VIRTUAL:   case TEMPLATE:  case IDENT:     case '~':
          member (cls, vis);
          break;

        default:
          match ();
          break;
        }
    }
}

static void
class_definition (containing_anonymous, is_struct, nested)
     sym_t* containing_anonymous;
     int is_struct;
     int nested;
{
  sym_t* super;
  sym_t* sub;

  if ((is_struct && !f_structs) || (nested && !f_nested_classes))
    sub = NULL;
  else
    {
      if (containing_anonymous)
        sub = containing_anonymous;
      else
        {
          sub = add_sym (yytext);
          sub->pos = buffer_pos ();
          sub->regexp = dupstr (line_start ());
          sub->filename = filename;
        }
    }

  if (looking_at (':'))
    {
      /* If at ':', inheritance list follows. */
      match ();

      for (;;)
        {
          switch (lookahead ())
            {
            case VIRTUAL: case PUBLIC: case PROTECTED: case PRIVATE: case ',':
              match ();
              break;
            case IDENT:
              /* The name of a base class. To avoid circular references that
		 come from something like `class X : public Base::X', do
		 not add a link in this case. */

              if (! ((is_struct && !f_structs)
		     || (nested && !f_nested_classes)))
                {
                  super = add_sym (yytext);

		  if (super != sub)
		    add_link (super, sub);
                }
              match ();
              break;

            case '<':
              skip_matching ();
              break;

	      /* A base class name may have the form A<>::B<>... */

	    case DCOLON:
	      match ();
	      break;

            case '{':
              goto l_class_body;

            default:
              /* A syntax error, possibly due to preprocessor constructs
		 like

		 #ifdef SOMETHING
		 class A : public B
		 #else
		 class A : private B.

		 Match until we see something like `;' or `{'. */

	      while (!(looking_at (';') || !looking_at (YYEOF)
		       || looking_at ('{')))
		match ();

	      goto l_class_body;
            }
        }
    }

 l_class_body:;

  if (looking_at ('{'))
    {
      if (is_struct && !f_structs)
        skip_matching ();
      else
        {
          match ();
          class_body (sub);

	  if (looking_at ('}'))
	    {
	      match ();
	      if (looking_at (';') && !nested)
		match ();
	    }
        }
    }
}

static void
operator_name (id, sc)
     char* id;
     int* sc;
{
  match ();

  if (looking_at (NEW) || looking_at (DELETE))
    {
      if (*sc != SC_FRIEND)
        *sc = SC_STATIC;

      strcpy (id, token_string (lookahead ()));
      match ();
    }
  else
    {
      strcpy (id, "operator");

      /* Beware access declarations of the form "X::f;" */

      do
        {
          strcat (id, " ");
          strcat (id, token_string (lookahead ()));
          match ();
        }
      while (!looking_at ('(') && !looking_at (';'));
    }

  assert (strlen (id) < MAX_ID_LENGTH);
}

static void
declaration ()
{
  char id[MAX_ID_LENGTH];
  sym_t* cls = NULL;
  char* regexp;
  unsigned pos;
  long hash;
  int is_constructor;
  int is_inline = 0;
  int is_const;
  int is_pure;
  int sc = 0;

  *id = 0;

  while (!looking_at (';') && !looking_at ('{') && !looking_at (YYEOF))
    {
      switch (lookahead ())
        {
        default:
          match ();
          break;

	case '[':
	  skip_matching ();
	  break;

	case ENUM:
	case TYPEDEF:
	  sc = SC_TYPE;
	  match ();
	  break;
	  
        case STATIC:
          sc = SC_STATIC;
          match ();
          break;

	case INT: 	case CHAR: 	case LONG: 	case UNSIGNED:
	case SIGNED: 	case CONST: 	case DOUBLE: 	case VOID:
	case SHORT:	case VOLATILE:
	  match ();
	  break;

        case CLASS: case STRUCT: case UNION:
          /* This is for the case `STARTWRAP class X : ...' or
             `declare (X, Y)\n class A : ...'. */
          if (*id)
            return;

        case '=':
          /* Assumed to be the start of an initialization in this context.
             Skip over everything upto ';'. */
          skip_to (';');
          break;

        case '~':
          *id = '~';
          *(id + 1) = 0;
          match ();
          break;

        case OPERATOR:
          operator_name (id, &sc);
          break;

        case INLINE:
          is_inline = 1;
          match ();
          break;

        case IDENT:
          if (*id == '~')
            strcat (id, yytext);
          else
            strcpy (id, yytext);

	  assert (strlen (id) < MAX_ID_LENGTH);
          match ();

          /* Template arguments */
          if (looking_at ('<'))
            skip_matching ();

          /* Maybe class name not yet seen. */
          if (looking_at (DCOLON))
            {
              cls = add_sym (id);
              match ();
            }
          break;

        case '(':
          /* Most probably the beginning of a parameter list. */
          if (cls)
            {
              match ();

              if (*id && cls)
                {
                  if (!(is_constructor = !strcmp (id, cls->name)))
                    regexp = dupstr (line_start ());
                }
              else
                is_constructor = 0;

              pos = buffer_pos ();
              hash = parm_list (&is_const, &is_pure);

              if (is_constructor)
                regexp = dupstr (line_start ());

              if (*id && cls)
                add_member_definition (cls, id, regexp, pos,
                                       hash, 0, SC_UNKNOWN,
                                       is_inline, is_const);
            }
          else
            {
              /* This may be a normal C functions, but also a macro
                 call of the form `declare (A, B)' --- such macros
                 can be found in some foundation class libraries. */

              match ();

              if (*id)
                {
                  regexp = dupstr (line_start ());
                  pos = buffer_pos ();
                  hash = parm_list (&is_const, &is_pure);
                  add_global_declaration (id, regexp, pos, hash, 0, sc,
					  is_inline, is_const);
                }

              /* This is for the case that the function really is
                 a macro with no `;' following it.  If a CLASS directly
                 follows, we would miss it otherwise. */

              if (looking_at (CLASS) || looking_at (STRUCT)
                  || looking_at (UNION))
                return;
            }

	  while (!looking_at (';') && !looking_at ('{')
                 && !looking_at (YYEOF))
            match ();

	  if (!cls && *id && looking_at ('{'))
	    add_global_definition (id, regexp, pos, hash, 0, sc,
				   is_inline, is_const);

          *id = 0;
          is_inline = 0;
          break;
        }
    }

  if (looking_at (';'))
    {
      /* The end of member variable or of an access declaration `X::f'.
	 To distinguish between them we had to know whether type information
	 has been seen. */

      if (*id)
        {
          char* regexp = dupstr (line_start ());
          unsigned pos = buffer_pos ();

	  if (cls)
	    add_member_definition (cls, id, regexp, pos, 0, 1,
				   SC_UNKNOWN, 0, 0);
	  else
	    add_global_definition (id, regexp, pos, 0, 1, sc, is_inline,
				   is_const);
        }
	
      match ();
    }
  else if (looking_at ('{'))
    {
      if (sc == SC_TYPE && *id)
        {
	  /* A named enum */
          regexp = dupstr (line_start ());
          pos = buffer_pos ();
          /* add_global_declaration (id, regexp, pos, 0, 1, sc, 0, 0); */
          add_global_definition (id, regexp, pos, 0, 1, sc, 0, 0);
        }

      skip_matching ();
    }
}

int
yyparse ()
{
  int anonymous;
  unsigned class_tk;

  for (;;)
    {
      switch (lookahead ())
        {
        case TEMPLATE:
          match ();
          if (looking_at ('<'))
            skip_matching ();
          break;

        case CLASS: case STRUCT: case UNION:
          class_tk = lookahead ();
          match ();
          anonymous = 1;

          /* More than one ident here to allow for MS-DOS specialties
	     like `far', `_export' etc. */

          while (looking_at (IDENT))
            {
              anonymous = 0;
              match ();
            }

          /* Don't add anonymous unions. */
          if ((looking_at (':') || looking_at ('{')) && !anonymous)
              class_definition (NULL, class_tk != CLASS, 0);
          else
            {
              if (skip_to (';') == ';')
                match ();
            }
          break;

        case YYEOF:
          return 1;

        default:
          declaration ();
          break;
        }
    }
}


