;;; CLASS BROWSER FOR C++
;;; $Id: br-struc.el,v 3.1 1995/02/17 18:19:36 mmann Exp $
;;;
;;; **********************************************************************
;;; Copyright (C) 1993, 1994 Gerd Moellmann. All rights reserved.
;;; Altenbergstr. 6, D-40235 Duesseldorf, Germany
;;; 100025.3303@COMPUSERVE.COM
;;; Suggestions, comments and requests for improvements are welcome.
;;; **********************************************************************
;;;
;;; This version works with both Emacs version 18 and 19, and I want
;;; to keep it that way. It requires the CL-19 Common Lisp compatibility
;;; package for Emacs 18 and 19.
;;;
;;; This file contains the structure definitions for the browser package.
;;; These structures must be the same as those generated by EROWSE.
;;; 

;;: This file may be made part of the Emacs distribution at the option
;;; of the FSF.

;; This code is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; this code, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

(require 'cl-19 "cl")
(require 'backquote)
(require 'br-macro)

(proclaim '(optimize (speed 3) (safety 0)))


;;;
;;; A @HEADER structure is the first in a file produced by EBROWSE.
;;; It contains
;;; - a version number that is compared against the version number
;;;   of the Lisp package when the file is loaded.
;;; - the name of the file containing regular expressions, if any.
;;;   If this name is NIL, regular expressions are stored within
;;;   the Lisp database.
;;; - a free slot that is filled out after the tree is loaded. This
;;;   slot MEMBER-OBARRAY is set to an obarray with an entry for
;;;   each member in the tree.
;;;   

(defstruct tree-header
  version                               ;version string
  command-line-options                  ;command line used to generate tree
  regexp-file                           ;file containing regexps or NIL
  member-obarray)                       ;filled out later


;;;
;;; Following the TREE-HEADER structure, the BROWSE file contains a
;;; number of TREE structures, each one describing one root class of
;;; the class hierarchy with all its subclasses.
;;;

(defstruct tree
  class                                 ;CLASS struct describing the class
  subclasses                            ;list of TREEs for subclasses
  member-variables                      ;list of MEMBERs structs
  member-functions                      ;
  static-variables                      ;
  static-functions                      ;
  friends                               ;
  types                                 ;
  superclasses                          ;list of TREEs for superclasses
  mark)                                 ;marked?


;;;
;;; A common structure definining an occurrance of some name in
;;; the source files. 
;;;

(defstruct browse 
  name                                  ;class or member name
  file                                  ;file in which to search
  pattern                               ;regexp to search for
  point)                                ;start point for search


;;;
;;; This is the structure stored in the CLASS slot of a TREE structure.
;;; It describes the location of the class declaration.
;;;

(defstruct (class (:include browse))
  source-file)


;;;
;;; This is the structure describing a single member. The TREE structure
;;; contains various lists of MEMBERS for the different types of
;;; members.
;;;

(defstruct (member (:include browse))
  flags
  definition-file
  definition-pattern
  definition-point
  hash)


;;;
;;; Some macros to access the FLAGS slot of a MEMBER.
;;;

(defmacro member-visibility (member)
  "Get state of `public', `protected', and `private'."
  (` (logand (member-flags (, member)) 7)))

(defmacro member-inline-p (member)
  "Is member marked as `inline'?"
  (` (/= 0 (logand (member-flags (, member)) 16))))

(defmacro member-virtual-p (member)
  "Is member marked as `virtual'?"
  (` (/= 0 (logand (member-flags (, member)) 32))))

(defmacro member-const-p (member)
  "Is member marked as `const'?"
  (` (/= 0 (logand (member-flags (, member)) 64))))

(defmacro member-pure-p (member)
  "Is member marked as `pure virtual'?"
  (` (/= 0 (logand (member-flags (, member)) 128))))

;;;
;;; Build a list of superclasses for TREE by searching the subclass
;;; lists incrementally. This function must be used instead of
;;; TREE-SUPERCLASSES to access the superclass list.
;;; 

(defun browse-superclasses (tree)
  (or (tree-superclasses tree)
      (setf (tree-superclasses tree)
            (loop with to-search = (list tree)
                  with result = nil
                  as search = (pop to-search)
                  while search finally return result
                  do (dotrees (ti @tree-obarray)
                       (when (memq search (tree-subclasses ti))
                         (unless (memq ti result)
                           (setf result (nconc result (list ti))))
                         (push ti to-search)))))))

(provide 'br-struc)

;;; end of `br-struc.el'.
