//----------------------------------------------------------------------------
//           Name: Zip()
//        Purpose: Compress a set of files into one "F20" file.
//
//         Author: Wilson H. Yuen
//   Date created: 03-21-93
//   Time created: 00:00:23am
//      Copyright: Wilson H. Yuen
//
//         Syntax: Zip( cSoruce, cTarget ) --> lRetVal
//
//       Argument: cSource - the file pattern(s) to compress.  Can contain
//                           path name and wildchard (??/*) characters.
//
//                 cTarget - the name of the compressed file.
//
//   Return Value: lRetVal - TRUE if successful.
//
//    Description: This function uses Funcky II Compress() function to
//                 create the compessed file.
//
//        Example: This example will compress all files with the file
//                 extension "DBF" into the file named "DBADBF.FEX" in
//                 the current directory.
//
//                    If ! Zip( "\DBA\*.DBF", "C:\DBADBF.F20" )
//                       ErrMsg( "Unable to create compressed file" )
//                    Endif
//
//----------------------------------------------------------------------------
#include "SetCurs.ch"
#include "FUNCky.ch"
#include "Backup.ch"


Function Zip( cSource, cTarget )
  local cInFile                                              //-- file to backup
  local nInHandle                                            //-- source file handle
  local nOutHandle                                           //-- target file handle
  local nHPos                                                //-- current file pointer position
  local nBytes                                               //-- number of bytes compressed
  local nHeader                                              //-- number of bytes written to target
  local x                                                    //-- FOR/NEXT counter
  local n                                                    //-- misc variable
  local cHeader                                              //-- name of target file
  local nFc                                                  //-- file count
  local cTarget2                                             //-- name of the target file less path
  local lRetVal     := .f.                                   //-- return value
  local cSourcePath := substr(cSource,1,rat("\",cSource))    //-- source path
  local cSaveScr    := SaveVideo(9,10,20,70)                 //-- current screen
  local nBuffer     := 0                                     //-- total number of bytes
  local aOptions    := { 'Yes', 'No' }                       //-- options
  local nCurCursor  := SetCursor(SC_NONE)                    //-- current cursor settings

  cSource := alltrim(cSource)
  cTarget := alltrim(cTarget)

  //-- extract the name of the compressed file.
  If (n := rat("\",cTarget)) > 0
     cTarget2 := SubStr( cTarget, n+1)
  else
     cTarget2 := cTarget
  Endif


  Begin Sequence
    // Open the Files and check for errors
    nOutHandle := FCREATE( cTarget, C_READWRITE)

    IF nOutHandle EQ ERROR
      ErrMsg("Error opening or creating files")
      break
    ENDIF

    nFc := FILECOUNT( cSource, A_NORMAL)

    IF nFc == 0
      ErrMsg("No files to compress")
      break
    ENDIF


    Msg( {"Compressing Files to " + cTarget2,""},, .f., 16 )


    FOR x := 1 TO nFc
      IF x == 1
        cInFile := FINDFIRST( cSource, A_NORMAL)
      ELSE
        cInFile := FINDNEXT()
      ENDIF

      // do not try to compress the compress file.
      If ! cInFile $ cTarget

        IF EMPTY(cInFile)
          EXIT
        ENDIF

        cInFile := "  "+alltrim(cInFile)+"  "
        @ 17,39-len(cInFile)/2 say cInFile COLOR "RG/W"
        cInFile := alltrim(cInFile)

        // Compress the (next) file
        nInHandle := FOPEN( cSourcePath + cInFile,O_READONLY)
        cHeader   := ALLTRIM(cInFile)+SPACE(12-LEN(ALLTRIM(cInFile)))+SPACE(8)
        nHeader   := FWRITE(nOutHandle,cHeader,LEN(cHeader))
        nHPos     := FTELL(nOutHandle)-8
        nBytes    := COMPRESS(nInHandle,nOutHandle,DEFAULT)

        FSEEK(nOutHandle,nHPos,0)
        FWRITE(nOutHandle,STR(nBytes,8),8)
        FSEEK(nOutHandle,0,2)

        // Close the files, convert nBytes to string and print results
        FCLOSE(nInHandle)

        IF nBytes GE 0
          nBuffer := nBuffer + nBytes
        ELSE
          ErrMsg("Error compressing "+cInFile)

          if ALERT("Do you want to continue compressing the rest of the files?",;
                  aOptions,;
                  { "W+/BR", "GR+/BR", "BG+/BR, W+/B" } ) == 2
            Break
          endif
        ENDIF
      Endif
    NEXT x

    FCLOSE(nOutHandle)
    RestVideo(9,10,20,70,cSaveScr)
    nBuffer := ULTOA(nBuffer)
    Notify()
    Msg({"Backup completed,",;
         nBuffer + " bytes compressed."})

    lRetVal := .t.

  End Sequence
  SetCursor(nCurCursor)

Return( lRetVal )
* eof Zip()
*
*
//----------------------------------------------------------------------------
//           Name: UnZip()
//        Purpose: Uncompress a "FEX" compressed file.
//
//         Author: Wilson H. Yuen
//   Date created: 03-21-93
//   Time created: 00:06:12am
//      Copyright: Softsystems, Inc.  1988-1992
//
//         Syntax: UnZip( cSource, cTarget ) --> lRetVal
//
//       Argument: cSource - path and name of the compressed file.
//
//                 cTarget   - path to uncompress the files to.
//
//   Return Value: lRetVal - TRUE if successful.
//
//    Description: This function uses Funcky II Decompress() function to
//                 create the compessed file.
//
//        Example: This example will decompress "DBADBF.FEX" into the
//                 subdirectory "\DBA\DBFS".
//
//                    If ! UnZip( "C:\DBADBF.FEX" ,"\DBA\DBFS")
//                       ErrMsg( "Unable to decompress file" )
//                    Endif
//
//----------------------------------------------------------------------------
Function UnZip( cSource, cTarget )
  local cOutFile
  local cSource2
  local n
  local nPos
  local nInHandle
  local nOutHandle
  local nBytes
  local nSize
  local x
  local nBuffer := 0
  local lRetVal       := .f.
  local cSaveScr      := SaveVideo(9,10,20,70)
  local cSourceSize := FileSize( cSource )
  local nCurCursor  := SetCursor(SC_NONE)

  cSource := alltrim(cSource)
  cTarget := alltrim(cTarget)


  //-- determine target path
  If cTarget == NIL
     cTarget := curdir()
  else
    if ! substr(cTarget,-1) == "\"
       cTarget := cTarget + "\"
    endif
  Endif

  //-- extract the compressed file name.
  If (n := rat("\",cSource)) > 0
     cSource2 := SubStr( cSource, n+1)
  else
     cSource2 := cSource
  Endif

  Begin sequence
    // Open the Files and check for errors
    nInHandle := FOPEN( cSource, O_READONLY)
    nPos      := 0

    Msg( {"Decompressing " + cSource2,""},, .f., 16 )

    DO WHILE ! nPos == cSourceSize

      // create a buffer
      cOutFile := SPACE(12)                                                    // (re)create a buffer

      FSEEK(nInHandle,nPos,0)

      // read the (next) file name into cOutFile
      FREAD(nInHandle,@cOutFile,12)

      IF EMPTY(cOutFile)
        ErrMsg( "No files to decompress")
        Break
      ENDIF

      cOutFile   := ALLTRIM(cOutFile)                                          // trim it up
      nOutHandle := FCREATE( cTarget + cOutFile,C_READWRITE)                   // create the (next) output file

      cOutFile := "  "+alltrim(cOutFile)+"  "
      @ 17,39-len(cOutFile)/2 say cOutFile COLOR "RG/W"
      cOutFile := alltrim(cOutFile)

      IF nInHandle EQ ERROR OR nOutHandle EQ ERROR
        ErrMsg("Error opening or creating "+cOutFile)
        break
      ENDIF

      // get the size of the (next) file to read
      nSize := SPACE(8)

      FREAD(nInHandle,@nSize,8)
      nSize  := VAL(nSize)
      nPos   += nSize+20
      nBytes := DCOMPRESS(nInHandle,nOutHandle,nSize)

      FCLOSE(nOutHandle)

      IF nBytes GE 0
         nBuffer := nBuffer + nBytes
      ELSE
        ErrMsg( "Error decompressing " + cOutFile )
        If  ALERT({"Do you want to continue decompressing the rest of the file?",;
                   "Do you want to overwrite this file?"},;
                   { 'Yes', 'No' },;
                   { "W+/BR", "GR+/BR", "BG+/BR, W+/B" } ) == 2
          Break
        endif

      ENDIF

    ENDDO

    // Close the files, convert nBytes to string and print results
    FCLOSE(nInHandle)
    RestVideo(9,10,20,70,cSaveScr)
    nBuffer := ULTOA(nBuffer)
    Notify()
    Msg({"Restore completed,",;
          nBuffer + " bytes decompressed."})

    lRetVal := .t.

  End Sequence

  SetCursor(nCurCursor)

Return( lRetVal )
* eof UnZip()
*
*
