/*******************************************************/
/*  REXX program to initialize tapes for backups       */
/*******************************************************/
'@ECHO OFF'
capacity = 250        /* Maximum capacity of each tape */

/*******************************************************/
/*  Initialize Visual REXX environment                 */
/*******************************************************/
Call RxFuncAdd 'VInit', 'VREXX', 'VINIT'
initcode = VInit()
If initcode = 'ERROR' Then
   signal CLEANUP
Call VDialogPos 50, 50 /* All dialog boxes centered */

/*******************************************************/
/* Set error traps to force clean up on failures       */
/*******************************************************/
Signal On Failure Name CLEANUP
Signal On Halt Name CLEANUP
Signal On Syntax Name CLEANUP

/*******************************************************/
/*  Initialize Visual REXX button values               */
/*******************************************************/
Ok         = 1
Cancel     = 2
OkOrCancel = 3
Yes        = 4
No         = 5
YesOrNo    = 6

/*******************************************************/
/* Initialize REXXUtil environment (if not present)    */
/*******************************************************/
rxload = RxFuncQuery('SysLoadFuncs')
If rxload Then
   Do
      Call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
      Call sysloadfuncs
   End

/*******************************************************/
/*     Extract the drives available on the system      */
/*******************************************************/
drives = SysDriveMap('C:', 'LOCAL')

/*******************************************************/
/*     Load number of drives into table root           */
/*******************************************************/
drive.0 = Words(drives)

/*******************************************************/
/*   Load the table with the available drive letters   */
/*******************************************************/
Do x=1 To drive.0 By 1
   Parse Value drives With drive.x drives
End

/*******************************************************/
/* Accumulate the total allocated space on all FAT     */
/* and HPFS drives.                                    */
/*******************************************************/
used_sum = 0
Do x=1 To drive.0 By 1
   filesys = FileType(drive.x)
   If filesys = 'FAT' | filesys = 'HPFS' Then
      Do
         drive_info = SysDriveInfo(drive.x)
         Parse Value drive_info With . free_space total_space .
         used_sum = used_sum + (total_space - free_space)
      End
End

/*******************************************************/
/* Convert allocated space from bytes to megabytes     */
/*******************************************************/
used_megs = Format((used_sum / 1048576),,0)

/*******************************************************/
/* Each tape can hold up to 250 megabytes.  Prompt for */
/* the number of tapes to erase, initialize with the   */
/* number of tapes we estimate we'll need.             */
/*******************************************************/
endloop = 0
Do Until endloop
   prompt.vstring = Format((used_megs / capacity)+1,,0)
   prompt.0 = 2
   prompt.1 = 'Enter number of tapes to  '
   prompt.2 = 'quick erase'
   button = VInputBox('Quick Erase Tapes', 'prompt', 2, OkOrCancel)
   If button = 'OK' Then
      Do
         If Datatype(prompt.vstring) = 'NUM' Then
            Do
               Do x=1 to prompt.vstring
                  msg.0 = 4
                  msg.1 = 'Insert Cartridge #'||x 'of' prompt.vstring
                  msg.2 = ''
                  msg.3 = 'Press OK to erase, or'
                  msg.4 = 'CANCEL to halt processing'
                  button = VMsgBox('Quick Erase Tapes', 'msg', OkOrCancel)
                  If button = 'OK' Then
                     'BACKMAST /QE'
                  Else
                     x = prompt.vstring
               End
               endloop = 1
            End
      End
   Else
      endloop = 1
End
CLEANUP:
   Call VExit
   If rxload Then
      Call SysDropFuncs  /* Drop only if we loaded */
   Exit
