#include "qwkvars.h"

/* ********************************************************************** */
/* Asylum INI Code (Part of COMMON) - by Michael Deweese Copyright 1994   */
/*                                                                        */
/* You may use this code in WWIV as installed under the COMMON.MOD file   */
/* You may not use this code in your own programs without the consent of  */
/* Michael Deweese.                                                       */
/* You can call these functions from your own WWIV modifications, but may */
/* not rip these out and include them in your mod.  In other words, to    */
/* access these functions, the user using your modification must have     */
/* mod installed                                                          */
/*                                                                        */
/*                                                                        */
/* USAGE:                                                                 */
/*    There are two basic ways to make use of these ini functions, the    */
/*    first and most simple is two call two high level functions that take*/
/*    care of all the work, they are:                                     */
/*                                                                        */
/*    get_ini_value                                                       */
/*    put_ini_value                                                       */
/*                                                                        */
/*    These two functions will open up the ini file, read it into memory, */
/*    do any modifications to it, as needed, and then write it back to    */
/*    disk.  Now this is on every single call, and thus, might be kind of */
/*    slow on large ini readings                                          */
/*                                                                        */
/*    eg                                                                  */
/*                                                                        */
/*    (Start ASYLUM.INI)                                                  */
/*    [LISTPLUS]                                                          */
/*    bar_color = 8                                                       */
/*    other_color = 5                                                     */
/*                                                                        */
/*    char s[10];                                                         */
/*    strcpy(s, "7");    // 7 to be used as a default if bar_color        */
/*                       // doesn't exist                                 */
/*    get_ini_value("ASYLUM.INI", "listplus", "bar_color", s, 10);        */
/*                                                                        */
/*    // s now equals "8"                                                 */
/*                                                                        */
/*    strcpy(s, "Zu Digital"                                              */
/*    get_ini_value("ASYLUM.INI", "Super Config", "WHO", s, 10);          */
/*                                                                        */
/*    // [Super Config] didn't exist, so it was created, likewise, WHO    */
/*    // didn't exist, so a 'WHO = Zu Digital' was added                  */
/*                                                                        */
/*                                                                        */
/*    strcpy(s, "1");                                                     */
/*    put_ini_value("ASYLUM.INI", "ListPlus", "other_color", s);          */
/*                                                                        */
/*    // other_color already existed in section [ListPlus], so it was     */
/*    // removed and replaced with other_color = 1                        */
/*                                                                        */
/*                                                                        */
/* Again, that reads and writes after every single call, if you need to   */
/* do alot of reading, then a little code such as the following is what   */
/* you want...                                                            */
/*                                                                        */
/*                                                                        */
/*  ini_inforec ini_info;                                                 */
/*  char s[10];                                                           */
/*                                                                        */
/*  // Read ini file into memory                                          */
/*  if(!open_ini_file("ASYLUM.INI", &ini_info))                           */
/*    return -1;    // Wasn't able to retrieve or create fname.ini        */
/*                                                                        */
/*                                                                        */
/*  // this will read the values...                                       */
/*  strcpy(s, "15");                                                      */
/*  read_ini_string(&ini_info, "ListPlus", "Color", s, 10);               */
/*                                                                        */
/*  // This will overwrite previous color with new value                  */
/*  strcpy(s, "NONE");                                                    */
/*  set_ini_string(&ini_info, "ListPlus", "Color", s, 10);                */
/*                                                                        */
/*  // Now write your changes to disk                                     */
/*  write_ini_file(&ini_info);                                            */
/*                                                                        */
/*  // And free your memory                                               */
/*  free_ini_info(&ini_info);                                             */
/*                                                                        */
/* ********************************************************************** */


int get_ini_value(char *fname, char *area, char *indent, char *value, int max_value)
{
  ini_inforec ini_info;

  if(!open_ini_file(fname, &ini_info))
    return -1;    /* Wasn't able to retrieve or create fname.ini */

  read_ini_string(&ini_info, area, indent, value, max_value);

  if(ini_info.changes_made)
    write_ini_file(&ini_info);   // value had to be added, so write it to disk

  free_ini_info(&ini_info);
  return 1;       /* Successful read */
}

int put_ini_value(char *fname, char *area, char *indent, char *value)
{
  ini_inforec ini_info;

  if(!open_ini_file(fname, &ini_info))
    return -1;    /* Wasn't able to retrieve or create fname.ini */

  set_ini_string(&ini_info, area, indent, value);
  write_ini_file(&ini_info);   // set_ini_string only sets the memory, it must
                               // be written to disk

  free_ini_info(&ini_info);
  return 1;       /* Successful read */
}

int get_ini_number(char *fname, char *area, char *indent, int default_value)
{
  ini_inforec ini_info;
  long x;

  if(!open_ini_file(fname, &ini_info))
    return -1;    /* Wasn't able to retrieve or create fname.ini */

  x = read_ini_number(&ini_info, area, indent, default_value);

  if(ini_info.changes_made)
    write_ini_file(&ini_info);   // value had to be added, so write it to disk

  free_ini_info(&ini_info);
  return x;       /* Successful read */
}

int put_ini_number(char *fname, char *area, char *indent, long value)
{
  ini_inforec ini_info;

  if(!open_ini_file(fname, &ini_info))
    return -1;    /* Wasn't able to retrieve or create fname.ini */

  set_ini_number(&ini_info, area, indent, value);
  write_ini_file(&ini_info);   // set_ini_string only sets the memory, it must
                               // be written to disk

  free_ini_info(&ini_info);
  return 1;       /* Successful read */
}


int get_ini_boolean(char *fname, char *area, char *indent, char *value)
{
  char s[51];

  strcpy(s, value);
  get_ini_value(fname, area, indent, s, 50);

  strupr(s);

  if(strcmp(s, "YES")==0 || strcmp(s, "TRUE")==0 || strcmp(s, "1")==0)
    return 1;

  if(strcmp(s, "NO")==0 || strcmp(s, "FALSE")==0 || strcmp(s, "0")==0)
    return 0;

  return -1;
}

int read_ini_boolean(ini_inforec *ini_info, char *area, char *indent, char *value)
{
  char s[51];

  strcpy(s, value);
  read_ini_string(ini_info, area, indent, s, 50);

  strupr(s);

  if(strcmp(s, "YES")==0 || strcmp(s, "TRUE")==0 || strcmp(s, "1")==0)
    return 1;

  if(strcmp(s, "NO")==0 || strcmp(s, "FALSE")==0 || strcmp(s, "0")==0)
    return 0;

  return -1;
}





void free_ini_info(ini_inforec *ini_info)
{
  if(ini_info->ini)
    free(ini_info->ini);
}



int write_ini_file(ini_inforec *ini_info)
{
  FILE *fp;

  fp = fopen(ini_info->fname, "w+t");
  if(!fp)
    return 0;               /*  Couldn't create ini file */

  fseek(fp, 0, SEEK_SET);
  fwrite((void *)ini_info->ini, 1, ini_info->fsize, fp);

  fclose(fp);
  return 1;
}

char *open_ini_file(char *fname, ini_inforec *ini_info)
{
  FILE *fp;

  memset((void *)ini_info, 0, sizeof(ini_inforec));
  strcpy(ini_info->fname, fname);

  fp = fopen(fname, "rt" );

  if(!fp)                        /* if it doesn't exist, create it */
  {
    fp = fopen(fname, "w+t");
    if(!fp)
      return NULL;               /*  Couldn't create ini file */
  }

  ini_info->fsize=filesize(fp);

  if(ini_info->fsize < 60000L)   /* Make 60,000 our file size */
    ini_info->ini = (char *) malloca(ini_info->fsize + EXTRA_INI_SPACE);

  if(!ini_info->ini)
  {
    fclose(fp);
    return NULL;
  }

  fseek(fp, 0, SEEK_SET);
  ini_info->fsize = fread((void *)ini_info->ini, 1, ini_info->fsize, fp);
  ini_info->ini[ini_info->fsize]=0;

  ini_info->allocated=ini_info->fsize + EXTRA_INI_SPACE;

  fclose(fp);
  return(ini_info->ini);
}

char * increase_ini_allocation(ini_inforec *ini_info)
{
  ini_info->allocated += EXTRA_INI_SPACE;

  if(ini_info->allocated > 0xffff)
    return NULL;

  ini_info->ini = (char *)realloc(ini_info->ini, ini_info->allocated);

  return(ini_info->ini);
}


void set_ini_string(ini_inforec *ini_info, char *area, char *indent, char *value)
{
  FILE *fp;
  char buff[255], *tmp, w1[128], w2[128];
  long area_start, pos, save_pos;
  int amount;


  /* If area == "" then don't bother with an [area] */
  if(area[0])
    area_start = find_ini_area(ini_info->ini, area);
  else
    area_start = ini_info->fsize;

  if(area_start >= ini_info->fsize)
  {

    if(area[0])
    {
      amount = sprintf(buff, "\n[%s]\n", area);

      if(ini_info->fsize + amount >= ini_info->allocated)
        if(!increase_ini_allocation(ini_info))
          return;

      strcat(ini_info->ini, buff);
      ini_info->fsize+= amount;
    }

    amount = sprintf(buff, "%s = %s\n", indent, value);

    if(ini_info->fsize + amount >= ini_info->allocated)
      if(!increase_ini_allocation(ini_info))
        return;

    strcat(ini_info->ini, buff);
    ini_info->fsize+= amount;

    ini_info->changes_made = 1;
    return;
  }


  pos = area_start;
  while(1)
  {
    save_pos = pos;
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)
      break;

    strip_string(buff);

    if(buff[0]=='[')  // Star of a new area, so break
    {
      save_pos = area_start;    // Put value right after [area], comment out this line to put at end
      break;
    }
    break_up_ini(buff, w1, w2);

    if(strcmpi(w1, indent) == 0)
    {
      if(!w2[0])
        strcpy(w2, value);

      memmove(ini_info->ini + save_pos, ini_info->ini + pos, ini_info->fsize - pos+1);
      ini_info->fsize = ini_info->fsize - (pos - save_pos);
      break;
    }
  }

  amount = sprintf(buff, "%s = %s\n", indent, value);

  if(ini_info->fsize + amount >= ini_info->allocated)
    if(!increase_ini_allocation(ini_info))
      return;

  memmove(ini_info->ini+amount+save_pos, ini_info->ini+save_pos,
                               ini_info->fsize+amount-save_pos+1);
  strncpy(ini_info->ini+save_pos, buff, amount);
  ini_info->fsize += amount;

  ini_info->changes_made = 1;
  return;         // Value was updated
}

void set_ini_number(ini_inforec *ini_info, char *area, char *indent, long value)
{
  FILE *fp;
  char buff[255], *tmp, w1[128], w2[128];
  long area_start, pos, save_pos;
  int amount;


  /* If area == "" then don't bother with an [area] */
  if(area[0])
    area_start = find_ini_area(ini_info->ini, area);
  else
    area_start = ini_info->fsize;

  if(area_start >= ini_info->fsize)
  {

    if(area[0])
    {
      amount = sprintf(buff, "\n[%s]\n", area);

      if(ini_info->fsize + amount >= ini_info->allocated)
        if(!increase_ini_allocation(ini_info))
          return;

      strcat(ini_info->ini, buff);
      ini_info->fsize+= amount;
    }

    amount = sprintf(buff, "%s = %ld\n", indent, value);

    if(ini_info->fsize + amount >= ini_info->allocated)
      if(!increase_ini_allocation(ini_info))
        return;

    strcat(ini_info->ini, buff);
    ini_info->fsize+= amount;

    ini_info->changes_made = 1;
    return;
  }


  pos = area_start;
  while(1)
  {
    save_pos = pos;
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)
      break;

    strip_string(buff);

    if(buff[0]=='[')  // Star of a new area, so break
    {
      save_pos = area_start;    // Put value right after [area], comment out this line to put at end
      break;
    }
    break_up_ini(buff, w1, w2);

    if(strcmpi(w1, indent) == 0)
    {
      memmove(ini_info->ini + save_pos, ini_info->ini + pos, ini_info->fsize - pos+1);
      ini_info->fsize = ini_info->fsize - (pos - save_pos);
      break;
    }
  }

  amount = sprintf(buff, "%s = %ld\n", indent, value);

  if(ini_info->fsize + amount >= ini_info->allocated)
    if(!increase_ini_allocation(ini_info))
      return;

  memmove(ini_info->ini+amount+save_pos, ini_info->ini+save_pos,
                               ini_info->fsize+amount-save_pos+1);
  strncpy(ini_info->ini+save_pos, buff, amount);
  ini_info->fsize += amount;

  ini_info->changes_made = 1;
  return;
}


long read_ini_number(ini_inforec *ini_info, char *area, char *indent, long value)
{
  char *ini, *tmp, buff[255], w1[128], w2[128];
  long fsize, area_start, pos;
  int amount;


  /* If area == "" then don't bother with an [area] */
  if(area[0])
    area_start = find_ini_area(ini_info->ini, area);
  else
    area_start = ini_info->fsize;

  if(area_start >= ini_info->fsize)
  {


    if(area[0])
    {
      amount = sprintf(buff, "\n[%s]\n", area);

      if(ini_info->fsize + amount >= ini_info->allocated)
        if(!increase_ini_allocation(ini_info))
          return value;

      strcat(ini_info->ini, buff);
      ini_info->fsize += amount;
    }

    amount = sprintf(buff, "%s = %ld\n", indent, value);

    if(ini_info->fsize + amount >= ini_info->allocated)
      if(!increase_ini_allocation(ini_info))
        return value;

    strcat(ini_info->ini, buff);
    ini_info->fsize += amount;


    ini_info->changes_made = 1;
    return(value);
  }



  pos = area_start;
  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)          // No more lines
    {
      ini_info->changes_made = 1;
      set_ini_number(ini_info, area, indent, value);
      return value;
    }

    strip_string(tmp);


    if(buff[0]=='[')  // Star of a new area
    {
      ini_info->changes_made = 1;
      set_ini_number(ini_info, area, indent, value);
      return value;
    }

    break_up_ini(buff, w1, w2);

    if(strcmpi(w1, indent) == 0)
      return(atol(w2));
  }
}



int read_ini_string(ini_inforec *ini_info, char *area, char *indent, char *value, int max_value)
{
  char *ini, *tmp, buff[255], w1[128], w2[128];
  long fsize, area_start, pos;
  int amount;


  /* If area == "" then don't bother with an [area] */
  if(area[0])
    area_start = find_ini_area(ini_info->ini, area);
  else
    area_start = ini_info->fsize;

  if(area_start >= ini_info->fsize)
  {

    if(area[0])
    {
      amount = sprintf(buff, "\n[%s]\n", area);

      if(ini_info->fsize + amount >= ini_info->allocated)
        if(!increase_ini_allocation(ini_info))
          return 0;

      strcat(ini_info->ini, buff);
      ini_info->fsize += amount;
    }

    amount = sprintf(buff, "%s = %s\n", indent, value);

    if(ini_info->fsize + amount >= ini_info->allocated)
      if(!increase_ini_allocation(ini_info))
        return 0;

    strcat(ini_info->ini, buff);
    ini_info->fsize += amount;

    ini_info->changes_made = 1;
    return 2;                 /* Value was added, area was created */
  }



  pos = area_start;
  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)          // No more lines
    {
      set_ini_string(ini_info, area, indent, value);
      return 3;
    }

    strip_string(tmp);

    if(buff[0]=='[')  // Star of a new area
    {
      set_ini_string(ini_info, area, indent, value);
      return 3;
    }

    break_up_ini(buff, w1, w2);

    if(strcmpi(w1, indent) == 0)
    {
      if(w2[0])
      {
        strncpy(value, w2, max_value);
        value[max_value]=0;
      }
      return 1;
    }
  }
}






char * get_ini_line(char *buff, int max_len, long *pos, char *ini)
{
  int c, buff_pos=0;
  char *temp;

  if(!ini[*pos])     // If at end of buffer, return NULL
    return NULL;

  while(1)
  {
    c = ini[*pos];

    if(c == 0 || c == '\n')
    {
      buff[buff_pos] = 0;
      if(c)    // Only increment if we are on a newline
        ++*pos;

      temp=strchr(buff, ';');
      if(temp)
        temp[0]=0;      // Null out all after ; (comment)

      return(buff);
    }

    buff[buff_pos] = c;

    ++buff_pos;
    ++*pos;

    if(buff_pos >= max_len)
    {
      buff[buff_pos] = 0;

      temp=strchr(buff, ';');
      if(temp)
        temp[0]=0;      // Null out all after ; (comment)

      return(buff);
    }
  }
}



void break_up_ini(char *buff, char *w1, char *w2)
{
  int pos=0, spos;
  int c;

  strip_string(buff);

  while(1)
  {
    c = buff[pos];

    if(c == '=' || c == 0)
    {
      buff[pos]=0;
      if(c)
        ++pos;
      break;
    }
    ++pos;
  }
  strcpy(w1, buff);

  spos=pos;

  while(1)
  {
    c = buff[pos];

    if(c == 0)
    {
      buff[pos] = 0;
      break;
    }

    ++pos;
  }

  strcpy(w2, buff+spos);

  strip_string(w1);
  strip_string(w2);
}


long find_ini_area(char *ini, char *area)
{
  char buff[255], *other_side, *tmp;
  long pos=0;

  if(!area[0])
    strlen(ini);

  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini);

    if(!tmp)
      return(pos);

    strip_string(buff);

    if(buff[0] != '[')
      continue;

    tmp=strchr(buff, ']');

    if(!tmp)
      return(pos);

    tmp[0]=0;

    if(strcmpi(buff+1, area) == 0)
      return(pos);

  }
}

void read_reg_ini(void)
{
  /* -------------------------------------------------------------------- */
  /* reg_code? holds your registration code, reg_number is the reg number */
  /* -------------------------------------------------------------------- */
  sysop_registered[0]=0;
  get_ini_value("QWKCNVT.INI", "Registration", "Registered to", sysop_registered, 50);

  reg_number = get_ini_number("QWKCNVT.INI", "Registration", "Registration number", 0);

  reg_code1[0]=0;
  get_ini_value("QWKCNVT.INI", "Registration", "Key code 1", reg_code1, 50);

  reg_code2[0]=0;
  get_ini_value("QWKCNVT.INI", "Registration", "key code 2", reg_code2, 50);
  check_registration();

// #ifdef REG
//  put_ini_number("QWKCNVT.INI", "Registration", "Serial Number", SERIAL_NUMBER);
// #endif
}

