/* ----------------------------------------------------------------------
 * FILE: as31.h
 * PACKAGE: as31 - 8031/8051 Assembler.
 *
 * DESCRIPTION:
 *	The sole header file for the 8031/8051 assembler package.
 *	It defines several structures used by the yacc stack.
 *	It defines several macros for testing the bitsize of numeric
 *	quantities.
 *
 *	Some macros to extract information from the mode structure.
 *
 * REVISION HISTORY:
 *	Jan. 19, 1990 - Created. (Ken Stauffer)
 *
 * AUTHOR:
 *	All code in this file written by Ken Stauffer (University of Calgary).
 *	January, 1990.
 *
 */

/* ----------------------------------------------------------------------
 * user / keyword symbol structures:
 */

struct opcode {
	char *name;
	int type;
	unsigned char *bytes;
};

struct symbol {
	char *name;
	int type;
	long value;
	struct symbol *next;
};

#define UNDEF	0
#define LABEL	1

/* ----------------------------------------------------------------------
 * addressing mode stuff:
 */

struct mode {
	unsigned char mode;		/* value to index with */
	unsigned char size;		/* # of bytes used */
	unsigned char orval;		/* value OR'd to obcode */
	unsigned char byte1;		/* extra byte 1 */
	unsigned char byte2;		/* extra byte 2 */
};

#define set_md(m,a)	((m).mode=(a))
#define set_sz(m,a)	((m).size=(a))
#define set_ov(m,a)	((m).orval=(a))
#define set_b1(m,a)	((m).byte1=(a))
#define set_b2(m,a)	((m).byte2=(a))

#define get_md(m)	((m).mode)
#define get_sz(m)	((m).size)
#define get_ov(m)	((m).orval)
#define get_b1(m)	((m).byte1)
#define get_b2(m)	((m).byte2)

/* ----------------------------------------------------------------------
 * yacc stack stuff:
 */

struct value {
	long v;
	unsigned char d;		/* expression defined flag */
};

union ystack {
	long value;
	struct value val;
	struct opcode *op;
	struct symbol *sym;
	struct mode mode;
	char *str;
};

/* ----------------------------------------------------------------------
 * IS_BIT_MAPPED_RAM:
 *	True is the byte 'a' is the byte address of a bit mapped
 *	RAM location.
 */
#define isbmram(a)	(((a)&0xf0)==0x20)

/* ----------------------------------------------------------------------
 * IS_BIT_MAPPED_SFR:
 *	True is the byte 'a' is the byte address of a bit mapped
 *	SPECIAL FUCTION REGISTER.
 */
#define isbmsfr(a)	(((a)&0x80) && !((a) & 0x07))

/* ----------------------------------------------------------------------
 * isbit8, ...
 *	Macros to check the sizes of values and to convert
 *	a value to a certain, size.
 *
 */
#define size8		(~0x00ff)
#define size11		(~0x07ff)
#define size13		(~0x1fff)
#define size16		(~0xffff)

#define size10		(~0x03ff)
#define size12		(~0x0fff)
#define size15		(~0x7fff)

#define isbit8(v)	( !( ((v)>=0) ? (v)&size8 : -(v)>=128) )
#define isbit11(v)	( !( ((v)>=0) ? (v)&size11 : (-(v))&size10 ) )
#define isbit13(v)	( !( ((v)>=0) ? (v)&size13 : (-(v))&size12 ) )
#define isbit16(v)	( !( ((v)>=0) ? (v)&size16 : (-(v))&size15 ) )

/* ----------------------------------------------------------------------
 * Size of user hash table.
 */
#define HASHTABSIZE		1000

/* ----------------------------------------------------------------------
 * Macros to nicely test which pass we are in.
 */
#define pass1			(!pass)
#define pass2			(pass)

/* -------- TOKENS ------------------------------------------------------
 *
 * This includes the header file generated by yacc -d.
 * NOPE is defined inside of as31.y, which does not
 * need to re-include the tokens twice, thus NOPE prevents this.
 *
 */
#ifdef NOPE
#else
#include "y_tab.h"
#endif

