// Copyright 1992 by Jon Dart.  All Rights Reserved

#include "stdafx.h"
#include "bhash.h"

static hash_t hash_codes[64][16] =
	{{0x09fa022008550102, 0x731a13283289375b, 0x779807be7146f3b3, 0x11be7d2b0a90b21b,
	0x6d1e3613c3afd02a, 0x08703d506d38f5d3, 0x236160dccfecd6f3, 0x3fdd4ce299a99a73,
	0x7a931066671db88c, 0x5ccc60267d79ab1f, 0x19cc51ab43de5890, 0x7c51579929b16c2e,
	0x2cea51be009fa589, 0x0043567a627644c9, 0x349b486a479a9460, 0x7ff12288554bce2b}, 
	{0x1cb876160d693d68, 0x3111310d6800fb7e, 0x0cd538ecf9411211, 0x5cf9051b3232675a,
	0x27167d3be3c23f36, 0x361243168876bce7, 0x29440d197b752109, 0x21891d72b31d4098,
	0x241f034c0ae0e2a6, 0x49477ebec59e828d, 0x3cf6656b553c3740, 0x0a2414e2079d2290,
	0x2d762940a1ce36f2, 0x125721a5928dbbb7, 0x75c02971d3de552b, 0x44313472fbfdf49d}, 
	{0x74be4d5fb74b82bc, 0x310c315b2c7ea349, 0x5ab6684f63c6628f, 0x407e495e562520a6,
	0x2a1f13c8b5a6a7eb, 0x57d247e7ffb8a9a7, 0x3ab4093bfb608f6b, 0x15b859973c73f103,
	0x72c83eeab5a07e8b, 0x763b3044006eca95, 0x2ad94c0275fcb7d5, 0x72f4284190a4f05a,
	0x6f67580ae54f39b2, 0x737d52df149cf512, 0x15061981ecacbad8, 0x22290a334fae497e}, 
	{0x52b327c3ebfac15a, 0x78f2721a6be9693e, 0x5664242d2525e256, 0x1ab30a7bf39f2652,
	0x07e37e813efa1944, 0x4c9836cce0871537, 0x6ddb588ce49d43e7, 0x23d25ed8d28415a7,
	0x69334d068c9dbdd9, 0x3b8f0cbf5d10fbf9, 0x3a9c1db956ae18ba, 0x07292c3e8140611d,
	0x06660ce61500ff05, 0x149d4f30386c8a3f, 0x4a9865e46a342575, 0x53401b552e747602}, 
	{0x7b3f790a0ff56b1e, 0x32ad455392ab0562, 0x39060ecf342d1513, 0x2e023cfa065a4432,
	0x5607163005df27be, 0x174c6eccb9e9db3a, 0x1ce112524c9bdec8, 0x373f3ebd92a696f8,
	0x0e0b0b6935964fa9, 0x152b2037892b10fd, 0x076806d830611ddb, 0x7b2b6f6117677dee,
	0x0a1b4a9af34159a5, 0x0aa06fa1c9429620, 0x129c6fe0a125788c, 0x74dd1360f6e8a6dc}, 
	{0x370b08fb073c7262, 0x2c24710d8eaab137, 0x7fc55e7d052ffdf2, 0x20d209640e8bc198,
	0x2e36079b0df4e254, 0x5ed642f0976653d3, 0x65ee61d6c84a82d8, 0x1f673ece1ab1dfe8,
	0x6bf62bdbd6cb6798, 0x22f76ab5b1e78265, 0x706447a7b7a507a2, 0x676274331091ce60,
	0x162d67eec7f299cf, 0x0e6c013917e8b21b, 0x2d3b2cbe67ae1429, 0x086a51da85208140}, 
	{0x52bf73603a4e4802, 0x573f2f50cd4d27c2, 0x6bc95d7f8ff8199d, 0x7d8a4c4008eb695a,
	0x2e16538bde59dc81, 0x6e1e7a3f070658a7, 0x2b2806606d19cf64, 0x0fb3589487fcd7ed,
	0x119c342315fbb7c0, 0x28db603f83ec4b73, 0x18b9346d218897fc, 0x4837713baab55f86,
	0x4a450fac5671913d, 0x7cea4501eea4f611, 0x1e9c25c6117fdbd7, 0x734e6a4d38b432e1}, 
	{0x1f0327ff8d2cde5a, 0x49e62e243d7ca085, 0x02396a1c47da6d41, 0x3294161672b284ca,
	0x774f4ec87cac2640, 0x340c4fc1184f42d8, 0x12ba5338d1f8e838, 0x1f893996775feaf8,
	0x11a77e0b16656fbf, 0x0ebf68e02d60e3ed, 0x678e3574239c0f54, 0x7e105101a24bb7f4,
	0x4a0a409ae4a0922c, 0x57026fff9b3dfc66, 0x2ee8783f75c82ea3, 0x7ef12440ecbd61f3}, 
	{0x707d6aa06654a546, 0x3e010f924ca0d685, 0x0c3d769d22a37a88, 0x12572b6f4895dcbb,
	0x2f8a221a6d0d530e, 0x4389727e554beb0a, 0x06ca2fa50a576e9e, 0x4a52435c0632007e,
	0x02bc37597fca41ae, 0x608b609d5bec4514, 0x080c4702f4ee3195, 0x4d56320e364be1bd,
	0x3d254d8239dc6e57, 0x419d2b3dedfcdffe, 0x6a465573e838ef16, 0x58bb7b3cffd0382a}, 
	{0x1fd8530daac68f20, 0x117869a1e59e0345, 0x16ff094a95a3409e, 0x533b04d108cebb82,
	0x006f4a49b61c72e5, 0x537b057fce7fad61, 0x358116f0ad248362, 0x6f770b6fd44c8272,
	0x7f8561d676695f2a, 0x4e2717803cb7eaac, 0x695a796048103d2c, 0x5e7266e9222b6776,
	0x4f3d5d2a980675fa, 0x05a113c330a8363d, 0x20df02a840fe803e, 0x12141ec84e0660b8}, 
	{0x09ba4d0dbe010dc4, 0x4633465a78dfdfc7, 0x73a43c6b97a9432f, 0x0fa84ec7b014e9f1,
	0x17a5181cddf91641, 0x3ecc1fcb10f46180, 0x11061862d0cfc7cf, 0x525e5b56fd055b47,
	0x26aa534aa1017a4e, 0x2b7b51907c68ccfa, 0x3ea270d6550ff602, 0x3dca761aa5e45131,
	0x2ffb6a5ac97d7bcf, 0x0ff83a9733881d05, 0x66da5927d3cac1a4, 0x2064726c37f70452}, 
	{0x0ecc18678805118f, 0x021b23c3f24aa491, 0x37573e489c068267, 0x460669d6be9eb15c,
	0x26d5305de843511e, 0x04624c6a2731618b, 0x4f82574108485db2, 0x1a7130991e33f4f2,
	0x1ad3357d24d2c4b6, 0x706f46d54a2763c0, 0x7f0965abd07c9c85, 0x5bc81a2a7cec2982,
	0x130743dc0f21cf13, 0x718824c342642cbb, 0x1460463875cc1456, 0x5d130db098b9cc2a}, 
	{0x13b548e46e520d5a, 0x0f1673e7c0470ca5, 0x3b3e51299b8780f0, 0x58bd6a882e23dc98,
	0x63a60bd25b18b3f8, 0x47270a65a23d8625, 0x6b1c0ad56affe655, 0x53173cc1562f36e5,
	0x02a70636a69aee7c, 0x38ec2358539caa43, 0x65b92429f066f29f, 0x0cd341a0e53cf97c,
	0x30080c7731514380, 0x5f3b4b4f2c837d42, 0x45994b227db35bde, 0x05893c1dcee4e3f5}, 
	{0x011d464c55e7f183, 0x5b0d1ccbcfbe5186, 0x1c814b55087e3ef6, 0x0e3519647edbb4f6,
	0x07c177443d1a4ec8, 0x4e014cc38ea12972, 0x21fd7e688de28385, 0x6bb7655541d08b14,
	0x08ce173f4b9b2a3d, 0x44d80720c4ed1b44, 0x71da0c96685d39bd, 0x09520f049d4b4ab3,
	0x42a63af274ec2a53, 0x31f71b413dada9fd, 0x5aac7d2fc0aef748, 0x3b2e7d3ab691f1e5}, 
	{0x43ae4c67a24331e3, 0x77e978798d142b39, 0x3c491715dab93e25, 0x10d772f2ad7e014b,
	0x50cd137a1367b50d, 0x03da7a8c78632514, 0x764c11408562d68c, 0x37bb43ddfd6dddf3,
	0x5bf10e5eba932814, 0x781d06354bc3ad09, 0x6693133b707032c8, 0x6daf58dfe110263b,
	0x0a8a1a169e535547, 0x66a675a34429cbd0, 0x77c205a5936eca1f, 0x036b048fae5520af}, 
	{0x4c0e6afc3b66bbd8, 0x1b9054f8e833d441, 0x3fbe32b1858881a8, 0x6f0927bb37440eea,
	0x0073553ee29ff8bf, 0x77996ec86e0bd231, 0x2e3236a6e86d0136, 0x6e8a25e229df9475,
	0x2eb8655c17c11c9d, 0x5aa128a01645dc54, 0x4b0d4060bd2f1f2e, 0x3a5129b8700316a7,
	0x4b5c48abf4641498, 0x1e2f2f7c8dbf7a1e, 0x050221cdcb21356f, 0x47a639a594491986}, 
	{0x0ee505d586d0043c, 0x1fec74504e8207a0, 0x1007307100ba882b, 0x1b341c4619e2a4a7,
	0x5c5b055630e1a85c, 0x5c267880fea10b6a, 0x53d775e2cc73a5cf, 0x2b8c0cebe07b9a0d,
	0x37ca6a0009e6b6f3, 0x184e6a68d11ba06a, 0x6a70304d83a8c1d9, 0x53a04017861c240a,
	0x4cc339793c7f3a00, 0x1d7a11d5e7b7d2cc, 0x2e9522efbf771ac3, 0x55483803c405071d}, 
	{0x04dc54b8687ffb6a, 0x02e40c8aaae8fbdc, 0x5a4d369cbf9f56da, 0x6bc1691bd2920cd5,
	0x2e2c408b03cdd8df, 0x08695abc35ac28e4, 0x35626a3cc565e321, 0x6e292e80c11b59b0,
	0x31d03e14b44027b1, 0x010a3b95ab6b730c, 0x53e31349796d5a35, 0x02040e84e1d1d9f8,
	0x5a683347ba8318bc, 0x4d6f59b59ed76b3b, 0x64a26e53449fda27, 0x5dba4f321d9f92a8}, 
	{0x2a9a636e46f4133f, 0x665f47ac6ccc6bf7, 0x0fb77f7cb9076b5e, 0x1b165ac35f0b1048,
	0x438e77a5df8218c2, 0x774a4c82a0330342, 0x64fb42fcebb05e7a, 0x19c97429e916bbd0,
	0x5b71575ebf9021f3, 0x08bf003051de4f80, 0x5a8e2d9dd38bac2e, 0x71e53b88bf1b3e85,
	0x43f350de36d17f86, 0x3af6382481675f51, 0x5b527d40b04a5727, 0x766302b8fcafe5da}, 
	{0x00c811bf072e3d16, 0x104743c181179174, 0x656e59576240c9e6, 0x479d71c53d83f953,
	0x6e2b6f6ccba07b03, 0x47b178dc4ebff6a8, 0x38cc4369d14635a3, 0x75d47b6ff6422961,
	0x77567fa84f14d755, 0x47541040f09aad88, 0x159a57924894f72f, 0x33ab21aadc70de43,
	0x5d0b0107f348c29b, 0x16f7522bdc2f4770, 0x0acd5d00d7a6f2cd, 0x18aa0a1f3f4da8e7}, 
	{0x0c0e13720fadeaca, 0x6a8312ce56302558, 0x54992676b11bf21b, 0x73bd62a969b492ca,
	0x03a840a84c47921b, 0x3c867ed1ca56dab9, 0x4afa42cc8f940ce9, 0x2db215d905f88bd7,
	0x4c2654b8098df8f2, 0x78b137cd35488669, 0x602f7d6e0e95fe23, 0x3bbe4f7378c9c247,
	0x7d580688b946b1b6, 0x365a40d07e773d7c, 0x2f3a2ed90f638da6, 0x21f950ee42110583}, 
	{0x55137051446f0cb6, 0x02fb3addd8fd6226, 0x1a615d211ae7e729, 0x05dd15f863d32300,
	0x5db0582067156e08, 0x3cb1716924800899, 0x79b02c6db88c0318, 0x50cb48f0b40f4d25,
	0x248a48561239b766, 0x7cbe36e0500f54e4, 0x59741f79d81f0176, 0x6287076a4e9d7324,
	0x00847829cbac9e13, 0x1207111db305dad8, 0x48c128142fb189bc, 0x53b776aee411a5e0}, 
	{0x688004220cf514b7, 0x0b9735f576e700e1, 0x37ee07a0937329bc, 0x48672839259976c9,
	0x59e9769da22ba143, 0x531a57ade9455bea, 0x67147f94639bbd7a, 0x52884e3a20de57be,
	0x4f2a204c11d8c5cb, 0x5763417fec97113f, 0x649251fdde41c313, 0x646c40179de3faed,
	0x463440b4f0d95b6c, 0x46e6441a4a223968, 0x1b8c11f88a3ec79c, 0x534b4ee681e5b3b2}, 
	{0x56fc7eae4b3ef428, 0x5a3f721f1cd63a0c, 0x7267443a920fbbff, 0x69c269f4303dd478,
	0x59fb30e702283cc9, 0x2ea92ca4252b2bd2, 0x794e4f8924b359db, 0x0a4f1341e73d1396,
	0x1ead76612baa54be, 0x30887fb53706363d, 0x28b13d40d48b8567, 0x61d624022213e236,
	0x32121eef6dad39fe, 0x15e04ece8f95f290, 0x2fc049cff83baa51, 0x2a1f611ef8a5d72d}, 
	{0x353063bd68c91ee4, 0x68db51623830c8ac, 0x52f445380d8b1d9d, 0x7c565fb27f7708e0,
	0x438f6fc60e2ad115, 0x22475f57673b53f2, 0x5a87439313427b86, 0x3389398d26846a21,
	0x69bc385c056f155a, 0x54150d87df05bc20, 0x10f71d8bf00c085c, 0x5f2e11b41a243313,
	0x2bc625a408870983, 0x63db1a4151a41e32, 0x518840dfcd581366, 0x459a68dfa6e93c05}, 
	{0x1bc40b1749978347, 0x555329c6b8dde544, 0x26bf50e2783552c4, 0x768b41f9907e5a56,
	0x004d7002cad37123, 0x24da52ccbcfd2b6f, 0x78e716fcc4374348, 0x6d9f16a47a8ac653,
	0x0afe2806c4653a3b, 0x31f27aff12bb1dad, 0x4c8f4326e7548e5e, 0x44da1bb4448f7817,
	0x1ef83b9906461e38, 0x39c0037bdd1958b3, 0x110a7c72dec263e7, 0x772455af69c7896c}, 
	{0x275f20835125942c, 0x618f4553094549d7, 0x7eed4180c9deda1c, 0x32c97d54610993ac,
	0x7ddc4262bb41ae6e, 0x514b5e0db1778ded, 0x0695190a4c03536b, 0x3bf7341002a6109d,
	0x611c5b27104c747b, 0x5e074c247dcf5327, 0x4e9f1259f071d958, 0x5f44088adc49b955,
	0x7b4f1b972d4b47d8, 0x44785b85ff3ab9f6, 0x426e15ce822b7c60, 0x74254b199dd7ea17}, 
	{0x78a923ca687543ef, 0x7377621218502fe9, 0x30a9055a76d4b7d3, 0x6f7832487051ff37,
	0x2de54bafe81499f2, 0x66834c225430d38e, 0x79ba2d064195cdbb, 0x05fb4f595ab1b2f5,
	0x4ebd3b870b64f4b8, 0x103c78fe4d69d751, 0x4e50036cbff529b7, 0x5ed252bfa0cc8161,
	0x34735465c274d89f, 0x54ea676607cfdb5e, 0x7ddd3a3b8ec1bfdb, 0x560520a221310839}, 
	{0x344c68b4f205026c, 0x14f4320a54634f7d, 0x55191eb874e96b94, 0x4f00355fb90c65c9,
	0x060f44b1d66bc42a, 0x47695c113332d6f6, 0x0c7d4a38b95b5284, 0x17125a05a10095cd,
	0x3a8a06ed5d6c6d0b, 0x247a6d963030a36f, 0x46c922a68aed4165, 0x57ee28dace0dd84e,
	0x420d48cd8b21a247, 0x5ffd6026c01fd6ce, 0x1f7f2b0358350ee5, 0x1a2d61d4526d0c85}, 
	{0x02ed1709d6d5c1fd, 0x73ec5b43a968e416, 0x496523e1396af78a, 0x57c90f20bb7210b7,
	0x00033a308be64011, 0x7ae440e45b02f048, 0x3d077be749470391, 0x1ea3799e736d241a,
	0x0f2a4f222ba40f11, 0x1aa709f3534d2f45, 0x7733105106eb61ce, 0x42ff6d63238649b1,
	0x1fc52f95cc32f60d, 0x7e9b72ce7af245ac, 0x477b3d6db4b4c988, 0x22034e360da19f2e}, 
	{0x11372a917765f37e, 0x624977c686c6a6b9, 0x2eb63f1eba27dc61, 0x743b7c13723bc8d3,
	0x19680cf48ba5a024, 0x2bde21a35aa9fa29, 0x4d7f615c07c7822e, 0x301807acef4f464e,
	0x3b4679ef1acb8de4, 0x16ad221d64667616, 0x62b500b369fc4cdd, 0x7c6d36e2dc2ddc9c,
	0x4d4213864d06a5ab, 0x6dab4064008f40d9, 0x59f95ac0f8f0d451, 0x32f05950b065ebea}, 
	{0x0fcf7314bdb4894b, 0x55f1159ad764d0e9, 0x6a342eb76d701d44, 0x72bf6cc15d9fd871,
	0x53e771c4dd46f35e, 0x293e19543caf4ebc, 0x440e2de087cbf026, 0x42d811b5b17c675d,
	0x3186411a50211722, 0x60727e1d90a2f1a6, 0x50783c1469b143fe, 0x449f4fe0b87a1ba3,
	0x4e2d5368517c015e, 0x0e145df3a2be62ba, 0x7f230045fa188f4a, 0x765d2aab19d19ae9}, 
	{0x336014590d41f440, 0x68ce36c70aaa1ba9, 0x250644f447153ade, 0x05bb05b179530966,
	0x3528716a06e9cc39, 0x65ec37019019c4a3, 0x6adc28b9e1c2f0c4, 0x324b5943d84d6fba,
	0x6891326b736560e4, 0x63e059fa86ab9937, 0x4ba31ebe3b19071d, 0x3fff36e5f36510d9,
	0x2a2d22039ff5dce1, 0x64c054802cc6c431, 0x231f3f4410dadc00, 0x79b01dcea47dd5e2}, 
	{0x349006294b0c26b7, 0x58c651560d7ebf7d, 0x4c55681cbe64355c, 0x43991f6b4291a503,
	0x46d268ac0b2f3db2, 0x78cf7db16371b803, 0x50102d2fa89ba2d5, 0x3dd953dd0098c75a,
	0x5b102faca7d798c7, 0x30de65bc72a6e98e, 0x235f18f894c4daa5, 0x76f41e774b6444d2,
	0x6ceb061e7d4f8670, 0x1a962115eb6fffa3, 0x76153d040f671c7e, 0x2e534241307f4606}, 
	{0x5007144cde94908c, 0x07c34f4e4d4a7868, 0x11481279c72e9069, 0x26bf4678b80f741d,
	0x168d08537236d645, 0x1cd1646d42bd0180, 0x45d32a8cf3c7a8a3, 0x08ea2b0c47b75aaf,
	0x07aa6ea3923671e6, 0x7b53456c013bd9ee, 0x6ad42f09aac17d81, 0x55e66d20fe6fc1a2,
	0x260e5a83afead3c5, 0x7c7e34b9ae012df3, 0x6c2e32cf4b6e3050, 0x69ac5b8d1a6f1509}, 
	{0x466e5e8aabd9251c, 0x7bac0eb7b8f37eed, 0x69085251d8c24c30, 0x0d973b5f5706c107,
	0x36035527409daaec, 0x30d9563c3c86fa3c, 0x624c2416573524fa, 0x1ae63c564b7f91ad,
	0x710879195ac21ddd, 0x1b2911116293e31a, 0x7929793a33a0321d, 0x2d3d3d67c8ff11db,
	0x693f4df87ba6121d, 0x7b607475c142e884, 0x3d916deb9c1e7b80, 0x652561394065ed1f}, 
	{0x5c6d58ab185b5540, 0x5e69619abae38a11, 0x0cbc49ede6f0eb5c, 0x3a8772a81d2c5595,
	0x3ad927effb854923, 0x37d03226dcd27ddb, 0x7fa43117ea54b626, 0x5f351946294955c8,
	0x1dd22cd7a43a4dc6, 0x0c4754b342540057, 0x698723d36470bb65, 0x31625dd4e9093b93,
	0x4e266346a7e21632, 0x2c243950f37a483a, 0x66684fa15628de9a, 0x3e247ecdfef8f8fc}, 
	{0x5aab4a7609191156, 0x7de10dfe43ffd757, 0x798d2f9567066e1a, 0x53f914dc87ba791a,
	0x3eba2d75a98dc6e6, 0x589c4b333229e181, 0x3c027cd5401481f2, 0x253f07617feb10f3,
	0x18af3ba297dd333e, 0x6e95105acea6ab67, 0x1b156f1cf2c058c4, 0x7abb50f01c07cc5b,
	0x706a7135777e3041, 0x47b350529071e878, 0x26d84d394fbab9aa, 0x761313d23340ded2}, 
	{0x0dd14fb4e592cd37, 0x4bfe4debbeb01ec2, 0x70a34d9150d45615, 0x64547e829365f76a,
	0x5f4c667fcfd4b2b0, 0x5e26686ccb9202d1, 0x798f469971e426a9, 0x206d00316abfaba2,
	0x70472b45d66c8060, 0x05fb380fb430dc8e, 0x30fc2f5c13a1cc25, 0x05b14d42a0efca47,
	0x6fb4228cb3d83105, 0x2af37a83676fe122, 0x030b6ffbdc84f03c, 0x725933cf3cd5c956}, 
	{0x4686582d91c77841, 0x5ea74f691ade98d6, 0x7725022816aba577, 0x5a004022be6719d8,
	0x3e3727d672fa1d7d, 0x375644d8b39639ee, 0x5e7261aa10b4c718, 0x6825313d889a90c7,
	0x37435586882666c8, 0x3a6133d9221a0d8f, 0x12634cdd7da157f4, 0x32ad3d533239c1eb,
	0x6fab76159fd16cba, 0x56cd6cecc43bcc9c, 0x4329552fd3b6e35a, 0x7c5e264cf6cd62ba}, 
	{0x597327a97335844e, 0x6fc43481c5effe96, 0x563b3fa3b059deee, 0x07651e450777a838,
	0x01241a42181e9bc7, 0x77150f7f793b5f7d, 0x54d4355234f30489, 0x77d07c0df5d5a8d8,
	0x044b682f53ca9691, 0x17af5fc1c40c38af, 0x6a7343e564d0bc1d, 0x46153fa910dcba5b,
	0x17f83e9600c8b11a, 0x70295094761bc3c9, 0x73592e1c8afe7391, 0x418a66d1bfc0d4b3}, 
	{0x1f4055f070dee4b9, 0x5d3e133aabcb8b86, 0x1b0f0c4a932d00a3, 0x22eb1172e9cceddd,
	0x51bb3a8ac6e03c8c, 0x54496b6a2a09cda0, 0x0add3cd7729100c8, 0x2ed7762a51465ec6,
	0x720665065b974156, 0x4dcd0bcec92dd5b0, 0x285324bd7fbe3c0b, 0x685226cef6513e2a,
	0x144222d81d9d5262, 0x3ff04283cad95f0c, 0x63c4400bd58d03ec, 0x534524e774c6c773}, 
	{0x74944ec9ed40085f, 0x28fb759d3adaf8a9, 0x16c60265b4f98e44, 0x46fa4631631db29a,
	0x5da3597802609246, 0x29db6fa053085efb, 0x72b40782e0fd5c21, 0x50a0691bb8479906,
	0x1f1d21d4474e1a33, 0x30a37c08de24dfd6, 0x7f2d13ad037a98a9, 0x25cd794a238d576b,
	0x14321da3baaf214c, 0x330853c28cbbba48, 0x289264440b1274f6, 0x26f74415747665af}, 
	{0x3a1751fed05be29b, 0x78e459b05f037e83, 0x5e8b503c880a88fb, 0x71fa1d0af3a341c4,
	0x56851bd2d23caef0, 0x76b4272903be6ab3, 0x4282389b13273a5f, 0x04955268c7adc58b,
	0x2e3748603b2d51c4, 0x381a68773019cfe4, 0x662648fca6941263, 0x6eed71a355098db3,
	0x4b707dbe1ede6f15, 0x5a5b09580c886ee3, 0x19ea080f00bd26bb, 0x16085be49be8589a}, 
	{0x547273557fafe348, 0x16e1317d89aed717, 0x4b84381706307273, 0x06532a869514622d,
	0x72087a61ba952207, 0x5dba110ec635cc69, 0x746e076a207e3ace, 0x561d639a9dd0c9c8,
	0x45fd5674def59924, 0x00187d246db39f1f, 0x186810f49c1b6c24, 0x18197e63c7bdec13,
	0x71a365f30d880d78, 0x6ad05c4d168993be, 0x53f42cb40b3dfdc6, 0x5de037db46b4c9e7}, 
	{0x2c4d1a97df3bfdc1, 0x70db630ca4c23ce9, 0x7ad9103da2bc4cd9, 0x4a6d372cc8a8602a,
	0x69d541edc109ff85, 0x25d72056a9f2dc42, 0x46a13f369ef27f39, 0x349f0238d68911b2,
	0x11171dd654e422ef, 0x48865a16c21bc849, 0x151b4bdb9c9fe659, 0x59bb42113920fc21,
	0x42734d08cd8f4db0, 0x3d4e39aaf785c43d, 0x36da677a00c15923, 0x1fe7578255f062ca}, 
	{0x2e4f038c557ea2e3, 0x18b848631fa666fc, 0x4db342f7537b97d7, 0x68b03f848817048e,
	0x7b93133e6cb9d7e6, 0x39f23c093cfe74e1, 0x3b433f49a1f3aaeb, 0x738547ca912e85bb,
	0x3e2c444f44bb9e3f, 0x754d1355ddf643a6, 0x1f666dfadb2f43ec, 0x503a1336e829c6ef,
	0x7d8a7dc62450017a, 0x4ebf02767ec31a44, 0x66c161ab36f91b5e, 0x61866e6123344bf6}, 
	{0x4b213dfbc878c209, 0x44622f8ce64390d4, 0x693a4e8d8ebf5599, 0x6f857417549897ab,
	0x68eb631bc043bb26, 0x28f33f308ae0ee69, 0x187b7ae9bf6fdcb0, 0x4a3601d86a988ed7,
	0x7fe443a7d2b83fb1, 0x0e5330eaeb6e8af9, 0x3e727f990e5ac349, 0x7bfe7c599050d511,
	0x668d16f557ac7a0f, 0x400a0bb9f70b2e37, 0x4bd3588d8215a681, 0x0c2364018e982f9f}, 
	{0x776b2dad9ca8d00e, 0x4dbf5a8f67007374, 0x36954547475608cc, 0x5152396c29e3e456,
	0x77857a4d44c83dc0, 0x25c178d222a0237f, 0x6871795f0ed6b8d4, 0x541a31eb801c177a,
	0x0ce969a6a59bb660, 0x3f822edc9b299787, 0x3d671d006c31295b, 0x416f3c02708c319a,
	0x45270b5d2b028a2d, 0x56171e7b31a41af9, 0x12381d6939c4da18, 0x6d2853e9f5b33879}, 
	{0x2bd40a6bb70ebd4e, 0x32b97f7290c34860, 0x62ed4d6cf690b199, 0x6480280c842e32e3,
	0x7108759dfbe76eb0, 0x07462bf711c46c45, 0x794e55f323185d23, 0x10990d8a6f948996,
	0x1fe05814e2a535e8, 0x58c17d33194fe312, 0x2b6c7676a842b68f, 0x68f544ba4655651f,
	0x64fe21c5e731810e, 0x79ce77c5785678ed, 0x2a18158631351a30, 0x35fb0da1349d0db6}, 
	{0x65045ffb7e2afaa4, 0x1537483fcef6c99b, 0x5f6921468e3cd2ae, 0x63780c7d62334d24,
	0x231d45d06dbfe171, 0x48690fa7e456a460, 0x5d3a3fec13a56fe6, 0x631b7e3e5556cda1,
	0x777404b82d936d63, 0x4df77ff8128867de, 0x5ba95044fa9d2bcf, 0x1ef83d074da279b2,
	0x15ba74f7509a326e, 0x3817489e9b6860f7, 0x479a3a2ec0194653, 0x7c0614b2a9edda0a}, 
	{0x23a40e25d8fb796c, 0x3b2052fe0f7e2fa9, 0x61320f1b87ab6c34, 0x6ca1674942277d6d,
	0x5f6b2fb19ef0a5ff, 0x08124eebaadb24f2, 0x6a5c7a94756b0feb, 0x1308218ed03a4d8c,
	0x564b395baea6906e, 0x370e0f32b5fa9daf, 0x654702b316d2ca87, 0x73e07f7445ecf7e6,
	0x2b0473b9ab1bef88, 0x41da360fe7a26d7a, 0x62e618a7ba9dc18d, 0x38af20a331bb48aa}, 
	{0x6c5c48b32900ac81, 0x0e5e31b5c0c4348d, 0x616f7934d4aa00d8, 0x02626cf721c38c93,
	0x7b9b4c07139a4dd5, 0x092908caef5cc49f, 0x3add5d315ddadd7d, 0x4bc84902fd98f3cc,
	0x030d13c5099d4f25, 0x4fed76e9af4e7fc7, 0x236e7a0932f154a2, 0x5c151a866a28e9d1,
	0x7c8460d4bd148719, 0x6bff591f284cb85a, 0x3825523a75736b69, 0x49601cfc2b9f7f47}, 
	{0x47d3176b573a843f, 0x1cd86a6eceb011cb, 0x1d4955d9ec8b91c6, 0x0b2506107c3ec4e7,
	0x51540799d45bebef, 0x3297504dc160df8b, 0x2ce4530c61e3fc66, 0x1cc37a227a472754,
	0x486305be64b8dc21, 0x787c77252eab86eb, 0x3545368e04870b8d, 0x2fff50c77bd0d904,
	0x65e15310cc654e5a, 0x2f6e3ed6ae2cdafb, 0x477f1c2fc8caa7f3, 0x6f80294672af2b17}, 
	{0x42b25617c8267282, 0x18767730a8a88066, 0x15e82f52c51c9fa8, 0x51514f1b5150ed3e,
	0x3e3f233064530fc7, 0x0f422c7badee4f59, 0x629a5b6d97f30df3, 0x79626e77649ed398,
	0x74f5550e64b7e780, 0x34a343efdeb8ab5f, 0x7df54c8cc1a6b131, 0x2c0618beb4dacf93,
	0x46c335349f6d1409, 0x290f683c458bed3f, 0x551b3fce075055b6, 0x50771907668373cb}, 
	{0x6da0347d60c668a4, 0x572046043a94bae2, 0x107323e6d3ad2caa, 0x034e18a01e3052eb,
	0x24033394c923cc5b, 0x4e13185cc28f6c2d, 0x4227099b94fb31f0, 0x390b13885975628b,
	0x5b6a1b7c2ed8a4dd, 0x2c4a054eed9d69e4, 0x24a564491edd86fc, 0x70931cf3d3be5513,
	0x02d4460a790c2c60, 0x19ca4a5a3a2f8a8b, 0x69221a5e0736d8be, 0x75ac73cae3330299}, 
	{0x5d4636668798d781, 0x52be38f0f2dc7942, 0x161265de0c0dba78, 0x33836728df96bcc2,
	0x6849218d8ae8b224, 0x41f202f88b4910aa, 0x75b204e06e6b0aa6, 0x17270add7623bba0,
	0x526a46d269dbc352, 0x2b59574b0a02b7c0, 0x147e3a0e513b4dd9, 0x020c3bef15737596,
	0x01b9185821a0661b, 0x66884e375c60cbc2, 0x4fbb1d291f2b1096, 0x4c89751447560133}, 
	{0x2a4b3399209cb176, 0x293825ffbe67f60a, 0x73ef3b82e68c4a3e, 0x585a733a12b87516,
	0x74b929e3ab44d41f, 0x61c84f5818a494f3, 0x6b630882ba31bae3, 0x331f29ff58814acc,
	0x349f18d63a00757c, 0x21b849ed600d11b5, 0x7ca81e23104f4632, 0x48db08393971b8b0,
	0x2f1d12e7dc0b1375, 0x182f50dbf7e74a47, 0x190f4d75245f604b, 0x26750c6f700419cd}, 
	{0x715957e09252685c, 0x1c7657360b9deb3d, 0x3b2f7f1a57fa5ea6, 0x4c3a295eb55045bb,
	0x36fa5d5db155c2c8, 0x487b5483f6a7d3ad, 0x556363ca8dbee2f2, 0x10597a751ee5f781,
	0x60ae275146066c76, 0x2350613c9f677007, 0x504a74d08f2a34f4, 0x116648597baf2874,
	0x7aa6707f70aa072a, 0x5ba9234fda0a1ffe, 0x1946448b6b80a866, 0x38d85d62bad4751a}, 
	{0x53152301c1b8ec90, 0x12610a9fc765925f, 0x40fd1eeed3a5f6dc, 0x4d8b2a1c46937704,
	0x20b520c5a4ba8f19, 0x34f55d8131db26fa, 0x29da7a007d00a39e, 0x163f39c765272bb3,
	0x39425c0ab32ddadb, 0x68071540f3374c7a, 0x468d365d855a5889, 0x0c1776d89aa64e77,
	0x57fc3fe7225e9176, 0x01dc0a9a5190e54a, 0x68872fb3cabf4af5, 0x1d1b3f7604debd4e}, 
	{0x742968c5154fb0ee, 0x14df72415e27a374, 0x1e7f1d46505e968c, 0x7eb649fdc13ad4c4,
	0x27912ce10794cc8e, 0x0a1c295b5846657e, 0x22f0426ca0689f32, 0x1038597e499fd1d5,
	0x250174cb263471c8, 0x4bc652020a25a051, 0x5a986f1127f073df, 0x4d55423dd14b34cb,
	0x3ec863e9b7868514, 0x7fb23fc42bc2b50f, 0x62fa503596cb2982, 0x50a43e31abb91b1c}, 
	{0x7d3d50f47116a5d0, 0x51d93424c0ca58fe, 0x30dd106944743de2, 0x66211188a57ca550,
	0x45370e7ae0828a23, 0x4eda6b187871ec5d, 0x3ecb48568be4f87a, 0x2dab7f216a2452dc,
	0x0e93035ac55a61d7, 0x4e7677891157e54f, 0x4b943f362a7ac75f, 0x4d870d10e0186303,
	0x2ab2134b7602323f, 0x6e126fd6b56728b1, 0x28c87b58a6d4a419, 0x34dc191e8d7d39b8}, 
	{0x1af857563c8d3d13, 0x1b386a5158b56b01, 0x194022a0a4b66e88, 0x6e353d46ef11b47b,
	0x774e2657b6a35b53, 0x2e1649c01f61923a, 0x3f962b0554e54ec2, 0x02020438e30d973a,
	0x64a16d8036605c24, 0x13fd59ddb67615b8, 0x1ca85b12c50816f5, 0x69176dda0403e534,
	0x1b6158d623306ab3, 0x09e33bd7bbc65912, 0x1e191a654d899e45, 0x0f2b43c206c140d5}, 
	{0x7e034bb25bb26811, 0x66e222cf15d11501, 0x3cd01233e6742aaa, 0x655a3dbd1b304b97,
	0x3f7f29428c98501a, 0x76b8605c59a0b239, 0x2b771dc29059c5d5, 0x04a4764c54320ae4,
	0x19d36d059d85944b, 0x6445410727a8a850, 0x14fd0aedac2aa00e, 0x606c2f21f98443f2,
	0x147e135205f180ab, 0x340f38cf8da6df97, 0x6b142aa462197912, 0x08fa65a7f49dd9a8}};

hash_t Board_Hash::HashCode( const Board &ABoard )
{
	hash_t h = 0; // hashcode

	for (int i = 0; i<16; i++)
	{
	    Square sq = ABoard.PiecePos(White,i);
	    if (!sq.IsInvalid())
	    {
	       Piece piece = ABoard[sq];
	       h ^= hash_codes[ sq ][ (int)piece ];
	    }
	    sq = ABoard.PiecePos(Black,i);
	    if (!sq.IsInvalid())
	    {
	       Piece piece = ABoard[sq];
	       h ^= hash_codes[ sq ][ (int)piece ];
	    }
	}
	if (ABoard.Side() == Black)
	   h |= 1;
        else
	   h &= ~1;
	return h;
}
	
hash_t Board_Hash::HashCode2( const Board & b)
{
    hash_t h = 0;
    for (int i = 0; i<16; i++)
    {
        Square sq = b.PiecePos(White,i);
        if (!sq.IsInvalid())
	{
	   Piece piece = b[sq];
	   h += (int)sq*(int)piece;
	}
        sq = b.PiecePos(Black,i);
        if (!sq.IsInvalid())
	{
	   Piece piece = b[sq];
	   h += 32*(int)sq*(int)piece;
	}
    }
    return h;
}

inline hash_t Xor( hash_t h, const Square &sq, const Piece &piece )
{
   ASSERT(sq.OnBoard());
   return h ^ hash_codes[sq][(int)piece];
}

void Board_Hash::UpdateHash( const Board &ABoard, 
		       const ExtendedMove &emove, 
		       const hash_t OldHash, hash_t &NewHash)
{
    hash_t register h = OldHash;
    if (!emove.IsNull())
    {
       Square kp;

       switch( emove.Special() )
       {
        case ExtendedMove::KCastle:
           kp = ABoard.KingPos(ABoard.Side());
           h = Xor(h, kp+3, Piece(Piece::Rook,ABoard.Side()));
           h = Xor(h, kp, Piece(Piece::King,ABoard.Side()));
           h = Xor(h, kp+1, Piece(Piece::Rook,ABoard.Side()));
           h = Xor(h, kp+2, Piece(Piece::King,ABoard.Side()));
           break;
        case ExtendedMove::QCastle:
           kp = ABoard.KingPos(ABoard.Side());
           h = Xor(h, kp-4, Piece(Piece::Rook,ABoard.Side()));
           h = Xor(h, kp, Piece(Piece::King,ABoard.Side()));
           h = Xor(h, kp-1, Piece(Piece::Rook,ABoard.Side()));
           h = Xor(h, kp-2, Piece(Piece::King,ABoard.Side()));
           break;
        case ExtendedMove::Normal:
           h = Xor(h, emove.StartSquare(), emove.PieceMoved());
           if (!emove.Capture().IsEmpty())
              h = Xor(h, emove.DestSquare(), emove.Capture());
           h = Xor(h, emove.DestSquare(), emove.PieceMoved());
           break;
        case ExtendedMove::EnPassant:
        {	
           Square dest = ABoard.EnPassantSq(ABoard.OppositeSide());
           h = Xor(h, emove.StartSquare(), emove.PieceMoved());
           h = Xor(h, dest, emove.Capture());
           h = Xor(h, emove.DestSquare(), emove.PieceMoved());
           break;
        }
        case ExtendedMove::Promotion:
           h = Xor(h, emove.StartSquare(), Piece(Piece::Pawn,ABoard.Side()));
           h = Xor(h, emove.DestSquare(), Piece(emove.PromoteTo(),ABoard.Side()));
           if (!emove.Capture().IsEmpty())
              h = Xor(h, emove.DestSquare(), emove.Capture());
           break;
       }
    }
    if (ABoard.Side() == White)
       h |= (hash_t)1;
    else
       h &= (hash_t)~1;
    NewHash = h;
}

