{$G+,X+,F+}

{Conditional defines that may affect this unit}
{$I AWDEFINE.INC}

{*********************************************************}
{*                   MODLIST.PAS 1.00                    *}
{*        Copyright (c) TurboPower Software 1995         *}
{*                 All rights reserved.                  *}
{*********************************************************}

unit Modlist;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Menus, StdCtrls, AdMisc, AdModDB, ModInfo;

type
  TMainForm = class(TForm)
    MainMenu: TMainMenu;
    FileMenu: TMenuItem;
    NewDB: TMenuItem;
    OpenDB: TMenuItem;
    SaveDB: TMenuItem;
    SaveDBAs: TMenuItem;
    N1: TMenuItem;
    ExitProgram: TMenuItem;
    EditMenu: TMenuItem;
    AddModem: TMenuItem;
    ChangeModem: TMenuItem;
    DeleteModem: TMenuItem;
    ModemList: TListBox;
    AddBtn: TButton;
    ChangeBtn: TButton;
    DeleteBtn: TButton;
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    Database: TModemDBase;
    procedure FormCreate(Sender: TObject);
    procedure NewDBClick(Sender: TObject);
    procedure OpenDBClick(Sender: TObject);
    procedure SaveDBClick(Sender: TObject);
    procedure SaveDBAsClick(Sender: TObject);
    procedure ExitProgramClick(Sender: TObject);
    procedure AddModemClick(Sender: TObject);
    procedure ChangeModemClick(Sender: TObject);
    procedure DeleteModemClick(Sender: TObject);
    procedure AddBtnClick(Sender: TObject);
    procedure ChangeBtnClick(Sender: TObject);
    procedure DeleteBtnClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure ModemListDblClick(Sender: TObject);
  public
    HaveFName : Boolean;
    Modified  : Boolean;
    FName     : TFileName;

    procedure DoneModem;
      {-Destroy the current modem}
    procedure NewModemPrim;
      {-Create a new modem database}
    procedure SaveModemPrim;
      {-Save the current modem database}
    procedure SaveModemAsPrim;
      {-Save the current modem database under a new name}
    function CheckSave : Boolean;
      {-See if the user wants to save a modified database}

    procedure AddModemPrim;
      {-Add a modem to the database}
    procedure ChangeModemPrim;
      {-Change a modem record}
    procedure DeleteModemPrim;
      {-Delete a modem record}
  end;

var
  MainForm: TMainForm;

implementation

{$R *.DFM}

const
  BufSize = 8 * 1024;

procedure CopyDBase(const SrcName, DestName : String);
  {-Copy a database file}
var
  InFile    : File;
  OutFile   : File;
  Buf       : Pointer;
  BytesRead : Word;

begin
  {allocate memory for the copy buffer}
  GetMem(Buf, BufSize);

  {try to open the input file}
  try
    Assign(InFile, SrcName);
    Assign(OutFile, DestName);
    Reset(InFile, 1);
  except
    FreeMem(Buf, BufSize);
    raise;
  end;

  {try to open the output file}
  try
    Rewrite(OutFile, 1);
  except
    FreeMem(Buf, BufSize);
    try Close(InFile); except end;
    raise;
  end;

  {copy data from source to destination}
  try
    while not Eof(InFile) do begin
      BlockRead(InFile, Buf^, BufSize, BytesRead);
      BlockWrite(OutFile, Buf^, BytesRead);
    end;

    FreeMem(Buf, BufSize);
    Close(InFile);
    Close(OutFile);
  except
    FreeMem(Buf, BufSize);
    Close(InFile);
    Close(OutFile);
    raise;
  end;
end;

procedure TMainForm.DoneModem;
  {-Destroy the current modem}
var
  F : File;

begin
  {if the database is open, delete the temporary file}
  if Database.Open then begin
    System.Assign(F, Database.FileName);
    try
      System.Erase(F);
    except
    end;

    {close the database}
    Database.Open := False;

    {no modem names in list}
    ModemList.Items.Clear;
  end;
end;

procedure TMainForm.NewModemPrim;
  {-Create a new modem database}
var
  UntitledFName : array[0..144] of Char;

begin
  DoneModem;

  {get a temporary filename for the database and add to that}
  GetTempFileName(#0, 'MOD', 0, UntitledFName);
  Database.FileName := StrPas(UntitledFName);
  Database.Open     := True;

  HaveFName := False;
  Modified  := False;
  Caption   := 'Untitled';
end;

procedure TMainForm.SaveModemPrim;
  {-Save the current modem database}
begin
  {if we don't have a filename, do save as...}
  if not HaveFName then
    SaveModemAsPrim
  else begin
    Cursor := crHourglass;

    try
      CopyDBase(Database.Filename, FName);
      Modified := False;
    except
      Cursor := crDefault;
      raise;
    end;

    Cursor := crDefault;
  end;
end;

procedure TMainForm.SaveModemAsPrim;
  {-Save the current modem database under a new name}
begin
  if not HaveFName then
    SaveDialog.Filename := ''
  else
    SaveDialog.FileName := JustFilename(FName);
  SaveDialog.InitialDir := JustPathname(FName);

  if SaveDialog.Execute then begin
    HaveFName := True;
    FName     := SaveDialog.Filename;
    try
      SaveModemPrim;
      Caption := JustFilename(FName);
    except
      HaveFName := False;
      raise;
    end;
  end;
end;

function TMainForm.CheckSave : Boolean;
  {-See if the user wants to save a modified database}
var
  Res : Word;

begin
  Res := Application.MessageBox('The current database has not been saved. Do you want to save it?',
                                'Warning', mb_IconQuestion or mb_YesNoCancel);
  CheckSave := (Res <> id_Cancel);
  if (Res = id_Yes) then
    SaveModemPrim;
end;

procedure TMainForm.AddModemPrim;
  {-Add a modem to the database}
var
  Frm       : TModemForm;
  ModemData : TModemInfo;

begin
  {initialize data structure}
  FillChar(ModemData, SizeOf(TModemInfo), 0);

  Frm := TModemForm.Create(Self, ModemData, Database);
  if (Frm.ShowModal = mrOK) then begin
    {add the modem to the database}
    Database.AddModem(ModemData);

    {add the modem's name to the listbox}
    ModemList.Items.Add(ModemData.Name);

    {change the highlighted item in the listbox}
    ModemList.ItemIndex := ModemList.Items.IndexOf(ModemData.Name);
    Modified := True;
  end;
  Frm.Free;
end;

procedure TMainForm.ChangeModemPrim;
  {-Change a modem record}
var
  Frm       : TModemForm;
  ModemData : TModemInfo;
  Name      : TModemName;

begin
  if (ModemList.ItemIndex = -1) then begin
    MessageBeep(0);
    Exit;
  end;

  Name := ModemList.Items[ModemList.ItemIndex];
  {read the record from the database}
  Database.GetModem(Name, ModemData);

  Frm := TModemForm.Create(Self, ModemData, Database);
  if (Frm.ShowModal = mrOK) then begin
    Database.UpdModem(Name, ModemData);

    {update the listbox if the name of the modem has changed}
    if (Name <> ModemData.Name) then begin
      ModemList.Items.Delete(ModemList.ItemIndex);
      ModemList.Items.Add(ModemData.Name);
      ModemList.ItemIndex := ModemList.Items.IndexOf(ModemData.Name);
    end;

    Modified := True;
  end;
end;

procedure TMainForm.DeleteModemPrim;
  {-Delete a modem record}
var
  Name : TModemName;

begin
  if (ModemList.ItemIndex =-1) then begin
    MessageBeep(0);
    Exit;
  end;

  if (Application.MessageBox('Do you really want to delete this modem record?',
                             'Confirm', mb_IconQuestion or mb_YesNo) = id_Yes) then begin
    Name := ModemList.Items[ModemList.ItemIndex];
    Database.DelModem(Name);
    ModemList.Items.Delete(ModemList.Items.IndexOf(Name));
    Modified := True;
  end;
end;

procedure TMainForm.FormCreate(Sender: TObject);
begin
  NewModemPrim;
end;

procedure TMainForm.NewDBClick(Sender: TObject);
begin
  if Modified then
    if not CheckSave then
      Exit;

  NewModemPrim;
end;

procedure TMainForm.OpenDBClick(Sender: TObject);
var
  TempFName : array[0..144] of Char;

begin
  if Modified then
    if not CheckSave then
      Exit;

  OpenDialog.FileName := '*.ini';
  if OpenDialog.Execute then begin
    FName := OpenDialog.FileName;
    DoneModem;
    Cursor := crHourGlass;

    try
      {get a temporary file name and copy the database there}
      GetTempFileName(#0, 'MOD', 0, TempFName);
      CopyDBase(FName, StrPas(TempFName));

      {open the database under the temporary name}
      Database.Filename := TempFName;
      Database.Open     := True;

      {load the list of modems into the listbox}
      ModemList.Items     := Database.Modems;
      ModemList.ItemIndex := -1;

      HaveFName := True;
      Caption   := JustFilename(FName);
    except
      NewModemPrim;
    end;

    Cursor := crDefault;
  end;
end;

procedure TMainForm.SaveDBClick(Sender: TObject);
begin
  SaveModemPrim;
end;

procedure TMainForm.SaveDBAsClick(Sender: TObject);
begin
  SaveModemAsPrim;
end;

procedure TMainForm.ExitProgramClick(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.AddModemClick(Sender: TObject);
begin
  AddModemPrim;
end;

procedure TMainForm.ChangeModemClick(Sender: TObject);
begin
  ChangeModemPrim;
end;

procedure TMainForm.DeleteModemClick(Sender: TObject);
begin
  DeleteModemPrim;
end;

procedure TMainForm.AddBtnClick(Sender: TObject);
begin
  AddModemPrim;
end;

procedure TMainForm.ChangeBtnClick(Sender: TObject);
begin
  ChangeModemPrim;
end;

procedure TMainForm.DeleteBtnClick(Sender: TObject);
begin
  DeleteModemPrim;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  DoneModem;
end;

procedure TMainForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  if Modified then
    CanClose := CheckSave
  else
    CanClose := True;
end;

procedure TMainForm.ModemListDblClick(Sender: TObject);
begin
  ChangeModemPrim;
end;

end.
