/*
 *  AllPaper shrinks and expands Windows wallpaper files to fit screens of
 *  any resolution.
 */

#include <windows.h>
#include <commdlg.h>
#include <stdlib.h>
#include <math.h>
#include "allpaper.h"

#define LINEWIDTH(bits)         ((((bits) + 31) >> 5) << 2)

#define IO_BLOCK_SIZE           0x8000
#define COUNTER_BASE            1000000

typedef struct {
    HGLOBAL hBmi;       // Handle to global memory block containing BITMAPINFO
    HGLOBAL hBits;      // Handle to global memory block containing bitmap bits
    HPALETTE hPalette;  // Palette handle
} DIB;

long FAR PASCAL WndProc (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL AboutDlgProc (HWND, WORD, WORD, LONG);

int ReadDib (HWND, char *szFileName, DIB *pDib);
int StretchDibAndSave (char *szFileName, DIB *dib, LONG, LONG);
void StretchBits (BYTE __huge *, int, int, BYTE __huge *, int, int, int);
void StretchScanLine (BYTE __huge *, int, BYTE __huge *, int, int, UINT,
                      DWORD);
void CopyBits (BYTE __huge *, DWORD, BYTE __huge *, DWORD, int);

BOOL GetOpenName (HWND, char *, int);
BOOL GetSaveName (HWND, char *, int);
void InitGroupRect (HWND, int, RECT *, BOOL);
void DrawGroupRect (HDC, RECT *);
void CenterDialog (HWND);
void SaveSettings (HWND);
void InitProgram (HWND);
void PositionWindow (HWND);

int nSize;                                      // Output size index
LONG lWidth[5] = { 640, 800, 1024, 1280, 0 };   // Output width array
LONG lHeight[5] = { 480, 600, 768, 1024, 0 };   // Output height array

RECT rectFrame;                 // Coordinates of frame surrounding bitmap
RECT rectPicture;               // Coordinates where bitmap is displayed
RECT rectControls;              // Coordinates of frame surrounding controls
OPENFILENAME ofn;               // Structure for obtaining file names

char szIniFile[128];            // INI file name
char szSection[] = "Settings";  // INI file section name
char *szEntry[] = {             // INI file entry names
    "WindowPos",
    "OutputSize"
};

char *szError[] = {
    "Error",
    "The file was not found or could not be opened",
    "An error occurred while the file was being read",
    "The end of the file was encountered unexpectedly",
    "Unsupported file format (file does not contain a DIB)",
    "Unsupported file format (DIB is compressed)",
    "Insufficient memory",
    "Error creating output file",
    "An error occurred while the file was being written",
    "Disk full error"
};

char *szWarning[] = {
    "Warning",
    "The end of the file was encountered unexpectedly. This probably "\
    "means there is an error in the bitmap which could prevent the image "\
    "from displaying properly."
};

char *szFilter[] = {
    "BMP Files (*.BMP)", "*.BMP",
    "All Files (*.*)", "*.*", ""
};

/*
 *  Function WinMain.
 */

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow)
{
    static char szAppName[] = "AllPaper";
    WNDCLASS wndclass;
    HWND hwnd;
    MSG msg;

    if (!hPrevInstance) {
        wndclass.style = 0;
        wndclass.lpfnWndProc = (WNDPROC) WndProc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = DLGWINDOWEXTRA;
        wndclass.hInstance = hInstance;
        wndclass.hIcon = LoadIcon (hInstance, szAppName);
        wndclass.hCursor = LoadCursor (NULL, IDC_ARROW);
        wndclass.hbrBackground = NULL;
        wndclass.lpszMenuName = NULL;
        wndclass.lpszClassName = szAppName;

        RegisterClass (&wndclass);
    }

    hwnd = CreateDialog (hInstance, szAppName, 0, NULL);

    InitProgram (hwnd);

    ShowWindow (hwnd, nCmdShow == SW_SHOWMAXIMIZED ? SW_SHOW : nCmdShow);
    UpdateWindow (hwnd);

    while (GetMessage (&msg, NULL, 0, 0))
        if (!IsDialogMessage (hwnd, &msg))
            DispatchMessage (&msg);

    return msg.wParam;
}

/*
 *  WndProc processes messages to the main window
 */

long FAR PASCAL WndProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    char szFileName[80];
    char szHeader[_MAX_FNAME + _MAX_EXT + 16];
    char szBuffer[8];

    static DIB dib;
    static HBRUSH hPatternBrush;
    static BOOL bDirty = FALSE;
    static HINSTANCE hInstance;

    HDC hdc;
    LPBITMAPINFOHEADER lpBmih;
    FARPROC lpfnAboutDlgProc;
    void FAR *lpBits;
    HBITMAP hBitmap;
    HCURSOR hCursor;
    HMENU hSysMenu;
    PAINTSTRUCT ps;
    POINT point;
    RECT rect;
    int i;

    switch (message) {

    case WM_CREATE:
        //
        // Get ready!
        //
        hInstance = ((LPCREATESTRUCT) lParam)->hInstance;
        i = GetModuleFileName (hInstance, szIniFile, sizeof (szIniFile));
        while (szIniFile[i--] != 0x2E);
        lstrcpy (&szIniFile[i+2], "INI");

        hPatternBrush = CreatePatternBrush (hBitmap = LoadBitmap (hInstance,
            "Background"));
        DeleteObject (hBitmap);

        hSysMenu = GetSystemMenu (hwnd, FALSE);
        DeleteMenu (hSysMenu, SC_SIZE, MF_BYCOMMAND);
        DeleteMenu (hSysMenu, SC_MAXIMIZE, MF_BYCOMMAND);
        AppendMenu (hSysMenu, MF_SEPARATOR, 0, NULL);
        AppendMenu (hSysMenu, MF_STRING, IDM_ABOUT, "&About AllPaper...");

        PositionWindow (hwnd);
        return 0;

    case WM_PAINT:
        //
        // Repaint the window.
        //
        hdc = BeginPaint (hwnd, &ps);
        DrawGroupRect (hdc, &rectFrame);
        DrawGroupRect (hdc, &rectControls);

        if (bDirty) {
            hCursor = SetCursor (LoadCursor (NULL, IDC_WAIT));
            SelectPalette (hdc, dib.hPalette, 0);
            RealizePalette (hdc);

            lpBmih = (LPBITMAPINFOHEADER) GlobalLock (dib.hBmi);
            lpBits = GlobalLock (dib.hBits);

            SetStretchBltMode (hdc, STRETCH_DELETESCANS);

            StretchDIBits (hdc, rectPicture.left, rectPicture.top,
                rectPicture.right - rectPicture.left,
                rectPicture.bottom - rectPicture.top, 0, 0,
                (int) lpBmih->biWidth, (int) lpBmih->biHeight,
                lpBits, (LPBITMAPINFO) lpBmih, DIB_RGB_COLORS, SRCCOPY);

            GlobalUnlock (dib.hBits);
            GlobalUnlock (dib.hBmi);
            SetCursor (hCursor);
        }

        EndPaint (hwnd, &ps);
        return 0;

    case WM_ERASEBKGND:
        //
        // Fill the window with the pattern stored in ALLPAPER.BMP.
        //
        GetClientRect (hwnd, &rect);
        UnrealizeObject (hPatternBrush);
        point.x = point.y = 0;
        ClientToScreen (hwnd, &point);
        SetBrushOrg ((HDC) wParam, point.x, point.y); 
        FillRect ((HDC) wParam, &rect, hPatternBrush);
        return 1;

    case WM_CTLCOLOR:
        //
        // Customize the control colors.
        //
        SetBkColor ((HDC) wParam, RGB (192, 192, 192));
        return GetStockObject (LTGRAY_BRUSH);

    case WM_COMMAND:
        switch (wParam) {

        case IDD_NEW:
            //
            // Load and display a new bitmap.
            //
            if (!GetOpenName (hwnd, szFileName, sizeof (szFileName)))
                return 0;

            if (bDirty) {
                GlobalFree (dib.hBmi);
                GlobalFree (dib.hBits);
                DeleteObject (dib.hPalette);
            }

            hCursor = SetCursor (LoadCursor (NULL, IDC_WAIT));
            i = ReadDib (hwnd, szFileName, &dib);
            SetCursor (hCursor);

            if (i) {
                MessageBox (hwnd, szError[i], szError[0], MB_OK);
                SetDlgItemText (hwnd, IDD_HEADER, "");
                bDirty = FALSE;
                EnableWindow (GetDlgItem (hwnd, IDD_SAVE), FALSE);
                InvalidateRect (hwnd, &rectPicture, FALSE);
                return 0;
            }

            _splitpath (szFileName, NULL, NULL, szHeader, szBuffer);
            lstrcat (szHeader, szBuffer);
            lpBmih = (LPBITMAPINFOHEADER) GlobalLock (dib.hBmi);
            wsprintf (&szHeader[lstrlen (szHeader)], " (%u x %u)",
                (unsigned) lpBmih->biWidth, (unsigned) lpBmih->biHeight);
            GlobalUnlock (dib.hBmi);
            SetDlgItemText (hwnd, IDD_HEADER, szHeader);

            bDirty = TRUE;
            EnableWindow (GetDlgItem (hwnd, IDD_SAVE), TRUE);
            InvalidateRect (hwnd, &rectPicture, FALSE);
            return 0;

        case IDD_SAVE:
            //
            // Shrink or expand the bitmap and save it to disk.
            //
            if (!GetSaveName (hwnd, szFileName, sizeof (szFileName)))
                return 0;

            if (nSize == 4) {
                lWidth[4] = (LONG) GetSystemMetrics (SM_CXSCREEN);
                lHeight[4] = (LONG) GetSystemMetrics (SM_CYSCREEN); 
            }

            hCursor = SetCursor (LoadCursor (NULL, IDC_WAIT));
            i = StretchDibAndSave (szFileName, &dib, lWidth[nSize],
                lHeight[nSize]);
            SetCursor (hCursor);

            if (i)
                MessageBox (hwnd, szError[i], szError[0], MB_OK);
            return 0;

        case IDD_640:
        case IDD_800:
        case IDD_1024:
        case IDD_1280:
        case IDD_SCREEN:
            //
            // Process a click of one of the Output Resolution radio buttons.
            //
            CheckRadioButton (hwnd, IDD_640, IDD_SCREEN, wParam);
            nSize = (int) wParam - IDD_640;
            return 0;

        case IDCANCEL:
            //
            // Minimize the window when the Esc key is pressed.
            //
            ShowWindow (hwnd, SW_MINIMIZE);
            return 0;
        }
        break;

    case WM_QUERYNEWPALETTE:
        //
        // If a bitmap is displayed, realize its palette prior to receiving
        // the input focus and invalidate the area of the window that contains
        // the bitmap image if a color remapping occurred.
        //
        if (bDirty) {
            hdc = GetDC (hwnd);
            SelectPalette (hdc, dib.hPalette, 0);

            if (i = RealizePalette (hdc))
                InvalidateRect (hwnd, &rectPicture, FALSE);

            ReleaseDC (hwnd, hdc);
            return i;
        }
        break;

    case WM_PALETTECHANGED:
        //
        // If a bitmap is displayed, realize its palette if another window
        // receives the input focus and realizes its own palette.
        //
        if ((wParam != (WPARAM) hwnd) && bDirty) {
            hdc = GetDC (hwnd);
            SelectPalette (hdc, dib.hPalette, 0);

            if (RealizePalette (hdc))
                InvalidateRect (hwnd, &rectPicture, FALSE);

            ReleaseDC (hwnd, hdc);
            return 0;
        }
        break;

    case WM_SYSCOMMAND:
        //
        // Display the About AllPaper dialog box.
        //
        if (wParam == IDM_ABOUT) {
            lpfnAboutDlgProc = MakeProcInstance (AboutDlgProc, hInstance);
            DialogBox (hInstance, "AboutBox", hwnd, lpfnAboutDlgProc);
            FreeProcInstance (lpfnAboutDlgProc);
            return 0;
        }
        break;

    case WM_CLOSE:
        //
        // Save program settings to ALLPAPER.INI and call DestroyWindow to
        // terminate the program.
        //
        SaveSettings (hwnd);
        DestroyWindow (hwnd);
        return 0;

    case WM_ENDSESSION:
        //
        // Save program settings to ALLPAPER.INI before terminating.
        //
        if (wParam)
            SaveSettings (hwnd);
        return 0;

    case WM_DESTROY:
        //
        // Free memory and delete GDI objects before terminating.
        //
        DeleteObject (hPatternBrush);

        if (bDirty) {
            GlobalFree (dib.hBmi);
            GlobalFree (dib.hBits);
            DeleteObject (dib.hPalette);
        }

        PostQuitMessage (0);
        return 0;
    }
    return DefDlgProc (hwnd, message, wParam, lParam);
}

/*
 *  GetOpenName obtains an open file name from the user.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *    szFileName = Pointer to file name buffer
 *    nMaxChar = Length of file name buffer
 *
 *  Returns:
 *
 *    TRUE = A valid file name was entered
 *    FALSE = A valid file name was not entered
 */

BOOL GetOpenName (HWND hwnd, char *szFileName, int nMaxChar)
{
    static bFirstTime = TRUE;

    szFileName[0] = 0x00;
    if (bFirstTime) {
        bFirstTime = FALSE;
        ofn.nFilterIndex = 1;
    }

    ofn.lStructSize = sizeof (OPENFILENAME);
    ofn.hwndOwner = hwnd;
    ofn.lpstrFilter = szFilter[0];
    ofn.lpstrCustomFilter = NULL;
    ofn.lpstrFile = szFileName;
    ofn.nMaxFile = nMaxChar;
    ofn.lpstrFileTitle = NULL;
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrTitle = NULL;
    ofn.Flags = OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY;
    ofn.lpstrDefExt = NULL;

    return (GetOpenFileName (&ofn));
}

/*
 *  GetSaveName obtains a save file name from the user.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *    szFileName = Pointer to file name buffer
 *    nMaxChar = Size of file name buffer
 *
 *  Returns:
 *
 *    TRUE = A valid file name was entered
 *    FALSE = A valid file name was not entered
 */

BOOL GetSaveName (HWND hwnd, char *szFileName, int nMaxChar)
{
    szFileName[0] = 0x00;

    ofn.lStructSize = sizeof (OPENFILENAME);
    ofn.hwndOwner = hwnd;
    ofn.lpstrFilter = szFilter[0];
    ofn.lpstrCustomFilter = NULL;
    ofn.lpstrFile = szFileName;
    ofn.nMaxFile = nMaxChar;
    ofn.lpstrFileTitle = NULL;
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrTitle = NULL;
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | OFN_NOTESTFILECREATE;
    ofn.lpstrDefExt = NULL;

    return (GetSaveFileName (&ofn));
}

/*
 *  ReadDib reads a DIB from disk and returns handles to:
 *
 *    1. A global memory block containing a BITMAPINFOHEADER structure and,
 *       if applicable, a color table
 *
 *    2. A global memory block containing the bitmap bits
 *
 *    3. A logical palette
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *    szFileName = Pointer to file name
 *    pDib = Pointer to uninitialized DIB structure
 *
 *  Returns:
 *
 *    0 = DIB successfully loaded
 *    ERROR_OPENING_FILE = The file was not found or could not be opened
 *    UNSUPPORTED_FORMAT = Unsupported file format (not a DIB file)
 *    COMPRESSED_FILE = Unsupported file format (DIB is compressed) 
 *    ERROR_READING_FILE = Error reading data from the file
 *    NOT_ENOUGH_MEMORY = Global memory allocation failed
 *    END_OF_FILE = End of file encountered unexpectedly
 *
 *  Notes:
 *
 *    OS/2-style DIBs are converted to Windows DIB format (BITMAPCOREINFO
 *    becomes BITMAPINFO).
 *
 *    If the DIB contains more than 256 colors, the palette handle returned
 *    in the DIB structure is that of Windows's default palette.
 *
 *    Currently, this routine does not support compressed DIBs. An attempt to
 *    read a file stored in BI_RLE4 or BI_RLE8 format returns the error code 
 *    COMPRESSED_FILE. 
 */

int ReadDib (HWND hwnd, char *szFileName, DIB *pDib)
{
    HFILE hFile;
    BITMAPFILEHEADER Bmfh;
    UINT nBytesToRead, nBytesRead;
    HGLOBAL hBmi, hPal, hBits;
    LPBITMAPINFO lpBmi;
    LPBITMAPINFOHEADER lpBmih;
    LPBITMAPCOREHEADER lpBmch;
    LPBITMAPCOREINFO lpBmci;
    DWORD dwFileType, dwColors, dwBytesRemaining;
    short nWidth, nHeight;
    WORD wPlanes, wBitCount;
    LPLOGPALETTE lpPalette;
    HPALETTE hPalette;
    BYTE __huge *hpbBuffer;
    int i;

    //
    //
    // Open the file.
    //
    if ((hFile = _lopen (szFileName, READ)) == HFILE_ERROR)
        return ERROR_OPENING_FILE;

    //
    // Read the bitmap file header into memory and check for the "BM"
    // signature in the bfType field.
    //
    Bmfh.bfType = 0;
    nBytesToRead = sizeof (BITMAPFILEHEADER);
    nBytesRead = _lread (hFile, &Bmfh, nBytesToRead);

    if (nBytesRead < nBytesToRead) {
        _lclose (hFile);
        return END_OF_FILE;
    }

    if (nBytesRead == HFILE_ERROR) {
        _lclose (hFile);
        return ERROR_READING_FILE;
    }

    if (Bmfh.bfType != 0x4D42) {
        _lclose (hFile);
        return UNSUPPORTED_FORMAT;
    }

    //
    // Read the BITMAPINFOHEADER structure into memory and verify that this
    // is a supported file format by checking the biSize field.
    //
    if ((hBmi = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,
        sizeof (BITMAPINFOHEADER) + (256 * sizeof (RGBQUAD)))) == NULL) {
        _lclose (hFile);
        return NOT_ENOUGH_MEMORY;
    }

    lpBmi = (LPBITMAPINFO) GlobalLock (hBmi);
    lpBmih = (LPBITMAPINFOHEADER) lpBmi;

    nBytesToRead = sizeof (lpBmih->biSize);
    nBytesRead = _lread (hFile, &(lpBmih->biSize), nBytesToRead);

    if (nBytesRead < nBytesToRead) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return END_OF_FILE;
    }

    if (nBytesRead == HFILE_ERROR) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return ERROR_READING_FILE;
    }

    dwFileType = lpBmih->biSize;
    if ((dwFileType != 0x28) && (dwFileType != 0x0C)) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return UNSUPPORTED_FORMAT;
    }           

    nBytesToRead = (UINT) (lpBmih->biSize) - sizeof (lpBmih->biSize);
    nBytesRead = _lread (hFile, &(lpBmih->biWidth), nBytesToRead);

    if (nBytesRead < nBytesToRead) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return END_OF_FILE;
    }

    if (nBytesRead == HFILE_ERROR) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return ERROR_READING_FILE;
    }

    //
    // If necessary, convert a BITMAPCOREHEADER structure to BITMAPINFOHEADER
    // format. Also set the BITMAPINFOHEADER structure's biSizeImage field.
    //
    if (dwFileType == 0x0C) {
        lpBmch = (LPBITMAPCOREHEADER) lpBmi;

        nWidth = lpBmch->bcWidth;
        nHeight = lpBmch->bcHeight;
        wPlanes = lpBmch->bcPlanes;
        wBitCount = lpBmch->bcBitCount;

        lpBmih->biSize = sizeof (BITMAPINFOHEADER);
        lpBmih->biWidth = (LONG) nWidth;
        lpBmih->biHeight = (LONG) nHeight;
        lpBmih->biPlanes = wPlanes;
        lpBmih->biBitCount = wBitCount;
        lpBmih->biCompression = BI_RGB;
        lpBmih->biSizeImage = 0;
        lpBmih->biXPelsPerMeter = 0;
        lpBmih->biYPelsPerMeter = 0;
        lpBmih->biClrUsed = 0;
        lpBmih->biClrImportant = 0;
    }

    lpBmih->biSizeImage = LINEWIDTH (lpBmih->biWidth * lpBmih->biBitCount) *
        lpBmih->biHeight;

    //
    // Make sure that the DIB is not compressed and that it uses 1, 4, 8, or
    // 24 bits per pixel.
    //
    if (lpBmih->biCompression != BI_RGB) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return COMPRESSED_FILE;
    }

    if ((lpBmih->biBitCount != 1) && (lpBmih->biBitCount != 4) &&
        (lpBmih->biBitCount != 8) && (lpBmih->biBitCount != 24)) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return UNSUPPORTED_FORMAT;
    }           

    //
    // Read the color table into memory if the DIB contains a color table,
    // and convert a BITMAPCOREINFO-style color table to BITMAPINFO format if
    // necessary.
    //
    dwColors = ((DWORD) 1) << lpBmih->biBitCount;

    if (dwColors <= 256) {
        nBytesToRead = dwColors * ((dwFileType == 0x28) ? sizeof (RGBQUAD) :
            sizeof (RGBTRIPLE));
        nBytesRead = _lread (hFile, &(lpBmi->bmiColors[0]), nBytesToRead);
              
        if (nBytesRead < nBytesToRead) {
            GlobalUnlock (hBmi);
            GlobalFree (hBmi);
            _lclose (hFile);
            return END_OF_FILE;
        }

        if (nBytesRead == HFILE_ERROR) {
            GlobalUnlock (hBmi);
            GlobalFree (hBmi);
            _lclose (hFile);
            return ERROR_READING_FILE;
        }

        if (dwFileType == 0x0C) {
            lpBmci = (LPBITMAPCOREINFO) ((BYTE FAR *) lpBmi +
                sizeof (BITMAPINFOHEADER) - sizeof (BITMAPCOREHEADER));

            for (i=(int) dwColors - 1; i>0; i--) {
                lpBmi->bmiColors[i].rgbReserved = 0;
                lpBmi->bmiColors[i].rgbRed = lpBmci->bmciColors[i].rgbtRed;
                lpBmi->bmiColors[i].rgbGreen = lpBmci->bmciColors[i].rgbtGreen;
                lpBmi->bmiColors[i].rgbBlue = lpBmci->bmciColors[i].rgbtBlue;
            }
        }
    }

    //
    // If the bitmap contains 256 or fewer colors, create a logical palette
    // for it. If the bitmap contains more than 256 colors, get Windows's
    // default palette handle.
    //
    if (dwColors <= 256) {
        if ((hPal = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,
            sizeof (LOGPALETTE) + (min (256, dwColors) *
            sizeof (PALETTEENTRY)))) == NULL) {
            GlobalUnlock (hBmi);
            GlobalFree (hBmi);
            _lclose (hFile);
            return NOT_ENOUGH_MEMORY;
        }

        lpPalette = (LPLOGPALETTE) GlobalLock (hPal);
        lpPalette->palVersion = 0x300;
        lpPalette->palNumEntries = (WORD) dwColors;

        for (i=0; i<(int) dwColors; i++) {
            lpPalette->palPalEntry[i].peRed = lpBmi->bmiColors[i].rgbRed;
            lpPalette->palPalEntry[i].peGreen = lpBmi->bmiColors[i].rgbGreen;
            lpPalette->palPalEntry[i].peBlue = lpBmi->bmiColors[i].rgbBlue;
            lpPalette->palPalEntry[i].peFlags = NULL;
        }

        hPalette = CreatePalette (lpPalette);
        GlobalUnlock (hPal);
        GlobalFree (hPal);
    }
    else
        hPalette = (HPALETTE) GetStockObject (DEFAULT_PALETTE);

    //
    // Allocate a global memory block to hold the bitmap bits.
    //
    if ((hBits = GlobalAlloc (GMEM_MOVEABLE, lpBmih->biSizeImage)) == NULL) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        DeleteObject (hPalette);
        _lclose (hFile);
        return NOT_ENOUGH_MEMORY;
    }

    //
    // Read the bitmap bits into memory.
    //
    hpbBuffer = (BYTE __huge *) GlobalLock (hBits);
    dwBytesRemaining = lpBmih->biSizeImage;
    _llseek (hFile, (LONG) Bmfh.bfOffBits, 0);

    while (dwBytesRemaining) {
        nBytesToRead = min (IO_BLOCK_SIZE, dwBytesRemaining);
        nBytesRead = _lread (hFile, hpbBuffer, nBytesToRead);

        if (nBytesRead < nBytesToRead) {
            MessageBox (hwnd, szWarning[1], szWarning[0], MB_OK);
            break;
        }           

        if (nBytesRead == HFILE_ERROR) {
            GlobalUnlock (hBits);
            GlobalFree (hBits);
            GlobalUnlock (hBmi);
            GlobalFree (hBmi);
            DeleteObject (hPalette);
            _lclose (hFile);
            return ERROR_READING_FILE;
        }
        dwBytesRemaining -= nBytesRead;
        hpbBuffer += nBytesRead;
    }

    //
    // Clean up, copy handles to the DIB structure, and return.
    //
    GlobalUnlock (hBits);
    GlobalUnlock (hBmi);
    _lclose (hFile);

    pDib->hBmi = hBmi;
    pDib->hBits = hBits;
    pDib->hPalette = hPalette;
    return 0;
}

/*
 *  StretchDibAndSave shrinks or expands a DIB to the specified dimensions
 *  and saves the resulting image to disk.
 *
 *  Input parameters:
 *
 *    szFileName = Pointer to save file name
 *    pDib = Pointer to initialized DIB structure
 *    lWidth = New bitmap width
 *    lHeight = New bitmap height
 *
 *  Returns:
 *
 *    0 = DIB successfully resized and saved
 *    NOT_ENOUGH_MEMORY = Global memory allocation failed
 *    ERROR_CREATING_FILE = File could not be created
 *    ERROR_WRITING_FILE = Error writing data to file
 *    DISK_FULL = Disk full
 */

StretchDibAndSave (char *szFileName, DIB *pDib, LONG lWidth, LONG lHeight)
{
    HFILE hFile;
    HGLOBAL hBmi, hBits;
    BITMAPFILEHEADER Bmfh;
    LPBITMAPINFOHEADER lpBmih;
    BYTE FAR *lpbSrc, FAR *lpbDest;
    BYTE __huge *hpbSrc, __huge *hpbDest, __huge *hpbBuffer;
    UINT nBytesToWrite, nBytesWritten;
    LONG lWidthSrc, lHeightSrc;
    DWORD dwBytesRemaining;
    int nSizeColorTable, i;

    //
    // Build a BITMAPINFOHEADER structure and, if applicable, a color table
    // for the new DIB.
    //
    if ((hBmi = GlobalAlloc (GMEM_MOVEABLE, sizeof (BITMAPINFOHEADER) +
        (256 * sizeof (RGBQUAD)))) == NULL)
        return NOT_ENOUGH_MEMORY;

    lpBmih = (LPBITMAPINFOHEADER) GlobalLock (hBmi);
    lpbSrc = (BYTE FAR *) GlobalLock (pDib->hBmi);
    lpbDest = (BYTE FAR *) lpBmih;

    lWidthSrc = ((LPBITMAPINFOHEADER) lpbSrc)->biWidth;
    lHeightSrc = ((LPBITMAPINFOHEADER) lpbSrc)->biHeight;

    for (i=0; i<sizeof (BITMAPINFOHEADER); i++)
        lpbDest[i] = lpbSrc[i];

    nSizeColorTable = (lpBmih->biBitCount <= 8) ?
        ((1 << lpBmih->biBitCount) * sizeof (RGBQUAD)): 0;

    if (nSizeColorTable) {
        lpbSrc += sizeof (BITMAPINFOHEADER);
        lpbDest += sizeof (BITMAPINFOHEADER);
        for (i=0; i<nSizeColorTable; i++)
            lpbDest[i] = lpbSrc[i];
    }

    GlobalUnlock (pDib->hBmi);

    lpBmih->biWidth = lWidth;
    lpBmih->biHeight = lHeight;
    lpBmih->biSizeImage = LINEWIDTH (lpBmih->biWidth * lpBmih->biBitCount) *
        lpBmih->biHeight;
    dwBytesRemaining = lpBmih->biSizeImage;

    //
    // Allocate a global memory block to hold the bitmap bits and stretch
    // the bits from the source bitmap.
    //
    if ((hBits = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT,
        lpBmih->biSizeImage)) == NULL) {
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        return NOT_ENOUGH_MEMORY;
    }

    hpbSrc = (BYTE __huge *) GlobalLock (pDib->hBits);
    hpbDest = (BYTE __huge *) GlobalLock (hBits);
    hpbBuffer = hpbDest;

    StretchBits (hpbDest, (int) lWidth, (int) lHeight, hpbSrc,
        (int) lWidthSrc, (int) lHeightSrc, (int) lpBmih->biBitCount);

    GlobalUnlock (pDib->hBits);

    //
    // Create the output file.
    //
    if ((hFile = _lcreat (szFileName, 0)) == HFILE_ERROR) {
        GlobalUnlock (hBits);
        GlobalFree (hBits);
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        return ERROR_CREATING_FILE;
    }

    //
    // Build the bitmap file header and write it to disk.
    //
    Bmfh.bfType = 0x4D42;
    Bmfh.bfSize = sizeof (Bmfh) + sizeof (BITMAPINFOHEADER) +
        nSizeColorTable + lpBmih->biSizeImage;
    Bmfh.bfReserved1 = 0;
    Bmfh.bfReserved2 = 0;
    Bmfh.bfOffBits = sizeof (Bmfh) + sizeof (BITMAPINFOHEADER) +
        nSizeColorTable;

    nBytesWritten = _lwrite (hFile, &Bmfh, sizeof (Bmfh));

    if (nBytesWritten < sizeof (Bmfh)) {
        GlobalUnlock (hBits);
        GlobalFree (hBits);
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return DISK_FULL;
    }

    if (nBytesWritten == HFILE_ERROR) {
        GlobalUnlock (hBits);
        GlobalFree (hBits);
        GlobalUnlock (hBmi);
        GlobalFree (hBmi);
        _lclose (hFile);
        return ERROR_WRITING_FILE;
    }

    //
    // Write the BITMAPINFOHEADER structure and color table to disk.
    //
    nBytesToWrite = sizeof (BITMAPINFOHEADER) + nSizeColorTable;
    nBytesWritten = _lwrite (hFile, lpBmih, nBytesToWrite);

    GlobalUnlock (hBmi);
    GlobalFree (hBmi);

    if (nBytesWritten < nBytesToWrite) {
        GlobalUnlock (hBits);
        GlobalFree (hBits);
        _lclose (hFile);
        return DISK_FULL;
    }

    if (nBytesWritten == HFILE_ERROR) {
        GlobalUnlock (hBits);
        GlobalFree (hBits);
        _lclose (hFile);
        return ERROR_WRITING_FILE;
    }

    //
    // Write the bitmap bits to disk.
    //
    while (dwBytesRemaining) {
        nBytesToWrite = min (IO_BLOCK_SIZE, dwBytesRemaining);
        nBytesWritten = _lwrite (hFile, hpbBuffer, nBytesToWrite);

        if (nBytesWritten < nBytesToWrite) {
            GlobalUnlock (hBits);
            GlobalFree (hBits);
            _lclose (hFile);
            return DISK_FULL;
        }

        if (nBytesWritten == HFILE_ERROR) {
            GlobalUnlock (hBits);
            GlobalFree (hBits);
            _lclose (hFile);
            return ERROR_WRITING_FILE;
        }
        dwBytesRemaining -= nBytesWritten;
        hpbBuffer += nBytesWritten;
    }

    //
    // Clean up and return.
    //
    GlobalUnlock (hBits);
    GlobalFree (hBits);
    _lclose (hFile);
    return 0;
}

/*
 *  StretchBits shrinks or expands the bits in a buffer representing a
 *  bitmap image.
 *
 *  Input parameters:
 *
 *    hpbDest = Pointer to buffer for destination bitmap
 *    nWidthDest = Width of destination bitmap 
 *    nHeightDest = Height of destination bitmap
 *    hpbSrc = Pointer to buffer for source bitmap
 *    nWidthSrc = Width of source bitmap 
 *    nHeightSrc = Height of source bitmap
 *    nBits = Bits per pixel
 *
 *  Returns:
 *
 *    Nothing
 */

void StretchBits (BYTE __huge *hpbDest, int nWidthDest, int nHeightDest,
                  BYTE __huge *hpbSrc, int nWidthSrc, int nHeightSrc,
                  int nBits)
{
    double dRatio;
    DWORD dwIncrement, dwIncrementWidth, dwIncCount;
    UINT nMultiplier, nMultiplierWidth, nOutputRows, i, j;
    UINT nBytesPerLineSrc, nBytesPerLineDest;

    //
    // Compute the replication factors for both width and height.
    //
    dRatio = (double) nWidthDest / (double) nWidthSrc;
    nMultiplierWidth = (UINT) floor (dRatio);
    dwIncrementWidth = (DWORD) ceil ((dRatio - floor (dRatio)) * COUNTER_BASE);

    dRatio = (double) nHeightDest / (double) nHeightSrc;
    nMultiplier = (UINT) floor (dRatio);
    dwIncrement = (DWORD) ceil ((dRatio - floor (dRatio)) * COUNTER_BASE);

    //
    // Compute the size (in bytes) of each scan line in both the source and
    // destination bitmaps.
    //
    nBytesPerLineSrc = (UINT) LINEWIDTH ((DWORD) nWidthSrc * (DWORD) nBits);
    nBytesPerLineDest = (UINT) LINEWIDTH ((DWORD) nWidthDest * (DWORD) nBits);

    //
    // Initialize counters.
    //
    dwIncCount = 0;
    nOutputRows = 0;

    //
    // Shrink or expand the bits one scan line at a time.
    //
    for (i=0; i<nHeightSrc; i++) {
        if (nMultiplier != 0) {
            for (j=0; j<nMultiplier; j++) {
                StretchScanLine (hpbDest, nWidthDest, hpbSrc, nWidthSrc,
                    nBits, nMultiplierWidth, dwIncrementWidth);
                hpbDest += nBytesPerLineDest;
                nOutputRows++;
            }
        }

        dwIncCount += dwIncrement;

        if ((dwIncCount >= COUNTER_BASE) && (nOutputRows < nHeightDest)) {
            StretchScanLine (hpbDest, nWidthDest, hpbSrc, nWidthSrc, nBits,
                nMultiplierWidth, dwIncrementWidth);
            hpbDest += nBytesPerLineDest;
            nOutputRows++;
            dwIncCount -= COUNTER_BASE;
        }
        hpbSrc += nBytesPerLineSrc;
    }
}

/*
 *  StretchScanLine shrinks or expands the bits in a buffer representing one
 *  scan line of a bitmap image.
 *
 *  Input parameters:
 *
 *    hpbDest = Pointer to buffer for destination bitmap
 *    nWidthDest = Width of destination bitmap 
 *    hpbSrc = Pointer to buffer for source bitmap
 *    nWidthSrc = Width of source bitmap
 *    nBits = Bits per pixel
 *    nMultiplier = Multiplicative pixel replication factor
 *    dwIncrement = Incremental pixel replication factor
 *
 *  Returns:
 *
 *    Nothing
 */

void StretchScanLine (BYTE __huge *hpbDest, int nWidthDest,
                      BYTE __huge *hpbSrc, int nWidthSrc, int nBits,
                      UINT nMultiplier, DWORD dwIncrement)
{
    UINT nOutputPixels, i, j;
    DWORD dwIncCount, dwBitIndexSrc, dwBitIndexDest;

    //
    // Initialize counters.
    //
    dwIncCount = 0;
    nOutputPixels = 0;
    dwBitIndexSrc = 0;
    dwBitIndexDest = 0;

    //
    // Read through the scan line from left to right, writing pixels to the
    // destination buffer as appropriate.
    //
    for (i=0; i<nWidthSrc; i++) {
        if (nMultiplier != 0) {
            for (j=0; j<nMultiplier; j++) {
                CopyBits (hpbDest, dwBitIndexDest, hpbSrc, dwBitIndexSrc,
                    nBits);
                dwBitIndexDest += nBits;
                nOutputPixels++;
            }
        }

        dwIncCount += dwIncrement;

        if ((dwIncCount >= COUNTER_BASE) && (nOutputPixels < nWidthDest)) {
            CopyBits (hpbDest, dwBitIndexDest, hpbSrc, dwBitIndexSrc, nBits);
            dwBitIndexDest += nBits;
            nOutputPixels++;
            dwIncCount -= COUNTER_BASE;
        }
        dwBitIndexSrc += nBits;
    }
}

/*
 *  CopyBits copies the specified number of bits from the source bitmap to
 *  the destination bitmap.
 *
 *  Input parameters:
 *
 *    hpbDest = Pointer to buffer for destination bitmap
 *    nBitIndexDest = Destination bit index
 *    hpbSrc = Pointer to buffer for source bitmap
 *    nBitIndexSrc = Source bit index
 *    nBits = Number of bits to copy
 *
 *  Returns:
 *
 *    Nothing
 *
 *  Notes:
 *
 *    For 8- and 24-bit moves, it is valid to assume that the source and
 *    destination addresses for the move fall on byte boundaries. For 4-bit
 *    moves, it is valid to assume that the source and destination addresses
 *    fall on byte or half-byte boundaries. This function handles 1-, 4-, 8-,
 *    and 24-bit moves exclusively. No error checking is performed to ensure
 *    that the value passed in the nBits parameter is 1, 4, 8, or 24.
 */

void CopyBits (BYTE __huge *hpbDest, DWORD dwBitIndexDest, BYTE __huge *hpbSrc,
               DWORD dwBitIndexSrc, int nBits)
{
    DWORD dwByteIndexSrc, dwByteIndexDest;
    BYTE byAndMaskSrc, byAndMaskDest, byOrVal;

    dwByteIndexSrc = dwBitIndexSrc >> 3;
    dwByteIndexDest = dwBitIndexDest >> 3;

    switch (nBits) {

    case 1:
        byAndMaskSrc = 0x80 >> (dwBitIndexSrc & 0x07);
        byOrVal = 0x80 >> (dwBitIndexDest & 0x07);
        byAndMaskDest = ~byOrVal;
        hpbDest[dwByteIndexDest] &= byAndMaskDest;
        if (hpbSrc[dwByteIndexSrc] & byAndMaskSrc)
            hpbDest[dwByteIndexDest] |= byOrVal;
        break;

    case 4:
        byAndMaskDest = ((dwByteIndexDest << 3) == dwBitIndexDest) ?
            0x0F : 0xF0;
        byAndMaskSrc = ((dwByteIndexSrc << 3) == dwBitIndexSrc) ?
            0xF0 : 0x0F;
        byOrVal = hpbSrc[dwByteIndexSrc] & byAndMaskSrc;

        if ((byAndMaskSrc == 0xF0) && (byAndMaskDest == 0xF0)) 
            byOrVal >>= 4;
        else if ((byAndMaskSrc == 0x0F) && (byAndMaskDest == 0x0F))
            byOrVal <<= 4;

        hpbDest[dwByteIndexDest] = (hpbDest[dwByteIndexDest] &
            byAndMaskDest) | byOrVal;
        break;

    case 8:
        hpbDest[dwByteIndexDest] = hpbSrc[dwByteIndexSrc];
        break;

    case 24:
        hpbDest[dwByteIndexDest] = hpbSrc[dwByteIndexSrc];
        hpbDest[dwByteIndexDest + 1] = hpbSrc[dwByteIndexSrc + 1];
        hpbDest[dwByteIndexDest + 2] = hpbSrc[dwByteIndexSrc + 2];
        break;
    }
}

/*
 *  InitGroupRect saves the coordinates of the control window in a RECT
 *  structure, and optionally destroys the control.
 *
 *  Input parameters:
 *
 *    hwnd = Handle of the control's parent
 *    nID = Control ID
 *    rect = Pointer to RECT structure to receive the coordinates
 *    bDestroy = Flag specifying whether the control should be destroyed
 *
 *  Returns:
 *
 *    Nothing
 */

void InitGroupRect (HWND hwnd, int nID, RECT *rect, BOOL bDestroy)
{
    GetWindowRect (GetDlgItem (hwnd, nID), rect);
    ScreenToClient (hwnd, (POINT FAR *) &rect->left);
    ScreenToClient (hwnd, (POINT FAR *) &rect->right);
    if (bDestroy)
        DestroyWindow (GetDlgItem (hwnd, nID));
}

/*
 *  DrawGroupRect draws a recessed group rectangle at the specified location.
 *
 *  Input parameters:
 *
 *    hdc = Device context handle
 *    rect = Pointer to RECT structure containing the rectangle's coordinates
 *
 *  Returns:
 *
 *    Nothing
 */

void DrawGroupRect (HDC hdc, RECT *rect)
{
    HPEN hPen;

    FillRect (hdc, rect, GetStockObject (LTGRAY_BRUSH)); 
    hPen = CreatePen (PS_SOLID, 1, RGB (128, 128, 128));
    SelectObject (hdc, hPen);
    MoveTo (hdc, rect->left, rect->bottom - 1);
    LineTo (hdc, rect->left, rect->top);
    LineTo (hdc, rect->right, rect->top);
    DeleteObject (SelectObject (hdc, GetStockObject (WHITE_PEN)));
    MoveTo (hdc, rect->right - 1, rect->top + 1);
    LineTo (hdc, rect->right - 1, rect->bottom - 1);
    LineTo (hdc, rect->left, rect->bottom - 1);
}

/*
 *  PositionWindow positions the window on the screen using the
 *  WindowPos entry stored in ALLPAPER.INI. If ALLPAPER.INI does not exist,
 *  if it does exist but does not contain a WindowPos entry, or if WindowPos
 *  specifies a position that is off (or nearly off) the screen,
 *  PositionWindows centers the window on the screen.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void PositionWindow (HWND hwnd)
{
    char szBuffer[32];
    static char szDefString[] = "WindowPos=32767,32767";
    int cxWindowPos, cyWindowPos, cxScreenSize, cyScreenSize;
    RECT rect;

    GetPrivateProfileString (szSection, szEntry[0], szDefString, szBuffer,
        sizeof (szBuffer), szIniFile);
    sscanf (szBuffer, "%u,%u", &cxWindowPos, &cyWindowPos);

    cxScreenSize = GetSystemMetrics (SM_CXSCREEN);
    cyScreenSize = GetSystemMetrics (SM_CYSCREEN);

    if ((cxWindowPos > (cxScreenSize - 32)) ||
        (cyWindowPos > (cyScreenSize - 32))) {
        GetWindowRect (hwnd, &rect);
        cxWindowPos = (cxScreenSize - (rect.right - rect.left)) >> 1;
        cyWindowPos = (cyScreenSize - (rect.bottom - rect.top)) >> 1;
    }

    SetWindowPos (hwnd, 0, cxWindowPos, cyWindowPos, 0, 0,
        SWP_NOSIZE | SWP_NOZORDER);
}

/*
 *  InitProgram initializes the program.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void InitProgram (HWND hwnd)
{
    int cxWindowPos, cyWindowPos;
    char szBuffer[32];
    RECT rect;

    //
    // Initialize group rectangle coordinates.
    //
    InitGroupRect (hwnd, IDD_RECT1, &rectFrame, TRUE);
    InitGroupRect (hwnd, IDD_RECT2, &rectControls, TRUE);
    CopyRect (&rectPicture, &rectFrame);
    InflateRect (&rectPicture, -8, -8);

    //
    // Set the default output resolution.
    //
    nSize = min (GetPrivateProfileInt (szSection, szEntry[1], 4, szIniFile),
        4);
    CheckRadioButton (hwnd, IDD_640, IDD_SCREEN, IDD_640 + nSize);
}

/*
 *  SaveSettings saves program settings to ALLPAPER.INI.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void SaveSettings (HWND hwnd)
{
    WINDOWPLACEMENT wp;
    char szBuffer[32];

    wp.length = sizeof (wp);
    GetWindowPlacement (hwnd, &wp);
    wsprintf (szBuffer, "%u,%u", (wp.rcNormalPosition).left,
        (wp.rcNormalPosition).top);
    WritePrivateProfileString (szSection, szEntry[0], szBuffer, szIniFile);

    WritePrivateProfileString (szSection, szEntry[1],
        _itoa (nSize, szBuffer, 10), szIniFile);
}

/*
 *  CenterDialog centers a dialog box window (or any window, for that matter)
 *  on the screen.
 *
 *  Input parameters:
 *
 *    hwnd = Window handle
 *
 *  Returns:
 *
 *    Nothing
 */

void CenterDialog (HWND hwnd)
{
    RECT rect;

    GetWindowRect (hwnd, &rect);
    SetWindowPos (hwnd, 0,
        (GetSystemMetrics (SM_CXSCREEN) - (rect.right - rect.left)) >> 1,
        (GetSystemMetrics (SM_CYSCREEN) - (rect.bottom - rect.top)) >> 1,
        0, 0, SWP_NOSIZE | SWP_NOZORDER);
}

/*
 *  AboutDlgProc processes messages to the About AllPaper dialog box.
 */

BOOL FAR PASCAL AboutDlgProc (HWND hwnd, WORD message, WORD wParam,
                              LONG lParam)
{
    static HPEN hPen;
    static RECT rectFrame;
    static HFONT hDlgFont;
    PAINTSTRUCT ps;
    LOGFONT lf;
    RECT rect;
    HDC hdc;

    switch (message) {

    case WM_INITDIALOG:
        //
        // Initialize and create GDI objects.
        //
        InitGroupRect (hwnd, IDD_FRAME, &rectFrame, TRUE);

        if (IsIconic (GetWindow (hwnd, GW_OWNER)))
            CenterDialog (hwnd);

        GetObject ((HFONT) SendMessage (hwnd, WM_GETFONT, 0, 0L),
            sizeof (LOGFONT), &lf);
        lstrcpy (lf.lfFaceName, "Arial");
        lf.lfHeight *= 3;
        lf.lfWidth *= 3;
        lf.lfItalic = 1;
        lf.lfOutPrecision = OUT_TT_PRECIS;
        if ((hDlgFont = CreateFontIndirect ((LPLOGFONT) &lf)) != NULL)
            SendDlgItemMessage (hwnd, IDD_TITLE, WM_SETFONT, hDlgFont, 0L);

        hPen = CreatePen (PS_SOLID, 1, RGB (128, 128, 128));
        return TRUE;

    case WM_CTLCOLOR:
        //
        // Paint all control backgrounds light gray. Also change the title
        // color to bright red.
        //
        SetBkColor ((HDC) wParam, RGB (192, 192, 192));
        if (GetDlgCtrlID (LOWORD (lParam)) == IDD_TITLE)
            SetTextColor ((HDC) wParam, RGB (255, 0, 0));
        return GetStockObject (LTGRAY_BRUSH);

    case WM_ERASEBKGND:
        //
        // Paint the window background light gray.
        //
        GetClientRect (hwnd, &rect);
        FillRect ((HDC) wParam, &rect, GetStockObject (LTGRAY_BRUSH));
        return TRUE;

    case WM_PAINT:
        //
        // Draw the rectangle surrounding the dialog box text.
        //
        hdc = BeginPaint (hwnd, &ps);

        SelectObject (hdc, hPen);
        SelectObject (hdc, GetStockObject (NULL_BRUSH));
        Rectangle (hdc, rectFrame.left, rectFrame.top,
            rectFrame.right, rectFrame.bottom);

        SelectObject (hdc, GetStockObject (WHITE_PEN));
        Rectangle (hdc, rectFrame.left + 1, rectFrame.top + 1,
            rectFrame.right + 1, rectFrame.bottom + 1);

        EndPaint (hwnd, &ps);
        return TRUE;

    case WM_COMMAND:
        //
        // Dismiss the dialog box in response to an IDOK or IDCANCEL message.
        //
        switch (wParam) {

        case IDOK:
        case IDCANCEL:
            EndDialog (hwnd, 0);
            return TRUE;
        }
        break;

    case WM_DESTROY:
        //
        // Delete GDI objects.
        //
        if (hDlgFont != NULL)
            DeleteObject (hDlgFont);
        DeleteObject (hPen);
        return TRUE;
    }
    return FALSE;
}
