/*--------------------------------------------------------------------
   Alged:  Algebra Editor    henckel@vnet.ibm.com

   Copyright (c) 1994 John Henckel
   Permission to use, copy, modify, distribute and sell this software
   and its documentation for any purpose is hereby granted without fee,
   provided that the above copyright notice appear in all copies.
*/
#include "alged.h"
#include <graphics.h>
/*
  xs is size
  xo is origin (left edge)
  xz is maxx/xs (number of pixels per unit length)
  np is number of points
*/
static double xs=0,xo,ys,yo,xz,yz,np,ts,to;
static double err,step;
static node ivar[10];       /* independent variables */
static int gm,polar,nv,rain;
static int co1,co2,co3;    /* colors */
extern char gdriver[80];
extern int gmode,psz,pst;

/*-----------------------------------------------------------------
  3d variables
*/
static double focalen = 5;
static double cx,cy,cz;    // camera location
static double ax,ay,az;    // camera orientation
static double m[4][4];     // transform matrix
#define EPS 1E-10
/* These are the zooming ratios.... rr1 is (1-rr)/2 */
#define rr 0.7
#define rr1 0.15
#define tt (M_PI/72)

/*-----------------------------------------------------------------
   find var
*/
double getvar(char *name) { int i;
  for (i=0; i<nv; ++i)
    if (!strcmp(name,ivar[i].name)) return ivar[i].value;
  return 0.0;
}

/*-----------------------------------------------------------------
  min and max
  */
double mind(double x, double y) {
  return min(x,y);
}
double maxd(double x, double y) {
  return max(x,y);
}
/*-----------------------------------------------------------------
   eval - return value of p, using ivar
*/
double eval(node *p) {
  double u,v;
  if (!p) return 0;
  switch (p->kind) {
  case VAR: return getvar(p->name);
  case NUM: return p->value;
  case ADD: return eval(p->lf) + eval(p->rt);
  case SUB: return eval(p->lf) - eval(p->rt);
  case MUL: return eval(p->lf) * eval(p->rt);
  case DIV:
    v = eval(p->rt);
    if (fabs(v)<err) return 0;
    return eval(p->lf) / v;
  case EXP:
    u = eval(p->lf);
    v = eval(p->rt);
    if (u<err && !whole(v)) modf(v,&v);    // result would be imaginary
    if (fabs(v * log10(fabs(u)+1)) > 300) return 0;
    if (!v) return 1;
    return pow(u,v);
  case EQU: return eval(p->lf);
  case FUN:
    if (!strcmp(p->name,"sin")) return sin(eval(p->lf));
    if (!strcmp(p->name,"cos")) return cos(eval(p->lf));
    if (!strcmp(p->name,"tan")) return tan(eval(p->lf));
    if (!strcmp(p->name,"asin")) return asin(eval(p->lf));
    if (!strcmp(p->name,"acos")) return acos(eval(p->lf));
    if (!strcmp(p->name,"atan")) return atan(eval(p->lf));
    if (!strcmp(p->name,"sinh")) return sinh(eval(p->lf));
    if (!strcmp(p->name,"cosh")) return cosh(eval(p->lf));
    if (!strcmp(p->name,"tanh")) return tanh(eval(p->lf));
    if (!strcmp(p->name,"rand")) return p->lf->value*rand()/RAND_MAX;
    if (!strcmp(p->name,"sign")) return p->lf->value<0?-1:p->lf->value>0?1:0;
    if (!strcmp(p->name,"ln")) {
      v = eval(p->lf);
      if (v<=0) return 0;
      return log(v);
    }
    if (!strcmp(p->name,"log")) {
      v = eval(p->lf);
      if (v<=0) return 0;
      return log10(v);
    }
    if (!strcmp(p->name,"abs")) return fabs(eval(p->lf));
    if (!strcmp(p->name,"r") && p->nump==2) return hypot(eval(p->lf),eval(p->rt));
    if (!strcmp(p->name,"min") && p->nump==2) return mind(eval(p->lf),eval(p->rt));
    if (!strcmp(p->name,"max") && p->nump==2) return maxd(eval(p->lf),eval(p->rt));
    if (!strcmp(p->name,"atan2") && p->nump==2) return atan2(eval(p->lf),eval(p->rt));
    if (!strcmp(p->name,"mod") && p->nump==2) return fmod(eval(p->lf),eval(p->rt));
  default: return 0;
  }
}

/*-----------------------------------------------------------------
   reset window,  use a weird number for xs to get better safety
*/
void resetx() { int i;
  xs = 7-M_PI/13; ys = xs*0.75;
  xo=xs/-2; yo=ys/-2; np=25;
  err = pow(10,-sigdig);  step=1.0;
  for (i=0; i<10; ++i) ivar[i].value = 0;
  cx=2.5; cy=2.58819; cz=-9.33013;   // camera location
  ax=.261799; ay=ax; az=0;        // camera orientation
  ts=xs; to=xo;
  co1=15; co2=2; co3=1; rain=0;
  focalen=5;
}
/*-----------------------------------------------------------------
   add a new var
*/
void addvar(char *name) { int i;
  for (i=0; i<nv; ++i) if (!strcmp(name,ivar[i].name)) break;
  if (i==nv && nv<10) strcpy(ivar[nv++].name,name);
}
/*-----------------------------------------------------------------
   look for the first var in p
*/
void setivar(node *p) { int i;
  if (p->kind==VAR) addvar(p->name);
  for (i=0; i<p->nump; ++i) setivar(p->parm[i]);
}

/*-----------------------------------------------------------------
   handle key
*/
void handlekey(char c) {
  switch (c) {
    case 72: yo+=ys/4; break;
    case 75: xo-=xs/4; break;
    case 77: xo+=xs/4; break;
    case 80: yo-=ys/4; break;
    case 82: xo+=xs*rr1; yo+=ys*rr1; xs*=rr; ys*=rr; break;
    case 83: xs/=rr; ys/=rr; xo-=xs*rr1; yo-=ys*rr1; break;
    case 71: np/=rr;  break;
    case 79: if (np>6) np*=rr;  break;
    case 73: ys/=rr; yo-=ys*rr1; break;
    case 81: yo+=ys*rr1; ys*=rr; break;
    case 'd': resetx(); break;
    case 'a': polar=!polar; break;
    case 'q': ivar[1].value -= step; break;
    case 'w': ivar[2].value -= step; break;
    case 'e': ivar[3].value -= step; break;
    case 'r': ivar[4].value -= step; break;
    case 't': ivar[5].value -= step; break;
    case 'y': ivar[6].value -= step; break;
    case 'u': ivar[7].value -= step; break;
    case 'i': ivar[8].value -= step; break;
    case 'o': ivar[9].value -= step; break;
    case 'p': step *= rr;            break;
    case '1': ivar[1].value += step; break;
    case '2': ivar[2].value += step; break;
    case '3': ivar[3].value += step; break;
    case '4': ivar[4].value += step; break;
    case '5': ivar[5].value += step; break;
    case '6': ivar[6].value += step; break;
    case '7': ivar[7].value += step; break;
    case '8': ivar[8].value += step; break;
    case '9': ivar[9].value += step; break;
    case '0': step /= rr;            break;
  }
}

/*-----------------------------------------------------------------
   MAIN GRAPH ROUTINE

   fx is an optional scaling function for the x coordinate, and fy
   is a required function for the y coord.  In case of polar coordinates
   the x is theta and the y is radius.
*/
#define rline(x,y,a,b) line(xz*(x-xo),yz*(ys+yo-(y)),xz*(a-xo),yz*(ys+yo-(b)))

void graph(node *fx,node *fy) {
  int gd=DETECT,c,i;
  double t,x1,y1,x2,y2,z;
  char ss[60];

  nv=0;
  if (fx) setivar(fx);
  setivar(fy);
  if (!nv) addvar("x");
  if (!fx) fx = newvar(ivar[0].name);
  if (!xs) {
    if (*gdriver && *gdriver!='*') {    // customized bgi file!
      gd=installuserdriver(gdriver,NULL);
      gm=gmode;
      if (!!(i=graphresult())) {
        printf(msg[17],i,grapherrormsg(i));
        pause;
        return;
      }
    }
    initgraph(&gd,&gm,"");
    if (!!(i=graphresult())) {
      printf(msg[17],i,grapherrormsg(i));
      pause;
      return;
    }
    if (!*gdriver) gmode = gm;         // auto detect and auto mode
    if (gm != gmode) setgraphmode(gmode);
    resetx();
    if (psz<2) { psz = getpalettesize()-1; pst=1; }
  }
  else setgraphmode(gmode);
  while (1) {
    xz = getmaxx()/xs;
    yz = getmaxy()/ys;
//    setlinestyle(DOTTED_LINE,0,1);
    setcolor(co2);
    rline(xo,0,xo+xs,0);
    rline(0,yo,0,yo+ys);
    rline(.1,1,-.1,1);
    rline(.1,-1,-.1,-1);
    rline(1,.1,1,-.1);
    rline(-1,.1,-1,-.1);
    rline(.1,2,-.1,2);
    rline(.1,-2,-.1,-2);
    rline(2,.1,2,-.1);
    rline(-2,.1,-2,-.1);

    rline(1.1,1,.9,1);
    rline(1,1.1,1,.9);
    rline(-1.1,1,-.9,1);
    rline(-1,1.1,-1,.9);
    rline(1.1,-1,.9,-1);
    rline(1,-1.1,1,-.9);
    rline(-1.1,-1,-.9,-1);
    rline(-1,-1.1,-1,-.9);

//    setlinestyle(SOLID_LINE,0,1);
    setcolor(co1);
    sprintf(ss,"%g  ",yo+ys);  outtextxy(getmaxx()/2,0,ss);
    sprintf(ss,"%g  ",yo   );  outtextxy(getmaxx()/2,getmaxy()-8,ss);
    sprintf(ss,"%g  ",xo+xs);  outtextxy(getmaxx()-50,getmaxy()/2,ss);
    sprintf(ss,"%g  ",xo   );  outtextxy(0,getmaxy()/2,ss);
    sprintf(ss,"%s %g  ",msg[18],step);
    if (polar) strcat(ss,msg[19]); outtextxy(0,0,ss);
    for (i=1; i<nv; ++i) {
      sprintf(ss,"%s = %g  ",ivar[i].name,ivar[i].value);
      outtextxy(0,i*8,ss);
    }
    srand(2);
    for (t=xo; t<=xs+xo; t+=xs/np) {
      ivar[0].value = t;
      if (polar) ivar[0].value = (2*(t-xo)/xs-1)*M_PI;
      x2 = eval(fx);
      y2 = eval(fy);
      if (polar) {
	z = x2;
	x2 = y2*cos(z);
	y2 = y2*sin(z);
      }
      x2 = xz*(x2-xo);
      y2 = yz*(ys+yo-y2);
      if (x2>30000) x2=30000;
      if (x2<-30000) x2=-30000;
      if (y2>30000) y2=30000;
      if (y2<-30000) y2=-30000;
      if (t!=xo) line(x1,y1,x2,y2);
      x1=x2; y1=y2;
      if (kbhit()) break;
    }
    c = getch();
    while (kbhit()) c = getch();
    if (c==27) break;                      /* escape */
    if (c=='g' && graph3d(fx,fy)) break;   /* 3d graphics */
    handlekey(c);
    cleardevice();
  }
  restorecrtmode();
  if (ti.screenheight>25) textmode(64);    /* ega 43 line mode */
}

/*-----------------------------------------------------------------
   compute matrix
*/
static void computematrix() {
  double sa,sb,sc,ca,cb,cc; int i;

  //  Precompute some sines and cosines
  ca=cos(ax); sa=sin(ax);
  cb=cos(ay); sb=sin(ay);
  cc=cos(az); sc=sin(az);

  //  Compute rotation transformation
  m[0][0]=cb*cc-sb*sa*sc;   /* R = Rb.Ra.Rc */
  m[0][1]=ca*sc;
  m[0][2]=sb*cc+cb*sa*sc;
  m[1][0]=-cb*sc-sb*sa*cc;
  m[1][1]=ca*cc;
  m[1][2]=-sb*sc+cb*sa*cc;
  m[2][0]=-sb*ca;
  m[2][1]=-sa;
  m[2][2]=cb*ca;

  //  Compute translation transformation
  for (i=0; i<3; ++i) m[i][3]=-cx*m[i][0]-cy*m[i][1]-cz*m[i][2];

  for (i=0; i<4; ++i) m[3][i]=m[2][i]/focalen;
}

static void view3d(double dx,double dy,double dz,double *xx,double *yy) {
  double x,y,z,t;
  x = dx*m[0][0] + dy*m[0][1] + dz*m[0][2] + m[0][3];
  y = dx*m[1][0] + dy*m[1][1] + dz*m[1][2] + m[1][3];
  z = dx*m[3][0] + dy*m[3][1] + dz*m[3][2] + m[3][3];

  //  Check if the point is behind the visual cone.
  t = hypot(x,y)/5;    // 5 is slope of the cone
  if (z < t) z=t;      // push point to the cone
  if (z < EPS) z=EPS;  // don't divide by zero
  *xx = x/z;
  *yy = y/z;
}

/*-----------------------------------------------------------------
   move camera forward or backward
*/
void movecam(double spd) {
  double y,d;
  y=sin(ax);
  d=cos(ax);
  cz+=spd*cos(ay)*d;
  cy-=spd*y;
  cx-=spd*sin(ay)*d;
}
/*-----------------------------------------------------------------
   rotate camera - with constant radius v=-1,1 vertically or 3,5 horiz
*/
void spincam(int v) {
  double r,t,a;
  r = hypot(cx,cz);
  if (v<2) {              // vertical
    t = atan2(cy,r) + tt*v;
    if (fabs(t)>M_PI/2) return;
    a = hypot(r,cy);
    cy = sin(t)*a;
    a *= cos(t)/r;
    cx *= a; cz *= a;
    ax+=tt*v;
  }
  else {               // horizontal
    t = atan2(cz,cx);
    t+=tt*(v-4);
    cx=cos(t)*r;
    cz=sin(t)*r;
    ay+=tt*(v-4);
  }
}
#define rline3(a,b,c,u,v,w) \
  view3d(a,b,c,&x,&y);  \
  view3d(u,v,w,&z,&yy); \
  rline(x,y,z,yy)
/*------------------------------------------------------------------
   3D Graphics

   This draws a 3d surface function where the altitude is z = fz,
   and x is the first variable found in fx and y is fy (if specified)
   or if fy is NULL then y is the second variable found in fz.  If
   there are insufficient variables in fz, then a zero is returned.

   Implementation: z is fz, x is ivar[0],y is fy or ivar[1].

   P.S.  I changed my mind in the middle of doing this.  Now y is fz,
   a function of x and z, and fy is unused.  I did this because y is the
   vertical axis.  Thus the code is confusing at places.
   Actually, I decided to make x=fy, so that I can make a bagel in polar
   mode.
*/
int graph3d(node *fy,node *fz) {
  int c,i,j,mm=0;        // mm 0=norm, 1=camera, 2=grid
  static double x,y,z,xx,yy,x1,y1,x2,y2,zz,*px,*py,cc;
  static char ss[60];
  static int np=0,onp=0;
  static double zmag,uo,us;
  int spd=0;

  while (1) {
    if (spd) {    // speedy mode (cleardevice is very slow)
      delay(50);
      setcolor(0);
      rline3(-2,0,0,2,0,0);         // erase axes (draw black)
      rline3(0,-2*zmag,0,0,2*zmag,0);
      rline3(0,0,-2,0,0,2);
      rline3(2.2,0,0,2,0,-0.1);   // arrows
      rline3(2.2,0,0,2,0, 0.1);
      rline3(0,0,2.2,-0.1,0,2);
      rline3(0,0,2.2, 0.1,0,2);
      rline3(0,2.2*zmag,0,-0.1,2*zmag,0);
      rline3(0,2.2*zmag,0, 0.1,2*zmag,0);
    }
    if (polar) { uo=-M_PI; us=-2*uo; to=uo/2; ts=us/2; }
    if (!np) { zmag=1; us=ts; uo=to; np=15; }    // reset
    computematrix();
    xz = getmaxx()/xs;
    yz = getmaxy()/ys;
    if (np<3) np = 3;
    if (np>3000) np = 3000;

    if (!spd) cleardevice();
    spd=0;
//    setlinestyle(DOTTED_LINE,0,1);
    setcolor(co2);
    rline3(-2,0,0,2,0,0);         // draw axes
    rline3(0,-2*zmag,0,0,2*zmag,0);
    rline3(0,0,-2,0,0,2);
    rline3(2.2,0,0,2,0,-0.1);       // arrows
    rline3(2.2,0,0,2,0, 0.1);
    rline3(0,0,2.2,-0.1,0,2);
    rline3(0,0,2.2, 0.1,0,2);
    rline3(0,2.2*zmag,0,-0.1,2*zmag,0);
    rline3(0,2.2*zmag,0, 0.1,2*zmag,0);

    if (!kbhit()) {
      setcolor(co1);
      sprintf(ss,"%s %g  ",msg[18],step );
      if (polar) strcat(ss,msg[19]); outtextxy(0,0,ss);
      for (i=2; i<nv; ++i) {
        sprintf(ss,"%s = %8.4f  ",ivar[i].name,ivar[i].value);
        outtextxy(0,i*10,ss);
      }
      x = 180/M_PI;
      sprintf(ss,"[%8.4f,%8.4f]  ",to,to+ts);  outtextxy(0,i++*10,ss);
      sprintf(ss,"[%8.4f,%8.4f]  ",uo,uo+us);  outtextxy(0,i++*10,ss);
      sprintf(ss," %8.4f  %d  ",zmag,np); outtextxy(0,i++*10,ss);
      sprintf(ss," %8.4f %8.4f %8.4f  ",cx,cy,cz);  outtextxy(0,i++*10,ss);
      sprintf(ss," %8.4f %8.4f %8.4f  ",ax*x,ay*x,az*x); outtextxy(0,i++*10,ss);
      if (mm) outtextxy(0,i++*10,msg[27]);
    } else spd=1;
    if (!kbhit()) {
      setcolor(co2);
      rline3(2,0,0.1,2,0,-0.1);
      rline3(0.1,2*zmag,0,-0.1,2*zmag,0);
      rline3(0.1,0,2,-0.1,0,2);
      rline3(1,0,0.1,1,0,-0.1);                  // axis ticks
      rline3(-1,0,0.1,-1,0,-0.1);
      rline3(0.1,zmag,0,-0.1,zmag,0);
      rline3(0.1,-1*zmag,0,-0.1,-1*zmag,0);
      rline3(0.1,0,1,-0.1,0,1);
      rline3(0.1,0,-1,-0.1,0,-1);
    }
    if (!kbhit() && !polar) {
      setcolor(co3);
      rline3(to   ,0,uo   ,to+ts,0,uo   );
      rline3(to+ts,0,uo   ,to+ts,0,uo+us);
      rline3(to+ts,0,uo+us,to   ,0,uo+us);
      rline3(to   ,0,uo+us,to   ,0,uo   );

      rline3(-1,-zmag, 1, 1,-zmag, 1);    // draw a unit box
      rline3( 1,-zmag, 1, 1, zmag, 1);
      rline3( 1, zmag, 1,-1, zmag, 1);
      rline3(-1, zmag, 1,-1,-zmag, 1);
      rline3(-1,-zmag,-1,-1,-zmag, 1);
      rline3( 1,-zmag,-1, 1,-zmag, 1);
      rline3(-1, zmag,-1,-1, zmag, 1);
      rline3( 1, zmag,-1, 1, zmag, 1);
      rline3(-1,-zmag,-1, 1,-zmag,-1);
      rline3( 1,-zmag,-1, 1, zmag,-1);
      rline3( 1, zmag,-1,-1, zmag,-1);
      rline3(-1, zmag,-1,-1,-zmag,-1);
    }
//    setlinestyle(SOLID_LINE,0,1);
    setcolor(co1);
    if (onp<np || onp>np*3) { onp = np;
      if (onp) { free(px); free(py); }
      px = malloc((np+1)*sizeof*px);
      py = malloc((np+1)*sizeof*py);
    }
    /*-----------------------
	 MAIN DRAW LOOP
    */
    srand(2);
    if (!kbhit())
    for (j=0,x=to; j<=np; ++j,x+=ts/np) {
      ivar[0].value = x;
      for (i=0,y=uo; i<=np; ++i,y+=us/np) {
	ivar[1].value = y;
/*	x = eval(fy);       didn't work, no time to fix */
        xx = x;
        zz = y;
	yy = cc = eval(fz);
        if (polar) {
          xx = cos(y)*cos(x)*yy;
          zz = sin(y)*cos(x)*yy;
          yy *= sin(x);
        }
	view3d(xx,yy*zmag,zz,&x2,&y2);
	x2 = xz*(x2-xo);
	y2 = yz*(ys+yo-y2);
        if (rain) setcolor(abs(zmag*cc*np/ts)%psz+pst);
	if (i) line(x1,y1,x2,y2);
	if (j) line(px[i],py[i],x2,y2);
	px[i]=x1=x2;
	py[i]=y1=y2;
        if (kbhit()) break;
      }
      if (kbhit()) break;
    }
    c = getch(); if (!c) c=getch();  // get ONE keystroke
//    if (kbhit()) spd=1;   // do a quick screen refresh (no clear)
    while(kbhit()) c=getch();   // flush kbd buffer
    if (c==27) return 1;    /* escape */
    if (c=='g') return 0;   /* 2d graphics */
    if (c=='d') np=0;       // force reset above
    if (c=='a' && polar) { ts*=2; to*=2; }  // make area square
    if (c<84 && c>70) c+=mm;
    switch (c) {
    case 'c': rain=1-rain; break;       // rainbow
    case 13: mm=200-mm; break;
    case 75: spincam(3); break;       // l,r,u,d
    case 77: spincam(5); break;
    case 72: spincam(1); break;
    case 80: spincam(-1); break;
    case 82: cx*=rr; cy*=rr; cz*=rr; break;    // ins,del
    case 83: cx/=rr; cy/=rr; cz/=rr; break;
    case 71: np/=rr; break;              //  home end
    case 79: if (np>7) np*=rr; break;
    case 73: zmag/=rr; break;   //  pgup dn
    case 81: zmag*=rr; break;
    case 275: ay+=tt; break;       // mm l,r,u,d
    case 277: ay-=tt; break;
    case 272: ax+=tt; break;
    case 280: ax-=tt; break;
    case 282: movecam(3); break;   // mm ins,del
    case 283: movecam(-3); break;
    case 273: az+=tt; break;      // mm pgup dn
    case 281: az-=tt; break;
    case 115: to-=ts/4; break;   // ctrl l r u d
    case 116: to+=ts/4; break;
    case 141: uo+=us/4; break;
    case 145: uo-=us/4; break;
    case 146: ts/=rr; us/=rr;
              to-=ts*rr1; uo-=us*rr1; break; // ctrl ins del
    case 147: to+=ts*rr1; uo+=us*rr1;
              ts*=rr; us*=rr; break;
    case 132: us/=rr; uo-=us*rr1; break;   // ctrl pgup dn
    case 118: uo+=us*rr1; us*=rr; break;
    case 987: fy+=1;        // avoid compiler warning
    default: handlekey(c);
    }
  }
  /* never here */
}
