/* 
 *     XaoS, a fast portable realtime fractal zoomer 
 *                  Copyright (C) 1996 by
 *
 *      Jan Hubicka          (hubicka@paru.cas.cz)
 *      Thomas Marsh         (tmarsh@austin.ibm.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef _plan9_
#include <u.h>
#include <libc.h>
#include <stdio.h>
#else
#include <stdio.h>
#ifndef _MAC
#include <malloc.h>
#else
#include <stdlib.h>
#endif
#include "aconfig.h"
#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif
#include <math.h>
#include <string.h>
#ifdef HAVE_ALLOCA_H
#include <alloca.h>
#endif
#include "config.h"
#ifndef _plan9_
/*#undef NDEBUG */
#include <assert.h>
#endif
#endif
#include "zoom.h"
#include "complex.h"
#include "formulas.h"
#include "plane.h"

#define callwait() if(d->wait_function!=NULL) d->wait_function();

int MAXITER = DEFAULT_MAX_ITER;

typedef struct {
    double y0, k, kk, y0k;
} symetry2;

struct symetryinfo2 {
    number_t xsym, ysym;
    int nsymetries;
    symetry2 *symetry;
};

static symetry2 sym_lines[100];
static struct symetryinfo2 cursymetry;
static number_t xmul, ymul, xdist, ydist;

#ifdef SCROLLING
int counter = 0;
#else
#define counter 0
#endif
static zoom_context *d;
static int /*INLINE */ calculate(number_t, number_t) CONSTF;
static int INLINE symetries(number_t, number_t) CONSTF;
#ifdef DEBUG
int mirrored2 = 0;
#endif

static int INLINE symetries(number_t x, number_t y)
{
    int i;

    for (i = 0; i < cursymetry.nsymetries; i++) {
	double b = cursymetry.symetry[i].k, c = x + b * y, x1, y1, dist;
	int x2, y2;
	x1 = (c - cursymetry.symetry[i].y0k) / (cursymetry.symetry[i].kk);
	y1 = (c - x1) / b;
	x1 = x - 2 * (x - x1);
	y1 = y - 2 * (y - y1);
	if (x1 < d->s.nc || y1 < d->s.ni || x1 >= d->s.mc || y1 >= d->s.mi)
	    continue;

	x2 = (x1 - d->s.nc) * xmul;
	y2 = (y1 - d->s.ni) * ymul;

	dist = myfabs(d->reallocx[x2].possition - x1);
	if (x2 > 0 && myfabs(d->reallocx[x2 - 1].possition - x1) < dist) {
	    x2--;
	    dist = myfabs(d->reallocx[x2].possition - x1);
	} else if (x2 < d->width - 1 && myfabs(d->reallocx[x2 + 1].possition - x1) < dist) {
	    x2++;
	    dist = myfabs(d->reallocx[x2].possition - x1);
	}
	if (dist > xdist)
	    continue;
	dist = myfabs(d->reallocy[y2].possition - y1);
	if (y2 > 0 && myfabs(d->reallocy[y2 - 1].possition - y1) < dist) {
	    y2--;
	    dist = myfabs(d->reallocy[y2].possition - y1);
	} else if (y2 < d->height - 1 && myfabs(d->reallocy[y2 + 1].possition - y1) < dist) {
	    y2++;
	    dist = myfabs(d->reallocy[y2].possition - y1);
	}
	if (dist > ydist)
	    continue;
	if (d->reallocx[x2].symto != -1)
	    x2 = d->reallocx[x2].symto;
	if (d->reallocy[y2].symto != -1)
	    y2 = d->reallocy[y2].symto;
	if (d->reallocx[x2].dirty ||
	    d->reallocy[y2].dirty /* == 1 */ )
	    continue;
#ifdef DEBUG
	mirrored2++;
#endif
	return ((unsigned char) *(d->vbuff + x2 + y2 * d->scanline));
    }
    return (-1);
}

static int /*INLINE */ calculate(number_t x, number_t y)
{
    int i;

    i = symetries(x, y);
    if (i != -1)
	return (i);
    if (d->plane) {
	recalculate(d, &x, &y);
    }
    if (d->mandelbrot)
	i = d->currentformula->calculate(x, y, x, y);
    else
	i = d->currentformula->calculate(x, y, d->pre, d->pim);
    if (i < 0)
	i += (-i / d->num_colors + 1) * d->num_colors;
    if (i == MAXITER)
	return d->colors[0];
    return (d->colors[1 + (i + counter) % (d->num_colors - 1)]);
}
struct dyn_data {
    number_t price;
    int previous;
};

#ifndef _UNDEFINED_
#define RANGES 2		/*shift equal to x*RANGE */
#define RANGE 4
#define DSIZEHMASK (0x7)	/*mask equal to x%(DSIZE) */
#else
#define RANGES 1		/*shift equal to x*RANGE */
#define RANGE 2
#define DSIZEHMASK (0x3)	/*mask equal to x%(DSIZE) */
#endif
#define DSIZE (2*RANGE)
#define DSIZES (RANGES+1)	/*shift equal to x*DSIZE */
#define getdata(n,i) ((((n)<<DSIZES)+(((i)&(DSIZEHMASK)))))
#define adddata(n,i) ((((n)<<DSIZES)+(((i)&(DSIZEHMASK)))))
#define getbest(i) (((size)<<DSIZES)+(i))
#ifndef _UNDEFINED_
#define PRICE(i,i1) ((((i)-(i1))*((i)-(i1)))/(pricerange))
#define NEWPRICE (/*range*range*/1)
#else
#define PRICE(i,i1) (myabs((i)-(i1)))
#define NEWPRICE (range)
#endif
#define NOSETMASK 0x80000000
#define MAXPRICE INT_MAX
CONST int dynsize = sizeof(struct dyn_data);

static /*INLINE */ void mkrealloc_table(number_t * pos, realloc_t * realloc, CONST int size, CONST number_t begin, CONST number_t end, CONST number_t sym)
{
    int i, p, ps, ps1, pe;
    number_t bestprice;
    realloc_t *r = realloc, *reallocs;
    struct dyn_data *dyndata, *dynptr;
    int bestdata = 0;
    int *best, *best1, *tmp;
    number_t step = (end - begin) / (number_t) size;
    number_t y, range = RANGE * step, yend;
    int symi = -1;
    int lastplus = 0;
    int data;
    int previous = 0;
    number_t pricerange;
    number_t myprice;

#ifdef DEBUG
    int nadded = 0, nsymetry = 0, nskipped = 0;
#endif
#ifdef HAVE_ALLOCA
    dyndata = alloca((size) * (DSIZE + 1) * sizeof(struct dyn_data));
    best = alloca(size * sizeof(int));
    best1 = alloca(size * sizeof(int));
#else
    dyndata = malloc((size) * (DSIZE + 1) * sizeof(struct dyn_data));
    best = malloc(size * sizeof(int));
    best1 = malloc(size * sizeof(int));
#endif
    if (dyndata == NULL) {
	fprintf(stderr, "XaoS fratal error:Could not allocate memory for"
		"temporary dynamical data of size:%i\n"
		"I am unable to handle this problem so please resize to lower window\n", (size) * (DSIZE + 1) * sizeof(struct dyn_data) + size * sizeof(int) + size * sizeof(int));
	return;
    }
    if (best == NULL) {
	fprintf(stderr, "XaoS fratal error:Could not allocate memory for"
		"temporary dynamical data of size:%i\n"
		"I am unable to handle this problem so please resize to lower window\n", (size) * (DSIZE + 1) * sizeof(struct dyn_data) + size * sizeof(int) + size * sizeof(int));
#ifndef HAVE_ALLOCA
	free(dyndata);
#endif
	return;
    }
    if (best1 == NULL) {
	fprintf(stderr, "XaoS fratal error:Could not allocate memory for"
		"temporary dynamical data of size:%i\n"
		"I am unable to handle this problem so please resize to lower window\n", (size) * (DSIZE + 1) * sizeof(struct dyn_data) + size * sizeof(int) + size * sizeof(int));
#ifndef HAVE_ALLOCA
	free(dyndata);
	free(best);
#endif
	return;
    }
/* get prices */

    if (begin > sym)
	symi = -2;
    p = 0;
    ps = 0;
    pe = 0;
    pricerange = range * range;
    bestprice = MAXPRICE;
    for (i = 0; i < size; i++) {
	bestprice = MAXPRICE;
	bestdata = 0;
	myprice = MAXPRICE;
	tmp = best1;
	best1 = best;
	best = tmp;
	y = begin + i * step;
	p = ps;			/*just inicialize parameters */
	if (y > sym && symi == -1)
	    symi = i - 1;	/*find symetry point */
	yend = y - range;
	if (yend < begin - step)	/*do no allow lines outside screen */
	    yend = begin - step;
	while (pos[p] <= yend && p < size)	/*skip lines out of range */
	    p++;
	ps1 = p;
	yend = y + range;
	/*printf("%i %i %i\n",p,ps,pe); *//*finds best previous path for situation
	   where no lines for this available */
	if (ps != pe && p > ps) {	/*previous point had lines */
	    assert(p >= ps);
	    if (p < pe) {
		previous = best[p - 1];
	    } else
		previous = best[pe - 1];
	    myprice = dyndata[previous].price;	/*find best one */
	} else {
	    if (i > 0) {	/*previous line had no lines */
		previous = getbest(i - 1);
		myprice = dyndata[previous].price;
	    } else
		previous = -1, myprice = 0;
	}
	if (p < pe && pos[p] < yend && p != size)	/*try to skip possition in last line
							   and add it to this. Sometimes is better */
	    if (previous != best[p]) {
		int previous1 = best[p];
		number_t price1 = dyndata[previous1].price + 2 * NEWPRICE - PRICE(pos[p], y - step);
		if (price1 < myprice) {
		    myprice = price1;
		    previous = previous1 | NOSETMASK;
		}
	    }
	data = getbest(i);	/*find store possition */
	if (pos[p] >= yend /*previous<=(size<<DSIZE) */ )	/*calculate price. */
	    /*preffer one doubled line instead of two */
	    myprice += NEWPRICE;	/*that results better in optimalizations */
	else
	    myprice += 2 * NEWPRICE;
	/*if(previous>data) abort(); */
	dyndata[data].price = myprice;	/*store data */
	dyndata[data].previous = previous;
	bestprice = myprice, bestdata = data;	/*calculate best available price */
	assert(bestprice >= 0);
	/*dyndata[data].best = data; */
	if (yend > end + step)	/*check bounds */
	    yend = end + step;
	data = adddata(p, i);	/*calcialte all lines good for this y */
	dynptr = &dyndata[data];
	if (ps != pe) {		/*in case that previous had laso possitions */
	    while (pos[p] < yend && p < size) {
		myprice = MAXPRICE;
		if (p > ps) {
		    assert(p >= ps);	/*find best one in previous */
		    if (p < pe) {
			previous = best[p - 1];
		    } else
			previous = best[pe - 1];
		    myprice = dyndata[previous].price;
		}
		if (p < pe)	/*also try to skip adding in previous but add
				   in this */
		    if (previous != best[p]) {
			int previous1 = best[p];
			number_t price1 = dyndata[previous1].price + NEWPRICE - PRICE(pos[p], y - step);
			if (price1 < myprice) {
			    myprice = price1;
			    previous = previous1 | NOSETMASK;
			}
		    }
		myprice += PRICE(pos[p], y);	/*store data */
		dynptr->price = myprice;
		/*if(previous>data) abort(); */
		dynptr->previous = previous;
		if (myprice < bestprice)	/*calcualte best */
		    bestprice = myprice, bestdata = data;
		assert(bestprice >= 0);
		assert(myprice >= 0);
		/*data = adddata(p,i)+1; */
		best1[p] = bestdata;
		p++;
		data += DSIZE;
		dynptr += DSIZE;
	    }
	} else {
	    number_t myprice1;	/*simplified loop for case that previous
				   y had no lines */
	    if (pos[p] < yend && p < size) {
		if (i > 0) {
		    previous = getbest(i - 1);
		    myprice1 = dyndata[previous].price;
		} else
		    previous = -1, myprice1 = 0;
		/*if(p==pe) ps1++,p++; */
		while (pos[p] < yend && p < size) {
		    myprice = myprice1 + PRICE(pos[p], y);
		    dynptr->price = myprice;
		    dynptr->previous = previous;
		    if (myprice < bestprice)
			bestprice = myprice, bestdata = data;
		    assert(bestprice >= 0);
		    assert(myprice >= 0);
		    /*data = adddata(p,i)+1; */
		    best1[p] = bestdata;
		    p++;
		    data += DSIZE;
		    dynptr += DSIZE;
		}
	    }
	}
	previous = ps;		/*store possitions for next loop */
	ps = ps1;
	ps1 = pe;
	pe = p;
	/*y += step; */
    }
    if (previous != ps1) {	/*this tries situation where is better to add new in last
				   line */
	if (pos[p] > end) {
	    while (pos[p] > end + range)
		p--;
	    if (pos[p + 1] < end + range)
		p++;
	}
	if (p < previous)
	    p = previous;
	if (p < ps1) {
	    previous = best[p];
	} else
	    previous = best[ps1 - 1];
	myprice = dyndata[previous].price + NEWPRICE;
	if (myprice < bestprice) {
	    bestprice = myprice;
	    data = getbest(i - 1);
	    dyndata[data].previous = previous;
	    bestdata = data;
	}
    }
    assert(bestprice >= 0);
    realloc += size;
    for (i = size; i > 0;) {	/*and finally traces the path */
	realloc--;
	i--;
	realloc->symto = -1;
	realloc->symref = -1;
	if ((bestdata & NOSETMASK) || bestdata >= ((size) << DSIZES)) {
	    /*realloc->possition=y; */
	    realloc->possition = begin + i * step;

	    realloc->recalculate = 1;
#ifdef DEBUG
	    nadded++;
#endif
	    realloc->dirty = 1;
	    lastplus++;
	    if (lastplus >= size)
		lastplus = 0;
	    realloc->plus = lastplus;
	    if (i < size - 1 && realloc->possition > (realloc + 1)->possition) {
		/*this fixes bug in previous algorithm. Sometimes are lines unsorted.
		   So bubblesort them */
		struct realloc tmp = *realloc;
		*realloc = *(realloc + 1);
		*(realloc + 1) = tmp;
		(realloc + 1)->possition += step;
	    }
	} else {
	    int p = bestdata >> DSIZES;
	    assert(p >= 0 && p < size);
	    realloc->possition = pos[p];
#ifdef _UNDEFINED_
	    if (pos[p] > 50) {
		printf("%i %i", p, bestdata);
		abort();
	    }
	    pos[p] = 60;
#endif
	    realloc->plus = p;
	    realloc->dirty = 0;
	    realloc->recalculate = 0;
	    lastplus = p;
	}
	bestdata = dyndata[bestdata & (~NOSETMASK)].previous;
    }
#ifndef HAVE_ALLOCA
    free(dyndata);
    free(best);
    free(best1);
#endif
    realloc = r;
    realloc++;
    for (i = 1; i <= symi; i++, realloc++) {	/*makes symetries */
	int j, min = 0;
	number_t dist = INT_MAX, tmp;
	if (realloc->symto != -1)
	    continue;
	y = realloc->possition;
	realloc->symto = 2 * symi - i;
	if (realloc->symto >= size - 1)
	    realloc->symto = size - 2;
	dist = myfabs(2 * sym - r[realloc->symto].possition - y);
	min = 0;
	for (j = realloc->symto > 10 ? -10 : -realloc->symto; j < 10 && realloc->symto + j < size - 1; j++) {
	    if ((tmp = myfabs(2 * sym - r[realloc->symto + j].possition - y)) < dist) {
		if (2 * sym - r[realloc->symto + j].possition > (realloc - 1)->possition && ( /*i==symi-1|| */ (2 * sym - r[realloc->symto + j].possition < (realloc + 1)->possition))) {
		    dist = tmp;
		    min = j;
		}
	    }
	}
	realloc->symto += min;
	if (realloc->symto <= symi || min == 10) {
	    realloc->symto = -1;
	    continue;
	}
	reallocs = &r[realloc->symto];
	if (reallocs->symto != -1 || reallocs->symref != -1) {
	    realloc->symto = -1;
	    continue;
	}
	if (myfabs(realloc->possition - (2 * sym - reallocs->possition)) < step * 3 / 4) {
	    if (!realloc->recalculate) {
		realloc->symto = -1;
		if (reallocs->symto != -1 || (
						 !reallocs->recalculate		/*&&
										   realloc->possition!=2*sym-reallocs->possition */ ))
		    continue;
		reallocs->plus = realloc->plus;
		reallocs->symto = i;
		reallocs->dirty = 1;
		realloc->symref = reallocs - r;
		/*reallocs->dirty = realloc->dirty; */
#ifdef DEBUG
		nadded -= reallocs->recalculate;
#endif
		reallocs->recalculate = 0;
		reallocs->possition = 2 * sym - realloc->possition;
	    } else {
		if (reallocs->symto != -1) {
		    realloc->symto = -1;
		    continue;
		}
#ifdef DEBUG
		nadded -= realloc->recalculate;
#endif
		realloc->dirty = 1;
		/*realloc->dirty = reallocs->dirty; */
		realloc->plus = reallocs->plus;
		realloc->recalculate = 0;
		reallocs->symref = i;
		realloc->possition = 2 * sym - reallocs->possition;
	    }
#ifdef DEBUG
	    nsymetry++;
#endif
	} else
	    realloc->symto = -1;
    }
    realloc = r + 1;
    for (i = 1; i < size - 1; i++, realloc++) {		/*optimizes new possitions to look better */
	if (realloc->recalculate && (!(realloc - 1)->recalculate || !(realloc + 1)->recalculate)) {
	    realloc->possition = ((realloc - 1)->possition + (realloc + 1)->possition) / 2;
	}
    }
#ifdef DEBUG
    printf("%i added %i skipped %i mirrored\n", nadded, nskipped, nsymetry);
#endif
}


static INLINE void moveoldpoints(void)
{
    char *vline, *vbuff = d->vbuff;
    int *size, *siz, *sizend;
    int *start, *startptr;
    realloc_t *ry, *rend;
    realloc_t *rx, *rend1;
    int plus1 = 0, plus2 = 0;

#ifdef HAVE_ALLOCA
    size = (int *) alloca(d->width * sizeof(int));
    start = (int *) alloca(d->width * sizeof(int));
#else
    size = (int *) malloc(d->width * sizeof(int));
    start = (int *) malloc(d->width * sizeof(int));
#endif
    if (size == NULL) {
	fprintf(stderr, "XaoS fratal error:Could not allocate memory for"
		"temporary dynamical data of size:%i\n"
		"I am unable to handle this problem so please resize to lower window\n", 2 * d->width * sizeof(int));
	return;
    }
    if (start == NULL) {
	fprintf(stderr, "XaoS fratal error:Could not allocate memory for"
		"temporary dynamical data of size:%i\n"
		"I am unable to handle this problem so please resize to lower window\n", 2 * d->width * sizeof(int));
#ifndef HAVE_ALLOCA
	free(size);
#endif
	return;
    }
    assert(size != NULL);
    assert(start != NULL);

    startptr = start;
    siz = size;
    for (rx = d->reallocx, rend1 = d->reallocx + d->width;
	 rx < rend1; rx++)
	if ((rx->recalculate) && plus1 < d->width + 1)
	    plus1++;
	else
	    break;
    *startptr = d->reallocx->plus;
    *siz = 0;
    for (; rx < rend1; rx++) {
	if ((rx->recalculate || rx->plus == *startptr + *siz) && *startptr + *siz < d->width)
	    (*siz)++;
	else {
	    siz++, startptr++;
	    if (rx->recalculate) {
		*startptr = 0;
		ry = rx;
		for (; rx < rend1; rx++)
		    if (rx->recalculate)
			(*startptr)--;
		    else
			break;
		if (rx < rend1)
		    *startptr += rx->plus, rx = ry;
		else {
		    plus2 = -(*startptr);
		    startptr--;
		    siz--;
		    break;
		}
	    }
	    *startptr = rx->plus;
	    assert(rx->plus >= 0 && rx->plus <= d->width);
	    *siz = 1;
	}

    }
    if (*siz)
	sizend = siz + 1;
    else
	sizend = siz;
#ifdef DEBUG
    printf("nsegments:%i plus1:%i plus2:%i\n", sizend - size, plus1, plus2);
#endif
    plus2 += d->scanline - d->width;
    for (ry = d->reallocy, rend = ry + d->height; ry < rend; ry++) {
	vline = d->back + ry->plus * d->scanline;
	if (!ry->recalculate) {
	    vbuff += plus1;
	    for (startptr = start, siz = size; siz < sizend; siz++, startptr++) {
		assert(vbuff >= d->vbuff);
		assert(vbuff + *siz <= d->vbuff + d->scanline * d->height);
		assert(vline + *startptr >= d->back);
		assert(vline + *startptr + *siz <= d->back + d->scanline * d->height);
		memcpy(vbuff, vline + *startptr, *siz), vbuff += *siz;
	    }
	    vbuff += plus2;
	} else
	    vbuff += d->scanline;
    }
#ifndef HAVE_ALLOCA
    free((void *) size);
    free((void *) start);
#endif
}


static /*INLINE */ void calculatenew(void)
{
    number_t x, y;
    int s;
    char *vbuff = d->vbuff, *vbuff1;
    realloc_t *rx, *ry, *rend, *rend1;
    int s2;
#ifdef DEBUG
    int tocalculate = 0;
    int avoided = 0;
    /*int s3; */
    mirrored2 = 0;
#endif


    d->max = 0;
    d->pos = 0;
    for (ry = d->reallocy, rend = ry + d->height; ry < rend; ry++) {
	if (ry->recalculate)
	    d->max++;
    }
    callwait();
    for (s = 0; s < 2; s++) {
	vbuff = d->vbuff + s * d->scanline;
	for (ry = d->reallocy + s, rend = ry + d->height - s; ry < rend; ry += 2) {
	    if (ry->recalculate) {
		y = ry->possition;
		vbuff1 = vbuff;
		for (s2 = 0; s2 < 2; s2++)
		    for (rx = d->reallocx + s2, vbuff = vbuff1 + s2, rend1 = d->reallocx + d->width;
			 rx < rend1; rx += 2) {
			assert(vbuff >= d->vbuff);
			assert(vbuff < d->vbuff + d->scanline * d->height);
			assert(rx >= d->reallocx && rx <= d->reallocx + d->width);
			assert(ry >= d->reallocy && ry <= d->reallocy + d->height);
			if (!rx->dirty /*!rx->recalculate && rx->symto == -1 */ ) {
#ifdef DEBUG
			    tocalculate++;
#endif
			    if (ry != d->reallocy && ry < rend - 1 &&
				rx != d->reallocx && rx < rend1 - 1 &&
				!(ry - 1)->dirty &&
				!(ry + 1)->dirty &&
			    (vbuff[-d->scanline] == vbuff[d->scanline] &&
			     ((
				  !(rx - 1)->dirty &&
				  !(rx + 1)->dirty &&
				  vbuff[-d->scanline + 1] == vbuff[d->scanline] &&
				  vbuff[-d->scanline - 1] == vbuff[d->scanline] &&
				  vbuff[d->scanline] == vbuff[d->scanline - 1] &&
				  vbuff[d->scanline] == vbuff[d->scanline + 1]) ||
			      (s2 &&
			       (
				   vbuff[-1] == vbuff[d->scanline] &&
				   vbuff[+1] == vbuff[d->scanline]
			       ))))) {
				*vbuff = vbuff[d->scanline];
#ifdef DEBUG
				avoided++;
#endif
			    } else {
				/*s3=mirrored2; */
				*vbuff = calculate(rx->possition, y);
				/*if(s3==mirrored2) vga_drawpixel(rx-d->reallocx,ry-d->reallocy,255); */
			    }
			}
			vbuff += 2;
		    }
		d->pos++;
		callwait();
		if (d->interrupt)
		    return;
		ry->recalculate = 2;
		ry->dirty = 0;
		vbuff = vbuff1 + d->scanline;
	    } else		/*if recalculate */
		vbuff += d->scanline;
	    vbuff += d->scanline;
	}			/*for ry */
    }				/*for s */
    d->max = 0;
    d->pass++;
    d->pos = 0;
    for (rx = d->reallocx, rend = rx + d->width; rx < rend; rx++) {
	if (rx->recalculate)
	    d->max++;
    }
    for (s = 0; s < 2; s++) {
	vbuff = d->vbuff + s;
	for (rx = d->reallocx + s, rend = rx + d->width - s; rx < rend; rx += 2) {
	    if (rx->recalculate) {
		x = rx->possition;
		vbuff1 = vbuff;
		for (s2 = 0; s2 < 2; s2++)
		    for (ry = d->reallocy + s2, vbuff = vbuff1 + s2 * d->scanline, rend1 = d->reallocy + d->height; ry < rend1; ry += 2) {
			assert(vbuff >= d->vbuff);
			assert(vbuff < d->vbuff + d->scanline * d->height);
			assert(rx >= d->reallocx && rx <= d->reallocx + d->width);
			assert(ry >= d->reallocy && ry <= d->reallocy + d->height);
			if (ry->symto == -1) {
#ifdef DEBUG
			    tocalculate++;
#endif
			    if (rx != d->reallocx && rx < rend - 1 &&
				ry != d->reallocy && ry < rend1 - 1 &&
				!(rx - 1)->dirty &&
				!(rx + 1)->dirty &&
				(vbuff[-1] == vbuff[1] &&
				 ((
				      !(ry - 1)->dirty &&
				      !(ry + 1)->dirty &&
				   vbuff[-1 + d->scanline] == vbuff[1] &&
				   vbuff[-1 - d->scanline] == vbuff[1] &&
				   vbuff[-1] == vbuff[1 + d->scanline] &&
				  vbuff[-1] == vbuff[1 - d->scanline]) ||
				  (s2 &&
				   (
				       vbuff[d->scanline] == vbuff[1] &&
				       vbuff[-d->scanline] == vbuff[1]
				   ))))) {
#ifdef DEBUG
				avoided++;
#endif
				*vbuff = vbuff[1];
			    } else {
				/*s3=mirrored2; */
				*vbuff = calculate(x, ry->possition);
				/*if(s3==mirrored2) vga_drawpixel(rx-d->reallocx,ry-d->reallocy,255); */
			    }
			}
			vbuff += d->scanline << 1;
		    }
		d->pos++;
		callwait();
		if (d->interrupt)
		    return;
		rx->recalculate = 0;
		rx->dirty = 0;
		/*vbuff -= d->width * d->height - 2; */
		vbuff = vbuff1 + 2;
	    } else
		vbuff += 2;
	}
    }
#ifdef DEBUG
    printf("Avoided caluclating of %i points from %i and mirrored %i %2.2f%% %2.2f%%\n", avoided, tocalculate, mirrored2, 100.0 * (avoided + mirrored2) / tocalculate, 100.0 * (tocalculate - avoided - mirrored2) / d->width / d->height);
#endif
}


static INLINE void dosymetry(void)
{
    char *vbuff = d->vbuff;
    realloc_t *rx, *ry, *rend;

    vbuff = d->vbuff;
    for (ry = d->reallocy, rend = ry + d->height; ry < rend; ry++) {
	if (ry->symto != -1) {
	    memcpy(vbuff, d->vbuff + d->scanline * ry->symto, d->width * sizeof(*vbuff));
	}
	vbuff += d->scanline;
    }
    vbuff = d->vbuff;
    for (rx = d->reallocx, rend = rx + d->width; rx < rend; rx++) {
	if (rx->symto != -1) {
	    char *vsrc = d->vbuff + rx->symto, *vend = vbuff + d->scanline * d->height;
	    for (; vbuff < vend; vbuff += d->scanline, vsrc += d->scanline)
		*vbuff = *vsrc;
	    vbuff -= d->scanline * d->height;
	}
	vbuff++;
    }
}

static void INLINE combine_methods(void)
{
    int i, j;
    struct symetryinfo *s1 = d->currentformula->out + d->coloringmode,
    *s2 = d->currentformula->in + d->incoloringmode;
    if (d->mandelbrot != d->currentformula->mandelbrot) {
	cursymetry.xsym = INT_MAX;
	cursymetry.ysym = INT_MAX;
	cursymetry.nsymetries = 0;
	return;
    }
    xmul = d->width / (d->s.mc - d->s.nc);
    ymul = d->height / (d->s.mi - d->s.ni);
    xdist = (d->s.mc - d->s.nc) / d->width / 6;
    ydist = (d->s.mi - d->s.ni) / d->height / 6;
    if (s1->xsym == s2->xsym)
	cursymetry.xsym = s1->xsym;
    else
	cursymetry.xsym = INT_MAX;
    if (s1->ysym == s2->ysym)
	cursymetry.ysym = s1->ysym;
    else
	cursymetry.ysym = INT_MAX;
    if (d->plane == P_PARABOL)
	cursymetry.xsym = INT_MAX;
    if (d->plane == P_LAMBDA) {
	if (cursymetry.xsym == 0 && cursymetry.ysym == 0)
	    cursymetry.xsym = 1;
	else
	    cursymetry.xsym = INT_MAX;
    }
    if (d->plane == P_INVLAMBDA)
	cursymetry.xsym = INT_MAX;
    if (d->plane == P_TRANLAMBDA) {
	if (cursymetry.xsym != 0 || cursymetry.ysym != 0)
	    cursymetry.xsym = INT_MAX;
    }
    if (d->plane == P_MEREBERG)
	cursymetry.xsym = INT_MAX;
    cursymetry.symetry = sym_lines;
    cursymetry.nsymetries = 0;
    if (d->plane == P_PARABOL || d->plane == P_LAMBDA || d->plane == P_INVLAMBDA ||
	d->plane == P_TRANLAMBDA || d->plane == P_MEREBERG)
	return;
    for (i = 0; i < s1->nsymetries; i++) {
	number_t y1 = s1->symetry[i].y0 + d->s.nc * s1->symetry[i].k;
	number_t y2 = s1->symetry[i].y0 + d->s.mc * s1->symetry[i].k;
	if ((y1 > d->s.ni ? (y1 < d->s.mi || y2 < d->s.mi) :
	     y2 > d->s.ni)) {
	    for (j = 0; j < s2->nsymetries; j++) {
		if (s1->symetry[i].k == s2->symetry[j].k &&
		    s1->symetry[i].y0 == s2->symetry[j].y0) {
		    cursymetry.symetry[cursymetry.nsymetries].k = s1->symetry[i].k;
		    cursymetry.symetry[cursymetry.nsymetries].y0 = s1->symetry[i].y0;
		    cursymetry.symetry[cursymetry.nsymetries].kk = s1->symetry[i].k * s1->symetry[i].k + 1;
		    cursymetry.symetry[cursymetry.nsymetries++].y0k = s1->symetry[i].y0 * s1->symetry[i].k;
		}
	    }
	}
    }
}

void do_fractal(zoom_context * context)
{
    char *tmp;
    number_t *posptr;
    realloc_t *r, *rend;

    d = context;

    /*many sanity checks */

    checkcontext(context);
    assert(d->xpos != NULL);	/*avarything correctly allocated? */
    assert(d->ypos != NULL);
    assert(d->switch_function != NULL);
    assert(d->vbuff != NULL);
    assert(d->back != NULL);
    assert(d->width > 0 && d->width < 65000);
    assert(d->height > 0 && d->height < 65000);
    assert(d->scanline >= d->width);
    assert(d->reallocx != NULL);
    assert(d->reallocy != NULL);
    /*options in allowed range? */
    assert(d->currentformula->magic == FORMULAMAGIC);
    assert(d->maxiter >= 0);
    assert(d->coloringmode >= 0);
    assert(d->incoloringmode >= 0);
    assert(d->coloringmode < OUTCOLORING);
    assert(d->incoloringmode < INCOLORING);
    assert(d->mandelbrot == 0 || d->mandelbrot == 1);
    assert(d->fullscreen == 0 || d->fullscreen == 1);
    assert(d->plane >= 0 && d->plane < PLANES);
    assert(d->num_colors > 0 && d->num_colors < 256);
    assert(d->s.mc >= d->s.nc);
    assert(d->s.mi >= d->s.ni);
    /*now we can safely start work... */


    d->interrupt = 0;
    if (d->dirty)
	d->dirty--;
    combine_methods();
    d->switch_function();
    d->max = 0;
    d->pass = 0;
    d->pos = 0;
    tmp = d->vbuff;
    d->vbuff = d->back;
    d->back = tmp;
    MAXITER = d->maxiter;
    coloringmode = d->coloringmode;
    incoloringmode = d->incoloringmode;
#ifdef SCROLLING
    counter++;
#endif

    mkrealloc_table(d->xpos, d->reallocx, d->width, d->s.nc, d->s.mc, cursymetry.xsym);
    callwait();
    d->pass++;
    mkrealloc_table(d->ypos, d->reallocy, d->height, d->s.ni, d->s.mi, cursymetry.ysym);
    callwait();

    for (r = d->reallocx, posptr = d->xpos, rend = d->reallocx + d->width;
	 r < rend; r++, posptr++)
	*posptr = r->possition;
    for (r = d->reallocy, posptr = d->ypos, rend = d->reallocy + d->height;
	 r < rend; r++, posptr++)
	*posptr = r->possition;
    callwait();
    d->pass++;
    moveoldpoints();
    callwait();
    d->pass++;
    calculatenew();
    if (d->interrupt) {
	init_tables(d);
	return;
    }
    callwait();
    d->pos = 0;
    d->max = 0;
    d->pass++;
    dosymetry();
    callwait();
}


static void recalculate_view(zoom_context * c)
{
    double xs = c->s.mc - c->s.nc, ys = (c->s.mi - c->s.ni) / c->height * c->width,
     xc = (c->s.mc + c->s.nc) / 2, yc = (c->s.mi + c->s.ni) / 2, size;
    checkcontext(c);
    assert(c->s.mc >= c->s.nc);
    assert(c->s.mi >= c->s.ni);

    if (xs > ys)
	size = xs;
    else
	size = ys;
    c->s.nc = xc - size / 2;
    c->s.mc = xc + size / 2;
    c->s.ni = yc - size / 2 * c->height / c->width;
    c->s.mi = yc + size / 2 * c->height / c->width;
}
static void recalculate_view1(zoom_context * c)
{
    double xs = c->s.mc - c->s.nc, ys = (c->s.mi - c->s.ni) * 1024 / 768,
     xc = (c->s.mc + c->s.nc) / 2, yc = (c->s.mi + c->s.ni) / 2, size;
    checkcontext(c);
    assert(c->s.mc >= c->s.nc);
    assert(c->s.mi >= c->s.ni);

    if (xs > ys)
	size = xs;
    else
	size = ys;
    c->s.nc = xc - size / 2;
    c->s.mc = xc + size / 2;
    c->s.ni = yc - size / 2 * 768 / 1024;
    c->s.mi = yc + size / 2 * 768 / 1024;
}

void init_tables(zoom_context * c)
{
    int i;
    checkcontext(c);

    c->dirty = 2;
    for (i = 0; i < c->width + 1; i++)
	c->xpos[i] = INT_MAX;
    for (i = 0; i < c->height + 1; i++)
	c->ypos[i] = INT_MAX;
}

void set_formula(zoom_context * c, CONST int num)
{
    checkcontext(c);
    assert(num < nformulas);
    assert(num >= 0);
    c->currentformula = formulas + num;
    c->mandelbrot = c->currentformula->mandelbrot;
    c->pre = c->currentformula->pre;
    c->pim = c->currentformula->pim;
    c->s.ni = c->currentformula->v.ni;
    c->s.nc = c->currentformula->v.nc;
    c->s.mi = c->currentformula->v.mi;
    c->s.mc = c->currentformula->v.mc;
    init_tables(c);
}

static int alloc_tables(zoom_context * c)
{
    checkcontext(c);
    c->xpos = (number_t *) malloc((c->width + 5) * sizeof(*c->xpos));
    if (c->xpos == NULL)
	return 0;
    c->ypos = (number_t *) malloc((c->height + 5) * sizeof(*c->ypos));
    if (c->ypos == NULL) {
	free(c->xpos);
	return 0;
    }
    c->reallocx = (realloc_t *) malloc(sizeof(realloc_t) * (c->width + 1));
    if (c->reallocx == NULL) {
	free(c->xpos);
	free(c->ypos);
	return 0;
    }
    c->reallocy = (realloc_t *) malloc(sizeof(realloc_t) * (c->height + 1));
    if (c->reallocy == NULL) {
	free(c->xpos);
	free(c->ypos);
	free(c->reallocx);
	return 0;
    }
    return 1;
}

static int free_tables(zoom_context * c)
{
    checkcontext(c);
    free((void *) c->xpos);
    free((void *) c->ypos);
    free((void *) c->reallocx);
    free((void *) c->reallocy);
}

int resize_to(zoom_context * c, CONST int width, CONST int height, CONST int scanline, char *vbuff, char *back)
{
    checkcontext(c);
    assert(width > 0 && width < 65000);
    assert(height > 0 && height < 65000);
    assert(scanline >= width);
    assert(vbuff != NULL);
    assert(back != NULL);
    /*try cause segfaults on unallocated memory */
    memset(vbuff, 0, scanline * height);
    memset(back, 0, scanline * height);
    free_tables(c);
    c->scanline = scanline;
    c->width = width;
    c->height = height;
    c->vbuff = vbuff;
    c->back = back;
    if (!alloc_tables(c))
	return 0;
    init_tables(c);
    return 1;
}

void set_view(zoom_context * c, CONST vinfo * s)
{
    checkcontext(c);
    c->s = *s;
    if (c->fullscreen)
	recalculate_view1(c);
    else
	recalculate_view(c);
}

void free_context(zoom_context * c)
{
    checkcontext(c);
    free_tables(c);
    free((void *) c);
}

zoom_context *
 make_context(CONST int width, CONST int height, CONST int scanline, CONST int formula, CONST int full, void (*switchptr) (), void (*waitfunc) (), char *vbuf, char *bckup)
{
    zoom_context *new;
    assert(width > 0 && width < 65000);
    assert(height > 0 && height < 65000);
    assert(scanline >= width);
    /*try cause segfaults on unallocated memory */
    memset(vbuf, 0, scanline * height);
    memset(bckup, 0, scanline * height);
    assert(vbuf != NULL);
    assert(bckup != NULL);
    assert(formula >= 0 && formula < nformulas);
    assert(switchptr != NULL);

    new = (zoom_context *) calloc(sizeof(zoom_context), 1);
    if (new == NULL)
	return 0;
    new->magic = CONTEXTMAGIC;
    new->fullscreen = full;
    new->scanline = scanline;
    new->width = width;
    new->maxiter = DEFAULT_MAX_ITER;
    new->coloringmode = 0;
    new->height = height;
    new->vbuff = vbuf;
    new->back = bckup;
    new->switch_function = switchptr;
    new->wait_function = waitfunc;
    if (!alloc_tables(new))
	return NULL;
    set_formula(new, formula);
    return (new);
}
