
/* 
 *     XaoS, a fast portable realtime fractal zoomer 
 *                  Copyright (C) 1996 by
 *
 *      Jan Hubicka          (hubicka@paru.cas.cz)
 *      Thomas Marsh         (tmarsh@austin.ibm.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef _plan9_
#include <u.h>
#include <libc.h>
#include <stdio.h>
#else

#ifndef _MAC
#include "aconfig.h"
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif
#include <math.h>
#include <string.h>
#endif
#include "config.h"
#include "complex.h"
#include "formulas.h"
#include "zoom.h"

int coloringmode;
int incoloringmode;

char *incolorname[] =
{
    "maxiter",
    "zmag",
};

char *outcolorname[] =
{
    "iter",
    "iter+real",
    "iter+imag",
    "iter+real/imag",
    "iter+real+imag+real/imag",
    "binary decomposition",
    "biomorphs",
    "biomorphs decompisition"
};


#define OUTPUT() if(iter>=MAXITER)\
		return(incoloringmode?incolor_output(zre,zim,iter):iter); else \
		return(!coloringmode?iter:color_output(zre,zim,iter))

int INLINE
 color_output(number_t zre, number_t zim, int iter)
{
    switch (coloringmode) {
    case 1:			/* real */
	return (iter + zre);
	break;
    case 2:			/* imag */
	return (iter + zim);
	break;
    case 3:			/* real / imag */
	if (myfabs(zim) < 0.00001)
	    return 0;
	return (iter + zre / zim);
	break;
    case 4:			/* all of the above */
	if (myfabs(zim) < 0.00001)
	    return 0;
	return (iter + zre + zim + zre / zim);
	break;
    case 5:
	if (zim < 0)
	    return (iter);
	else
	    return (MAXITER - iter);
	break;
    case 6:
	if (myabs(zim) < 2.0 || myabs(zre) < 2.0)
	    return (MAXITER - iter);
	else
	    return (iter);
	break;
    default:
	break;
    }
    return iter;
}

int INLINE
 incolor_output(number_t zre, number_t zim, int iter)
{
    switch (incoloringmode) {
    case 1:			/* zmag */
	return ((zre * zre + zim * zim) * (number_t) (MAXITER >> 1) + 1);
	break;
    default:
	break;
    }
    return iter;
}


#ifdef __UNDEFINED__
/*#if defined(__i386__) && defined(__GNUC__) && 0

   static int mand_calc(number_t volatile cre, number_t volatile cim, number_t volatile pre, number_t volatile pim) REGISTERS(3)
   {
   unsigned int volatile iter = MAXITER ;
   volatile number_t zre, zim,limit=4,limit1=2;
   number_t r,s;
   zre = cre;
   zim = cim;
   if(!incoloringmode&&zre==cre&&zim==cim) {
   r=cre*cre+cim*cim;
   s=sqrt(r-cre/2 + 1.0/16.0);
   }
   if(!incoloringmode&&zre==cre&&zim==cim&&((cre+1)*(cre+1)+cim*cim<1.0/16.0||((16*r*s)<(5*s-4*cre+1)))) iter=0; else 
   {

   __asm__ __volatile__ ("
   movl %9,%%ecx
   fxch %3              # uloz zi*zi misto zr a nacti zr
   .align 8
   iterace:                     # v ST je ted zr

   fst  %3              #uloz do zr
   fmul %4,%%st # *zi
   fadd %%st,%%st               # *2
   fadd %6,%%st # +ci
   # ted mam v ST zr*zi*2 +ci
   fxch %4              # nacti zi, v ST(2) je ted zr*zi*2+ci -zustane zde
   fmul %%st,%%st               # zi*zi
   fxch %3              # uloz zi*zi misto zr a nacti zr
   fmul %%st(0),%%st    # zr*zr
   fadd %3,%%st # +zi*zi
   fcom %7              # porovnej s hranici
   fnstsww      %%ax
   testb        $3,%%ah
   jz   dal             # omezeni hranici je prekroceno
   fsub %3,%%st # -zi*zi
   fsub %3,%%st # -zi*zi
   fadd %5,%%st # +cr
   loop iterace # dokud je pocet iter. kroku>0 opakuj
   .align 8
   dal:
   fsub %3,%%st # -zi*zi
   fsub %3,%%st # -zi*zi
   fadd %5,%%st # +cr
   fstl         %1
   fxch %4
   fstl %2
   " : 
   "=cx" (iter),
   "=g" (zre),
   "=g" (zim)
   :"f" ((double)zre),
   "f" ((double)zim),
   "f" ((double)pre),
   "f" ((double)pim), 
   "f" ((double)limit),
   "f" ((double)limit1),
   "g" ((unsigned int)MAXITER)
   :
   "ax","memory"
   );
   }
   iter = MAXITER - iter;
   OUTPUT();
   } */
#else
static int mand_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim) REGISTERS(2);
static int mand_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0;
    register unsigned int iter = MAXITER /*& (~(int) 3) */ ;
    register number_t zre, zim;
    register number_t r, s;
    zre = cre;
    zim = cim;
    if (!incoloringmode && pre == cre && pim == cim) {
	r = cre * cre + cim * cim;
	s = sqrt(r - cre / 2 + 1.0 / 16.0);
    }
    if (!incoloringmode && pre == cre && pim == cim && ((cre + 1) * (cre + 1) + cim * cim < 1.0 / 16.0 || ((16 * r * s) < (5 * s - 4 * cre + 1))))
	iter = 0;
    else {
	while ((iter) && (rp + ip < 4)) {
	    ip = (zim * zim);
	    zim = (zim * zre) * 2 + pim;
	    rp = (zre * zre);
	    zre = rp - ip + pre;
	    iter--;

	}
    }
    iter = MAXITER - iter;
    OUTPUT();
}
#endif

static int mand3_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    rp = zre * zre;
    ip = zim * zim;
    while ((iter) && (rp + ip < 4)) {
	rp = zre * (rp - 3 * ip);
	zim = zim * (3 * zre * zre - ip) + pim;
	zre = rp + pre;
	rp = zre * zre;
	ip = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int mand4_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    rp = zre * zre;
    ip = zim * zim;
    while ((iter) && (rp + ip < 4)) {
	rp = rp * rp - 6 * rp * ip + ip * ip + pre;
	zim = 4 * zre * zre * zre * zim - 4 * zre * ip * zim + pim;
	zre = rp;
	rp = zre * zre;
	ip = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int mand5_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0, t;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    while ((iter) && (rp + ip < 4)) {
	c_pow4(zre, zim, rp, ip);
	c_mul(zre, zim, rp, ip, t, zim);
	zre = t + pre;
	zim += pim;
	rp = zre * zre;
	ip = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int mand6_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0, t;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    while ((iter) && (rp + ip < 4)) {
	c_pow3(zre, zim, rp, ip);
	c_pow3(rp, ip, t, zim);
	zre = t + pre;
	zim += pim;
	rp = zre * zre;
	ip = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int barnsley1_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    do {
	if (zre >= 0) {
	    c_mul(zre - 1, zim, pre, pim, rp, zim);

	} else {
	    c_mul(zre + 1, zim, pre, pim, rp, zim);
	}
	zre = rp;
	iter--;
    }
    while ((iter) && (zre * zre + zim * zim < 4));
    iter = MAXITER - iter;
    OUTPUT();
}

static int newton_calc(register number_t zre, register number_t zim, register number_t pre, register number_t pim)
{
    register number_t a, n, d, sqrr, sqri, zre1, zim1;
    register int iter = MAXITER;

    sqri = zim * zim;
    do {
	zre1 = zre;
	zim1 = zim;
	sqri = zim * zim;
	sqrr = zre * zre;
	n = 3 * ((sqrr + sqri) * (sqrr + sqri));
#ifdef ZEROCHECK
	if (n > 0.00000001) {
#endif
	    zim = 2 * zim / 3 - 2 * zre * zim / n;
	    zre = 2 * zre / 3 + (sqrr - sqri) / n;
#ifdef ZEROCHECK
	}
#endif
	a = (zre - zre1) * (zre - zre1) + (zim - zim1) * (zim - zim1);
	iter--;
    }
    while (iter && a > 1E-6);
    iter = MAXITER - iter;
    OUTPUT();
}


static int phoenix_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register int iter = MAXITER;
    register number_t zre, zim, zpr, zpm, rp = 0.0, ip = 0.0;

    zre = cre;
    zim = cim;
    zpr = zre * zre, zpm = zim * zim;
    while (iter && (zpr + zpm < 4)) {
	zpr = zpr - zpm + pre + pim * rp;
	zpm = 2 * zre * zim + pim * ip;
	rp = zre, ip = zim;
	zre = zpr;
	zim = zpm;
	zpr = zre * zre, zpm = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int octo_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp, ip, tr, ti;
    register int iter = MAXITER;
    register number_t zre = cre, zim = cim, zpr = 0, zpm = 0;

    while (iter && ((zpr * zpr + zpm * zpm) < 4)) {
	rp = zre;
	ip = zim;
	c_pow3(zre, zim, tr, ti);
	c_add(tr, ti, zpr, zpm, zre, zim);
	zpr = rp;
	zpm = ip;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
    return (iter);
}

symetry sym6[] =
{
    {0, 1.73205080758},
    {0, -1.73205080758}
};

symetry sym8[] =
{
    {0, 1},
    {0, -1}
};

symetry sym16[] =
{
    {0, 1},
    {0, -1},
    {0, 0.414214},
    {0, -0.414214},
    {0, 2.414214},
    {0, -2.414214}
};

struct formula formulas[] =
{
    {
	FORMULAMAGIC,
	mand_calc,
	{"Mandelbrot", "Julia"},
	{0.5, -2.0, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, 0, 0, NULL}
	},
	{
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL}
	},
    },
    {
	FORMULAMAGIC,
	mand3_calc,
	{"Mandelbrot^3", "Julia^3"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {0, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {0, 0, 0, NULL}
	},
	{
	    {0, 0, 0, NULL},
	    {0, 0, 0, NULL}
	},
    },
    {
	FORMULAMAGIC,
	mand4_calc,
	{"Mandelbrot^4", "Julia^4"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {INT_MAX, 0, 2, sym6},
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, 0, 0, NULL}
	},
	{
	    {INT_MAX, 0, 2, sym6},
	    {INT_MAX, 0, 2, sym6}
	}
    },
    {
	FORMULAMAGIC,
	mand5_calc,
	{"Mandelbrot^5", "Julia^5"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {0, 0, 2, sym8},
	    {INT_MAX, 0, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {0, 0, 2, sym8},
	},
	{
	    {0, 0, 2, sym8},
	    {0, 0, 2, sym8}
	},
    },
    {
	FORMULAMAGIC,
	mand6_calc,
	{"Mandelbrot^6", "Julia^6"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {0, 0, 6, sym16},
	    {INT_MAX, 0, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {0, 0, 0, NULL},
	},
	{
	    {0, 0, 6, sym16},
	    {0, 0, 6, sym16}
	},
    },
    {
	FORMULAMAGIC,
	octo_calc,
	{"Octal", "Octal"},
	{1.25, -1.25, 1.25, -1.25},
	0, 0.0, 0.0,
	{
	    {0, 0, 6, sym16},
	    {INT_MAX, 0, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {0, 0, 0, NULL},
	},
	{
	    {0, 0, 6, sym16},
	    {0, 0, 6, sym16}
	},
    },
    {
	FORMULAMAGIC,
	newton_calc,
	{"Newton", "Newton"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {INT_MAX, 0, 2, sym6},
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, 0, 2, sym6}
	},
	{
	    {INT_MAX, 0, 2, sym6},
	    {INT_MAX, 0, 2, sym6}
	},
    },
    {
	FORMULAMAGIC,
	barnsley1_calc,
	{"Barnsley1 Mandelbrot", "Barnsley1"},
	{1.25, -1.25, 1.25, -1.25},
	0, -0.6, 1.1,
	{
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL}
	},
	{
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL}
	},
    },
    {
	FORMULAMAGIC,
	phoenix_calc,
	{"MandPhoenix", "Phoenix"},
	{1.25, -1.25, 1.25, -1.25},
	0, 0.56667000000000001, -0.5,
	{
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, 0, 0, NULL},
	},
	{
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL}
	},
    },
};

struct formula *currentformula = formulas;
CONST int nformulas = sizeof(formulas) / sizeof(struct formula);
