{******************************************************************}
{                                                                  }
{                          Toolbar unit                            }
{                  (c) 1991 Applied Visions, Inc.                  }
{                                                                  }
{******************************************************************}

unit ToolBar;

interface

uses WObjects, WinTypes, WinProcs,
     OGL1, OGL2, OGL3;

const
    ToolSize = 11;
type
IDArray = array [0..ToolSize] of integer;
BitmapArray = array [0..ToolSize] of Pchar;

PToolBar = ^TToolBar;
TToolBar = object(TGWindow)
  ToolIDs : IDArray;
  ToolBitmaps : BitmapArray;
  SelectedTool: PGBitmap;
  constructor Init(AParent: PGWindow);
  destructor  Done;                                      virtual;
  procedure setIDs(IDs : IDArray);
  procedure setBitmaps(Bitmaps : BitmapArray);
  procedure SetupWindow;                                 virtual;
  procedure GPaint(APort: PPort; BadRect: PMathRect);    virtual;
  procedure MarkSelection(Selection: PGBitmap; APort: PPort); virtual;
  procedure BeginDrag(APoint: PGPoint; KeyStates: Word); virtual;
  procedure Drag(APoint: PGPoint; KeyStates: Word);      virtual;
  procedure EndDrag(APoint: PGPoint; KeyStates: Word);   virtual;
  function  SelectTool(ATool: PGBitmap): Boolean;        virtual;
  procedure SelectID(ID: Integer);                       virtual;
end;

implementation

{const
  ToolSize = 10;
  IDs : array [0..ToolSize] of Integer = (710, 732, 711, 719, 713,
				    714, 715, 716, 718, 721, 24340);

  Maps: array [0..ToolSize] of PChar   = ('open', 'save', 'find',
				    'note', 'dial', 'help', 'new',
				    'modify', 'delete', 'setup', 'exit');
 }

{------------------------------------------------------------------
 Creates an instance of the ToolBar, binding it to the given Parent
 window.
 ------------------------------------------------------------------
}

constructor TToolBar.Init(AParent: PGWindow);
begin
  TGWindow.Init(AParent, 'ToolBar');
  Attr.Style := Attr.Style or ws_Child or ws_Border;

  SelectedTool := nil;
end;

procedure TToolBar.setIDs(IDs : IDArray);
begin
  ToolIDs := IDs;
end;

procedure TToolBar.setBitmaps(Bitmaps : BitmapArray);
begin
  ToolBitmaps := Bitmaps;
end;


{------------------------------------------------------------------
 Completes the construction of the ToolBar by loading the bitmaps,
 identified in the Maps list, into the Picture instance variable.
 As each bitmap is loaded, it is assigned its Menu Command ID as its
 Graphic Id data field.  This is used when the bitmap is selected,
 to send a command to the parent window.  
 ------------------------------------------------------------------
}
procedure TToolBar.SetupWindow;
var
  ABitmap: PGBitmap;
  I      : Integer;
  APnt   : TGPoint;
  Black  : TColor;
  Gray   : TColor;
  ABrush : TBrush;
begin
  Gray.InitDefault;
  ABrush.InitDefault;
  Gray.SetRGB(192, 192, 192);
  ABrush.SetColor(@Gray);

  SetGCursor(cs_Finger);  { Selects default cursor for the window }
  SetBrush(@ABrush);       { Sets the window's background brush }

  APnt.Init(0,0);
  for I := 0 to ToolSize do
  begin
    New(ABitmap, Init(1,0, 0,0, tc_Tools));
    ABitmap^.LoadBMPResource(ToolBitmaps[I], Space);
    ABitmap^.Id := ToolIDs[I];
    if Picture^.Count > 0 then
    begin
      APnt.Y := Picture^.Last^.Bottom;
      ABitmap^.Offset(@APnt);
    end;
    Picture^.Add(ABitmap);
  end;

  APnt.Done;
  ABrush.Done;
  Gray.Done;
end;

{------------------------------------------------------------------
 Dispose of the ToolBar.  Since all it owns are its bitmaps, the
 disposal of Picture in TGWindow.Done will take care of that.
 NOTE that the SelectedTool IVar only points into Picture, it does
 not represent a separate object, and need not be disposed.
 ------------------------------------------------------------------
}
destructor TToolBar.Done;
begin
  TGWindow.Done;
end;

{------------------------------------------------------------------
 Draw the ToolBar when invalidated.  We override the ancestral
 GPaint for two reasons: we can use FastDraw since we don't care
 about the pen or brush, and we want to invert the selected
 tool to indicate its selection.
 ------------------------------------------------------------------
}
procedure TToolBar.GPaint(APort: PPort; BadRect: PMathRect);
begin
  Picture^.FastDraw(APort);
  if SelectedTool <> nil then
    MarkSelection(SelectedTool, APort);
end;

{------------------------------------------------------------------
 Highlights the given bitmap to mark it as a selection.  This is
 done through a slightly complex double-inversion aimed at inverting
 only the border of the button to make it look pushed-in.
 ------------------------------------------------------------------
}
procedure TToolBar.MarkSelection(Selection: PGBitmap; APort: PPort);
var
  BRect: TRectangle;
begin
  if Selection <> nil then
  begin
    BRect.InitDefault;
    Selection^.GetBoundsRect(BRect);
    BRect.Inflate(-1,-1);
    BRect.Invert(APort);
    BRect.Inflate(-1,-1);
    BRect.Invert(APort);
    BRect.Done;
  end;
end;

{------------------------------------------------------------------
 If a new tool is selected, re-invert the previous selection and
 invert the new one.
 ------------------------------------------------------------------
}
function TToolBar.SelectTool(ATool: PGBitmap): Boolean;
begin
  if ATool <> nil then
    if ATool <> SelectedTool then
    begin
      Port^.Associate(@Self);
      if SelectedTool <> nil then
        MarkSelection(SelectedTool, Port);

      SelectedTool := ATool;
      MarkSelection(SelectedTool, Port);
      Port^.Dissociate;
      SelectTool := True;
    end
    else
      SelectTool := False
  else
    SelectTool := False;
end;

{------------------------------------------------------------------
 Used to synchronize the Palette with selections made through
 the menu.
 ------------------------------------------------------------------
}
procedure TToolBar.SelectID(ID: Integer);
var
  Index: Integer;
  I    : Integer;
begin
  Index := -1;
  for I := 0 to ToolSize do
    if ToolIDs[I] = ID then
      Index := I;
  if Index >= 0 then
    SelectTool(PGBitmap(Picture^.At(Index)));
end;

{------------------------------------------------------------------
 If the mouse has been clicked over a tool's Graphic, select that
 Graphic and highlight it.  The message for that button will only
 be set if it is still selected in EndDrag.
 ------------------------------------------------------------------
}
procedure TToolBar.BeginDrag(APoint: PGPoint; KeyStates: Word);
var
  ATool : PGraphic;
  Index : Integer;
begin
  ATool := Picture^.ThatContains(APoint);
  Index := -1;

  if ATool <> nil then
    if SelectTool(PGBitmap(ATool)) then
      Index := Picture^.IndexOf(SelectedTool);
end;

{------------------------------------------------------------------
 If the mouse has been released over a tool's Graphic, and that
 button is currently selected, then clear the highlight and send
 its associated command to its parent.  Recall that the Command ID
 is stored in the Bitmap object as its Id instance variable.

 Note that, since this toolbar models "momentary" buttons, we clear
 the highlight of the button when it is released.
 ------------------------------------------------------------------
}
procedure TToolBar.EndDrag(APoint: PGPoint; KeyStates: Word);
var
  ATool : PGBitmap;
  Index : Integer;
begin
  ATool := PGBitmap(Picture^.ThatContains(APoint));
  Index := -1;

  if (ATool <> nil) and (ATool = SelectedTool) then
  begin
    Port^.Associate(@Self);
    MarkSelection(SelectedTool, Port);  { "Release" the button }
    Port^.Dissociate;
    SendMessage(Parent^.HWindow, wm_Command, SelectedTool^.Id, 0);
    SelectedTool := nil;
  end;
end;

{------------------------------------------------------------------
 If the mouse is moved outside the current selection, then clear
 the highlight of that selection.
 ------------------------------------------------------------------
}
procedure TToolBar.Drag(APoint: PGPoint; KeyStates: Word);
var
  ATool: PGBitmap;
begin
  ATool := PGBitmap(Picture^.ThatContains(APoint));

  if (SelectedTool <> nil) and (ATool <> SelectedTool) then
  begin
    Port^.Associate(@Self);
    MarkSelection(SelectedTool, Port);
    Port^.Dissociate;
    SelectedTool := nil;
  end;
end;

end.