{**************************************************}
{                     OrgWindow                    }
{                    Written in                    }
{             Turbo Pascal for Windows             }
{                       by                         }
{                  Zack Urlocker                   }
{**************************************************}
Unit OrgWin;

interface

uses OGL1, OGL2, OGL3, RGBDialog,   { ObjectGraphics }
     WObjects, StdDlgs, Strings, WinTypes, WinProcs;

const
 cm_Cut    = 201;      { constants for main window menus} 
 cm_Copy   = 202;
 cm_Paste  = 203;
 cm_Clear  = 204;
 cm_Select = 109; 
 cm_Line   = 110;
 cm_DotLine= 111;
 cm_EmpRect = 112;
 cm_Rect   = 113;
 cm_RndRect= 114;
 cm_Text   = 115;
 cm_Data   = 309;
 FieldLen  =  24;      { Dialog constants }
 id_Name   = 104;
 id_Posn   = 106;
 id_Dept   = 108;

Type

  { Transfer record for Dialog }
  ItemTransferBuffer = record
    NameStr, PosnStr, DeptStr : array[0..FieldLen-1] of char;
  end;

  { Keep an array of all the items, indexed by graphic ID }
  ItemsArray = array[0..50] of ItemTransferBuffer;

  { The dialog is used for input of new employee info }
  PDataDialog = ^TDataDialog;
  TDataDialog = object(TDialog)
    NameEdit, PosnEdit, DeptEdit : PEdit;
    constructor Init(AParent: PWindowsObject; ATitle:PChar);
  end;

  { Custom input field; loses focus when Enter is pressed }
  PRetEdit = ^TRetEdit;
  TRetEdit = object(TEdit)
    procedure wmChar(var Msg:TMessage); virtual wm_Char;
  end;

  { Employee graphic object }
  PEmpRect = ^TEmpRect;
  TEmpRect = object(TRectangle)
    function  Copy: Pointer; virtual;
    procedure drawOn(APort:PPort); virtual;
    constructor Load(var S: TStream);
    procedure Store(var S: TStream); virtual;
  end;

  { Drawing area window }
  POrgWindow = ^TOrgWindow;
  TOrgWindow = object(TGWindow)
    LabelEC: PRetEdit;   { Edit control          }
    Graphic: PGraphic;   { Graphic being drawn   }
    Chosen: PSubPicture; { selected item         }
    Marks: PPolymark;    { to mark selected item }
    HitPt: PGPoint;      { mouse click location  }
    Saved: Boolean;      { has work been saved?  }
    ViewGrid : Boolean;  { is grid view on?      }
    Tool: Integer;       { what tool is selected }
    CurrPen: PPen;       { current tool settings }
    CurrBrush: PBrush;
    CurrTextPen: PTextPen;
    ItemBuffer : ItemTransferBuffer; { for Dialog }
    ItemsCount : Integer;
    constructor Init(AParent: PWindowsObject; ATitle: PChar);
    procedure SetupWindow; virtual;
    destructor Done; virtual;
    procedure GPaint(APort: PPort; BadRect: PMathRect);virtual;
    {mouse messages}
    procedure BeginDrag(MousePt: PGPoint; KeyStates: Word); virtual;
    procedure Drag(MousePt: PGPoint; KeyStates: Word); virtual;
    procedure EndDrag(MousePt: PGPoint; KeyStates: Word); virtual;
    procedure WMLButtonDblClick(var Msg: TMessage); virtual WM_LBUTTONDBLCLK;
    procedure WMSetFocus(var Msg: TMessage); virtual WM_SETFOCUS;
    {Methods to interface with Main Window}
    procedure ClearAll;
    procedure SetTool(ToolID: Integer);virtual;
    procedure SetColor(Color:PLogColor);virtual;
    procedure SetWidth(Width:Integer);
    procedure SetCurrTextPen(ATPen: PTextPen);virtual;
    procedure Print(Printer: PGPrinter);virtual;
    procedure CutChoice; virtual;
    procedure CopyChoice; virtual;
    procedure PasteChoice; virtual;
    procedure ClearChoice; virtual;
    procedure SetScrollState(ScrollState: Boolean);virtual;
    procedure SetSpace(NewSpace: PGraphSpace);virtual;
  end;

var
    Items : ItemsArray;  { Global }

implementation

{--------------------------------------------------}
{ TRetEdit method implementations:                 }
{--------------------------------------------------}

{ if Enter is pressed send focus back; otherwise process input }
procedure TRetEdit.wmChar(var Msg:TMessage);
var key : word;
begin
  key := Msg.wParam;
  if (key = word(#13)) or (key = word(#27))
  then
    { exit }
    setFocus(Application^.mainWindow^.HWindow)
  else
    defWndProc(Msg);
end;

{ The edit controls will contain the transfer data. }
constructor TDataDialog.Init(AParent: PWindowsObject; ATitle:PChar);
begin
  TDialog.Init(AParent, ATitle);
  new(NameEdit, initResource(@Self, id_Name, fieldLen));
  new(PosnEdit, initResource(@Self, id_Posn, fieldLen));
  new(DeptEdit, initResource(@Self, id_Dept, fieldLen));
end;

{--------------------------------------------------}
{ TEmpRect method implementations:                 }
{--------------------------------------------------}

{ Create a copy of the new EmpRect }
function TEmpRect.Copy:Pointer;
var
  NewRect: PEmpRect;
begin
  New(NewRect, InitDefault);
  NewRect^.CopyFrom(@Self);
  Copy := NewRect;
end;

{ Draw 2 rectangles for 3D effect }
procedure TEmpRect.drawOn(APort:PPort);
var TEmpRect : PRectangle;
    OffPt : TGPoint;
    col: TColor;
begin
  { draw the first one with a brush }  
  New(TEmpRect, initDefault);
  TEmpRect^.CopyFrom(@Self);
  TEmpRect^.inflate(-4, -4);
  TEmpRect^.brush^.Pattern := bp_thirty;
  TEmpRect^.brush^.setColor(TEmpRect^.Pen^.color{@col});
  TEmpRect^.Draw(APort);
  { draw the second one moved over, white }
  OffPt.init(-4,-4);
  TEmpRect^.offset(@OffPt);
  TEmpRect^.brush^.Pattern := bp_solid;
  col.init(ps_white); 
  TEmpRect^.brush^.setColor(@col);
  TEmpRect^.Draw(APort);
  OffPt.Done;
end;

{ Load additional information for employee }
constructor TEmpRect.Load(var S: TStream);
var S1 : PChar;
begin
  TRectangle.Load(S);
  S1 := S.StrRead;
  StrCopy(Items[ID].NameStr, S1);
  S1 := S.StrRead;
  StrCopy(Items[ID].PosnStr, S1);
  S1 := S.StrRead;
  StrCopy(Items[ID].DeptStr, S1);
end;

{ Store additional information for employee}
procedure TEmpRect.Store(var S: TStream);
begin
  TRectangle.Store(S);
  S.StrWrite(Items[ID].NameStr);
  S.StrWrite(Items[ID].PosnStr);
  S.StrWrite(Items[ID].DeptStr);
end;

{--------------------------------------------------}
{ TOrgWindow method implementations:               }
{--------------------------------------------------}

{ Initialize fields, setup style, scrollers etc  }
constructor TOrgWindow.Init(AParent: PWindowsObject;ATitle: PChar);
var GridSize:TGPoint;
begin
  TGWindow.Init(AParent, ATitle);
  Attr.Style := Attr.Style or ws_Child   or ws_Border 
                           or ws_VScroll or ws_HScroll; 

  { Create scroller object specifiying size, range }                            
  Scroller := New(PGScroller, Init(@Self, 20, 20,
                  Space^.WorldRect^.Width,
                  Space^.WorldRect^.Height));
  Scroller^.XLine   := Space^.Granularity;
  Scroller^.YLine   := Space^.Granularity;
  Scroller^.AutoMode:= False;

  Marks := New(PPolymark, InitDefault);
  New(Chosen, InitDefault);

  { Always start with a grid }
  GridSize.init(20,20);
  Space^.setGrid(@GridSize);
  GridSize.done;
  ViewGrid := True;
  Saved := True;
  Tool := cm_EmpRect;
  Graphic := New(PEmpRect, init(0,0,0,0, tc_Tools));
  ItemsCount := 0;
  LabelEC := nil;
  New(CurrPen,     InitDefault);
  New(CurrBrush,   InitDefault);
  New(CurrTextPen, InitDefault);
end;

{ Clean up all objects }
destructor TOrgWindow.Done;
begin
  Dispose(Graphic, Done);
  Dispose(CurrPen, Done);
  Dispose(CurrBrush, Done);
  Dispose(CurrTextPen, Done);
  TGWindow.Done;
end;


{ This is the central method for handling mouse clicks and dragging }
procedure TOrgWindow.BeginDrag(MousePt: PGPoint; KeyStates: Word);
var  APt    : TGPoint;
     AString: array [0..50] of Char;

{ Respond to mouse click when Text tool is selected.
  Note that you execute this code once when selecting text
  and a second time when selecting elsewhere to finish }
  procedure BeginTextDrag;
  begin
    if LabelEC = nil then
    { Initial mouseclick to start text entry }
    begin
      { Create and display edit field for input }
      Space^.WorldToDisplay(MousePt, APt);
      LabelEC:= New(PRetEdit, Init(@Self, 555, '',
   		    APt.X, APt.Y, 120, 24, 50, False));
      Graphic^.SetOrigin(MousePt);
      Graphic^.SetTextPen(CurrTextPen);       
      Application^.MakeWindow(LabelEC);
      SetFocus(LabelEC^.HWindow);
    end
    else
    { Second mouseclick elsewhere to end text entry }
    begin
      { find out what text was entered, show it }
      LabelEC^.GetText(@AString, 50);
      Dispose(LabelEC, Done);
      LabelEC := nil;
      if StrLen(AString) > 0 then
      begin
        PLabel(Graphic)^.SetText(AString);
        Graphic^.SetTextPen(CurrTextPen); 
        Graphic^.Draw(Port);
        Picture^.Add(Graphic^.Copy);
      end;
    end;
  end;

{ Respond to mouse click when select tool is selected. }
  procedure BeginSelectDrag;
  var
    BRect: TMathRect;
  begin
    BRect.InitDefault;
    Chosen^.GetBoundsRect(BRect);

    HitPt := MousePt^.Copy;  { Used for dragging selections }

{ Non-Zero count means we've got something selected that
  we will move; otherwise we're making a new selection }
    if (Chosen^.Count > 0) and
       (BRect.Contains(MousePt)) then
    begin
      { start moving the current selected item }
      Dispose(Graphic, Done);        { Get rid of the Chooser }
      Graphic := New(PRegion, Init(tc_Tools));
      Chosen^.GetFastRegion(Graphic^);
    end
    else
    begin
      { select a new item }
      if Marks^.Count > 0 then
        Marks^.Draw(Port);   { Erase the old marks, if any...      }
      Marks^.FreeAll;        { and clear so GPaint won't draw them }
      Chosen^.FreeAll;       { Clear any existing selection        }
      Dispose(Graphic, Done);
      Graphic := New(PChooser, Init(MousePt^.X, MousePt^.Y));
    end;
    BRect.Done;
  end;


{ BeginDrag }
begin
 { Always snap to grid unless selecting }
  if (Tool <> cm_Select) or (Chosen^.Count > 0) then
    Space^.snapToGrid(MousePt^);

  if Tool = cm_Text then
    beginTextDrag
  else
  if Tool = cm_Select then 
    beginSelectDrag
  else
    if Tool = cm_EmpRect then  { give unique ID for storing data }
    begin
       inc(ItemsCount);
       Graphic^.ID := ItemsCount;
    end;

  Graphic^.Pen^.Combo := cb_Nxor;
  Graphic^.SetOrigin(MousePt);
  Graphic^.SetCorner(MousePt);
  Port^.Associate(@Self);
  Graphic^.Draw(Port);
end;

{ When dragging something redraw it }
procedure TOrgWindow.Drag(MousePt: PGPoint; KeyStates: Word);
var
  OffsetPt: TGPoint;
begin
 { Always snap to grid unless selecting }
  if (Tool <> cm_Select) or (Chosen^.Count > 0) then
    Space^.snapToGrid(MousePt^);

  if (Tool = cm_Select) and (Chosen^.Count > 0) then
  begin
  { start moving the selected item }
    begin
      HitPt^.OffsetFrom(MousePt, OffsetPt);
      Graphic^.Offset(@OffsetPt);
      HitPt^.CopyFrom(MousePt);
    end;
  end;

  { Redraw for dragging effect }
  if Tool <> cm_Text then
  begin
    Graphic^.FastDraw(Port);
    Graphic^.SetCorner(MousePt);
    Graphic^.FastDraw(Port);
  end;
end;

{ This is the main routine for whenever the user release the mouse. }  
procedure TOrgWindow.EndDrag(MousePt: PGPoint; KeyStates: Word);
var
  APolyshape: PPolyshape;
  BRect     : TRectangle;
  OrigPt, APt : TGPoint;
  AString: array [0..50] of Char;

  { End a drag action with the select tool }
  procedure EndSelectDrag;
  var
    TempChosen    : PSubPicture;
    XScale, YScale: Integer;
    OffsetPt: TGPoint;
  begin
{ Non-zero count means we're moving an existing selection;
  otherwise we're making a new selection by putting it in Chosen }

    if Chosen^.Count > 0 then
   { Relocate a moved selection.  First, clear away the marks,
     then invalidate the Chosen in its old position, move it,
     invalidate in its new position, and then re-mark it. }
    begin
      Marks^.Draw(Port);
      Chosen^.Invalidate(Port);
      Chosen^.PositionAt(Graphic^.Origin);
      Picture^.FindBounds;     { In case Chosen moved outside them }
      Marks^.FreeAll;
      Chosen^.Invalidate(Port);
      Chosen^.MarkCorners(Marks);
      Marks^.Draw(Port);
    end
    else
    { See what was selected.  Tell the Chooser (current Graphic) to
      collect the chosen objects into Chosen, then mark the selection
      and draw the marks. } 
    begin
      PChooser(Graphic)^.Choose(Picture, Chosen^);
      Chosen^.MarkCorners(Marks);
      Marks^.Draw(Port);
   end;
   { Chooser is still visible, so erase it when we're done}
    Graphic^.Pen^.Combo := cb_NXOr;
    Graphic^.Draw(Port);
  end;

begin { EndDrag }
  BRect.InitDefault;

  if Tool = cm_Select then
    EndSelectDrag

  else 
  if Tool <> cm_Text then
  begin
    Graphic^.Pen^.Combo := cb_Copy;
    { Dotted lines are not affected by width }
    if Tool = cm_DotLine then
      Graphic^.Pen^.setColor(CurrPen^.Color)
    else
      Graphic^.SetPen(CurrPen);
    Graphic^.Draw(Port);
    Port^.Dissociate;
    Picture^.Add(Graphic^.Copy);
    BRect.Done;
  end;
end;

{ Respond to Right click action by displaying dialog box}
procedure TOrgWindow.WMLButtonDblClick(var Msg: TMessage);
var
  TempGraphic : PGraphic;
  CurItem : Word;
  EmpRect : PEmpRect;
  Dlg: TDataDialog;
  Reply : Word;
begin
   TempGraphic := Chosen^.first;
   if TempGraphic <> nil then
   if (typeOf(TempGraphic^) = typeof(TEmpRect)) then
   begin
     { Get information into buffer }
     Dlg.Init(@Self, 'DataDlg');
     CurItem := TempGraphic^.ID;
     strCopy(ItemBuffer.NameStr, Items[CurItem].NameStr);
     strCopy(ItemBuffer.PosnStr, Items[CurItem].PosnStr);
     strCopy(ItemBuffer.DeptStr, Items[CurItem].DeptStr);
     Dlg.TransferBuffer := @ItemBuffer;
     Reply := Dlg.Execute;
     Dlg.Done;
     if Reply = id_Ok then
     begin
     { Get information out of buffer }
      strCopy(Items[CurItem].NameStr, ItemBuffer.NameStr);
      strCopy(Items[CurItem].PosnStr, ItemBuffer.PosnStr);
      strCopy(Items[CurItem].DeptStr, ItemBuffer.DeptStr);
     end;
   end
   else
    MessageBeep(0);
end;


{ Trap the WMSetFocus message so that when the user
  hits return in the edit field we can get the input }
procedure TOrgWindow.WMSetFocus(var Msg: TMessage);
var AString:array[0..50] of char;
 TempGraphic : PGraphic;
 TempPt : TGPoint;
begin
if LabelEC <> nil then
 begin
{ Restart here when you get the focus!}
  LabelEC^.GetText(@AString, 50);
  Dispose(LabelEC, Done);
  LabelEC := nil;
  if StrLen(AString) > 0 then
  begin
    TempGraphic := New(PLabel, Init(0,0, '', tc_Tools));
    { Actually, should center it here }
    TempGraphic^.SetOrigin(Graphic^.Origin);
    TempGraphic^.SetTextPen(CurrTextPen);
    PLabel(TempGraphic)^.SetText(AString);
    Picture^.Add(TempGraphic^.Copy);
    TempGraphic^.Draw(Port);
    TempGraphic^.done;
  end;
 end;
end;

{ Take care of last minute initialization after window is created }
procedure TOrgWindow.SetupWindow;
begin
  TGWindow.SetupWindow;
  SetGCursor(cs_Graphic);
end;

{ Override default painting to show grid, marks, page border }
procedure TOrgWindow.GPaint(APort: PPort; BadRect: PMathRect);
var
  PageBorder: TRectangle;
begin
 { Show the grid if view is turned on }
  if ViewGrid then
    APort^.DrawGrid;

  TGWindow.GPaint(APort, BadRect);
  if Marks^.Count > 0 then
    Marks^.Draw(APort);

  PageBorder.Init(Space^.WorldRect^.Origin^.X, Space^.WorldRect^.Origin^.Y,
                  Space^.WorldRect^.Corner^.X, Space^.WorldRect^.Corner^.Y,
                  tc_Tools);
  PageBorder.Brush^.Pattern := bp_Invisible;
  PageBorder.Draw(APort);
  PageBorder.Done;
end;

{ Start a new drawing }
procedure TOrgWindow.ClearAll;
var i : integer;
begin
    Picture^.FreeAll;
    Marks^.FreeAll;
    Chosen^.FreeAll;
    for i := 1 to 50 do
    begin
       StrPCopy(Items[i].NameStr, '');
       StrPCopy(Items[i].PosnStr, '');
       StrPCopy(Items[i].DeptStr, '');
    end;
    invalidate;
end;

{ Set the tool and create the appropriate graphic }
procedure TOrgWindow.SetTool(ToolID: Integer);
begin
  Tool := ToolID; 
{ Check to see if the Text Edit control is active.  If it is, destroy
  it whenever a tool is selected (even Text; we will just recreate it).}
  if LabelEC <> nil then
  begin
    Dispose(LabelEC, Done);
    LabelEC := nil;
  end;
{ Clean up selection and dragging.  Whatever the current drawing object is,
  draw it to clear it away, then delete it and reconstruct it based on what
  was selected. }
  Port^.Associate(@Self); 
  Marks^.Draw(Port);      
  Marks^.FreeAll;         
  Chosen^.FreeAll;        
  if Graphic <> nil then
  begin
    Dispose(Graphic, Done);
    Graphic := nil;
  end;
  Port^.Dissociate;

  { Create the appropriate tool and set the cursor } 
  case Tool of
    cm_Select   : begin
		    Graphic := New(PChooser, InitDefault);
                    Marks^.SetMark(nil);
		    SetGCursor(cs_Pointer);
		  end;
    cm_Line     : begin
		    Graphic := New(PLine, Init(0,0, 0,0, tc_Tools));
                    Graphic^.pen^.style := ls_Solid;
		    SetGCursor(cs_ThinCross);
		  end;
    cm_DotLine  : begin
		    Graphic := New(PLine, Init(0,0, 0,0, tc_Tools));
                    Graphic^.pen^.style := ls_Dot;
                    Graphic^.pen^.width := 0; 
   		    SetGCursor(cs_ThinCross);
		  end;
    cm_EmpRect:    begin
		    Graphic := New(PEmpRect, Init(0,0, 0,0, tc_Tools));
		    SetGCursor(cs_ThinCross);
		  end;
    cm_Rect:      begin
		    Graphic := New(PRectangle, Init(0,0, 0,0, tc_Tools));
		    SetGCursor(cs_ThinCross);
		  end;
    cm_RndRect  : begin
		    Graphic := New(PRoundRect, Init(0,0, 0,0, tc_Tools));
		    SetGCursor(cs_ThinCross);
		  end;
    cm_Text     : begin
		    Graphic := New(PLabel, Init(0,0, '', tc_Tools));
		    SetGCursor(cs_IBeam);
		  end;
  end;
end;

{ Set the color of all tools }
procedure TOrgWindow.SetColor(Color:PLogColor);
begin
  CurrPen^.setColor(Color);
  Graphic^.Pen^.setColor(Color);
  CurrTextPen^.setColor(Color);
end;

{ Set the width of Pen }
procedure TOrgWindow.SetWidth(Width:Integer);
begin
  CurrPen^.Width := Width;
  if tool <> cm_DotLine then
    Graphic^.Pen^.Width := width;
end;

procedure TOrgWindow.SetCurrTextPen(ATPen: PTextPen);
begin
end;


{ Update scrollers }
procedure TOrgWindow.SetScrollState(ScrollState: Boolean);
begin
  if ScrollState then
  begin
    ShowScrollBar(HWindow, sb_Both, True);
    PGScroller(Scroller)^.XPos := Space^.MappingRect^.Left;
    PGScroller(Scroller)^.YPos := Space^.MappingRect^.Top;
  end
  else
    ShowScrollBar(HWindow, sb_Both, False);
end;


{------------------------------------------------------------------
 Redefine the ancestral SetSpace method, so that we may plug any
 new GraphSpace units into our scrollers.  Calls on the ancestral
 method to actually set the space.
 ------------------------------------------------------------------
}
{ Is this really necessary? }
procedure TOrgWindow.SetSpace(NewSpace: PGraphSpace);
begin
  TGWindow.SetSpace(NewSpace);
  Scroller^.XLine := Space^.Granularity;
  Scroller^.YLine := Space^.Granularity;
  Scroller^.SetRange(Space^.WorldRect^.Width, Space^.WorldRect^.Height);
end;

{ Print the picture }
procedure TOrgWindow.Print(Printer: PGPrinter);
var
  PrintingOff: TGPoint;
  PrintSpace : PGraphSpace;
begin
{ Un-do any scrolling or zooming present in the window's
  Space by resetting the MappingRect to be the world.
}
  PrintSpace := Space^.Copy;
  PrintSpace^.SetMappingRect(PrintSpace^.WorldRect);
  PrintSpace^.SetZoomLevel(100, 100);
  Printer^.SetSpace(PrintSpace);
  Dispose(PrintSpace, Done);

  Port^.AssociatePrinter(Printer);
  Port^.SetSpace(Printer^.Space);
{
 Reposition the Mapping Rectangle of the Port to account for the
 physical offset of the printer.  NOTE that these conversions are
 not done until after association because they deal with Display
 coordinates, and these do not accurately reflect the resolution
 of the Printer until it is associated.
}
  Printer^.GetPhysicalOffset(PrintingOff);
  Port^.Space^.DisplayToWorld(@PrintingOff, PrintingOff);
  Port^.Space^.MoveMappingRect(@PrintingOff);

  Picture^.Draw(Port);
  Port^.DissociatePrinter;
end;

{ Cut an item }
procedure TOrgWindow.CutChoice;
begin
  CopyChoice;
  ClearChoice;
end;

{------------------------------------------------------------------
 Copy TheChosen out of the Picture and post it to the Clipboard.
 ------------------------------------------------------------------}
procedure TOrgWindow.CopyChoice;
begin
  if Chosen^.Count > 0 then
  begin
    Port^.Associate(@Self);
    Port^.ToClipboard(Chosen);
    Port^.Dissociate;
  end;
end;

{------------------------------------------------------------------
 Paste the contents of the Clipboard into the Picture.  Put the
 new objects in the center of the Window, and automatically marks
 them for subsequent positioning.
 ------------------------------------------------------------------}
procedure TOrgWindow.PasteChoice;
var
  CenterPt: TGPoint;
  DRect   : TMathRect;
  ASpace  : TGraphSpace;

  procedure AddIt(AGraphic: PGraphic); far;
  begin
    Picture^.Add(AGraphic);
  end;
begin
  Port^.Associate(@Self);
  Marks^.Draw(Port);      { Erase current Marks if any,  }
  Marks^.FreeAll;         { then clear for later re-mark.}
  Chosen^.FreeAll;
{
 By reading the contents of the Clipboard into a SubPicture
 (TheChosen), we can then transfer the objects into the Picture
 without copying them, since TheChosen does not assume ownership.
 Also, note that we ignore the Space coming in from the clipboard,
 choosing instead to impose the current space on the objects.  A
 more complete application might use the incoming space to do some
 sort of units conversion on the objects.
}
  ASpace.InitDefault;
  Port^.FromClipboard(ASpace, Chosen^);
  if Chosen^.Count > 0 then
  begin
    DRect.InitDefault;
    GetDisplayRect(DRect);
    CenterPt.Init(DRect.Width div 2, DRect.Height div 2);
    Space^.DisplayToWorld(@CenterPt, CenterPt);
    Chosen^.CenterAt(@CenterPt);
    if Chosen^.Count > 0 then
    begin
      Chosen^.ForEach(@AddIt);
      Chosen^.MarkCorners(Marks);
      Chosen^.Draw(Port);
      Marks^.Draw(Port);
      PGWindow(Parent)^.SetPicture(Chosen);
    end;
    DRect.Done;
  end;
  Port^.Dissociate;
  ASpace.Done;
end;

{------------------------------------------------------------------
 Clear TheChosen from the Picture, and from the screen.  Remove
 TheMarks, since there's nothing left to mark.
 ------------------------------------------------------------------}
procedure TOrgWindow.ClearChoice;
var
  BRect: TRectangle;

  procedure RemoveIt(AGraphic: PGraphic); far;
  begin
    Picture^.Delete(AGraphic);
  end;
begin
  if Chosen^.Count > 0 then
  begin
    BRect.InitDefault;
    Marks^.GetBoundsRect(BRect);
    Marks^.FreeAll;
    Chosen^.ForEach(@RemoveIt);
    Chosen^.FreeAll;
    Port^.Associate(@Self);
    BRect.Invalidate(Port);
    Port^.Dissociate;
    BRect.Done;

    PGWindow(Parent)^.SetPicture(Chosen);
  end;
end;


const
  { Define streaming mechanism for Employee Rect type }
  REmpRect      : TStreamRec = (
                   ObjType: st_FirstOGLType + 99;
                   VmtLink: Ofs(TypeOf(TEmpRect)^);
                   Load   : @TEmpRect.Load;
                   Store  : @TEmpRect.Store
                );

begin
  registerType(REmpRect);
end.